/*---------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_procevmon.c                                               --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Translates text files generated by db2evmon into .del files for   --
-- IMPORT / LOAD into tables in DB2.                                 --
--                                                                   --
-- 060502 - add doExit() for Solaris
---------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <sqlmon.h>

#define FALSE 0
#define TRUE  1

#define STMT_EV_MARKER "Statement Event ..."

/* Marker strings for the different lines that can occur in statement
** event monitor output.
*/

char AGENT_ID[]            = "  Appl Handle: ";
char APPL_ID[]             = "  Appl Id: ";
char SEQUENCE_NO[]         = "  Appl Seq number: ";
char EMPTY[]               = "\n";
char EVMON_FLUSHES[]       = "  Record is the result of a flush: ";
char DASHES[]              = "  -------------------------------------------";
char STMT_TYPE[]           = "  Type     : ";
char STMT_OPERATION[]      = "  Operation: ";
char SECTION_NUMBER[]      = "  Section  : ";
char CREATOR[] 	 	   = "  Creator  : ";
char PACKAGE_NAME[]        = "  Package  : ";
char CONSISTENCY_TOKEN[]   = "  Consistency Token  : ";
char PACKAGE_VERSION_ID[]  = "  Package Version ID  : ";
char CURSOR_NAME[]         = "  Cursor   : ";
char BLOCKING_CURSOR[]     = "  Cursor was blocking: ";
char STMT_TEXT[]           = "  Text     : ";
char START_TIME[]          = "  Start Time: ";
char STOP_TIME[]           = "  Stop Time:  ";
char EXEC_TIME[]           = "  Exec Time:  ";
char AGENTS_TOP[]          = "  Number of Agents created: ";
char USER_CPU_TIME[]       = "  User CPU: ";
char SYSTEM_CPU_TIME[]     = "  System CPU: ";
char FETCH_COUNT[]         = "  Fetch Count: ";
char TOTAL_SORTS[]         = "  Sorts: ";
char TOTAL_SORT_TIME[]     = "  Total sort time: ";
char SORT_OVERFLOWS[]      = "  Sort overflows: ";
char ROWS_READ[]           = "  Rows read: ";
char ROWS_WRITTEN[]        = "  Rows written: ";
char INT_ROWS_DELETED[]    = "  Internal rows deleted: ";
char INT_ROWS_UPDATED[]    = "  Internal rows updated: ";
char INT_ROWS_INSERTED[]   = "  Internal rows inserted: ";
char BP_DATA_LR[] 	   = "  Bufferpool data logical reads: ";
char BP_DATA_PR[] 	   = "  Bufferpool data physical reads: ";
char BP_IDX_LR[]  	   = "  Bufferpool index logical reads: ";
char BP_IDX_PR[]  	   = "  Bufferpool index physical reads: ";
char BP_TEMP_LR[] 	   = "  Bufferpool temporary data logical reads: ";
char BP_TEMP_PR[] 	   = "  Bufferpool temporary data physical reads: ";
char BP_TIDX_LR[] 	   = "  Bufferpool temporary index logical reads: ";
char BP_TIDX_PR[] 	   = "  Bufferpool temporary index physical reads: ";
char SQLCA[]               = "  SQLCA";
char SQLcode[]             = "   sqlcode: ";
char SQLstate[]            = "   sqlstate: ";

/* Structure to contain the data we extract from a statement event.
*/

struct {
	char AGENT_ID[24];
	char AGENTS_TOP[24];
	char APPL_ID[36];
	char BLOCKING_CURSOR[10];
	char CONSISTENCY_TOKEN[20];
	char CREATOR[32];
	char CURSOR_NAME[20];
	char EVMON_FLUSHES[24];
	char FETCH_COUNT[24];
	char INT_ROWS_DELETED[24];
	char INT_ROWS_INSERTED[24];
	char INT_ROWS_UPDATED[24];
	char PACKAGE_NAME[20];
	char PACKAGE_VERSION_ID[70];
	char PARTIAL_RECORD[10];
	char BP_DATA_LR[24];
	char BP_DATA_PR[24];
	char BP_IDX_LR[24];
	char BP_IDX_PR[24];
	char BP_TEMP_LR[24];
	char BP_TEMP_PR[24];
	char BP_TIDX_LR[24];
	char BP_TIDX_PR[24];
	char ROWS_READ[24];
	char ROWS_WRITTEN[24];
	char SECTION_NUMBER[24];
	char SEQUENCE_NO[8];
	char SORT_OVERFLOWS[24];
	char SQL_REQ_ID[24];
	char SQLCABC[12];
	char SQLCAID[12];
	char SQLcode[12];
	char SQLERRD1[12];
	char SQLERRD2[12];
	char SQLERRD3[12];
	char SQLERRD4[12];
	char SQLERRD5[12];
	char SQLERRD6[12];
	char SQLERRM[80];
	char SQLERRP[12];
	char SQLstate[10];
	char SQLWARN[16];
	char START_TIME[32];
	char STMT_OPERATION[24];
	char STMT_TYPE[24];
	char STOP_TIME[32];
	char SYSTEM_CPU_TIME[24];
	char TOTAL_SORT_TIME[24];
	char TOTAL_SORTS[24];
	char USER_CPU_TIME[24];
	char STMT_TEXT[32800];
} row, templateRow = {
	"0", /* char AGENT_ID[24]; */
	"0", /* char AGENTS_TOP[24]; */
	"\"\"", /* char APPL_ID[36]; */
	"0", /* char BLOCKING_CURSOR[10]; */
	"\"\"", /* char CONSISTENCY_TOKEN[10]; */
	"\"\"", /* char CREATOR[32]; */
	"\"\"", /* char CURSOR_NAME[20]; */
	"0", /* char EVMON_FLUSHES[24]; */
	"0", /* char FETCH_COUNT[24]; */
	"0", /* char INT_ROWS_DELETED[24]; */
	"0", /* char INT_ROWS_INSERTED[24]; */
	"0", /* char INT_ROWS_UPDATED[24]; */
	"\"\"", /* char PACKAGE_NAME[10]; */
	"\"\"", /* char PACKAGE_VERSION_ID[70]; */
	"0", /* char PARTIAL_RECORD[10]; */
	"0", /* char BP_DATA_LR[24]; */
	"0", /* char BP_DATA_PR[24]; */
	"0", /* char BP_IDX_LR[24]; */
	"0", /* char BP_IDX_PR[24]; */
	"0", /* char BP_TEMP_LR[24]; */
	"0", /* char BP_TEMP_PR[24]; */
	"0", /* char BP_TIDX_LR[24]; */
	"0", /* char BP_TIDX_PR[24]; */
	"0", /* char ROWS_READ[24]; */
	"0", /* char ROWS_WRITTEN[24]; */
	"0", /* char SECTION_NUMBER[24]; */
	"\"\"", /* char SEQUENCE_NO[8]; */
	"0", /* char SORT_OVERFLOWS[24]; */
	"0", /* char SQL_REQ_ID[24]; */
	"136", /* char SQLCABC[12]; */
	"'SQLCA'", /* char SQLCAID[12]; */
	"0", /* char SQLcode[12]; */
	"0", /* char SQLERRD1[12]; */
	"0", /* char SQLERRD2[12]; */
	"0", /* char SQLERRD3[12]; */
	"0", /* char SQLERRD4[12]; */
	"0", /* char SQLERRD5[12]; */
	"0", /* char SQLERRD6[12]; */
	"0", /* char SQLERRM[80]; */
	"\"\"", /* char SQLERRP[12]; */
	"\"\"", /* char SQLstate[10]; */
	"\"\"", /* char SQLWARN[16]; */
	"\"\"", /* char START_TIME[32]; */
	"0", /* char STMT_OPERATION[24]; */
	"0", /* char STMT_TYPE[24]; */
	"", /* char STOP_TIME[32]; */
	"0", /* char SYSTEM_CPU_TIME[24]; */
	"0", /* char TOTAL_SORT_TIME[24]; */
	"0", /* char TOTAL_SORTS[24]; */
	"0", /* char USER_CPU_TIME[24]; */
	"\"\""  /* char STMT_TEXT[32800]; */
};

/* Statement event monitor output in DPF systems includes SUBSECTION events as well
** as statement events.   Accommodate those here.
*/

#define SUBSECT_EV_MARKER "Subsection Event ..."

char SS_AGENT_ID[]           = "  Appl Handle: ";
char SS_EVMON_FLUSHES[]      = " Record is the result of a flush: ";
char SS_SUBSECT_NUM[]        = "  Subsection number: ";
char SS_NODE_NUM[]           = "  Node number: ";
char SS_ELAPSED_TIME[]       = "  Elapsed execution time: ";
char SS_USER_CPU_TIME[]      = "  User CPU time: ";
char SS_SYSTEM_CPU_TIME[]    = "  System CPU time: ";
char SS_TQ_SEND_OVFL[]       = "  Tablequeue send buffers overflowed: ";
char SS_TQ_ROWS_READ[]       = "  Tablequeue rows read: ";
char SS_TQ_ROWS_WRITTEN[]    = "  Tablequeue rows written: ";
char SS_NUM_AGENTS_CREATED[] = "  Number of agents created: ";

struct {
	char SS_AGENT_ID[24];
	char SS_EVMON_FLUSHES[24];
	char SS_SUBSECT_NUM[24];
	char SS_NODE_NUM[24];
	char SS_ELAPSED_TIME[32];
	char SS_USER_CPU_TIME[32];
	char SS_SYSTEM_CPU_TIME[32];
	char SS_TQ_SEND_OVFL[24];
	char SS_TQ_ROWS_READ[24];
	char SS_TQ_ROWS_WRITTEN[24];
	char SS_NUM_AGENTS_CREATED[24];
} ss_row,ss_templateRow = {
	"", /* char SS_AGENT_ID[24] */
	"", /* char SS_EVMON_FLUSHES[24] */
	"", /* char SS_SUBSECT_NUM[24] */
	"", /* char SS_NODE_NUM[24] */
	"", /* char SS_ELAPSED_TIME[32] */
	"", /* char SS_USER_CPU_TIME[32] */
	"", /* char SS_SYSTEM_CPU_TIME[32] */
	"", /* char SS_TQ_SEND_OVFL[24] */
	"", /* char SS_TQ_ROWS_READ[24] */
	"", /* char SS_TQ_ROWS_WRITTEN[24] */
	""  /* char SS_NUM_AGENTS_CREATED[24] */
};



/*************************
** Macros to make processing of the input strings easier.
*/
#define BUF_IS( A )       !strncmp( buf,A,sizeof(A)-1 )
#define FOLLOWING( A )    (lastSeen=A,printed=TRUE,strlen(buf+sizeof(A))),buf + sizeof(A) - 1
#define OPEN_QUOTE( A )   (isNumeric(buf+sizeof(A)-1)?0:(escapeQuotes(buf+sizeof(A)-1),1)),"\""
#define CLOSE_QUOTE( A )  (isNumeric(buf+sizeof(A)-1)?0:1),"\""
#define SAVE( STRUCT,A ) (strlen(buf + sizeof(A) - 1) < sizeof(STRUCT.A) ? \
			  sprintf( STRUCT.A,"%.*s%.*s%.*s",OPEN_QUOTE(A),FOLLOWING(A),CLOSE_QUOTE(A) ) : \
			  ( printf("error: value '%s' too large for field at line %d\n",\
			  FOLLOWING(A),__LINE__), doExit(1) ) )

/*************************
** doExit
**
** 'Dummy' exit function to get around compiler complaint about use of exit() in SAVE() macro on Solaris.
**
*************************/
int doExit(int code)
{
  exit(code);
  return 0;
}

/*************************
** isNumeric
**
** Returns TRUE if the parameter string pointer indicates a numeric value, FALSE otherwise.
**
*************************/
int isNumeric( char *p )
{
  while( *p )
  {
    while( isspace(*p) )
      ++p;
    if( !isdigit(*p) && (*p != '-') && (*p != '.') && (*p != '\n') && (*p) )
      return( FALSE );

    while( isdigit(*p) || (*p == '.') || (*p == '-') || (*p == '+') )
      ++p;

    if( !isspace(*p) && (*p != '\n') && (*p) )
      return( FALSE );
  }

  return( TRUE );
}


/*************************
** escapeQuotes
**
** Locates any quotes (") in the input string, and doubles them so that when the string is imported or loaded
** back into DB2, the quote doesn't end the string.
**
*************************/
int escapeQuotes( char *buf )
{
  char *p;
  for( p=buf; *p; ++p )
  {
    if( *p == '"' )
    {
      ++p;
      if( *p != '"' )
      {
        memmove( p+1,p,strlen(p)+1 );
        *p = '"';
      }
    }
  }
  return 0;
}



/*************************
** removeSeconds
**
** Scans forward in a timestmp and removes the 's' marking the seconds.  Import doesn't like this, so we have to
** get rid of it.
**
*************************/
void removeSeconds( char *p )
{
  while( *p != 's' && *p )
    ++p;
  *p = '\0';
}


/*************************
** fixTimeStamp
**
** Change from statement event monitor format (MM[-/]DD[-/]YYYY HH:MM:SS.hh) to 
** import/load format YYYY-MM-DD-HH.MM.SS.hh 
**
*************************/
void fixTimeStamp( char *buf )
{
  int i;
  char year[5];
  char mmdd[7];
  char *p = buf;

  while( isspace(*p) )
    ++p;

  for( i=0; i<6; ++i )
    mmdd[i] = *p++;

  for( i=0; i<4; ++i )
    year[i] = *p++;

  memcpy( buf,year,4 );
  buf[4] = '-';
  memcpy( buf+5,mmdd,6 );
  buf[7] = '-';
  buf[10] = '-';

  for( ; *p; ++p )
    if( *p == ':' )
      *p = '.';
}


/*************************
** convertStmtOperation
**
** Maps the input operation name, which is extracted from the statement event monitor
** output, to the numeric format that statement event monitor tables use.  These constants
** come from sqlmon.h
**
*************************/
int convertStmtOperation( char *operation )
{
  if( !strcmp(operation,"\"Prepare\"") ) 				return SQLM_PREPARE;
  if( !strcmp(operation,"\"Execute\"") ) 				return SQLM_EXECUTE;
  if( !strcmp(operation,"\"Execute Immediate\"") )			return SQLM_EXECUTE_IMMEDIATE;
  if( !strcmp(operation,"\"Open\"") ) 					return SQLM_OPEN;
  if( !strcmp(operation,"\"Fetch\"") ) 					return SQLM_FETCH;
  if( !strcmp(operation,"\"Close\"") ) 					return SQLM_CLOSE;
  if( !strcmp(operation,"\"Describe\"") ) 				return SQLM_DESCRIBE;
  if( !strcmp(operation,"\"Static Commit\"") ) 				return SQLM_STATIC_COMMIT;
  if( !strcmp(operation,"\"Static Rollback\"") ) 			return SQLM_STATIC_ROLLBACK;
  if( !strcmp(operation,"\"Set\"") ) 					return SQLM_SET;
  if( !strcmp(operation,"\"Free Locator\"") ) 				return SQLM_FREE_LOCATOR;
  if( !strcmp(operation,"\"Prepare to Commit\"") ) 			return SQLM_PREP_COMMIT;
  if( !strcmp(operation,"\"Call\"") ) 					return SQLM_CALL;
  if( !strcmp(operation,"\"Select\"") ) 				return SQLM_SELECT;
  if( !strcmp(operation,"\"Runstats\"") ) 				return SQLM_RUNSTATS;
  if( !strcmp(operation,"\"Reorg\"") ) 					return SQLM_REORG;
  if( !strcmp(operation,"\"Rebind Package\"") ) 			return SQLM_REBIND;
  if( !strcmp(operation,"\"Redistribute\"") ) 				return SQLM_REDIST;
  if( !strcmp(operation,"\"Get Administrative Authorization\"") ) 	return SQLM_GETAA;
  if( !strcmp(operation,"\"Get Table Authorization\"") ) 		return SQLM_GETTA;
#ifdef SQLM_GETNEXTCHUNK
  /* Only include this if we're at a version of DB2 that knows what GETNEXTCHUNK is */
  if( !strcmp(operation,"\"Get Next Chunk\"") ) return SQLM_GETNEXTCHUNK;
#endif
  return 0;
}

/*************************
** convertStmtType
**
** Maps the input type name, which is extracted from the statement event monitor
** output, to the numeric format that statement event monitor tables use.  These constants
** come from sqlmon.h
**
*************************/
int convertStmtType( char *type )
{
  if( !strcmp(type,"\"Static\"") )  return SQLM_STATIC;
  if( !strcmp(type,"\"Dynamic\"") ) return SQLM_DYNAMIC;
  if( !strcmp(type,"\"Non-SQL\"") ) return SQLM_NON_STMT;
  return SQLM_STMT_TYPE_UNKNOWN;
}



int main( int argc,char **argv )
{
  int i;

  FILE *in = NULL, *out_stmt = NULL, *out_subsect = NULL;

  char buf[65535];
  char *lastSeen = NULL;

  int printed = FALSE;

  /*
  ** Sanity check the input arguments
  */
  if( argc < 3 )
  {
    printf("Usage: %s input-file output-file [outut-subsect-file]\n",argv[0]);
    exit(1);
  }

  if( (in = fopen(argv[1],"r")) == NULL )
  {
    printf("Couldn't open input file %s\n",argv[1]);
    exit(1);
  }

  if( (out_stmt = fopen(argv[2],"w")) == NULL )
  {
    printf("Couldn't open stmt file %s\n",argv[1]);
    exit(1);
  }

  if( (argc > 3) && (out_subsect = fopen(argv[3],"w")) == NULL )
  {
    printf("Couldn't open stmt file %s\n",argv[1]);
    exit(1);
  }


  /* 
  ** Go till we find a statement event 
  */
  while( fgets(buf,sizeof(buf),in) )
  {
    /* 
    ** This is a new statement event 
    */
    if( isdigit(buf[0]) &&
        (strstr(buf,STMT_EV_MARKER)))
    {
      /* 
      ** Initialize the structure we're going to populate. 
      */
      row = templateRow;

      /* 
      ** Get the event number 
      */

      for( i=0; isdigit(buf[i]); ++i )
        fprintf(out_stmt,"%c",buf[i]);
      fprintf( out_stmt,",");

      /* 
      ** Get a line of input 
      */

      while( fgets(buf,sizeof(buf),in) )
      {
	/*
	** Each time we get a line, we check to see what type it is.  We use the macros
	** above to recognize the input.   If it's something we want to save and
	** output, then we use the SAVE macro.
	*/

        if( BUF_IS(AGENT_ID) )               	SAVE( row,AGENT_ID );
        else if( BUF_IS(APPL_ID) )              SAVE( row,APPL_ID );
        else if( BUF_IS(SEQUENCE_NO) )          SAVE( row,SEQUENCE_NO );
        else if( BUF_IS(EMPTY) 
		&& (lastSeen == SEQUENCE_NO) ) 	;
        else if( BUF_IS(EVMON_FLUSHES) )        SAVE( row,EVMON_FLUSHES );
        else if( BUF_IS(DASHES) )	        lastSeen = DASHES;
        else if( BUF_IS(STMT_TYPE) )            SAVE( row,STMT_TYPE );
        else if( BUF_IS(STMT_OPERATION) )       SAVE( row,STMT_OPERATION );
        else if( BUF_IS(SECTION_NUMBER) )       SAVE( row,SECTION_NUMBER );
        else if( BUF_IS(CREATOR) )              SAVE( row,CREATOR );
        else if( BUF_IS(PACKAGE_NAME) )	        SAVE( row,PACKAGE_NAME );
        else if( BUF_IS(CONSISTENCY_TOKEN) )    SAVE( row,CONSISTENCY_TOKEN );
        else if( BUF_IS(PACKAGE_VERSION_ID) )   SAVE( row,PACKAGE_VERSION_ID );
        else if( BUF_IS(CURSOR_NAME) )          SAVE( row,CURSOR_NAME );
        else if( BUF_IS(BLOCKING_CURSOR) )      SAVE( row,BLOCKING_CURSOR );
        else if( BUF_IS(STMT_TEXT) )            SAVE( row,STMT_TEXT );
        else if( BUF_IS(START_TIME) )           { fixTimeStamp(buf+sizeof(START_TIME)-1);
                                                  SAVE( row,START_TIME ); }
        else if( BUF_IS(STOP_TIME) )            { fixTimeStamp(buf+sizeof(STOP_TIME)-1);
                                                  SAVE( row,STOP_TIME ); }
        else if( BUF_IS(EXEC_TIME) )            ;
        else if( BUF_IS(AGENTS_TOP) )           SAVE( row,AGENTS_TOP );
        else if( BUF_IS(USER_CPU_TIME) )        { removeSeconds(buf+sizeof(USER_CPU_TIME));
                                                  SAVE( row,USER_CPU_TIME ); }
        else if( BUF_IS(SYSTEM_CPU_TIME) )      { removeSeconds(buf+sizeof(SYSTEM_CPU_TIME));
                                                  SAVE( row,SYSTEM_CPU_TIME ); }
        else if( BUF_IS(FETCH_COUNT) )          SAVE( row,FETCH_COUNT );
        else if( BUF_IS(TOTAL_SORTS) )          SAVE( row,TOTAL_SORTS );
        else if( BUF_IS(TOTAL_SORT_TIME) )      SAVE( row,TOTAL_SORT_TIME );
        else if( BUF_IS(SORT_OVERFLOWS) )       SAVE( row,SORT_OVERFLOWS );
        else if( BUF_IS(ROWS_READ) )            SAVE( row,ROWS_READ );
        else if( BUF_IS(ROWS_WRITTEN) )         SAVE( row,ROWS_WRITTEN );
        else if( BUF_IS(INT_ROWS_DELETED) )     SAVE( row,INT_ROWS_DELETED );
        else if( BUF_IS(INT_ROWS_UPDATED) )     SAVE( row,INT_ROWS_UPDATED );
        else if( BUF_IS(INT_ROWS_INSERTED) )    SAVE( row,INT_ROWS_INSERTED );
        else if( BUF_IS(BP_DATA_LR) )           SAVE( row,BP_DATA_LR );
        else if( BUF_IS(BP_DATA_PR) )           SAVE( row,BP_DATA_PR );
        else if( BUF_IS(BP_TEMP_LR) )           SAVE( row,BP_TEMP_LR );
        else if( BUF_IS(BP_TEMP_PR) )           SAVE( row,BP_TEMP_PR );
        else if( BUF_IS(BP_IDX_LR) )            SAVE( row,BP_IDX_LR );
        else if( BUF_IS(BP_IDX_PR) )            SAVE( row,BP_IDX_PR );
        else if( BUF_IS(BP_TIDX_LR) )           SAVE( row,BP_TIDX_LR );
        else if( BUF_IS(BP_TIDX_PR) )           SAVE( row,BP_TIDX_PR );
        else if( BUF_IS(SQLCA) )                ;
        else if( BUF_IS(SQLcode) )              SAVE( row,SQLcode );
        else if( BUF_IS(SQLstate) )             SAVE( row, SQLstate );
        else if( printed )
        {
	  /*
	  ** It's not one of the recognized strings, so we 
	  ** output the contents of the structure as a row
	  ** in the .del file, and get ready to start again.
	  */

          printed = FALSE;
	  lastSeen = NULL;

	  fprintf( out_stmt,"%s,", row.AGENT_ID );
	  fprintf( out_stmt,"%s,", row.AGENTS_TOP );
	  fprintf( out_stmt,"%s,", row.APPL_ID );
	  fprintf( out_stmt,"%s,", (!strcmp(row.BLOCKING_CURSOR,"TRUE") ? "1":"0") );
	  fprintf( out_stmt,"%s,", row.CONSISTENCY_TOKEN );
	  fprintf( out_stmt,"%s,", row.CREATOR );
	  fprintf( out_stmt,"%s,", row.CURSOR_NAME );
	  fprintf( out_stmt,"%s,", (!strcmp(row.EVMON_FLUSHES,"TRUE") ? "1":"0") );
	  fprintf( out_stmt,"%s,", row.FETCH_COUNT );
	  fprintf( out_stmt,"%s,", row.INT_ROWS_DELETED );
	  fprintf( out_stmt,"%s,", row.INT_ROWS_INSERTED );
	  fprintf( out_stmt,"%s,", row.INT_ROWS_UPDATED );
	  fprintf( out_stmt,"%s,", row.PACKAGE_NAME );
	  fprintf( out_stmt,"%s,", row.PACKAGE_VERSION_ID );
	  fprintf( out_stmt,"%s,", row.PARTIAL_RECORD );
	  fprintf( out_stmt,"%s,", row.BP_DATA_LR );
	  fprintf( out_stmt,"%s,", row.BP_DATA_PR );
	  fprintf( out_stmt,"%s,", row.BP_IDX_LR );
	  fprintf( out_stmt,"%s,", row.BP_IDX_PR );
	  fprintf( out_stmt,"%s,", row.BP_TEMP_LR );
	  fprintf( out_stmt,"%s,", row.BP_TEMP_PR );
	  fprintf( out_stmt,"%s,", row.BP_TIDX_LR );
	  fprintf( out_stmt,"%s,", row.BP_TIDX_PR );
	  fprintf( out_stmt,"%s,", row.ROWS_READ );
	  fprintf( out_stmt,"%s,", row.ROWS_WRITTEN );
	  fprintf( out_stmt,"%s,", row.SECTION_NUMBER );
	  fprintf( out_stmt,"\"%s\",", row.SEQUENCE_NO );
	  fprintf( out_stmt,"%s,", row.SORT_OVERFLOWS );
	  fprintf( out_stmt,"%s,", row.SQL_REQ_ID );
	  fprintf( out_stmt,"%s,", row.SQLCABC );
	  fprintf( out_stmt,"%s,", row.SQLCAID );
	  fprintf( out_stmt,"%s,", row.SQLcode );
	  fprintf( out_stmt,"%s,", row.SQLERRD1 );
	  fprintf( out_stmt,"%s,", row.SQLERRD2 );
	  fprintf( out_stmt,"%s,", row.SQLERRD3 );
	  fprintf( out_stmt,"%s,", row.SQLERRD4 );
	  fprintf( out_stmt,"%s,", row.SQLERRD5 );
	  fprintf( out_stmt,"%s,", row.SQLERRD6 );
	  fprintf( out_stmt,"%s,", row.SQLERRM );
	  fprintf( out_stmt,"%s,", row.SQLERRP );
	  fprintf( out_stmt,"%s,", row.SQLstate );
	  fprintf( out_stmt,"%s,", row.SQLWARN );
	  fprintf( out_stmt,"%s,", row.START_TIME );
	  fprintf( out_stmt,"%d,", convertStmtOperation(row.STMT_OPERATION) );
	  fprintf( out_stmt,"%d,", convertStmtType(row.STMT_TYPE) );
	  fprintf( out_stmt,"%s,", row.STOP_TIME );
	  fprintf( out_stmt,"%s,", row.SYSTEM_CPU_TIME );
	  fprintf( out_stmt,"%s,", row.TOTAL_SORT_TIME );
	  fprintf( out_stmt,"%s,", row.TOTAL_SORTS );
	  fprintf( out_stmt,"%s,", row.USER_CPU_TIME );
	  fprintf( out_stmt,"%s",  row.STMT_TEXT );
          fprintf( out_stmt,"\n");
          break;
        }
      }
    }
    else if (out_subsect != NULL)
    {
      /*
      ** It's a subsection event, so process that.  Very similar to a statement event, 
      ** but different matching strings, etc.
      */

      if( isdigit(buf[0]) &&
        (strstr(buf,SUBSECT_EV_MARKER)))
      {
        ss_row = ss_templateRow;

	for( i=0; isdigit(buf[i]); ++i )
	  fprintf(out_subsect,"%c",buf[i]);
	fprintf(out_subsect,",");

	/* 
	** Get a line of input 
	*/

	while( fgets(buf,sizeof(buf),in) )
	{
	  if( BUF_IS(SS_AGENT_ID) )                  SAVE( ss_row,SS_AGENT_ID );
	  else if( BUF_IS(EMPTY) && (lastSeen == SS_AGENT_ID) ) ;
	  else if( BUF_IS(SS_EVMON_FLUSHES) )        SAVE( ss_row,SS_EVMON_FLUSHES );
	  else if( BUF_IS(SS_SUBSECT_NUM) )          SAVE( ss_row,SS_SUBSECT_NUM );
	  else if( BUF_IS(SS_NODE_NUM) )             SAVE( ss_row,SS_NODE_NUM );
	  else if( BUF_IS(SS_ELAPSED_TIME) )         SAVE( ss_row,SS_ELAPSED_TIME );
	  else if( BUF_IS(SS_USER_CPU_TIME) )        { removeSeconds(buf+sizeof(SS_USER_CPU_TIME));
						       SAVE( ss_row,SS_USER_CPU_TIME ); }
	  else if( BUF_IS(SS_SYSTEM_CPU_TIME) )      { removeSeconds(buf+sizeof(SS_SYSTEM_CPU_TIME));
						       SAVE( ss_row,SS_SYSTEM_CPU_TIME ); }
	  else if( BUF_IS(SS_TQ_SEND_OVFL) )         SAVE( ss_row,SS_TQ_SEND_OVFL );
	  else if( BUF_IS(SS_TQ_ROWS_READ) )         SAVE( ss_row,SS_TQ_ROWS_READ );
	  else if( BUF_IS(SS_TQ_ROWS_WRITTEN) )      SAVE( ss_row,SS_TQ_ROWS_WRITTEN );
          else if( BUF_IS(SS_NUM_AGENTS_CREATED) )   SAVE( ss_row,SS_NUM_AGENTS_CREATED );
	  else if( printed )
	  {
	    printed = FALSE;
	    lastSeen = NULL;

	    fprintf( out_subsect,"%s,",ss_row.SS_AGENT_ID );
	    fprintf( out_subsect,"%s,",ss_row.SS_EVMON_FLUSHES );
	    fprintf( out_subsect,"%s,",ss_row.SS_SUBSECT_NUM );
	    fprintf( out_subsect,"%s,",ss_row.SS_NODE_NUM );
	    fprintf( out_subsect,"%s,",ss_row.SS_ELAPSED_TIME );
	    fprintf( out_subsect,"%s,",ss_row.SS_USER_CPU_TIME );
	    fprintf( out_subsect,"%s,",ss_row.SS_SYSTEM_CPU_TIME );
	    fprintf( out_subsect,"%s,",ss_row.SS_TQ_SEND_OVFL );
	    fprintf( out_subsect,"%s,",ss_row.SS_TQ_ROWS_READ );
	    fprintf( out_subsect,"%s,",ss_row.SS_TQ_ROWS_WRITTEN );
	    fprintf( out_subsect,"%s", ss_row.SS_NUM_AGENTS_CREATED );
	    fprintf(out_subsect,"\n");
	    break;
	  }
	}
      }
    }
  }

  return 0;
}
