// ========================================================================
// LOG_SampleReader.java
// 
// (C) COPYRIGHT International Business Machines Corp. 1999
// All Rights Reserved
// 
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
// 
// PURPOSE
//      Sample program to read log records from the binary log files
// 
//  The file must be located in the following Network Dispatcher directory:
//   
//    - nd/servers/lib/BinaryLog/ (nd\servers\lib\BinaryLog on NT)
//
// ========================================================================
import com.ibm.internet.nd.log.*;

import java.io.*;
import java.util.*;
import java.text.*;

/**
 * The LOG_SampleReader class is an example of a main program to extract
 * information from a Network Dispatcher component's binary log files.
 *
 * The program accepts two times as input arguments; and retrieves all
 * available log records that were logged between the two times.
 *
 * The directory where the log files are stored is provided to the
 * program in the java environment variable "END_LOG_DIRECTORY", using
 * the -D flag of the java command.
 *
 * An example use of this class would be
 * 
 * java -cp <classpath> -D<log_directory> LOG_SampleReader 1999/02/26 8:00 1999/02/26 17:00
 * 
 * The above command would display all the server information logged between
 * 8:00 AM and 5:00 PM on February 2nd, 1999.
 * 
 * The <classpath> argument must point first to the directory containing the class
 * file for LOG_SampleReader.java, then to the ibmnd.jar file.
 * A sample command to invoke the logreader is provided in the samples/BinaryLog directory.
 */

public class LOG_SampleReader
{
  // ----------------------
  // Public Class variables
  // ======================

  /**
   * The SimpleDateFormat used to read dates from the command line.
   *
   * This format is used to parse the dates entered on the command line.
   * Its value is "yyyy/MM/dd HH:mm".  This means that the two
   * dates provided on the command line would look like:
   * 
   * 1999/02/26 8:00 1999/02/26 17:00
   * 
   * You must edit the source code to change the format used.
   * 
   * Since the format contains spaces between the date and time fields,
   * the main program reads them as args[0] through args[3] of the input
   * arguments.  If you modify the format to contain fewer tokens or
   * more tokens, you must modify the main program to read your arguments
   * from the correct input arg.
   */
  // Declared public so javadoc will print it by default  
  public static SimpleDateFormat sdfInput = new SimpleDateFormat("yyyy/MM/dd HH:mm");

  // -----------
  // Constructor
  // ===========
  /**
   * The object is never actually constructed, since all the work is done
   * in the main program.
   */
  LOG_SampleReader(){}
    
  // --------------------
  // Main Program
  // ====================

  /**
   * The main program takes input form the command line and writes out the
   * requested log records from the specified log directory's binary log files.
   *
   * The program instantiates a LOG_Reader object, initializes it with the
   * start and end times, then retrieves LOG_Record objects from the LOG_Reader
   * until a null record is returned.  The LOG_Record objects retrieved are cast
   * to the appropriate type, and the information from the LOG_Record is either
   * stored for later output, or written to the screen.
   *
   * The LOG_Reader object's constructor takes as input the directory where
   * the log files are stored.
   *
   * The LOG_Reader object's initialize function takes as input two long values
   * that represent the start and end times for the records to be retrieved.  These
   * times are in Coordinated Universal Time (CUT).
   * The input arguments from the command line have been converted from strings to
   * the appropriate longs using the java.text.SimpleDateFormat and java.util.Date
   * classes.
   *
   * The LOG_Reader object's initialize method returns the first record in the
   * logs that falls within the time range specified. If no records exist between
   * the times specified, null is returned.
   *
   * The main program then starts a loop which processes the current LOG_Record,
   * then calls the LOG_Reader object's readRecord method to get the next LOG_Record.
   * This loop will continue until the LOG_Record retrieved is null.
   *
   * The following are the types of LOG_Record objects that can be retrieved:
   *
   * LOG_TimeStampRecord
   *   The LOG_TimeStampRecord contains the following methods:
   * 
   *   getTimestamp() - returns a long value representing the time in CUT when
   *     the record was written to the binary log.
   * 
   * LOG_ExecutorIDRecord
   *   The LOG_ExecutorIDRecord contains the following methods:
   * 
   *   getNFA() - returns a string representing the Non-Forwarding address
   *     of the executor.
   *   getNumClusters() - returns an int representing the number of clusters defined
   *     in the executor.
   * 
   * LOG_ClusterIDRecord
   *   The LOG_ClusterIDRecord contains the following methods:
   *   
   *   getIPAddress() - returns a string representing the cluster address.
   *   getNumPorts() - returns an int representing the number of ports defined
   *     for this cluster.
   * 
   * LOG_PortIDRecord
   *   Each LOG_PortIDRecord describes a port that has been defined under the cluster
   *   address that was returned by the most recently retrieved LOG_ClusterIDRecord.
   *   The LOG_PortIDRecord contains the following methods:
   * 
   *   getPortNumber() - returns an int representing the port number.
   *   getNumServers() - returns an int representing the number of servers defined
   *     for this port.
   * 
   * LOG_ServerReportRecord
   *   Each LOG_ServerReportRecord describes a server that has been defined under
   *   the port number that was returned by the most recently retrieved LOG_PortIDRecord.
   *   The LOG_ServerReportRecord contains the following methods:
   * 
   *   getServerId()  - returns a string representing the serverId.
   *   getIPAddress() - returns a string representing the server address.
   *   getWeight()    - returns an int representing the weight
   *     of the server when the record was written to the binary log  
   *   getTotalConnections() - returns an int representing the total connections
   *     of the server when the record was written to the binary log  
   *   getActiveConnections() - returns an int representing the active connections
   *     of the server when the record was written to the binary log  
   *   getPortLoad() - returns an int representing the port load
   *     of the server when the record was written to the binary log  
   *   getSystemLoad() - returns an int representing the system load
   *     of the server when the record was written to the binary log  
   *   getHasPortLoad() - returns a boolean specifying whether port loads were
   *     being provided for the server when the record was written to the binary log  
   *   getHasSystemLoad() - returns a boolean specifying whether system loads were
   *     being provided for the server when the record was written to the binary log  
   * 
   * 
   * LOG_Record objects of each type will be retrieved from the binary logs in
   * the following order.
   * 
   *   LOG_TimeStampRecord
   *   LOG_ExecutorIDRecord
   *   LOG_ClusterIDRecord (if any clusters exist)
   *   LOG_PortIDRecord (if any ports exist for the previously read cluster)
   *   LOG_ServerReportRecord (one for each server defined on the previously read port)
   *   Additional LOG_PortIDRecords and their LOG_ServerReportRecords for other ports
   *     defined on the previously read cluster
   *   Additional LOG_ClusterIDRecords and their LOG_PortIDRecords and their LOG_ServerReportRecords
   *     for other defined clusters.
   *   Next sequence in same order, starting with a LOG_TimeStampRecord.
   * 
   */
  public static void main(String[] args) throws ParseException {
    // The directory where the logs are kept is communicated through a java environment variable.
    // This allows the main class to be called with just the times to report.  If the directory
    // name doesn't end with a path separator, add one on.
    String sLogDirectory = System.getProperty("END_LOG_DIRECTORY");
    if (!sLogDirectory.endsWith(File.separator)) sLogDirectory = sLogDirectory+File.separator;

    // Get the start and end time from the command line and convert to longs.
    Date date;
    date = sdfInput.parse(args[0]+" "+args[1]);
    long lStartTime = date.getTime();
    date = sdfInput.parse(args[2]+" "+args[3]);
    long lEndTime = date.getTime();

    // Initialize all the fields we will be retrieving and writing out
    SimpleDateFormat sdfTimestamp = new SimpleDateFormat("yyyy/MM/dd-HH:mm:ss.SSS");
    long lTimeStamp = 0;
    String sNFA = null;
    int iNumClusters =0;
    String sClusterIPAddress = null;
    int iNumPorts = 0;
    int iPortNumber = 0;
    int iNumServers = 0;
    String sServerId = null;  //75318 20010712
    String sServerIPAddress = null;
    int iWeight = 0;
    int iTotalConnections = 0;
    int iActiveConnections = 0;
    int iPortLoad = 0;
    int iSystemLoad = 0;
    boolean bHasPortLoad = false;
    boolean bHasSystemLoad = false;

    // Instantiate a LOG_Reader and get the first LOG_Record
    LOG_Reader logReader = new LOG_Reader(sLogDirectory);
    LOG_Record logRecord = logReader.initialize(lStartTime,lEndTime);

    // Process each LOG_Record, accumulate information until we get a
    // LOG_ServerReportRecord, then write out everything.
    while (null != logRecord) {
      if (logRecord instanceof LOG_TimeStampRecord) {
        // Got a TimeStamp
        lTimeStamp = ((LOG_TimeStampRecord)logRecord).getTimeStamp();
      }
      else if (logRecord instanceof LOG_ExecutorIDRecord) {
        // Got an Executor ID
        sNFA         = ((LOG_ExecutorIDRecord)logRecord).getNFA();
        iNumClusters = ((LOG_ExecutorIDRecord)logRecord).getNumClusters();
      }
      else if (logRecord instanceof LOG_ClusterIDRecord) {
        // Got a Cluster ID
        sClusterIPAddress = ((LOG_ClusterIDRecord)logRecord).getIPAddress();
        iNumPorts         = ((LOG_ClusterIDRecord)logRecord).getNumPorts();
      }
      else if (logRecord instanceof LOG_PortIDRecord) {
        // Got a Port ID
        iPortNumber = ((LOG_PortIDRecord)logRecord).getPortNumber();
        iNumServers = ((LOG_PortIDRecord)logRecord).getNumServers();
      }
      else if (logRecord instanceof LOG_ServerReportRecord) {
        // Got a Server Report
        sServerId          = ((LOG_ServerReportRecord)logRecord).getServerId();  //75318 20010712
        sServerIPAddress   = ((LOG_ServerReportRecord)logRecord).getIPAddress();
        iWeight            = ((LOG_ServerReportRecord)logRecord).getWeight();
        iTotalConnections  = ((LOG_ServerReportRecord)logRecord).getTotalConnections();
        iActiveConnections = ((LOG_ServerReportRecord)logRecord).getActiveConnections();
        iPortLoad          = ((LOG_ServerReportRecord)logRecord).getPortLoad();
        iSystemLoad        = ((LOG_ServerReportRecord)logRecord).getSystemLoad();
        bHasPortLoad       = ((LOG_ServerReportRecord)logRecord).getHasPortLoad();
        bHasSystemLoad     = ((LOG_ServerReportRecord)logRecord).getHasSystemLoad();
        System.out.println(sdfTimestamp.format(new Date(lTimeStamp))+","+
                           sNFA+","+
                           sClusterIPAddress+","+
                           iPortNumber+","+
                           sServerId+","+           //75318 20010712
                           sServerIPAddress+","+
                           iWeight+","+
                           iTotalConnections+","+
                           iActiveConnections+","+
                           iPortLoad+","+
                           bHasPortLoad+","+
                           iSystemLoad+","+
                           bHasSystemLoad);
      }
      else {
        // Got some other kind of LOG_Record (there are none yet)
      }
      logRecord = logReader.readRecord();
    }
  }
}

