package com.ibm.ulc.application;

import com.ibm.ulc.util.UlcHashtable;
/**
 * ULC table models internally use oid to identify rows independently from the row's index in the model's rows.
 * This class manages the index-to-oid mappings in an array. This solution emphasizes speed over memory requirement
 * since the array is preallocated to whatever rowCount is specified by the receiver's itemList.
 * In most cases, this class would be the one to use. Only in cases of enormous data on the table model, which the application
 * developer is sure will never be viewed by the user (but then, why even show so many rows?) would the lookup solution be preferable.
 */
public class ULCArrayedIndexToOidMap extends ULCIndexToOidMapAbstract {
	protected int[] fIndexToOidMap = new int[0];
/**
 * Answer the receiver's oids in an int array. 
 * The index is implicit in the position of each oid.
 */
protected int[] getOidArray(int invalidOid) {
	if (invalidOid == fInvalidOid)
		return fIndexToOidMap;
	else {
		int[] answer = new int[fIndexToOidMap.length];
		for (int i = 0; i < fIndexToOidMap.length; i++) {
			if (fIndexToOidMap[i] == fInvalidOid)
				answer[i] = invalidOid;
			else
				answer[i] = fIndexToOidMap[i];
		}
		return answer;
	}
}
/**
 * Return the object Id for the row Index or @defaultOid if not found
 *
 * @param 	index		int		the row index whose oid is required
 * @param	defaultOid	int		the default oid to be answered
 */
public int getOidForIndex(int index, int defaultOid) {
	int oid = defaultOid;
	if (index >= 0 && index < fIndexToOidMap.length) {
		oid = fIndexToOidMap[index];
		if (oid == defaultOid)
			oid = getItemList().handleMissingOid(index);
	}
	return oid;
}
/**
 * Insert a range of new rows into the receiver.
 * Assign oids to the items inserted and inform the itemList's model about the new mapping
 *
 * @param itemList	The itemList of the receiver	
 * @param start 	The index of the first row to be inserted
 * @param end 		The index of the last row to be inserted
 */
protected void insertRows(ULCDefaultItemList itemList, int start, int end) {
	int[] newArray = newIndexToOidMap();
	if (fIndexToOidMap.length > 0 && start > 0)
		System.arraycopy(fIndexToOidMap, 0, newArray, 0, start);
	if (fIndexToOidMap.length > start)
		System.arraycopy(fIndexToOidMap, start, newArray, end + 1, fIndexToOidMap.length - start);
	for (int i = start; i <= end; i++) {
		int oid = itemList.getNextOid();
		newArray[i] = oid;
		itemList.getModel().indexWasMapped(i, oid);
	}
	fIndexToOidMap = newArray;
	rebuildOidToIndexMap();
}
/**
 * Insert the given oids to the receiver. Start this insertion at the specified index.
 *
 * @param insertionIndex 	the index at which the oids are to be inserted
 * @param oids				the array of oids to be inserted
 */
public void insertRowsAt(int insertionIndex, int[] oids) {
	int[] newArray = newIndexToOidMap();
	if (fIndexToOidMap.length > 0 && insertionIndex > 0)
		System.arraycopy(fIndexToOidMap, 0, newArray, 0, insertionIndex);
	if (fIndexToOidMap.length > insertionIndex)
		System.arraycopy(fIndexToOidMap, insertionIndex, newArray, oids.length, fIndexToOidMap.length - insertionIndex);
	for (int i = 0; i <= oids.length; i++) {
		newArray[insertionIndex + i] = oids[i];
	}
	fIndexToOidMap = newArray;
}
/**
 * Invert the the receiver's rows.
 */
protected void invertRows() {
	int[] newIndexToOidMap = new int[fIndexToOidMap.length];
	int counter = fIndexToOidMap.length;
	for (int i = 0; i < fIndexToOidMap.length; i++) {
		newIndexToOidMap[--counter] = fIndexToOidMap[i];
	}
	fIndexToOidMap = newIndexToOidMap;
	rebuildOidToIndexMap();
}
/**
 * Create and answer a new index-oid map for the receiver
 */
protected int[] newIndexToOidMap() {
	int invalid = getItemList().INVALID_OID;
	int[] answer = new int[getItemList().getRowCount()];
	for (int i = 0; i < answer.length; i++) {
		answer[i] = invalid;
	}
	return answer;
}
/**
 * Rebuild the row index map based on values from the Oid map.
 */
protected void rebuildIndexToOidMap() {
	fIndexToOidMap = newIndexToOidMap();
}
/**
 * Rebuild the row index map based on values from the Oid map.
 */
protected void rebuildOidToIndexMap() {
	clearOidToIndexMap(fIndexToOidMap.length);
	for (int i = 0; i < fIndexToOidMap.length; i++) {
		if (fIndexToOidMap[i] != fInvalidOid)
			fOidToIndexMap.put(new Integer(fIndexToOidMap[i]), new Integer(i));
	}
}
/**
 * Update the receiver's maps by removing all mappings for @rowIds
 *
 * @param rowIds the oids of the rows to remove
 */
protected void removeRows(int[] rowIds) {
	for (int i = 0; i < rowIds.length; i++) {
		int index = getIndexForOid(rowIds[i]);
		fIndexToOidMap[index] = fInvalidOid;
	}
	int[] newIndices = newIndexToOidMap();
	int counter = 0;
	for (int i = 0; i < fIndexToOidMap.length; i++) {
		if (fIndexToOidMap[i] != fInvalidOid)
			newIndices[counter++] = fIndexToOidMap[i];
	}
	fIndexToOidMap = newIndices;
	rebuildOidToIndexMap();
}
/**
 * Set the mapping for @index to @oid
 */
protected void setIndexToOidMapping(int index, int oid) {
	fIndexToOidMap[index] = oid;
	fOidToIndexMap.put(new Integer(oid), new Integer(index));
}
/**
 * Set the receiver's rows at the given rowIds. The index is implicit in the position of each oid.
 * Invert the order of the elements if so indicated by the inverted flag.
 *
 * @param rowIds	int[]	The rowIds of the rows to be set
 * @param inverted	boolean	Whether the order should be reversed
 */	
protected void setRows(int[] rowIds, boolean inverted) {
	fIndexToOidMap = rowIds;
	if (inverted)
		invertRows();
	else
		rebuildOidToIndexMap();
}
/**
 * Answer true if the receiver's current state indicates that the receiver 
 * may no longer be in synch with its model. This is assumed if the size 
 * of the receiver's mappings is equal to the @currentRowCount of the model.
 *
 * @param currentRowCount	int		The current number of rows in the receiver
 *
 */
protected boolean shouldNotifyContentsChanged(int currentRowCount) {
	return fIndexToOidMap.length != currentRowCount;
}
}
