package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCDefaultItemList is the default implementation of ItemList for use with the
 * exclusivly index-based ULCTableModel hierarchy. It is also used by all AbstractTableModel
 * subclasses that must provide indexed and sparse access to their rows at the same time.
 *
 * If secondary itemLists (such as ULCSortedItemList or ULCHierarchicalItemList) use the 
 * receiver as source for the oid-to-row mapping, the complete mapping for the receiver is done
 * at the beginning. If the receiver is the sole customer of its TableModel, the mapping will
 * be done lazily.
 *
 * List Widgets have to access rows by index through an item list.
 *
 * @see ULCTableModel
 */
public class ULCDefaultItemList extends ULCIndexedItemList {
/**
 * Construct a new ULCDefaultItemList
 */
public ULCDefaultItemList() {
}
/**
 * The contents of the receiver's tableModel has changed. The receiver must reinitialize
 * itself. It is called when the receiver has been uploaded upon changes in the receiver's
 * TableModel, or before the receiver is saving its state.
 *
 * The default ItemList installs all mapping information.
 *
 * @see #saveState
 */
protected void contentsChanged() {
	getObjectIdentifiers(0, Math.min(getPrefetchCount(), getRowCount()));
}
/**
 * Answer the next available object identifier for the receiver 
 */
protected int getNextOid() {
	fLastOid++;
	return fLastOid;
}
/**
 * An oid was not defined at the given index.
 * The receiver creates a new oid for this index and answers it.
 *
 * @param index		int		The index at which the oid is missing
 */
protected int handleMissingOid(int index) {
	int rowId = getNextOid();
	getOidMap().setIndexToOidMapping(index, rowId);
	getModel().indexWasMapped(index, rowId);
	return rowId;
}
/*
 * extract and answer all oids defined by the ranges of indices
 *
 * @param ranges A Vector containing <code>UlcRange</code> objects
 */
private int[] indexRangesToOids(Vector ranges) {
	Vector allOids = new Vector();
	for (int i = 0; i < ranges.size(); i++) {
		UlcRange range = (UlcRange) ranges.elementAt(i);
		Anything rowIds = new Anything(getObjectIdentifiers(range.fStartIndex, (range.fEndIndex - range.fStartIndex) + 1));
		for (int j = 0; j < rowIds.size(); j++) {
			allOids.addElement(new Integer(rowIds.get(j).asInt(-1)));
		}
	}
	int[] answer = new int[allOids.size()];
	for (int i = 0; i < answer.length; i++) {
		answer[i] = ((Integer) allOids.elementAt(i)).intValue();
	}
	return answer;
}
/**
 * Insert a range of new rows into the receiver.
 *
 * @param start The index of the first row to be inserted
 * @param end 	The index of the last row to be inserted
 */
protected void insertRows(int start, int end) {
	getOidMap().insertRows(this, start, end);
}
/**
 * The default Item list is always active.
 */
protected boolean isActive() {
	return true;
}
/**
 * This method must be called whenever the underlying model is changed.
 *
 * @param model The new <code>ULCAbstractTableModel</code>
 */
protected void modelChanged(ULCAbstractTableModel model) {
}
/**
 * Notify UI TableModel that either contents or size of
 * ULC TableModel has changed.
 * This method must be used when attempting to delete rows of non contiguous
 * since deleting non contiguos ranges will cause the indexes to change
 * during the delete.
 * ranges. eg: (1,2,3,19,25,26);
 *
 * @param type 	The kind of change.
 * <pre>
 * type can be one of:
 *	TABLE_MODEL_CONTENTS_CHANGED;
 *	TABLE_MODEL_ROWS_ADDED;
 *	TABLE_MODEL_ROWS_REMOVED;
 *	TABLE_MODEL_ROWS_CHANGED;
 *	TABLE_MODEL_CELL_CHANGED;
 * </pre>
 * @param colId The column's identifier.
 * @param ranges A Vector containing <code>UlcRange</code> objects
 */
protected void notify(int type, String attributeName, Vector ranges) {
	if (type == TABLE_MODEL_CONTENTS_CHANGED) {
		getOidMap().clear();
		fModel.internalNotifyContentsChanged();
		return;
	}
	if (type == TABLE_MODEL_ROWS_ADDED) {
		for (int i = 0; i < ranges.size(); i++) {
			UlcRange range = (UlcRange) ranges.elementAt(i);
			insertRows(range.fStartIndex, range.fEndIndex);
		}
	}
	int[] oids = indexRangesToOids(ranges);
	if (type == TABLE_MODEL_ROWS_ADDED) {
		fModel.internalNotifyRowsAdded(oids);
		return;
	}
	if (type == TABLE_MODEL_ROWS_REMOVED) {
		fModel.internalNotifyRowsRemoved(oids);
	}
	if (type == TABLE_MODEL_ROWS_CHANGED || type == TABLE_MODEL_CELL_CHANGED) {
		fModel.internalNotifyRowsChanged(oids, attributeName);
	}
}
/**
 * Remove the receiver from the given model.
 *
 * @param model		ULCAbstractTableModel	The model from which I have to remove myself
 */
public void removeFromModel(ULCAbstractTableModel model) {
	if (model != null)
		model.resetDefaultItemList(this);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	if (shouldNotifyContentsChanged())
		notify(TABLE_MODEL_CONTENTS_CHANGED, null, new Vector());
	super.saveState(a);
}
/**
 * Set to true, if the receiver is set to active.
 *
 * @param active	boolean		New active state of the receiver
 */
protected void setActive(boolean active) {
	if (!isActive()) {
		super.setActive(active);
	}
}
/**
 * Answer true if the receiver's current state indicates that the receiver 
 * may no longer be in synch with its model.
 */
protected boolean shouldNotifyContentsChanged() {
	return getOidMap().shouldNotifyContentsChanged(getRowCount());
}
}
