package com.ibm.ulc.ui.lists;

/*
 * Copyright (c) 1997 Object Technology International Inc.
 */
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.comm.DeferredRequest;
import com.ibm.ulc.ui.base.*;
import com.ibm.ulc.ui.*;
public abstract class UIAbstractList extends UIComponent implements IEnableListener, IItemListListener {
	protected JScrollPane fScrollPane = null;
	protected Vector fEmptyStateListeners = null;
	protected Vector fSelectionListeners = null;
	protected int[] fSelectedRowIds = new int[0];
	protected IRow[] fPendingSelection = new IRow[0];
	protected UIItemList fItemList = null;
	protected IJList fWidget = null;
	protected UITrigger fTrigger = null;
	protected Vector fRowModels = new Vector();
/**
 * Add the given component as listener, who will be informed
 * when the selection of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * required protocol of IEnableListener
 *
 * @param component : The Component which will act as listener.
 */
public void addEnableListener(IEnableListenerTarget c) {
	Assert.isNotNull(c);
	if (fEmptyStateListeners == null)
		fEmptyStateListeners = new Vector();
	fEmptyStateListeners.addElement(c);
	c.setEnabled(shouldEnableListener());
}
/**
 * Dont do anything because we are already registered as a mouse listener.
 */
protected void addMouseListenerForPopupMenu() {
}
/*
  * 	In order to have rowModels keep track of tableModel changes in the receiver
  *	the receiver needs to keep track of them to notify when a setModel happens.
  */
protected void addRowModel(com.ibm.ulc.ui.lists.UIRowModel rowModel) {
	addSelectionListener(rowModel);
	fRowModels.addElement(rowModel);
	if (getItemList() != null) {
		getItemList().getItemCache().addListener(rowModel);
	}
}
/**
 * Add a component who will be notified when the current selection
 * of the receiver changes.
 */
public void addSelectionListener(ISelectionListener component) {
	Assert.isNotNull(component);
	if (fSelectionListeners == null)
		fSelectionListeners = new Vector();
	fSelectionListeners.addElement(component);
	component.updateSelection(getSelectedRowIds());
}
private void basicSelectionChanged(final String event) {
	int[] ix = getSelectedRowIds();
	boolean valid= true;
	int i= 0;
	while (valid && i < ix.length)	{
		if (ix[i] == -1)
			valid= false;
		i++;
	}
	if (valid) {
		fSelectedRowIds = ix;
		enableListeners(ix.length > 0);
		updateSelectionListeners(fSelectedRowIds);
		sendEventULC(event, "rows", prepareCurrentSelection());
		if (fTrigger != null)
			fTrigger.trigger(fConnection, "action".equals(event) ? TRIGGER_ON_ACTION : TRIGGER_ON_SELECTION, this, null);
	}
	else {
		DeferredRequest r= new DeferredRequest(fConnection) {
			public void safeDispatch() {
				selectionChanged(event);
			}
		};
		fConnection.postRequest(r);			
	}
}
public void clearSelection() {
	setSelectedItems(new IRow[0]);
}
void enableListeners(boolean state) {
	if (fEmptyStateListeners != null) {
		Enumeration e = fEmptyStateListeners.elements();
		while (e.hasMoreElements()) {
			IEnableListenerTarget c = (IEnableListenerTarget) e.nextElement();
			c.setEnabled(state);
		}
	}
}
public void free() {
	if (fItemList != null) {
		fItemList.removeListener(this);
		fItemList = null;
	}
	if (fScrollPane != null) {
		try {
			JViewport viewPort = fScrollPane.getViewport();
			viewPort.remove(fScrollPane);
			viewPort.setBackingStoreEnabled(false);
		} catch (Exception e) { 
			// sometimes a nullpointer  exception occurs inside getViewPort. ignore it
		}
		fScrollPane.setViewportView(null);
		fScrollPane.setViewportBorder(null);
		fScrollPane.setViewport(null);
		fScrollPane.setVisible(false);
		fScrollPane = null;
	}
	fEmptyStateListeners = null;
	super.free();
}
/**
 * Return the Component whose background can be set to show the mandatory property.
 * @return java.awt.Component
 */
protected Component getBackgroundComponent() {
	return getBasicComponent();
}
public Component getBasicComponent() {
	return (Component) fWidget;
}
public Component getComponent() {
	return fScrollPane;
}
protected int getDefaultRowHeight() {
	double d = 1.5 * getBasicComponent().getFont().getSize();
	int height = Math.round(new Double(d).floatValue());
	return height;
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>UIItemList</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see UIItemList
 *
 */
public UIItemList getItemList() {
	return fItemList;
}
/*
  * 	Answer the collection of rowModels currently listening on the receiver
  */
protected UIRowModel[] getRowModels() {
	UIRowModel[] answer = new UIRowModel[fRowModels.size()];
	fRowModels.copyInto(answer);
	return answer;
}
public int[] getSelectedIndices() {
	int[] answer = new int[fSelectedRowIds.length];
	for (int i = 0; i < fSelectedRowIds.length; i++) {
		answer[i] = fItemList.oidToIndex(fSelectedRowIds[i]);
	}
	return answer;
}
public int[] getSelectedRowIds() {
	int[] sel = fWidget.getSelectedIndices();
	if (sel != null) {
		int[] selectedOids = new int[sel.length];
		for (int i = 0; i < sel.length; i++) {
			selectedOids[i] = fItemList.getRowAt(sel[i]).getOid();
		}
		return selectedOids;
	}
	else {
		return new int[0];
	}
}
public IJList getWidget() {
	return fWidget;
}
public void handleEnsureIndexIsVisible(Anything args) {
	getWidget().ensureIndexIsVisible(fItemList.oidToIndex(args.asInt(-1)));
}
protected UIItemList handleMissingItemList(ORBConnection conn, UITableModel tableModel) {
	UITableModel model = tableModel;
	if (model == null) {
		trouble("restoreState", "a list must have either an itemList or listModel assigned");
		// create a default table model instance to allow the view to come up: WHAT FOR?
		model = new UITableModel();
		model.fData = new UiItemCache(model);
		model.setItemList(conn, new Anything());
	}
	return model.getItemList();
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("ensureItemIsVisible")) {
		handleEnsureIndexIsVisible(args);
		return;
	}
	if (request.equals("setModel") || request.equals("setItemList")) {
		handleSetItemList(conn, args, true);
		return;
	}
	if (request.equals("setSelectedItems") || request.equals("setSelectedIndices")) {
		setSelectedItems(args);
		return;
	}
	if (request.equals("setSelectionMode")) {
		setSelectionMode(args.asInt(-1));
		return;
	}
	if (request.equals("setSelectionForeground")) {
		setSelectionForegroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setSelectionBackground")) {
		setSelectionBackgroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setRowHeight")) {
		setRowHeight(args.asInt(0));
		return;
	}
	//if (request.equals("ensureIndexIsVisible")) {
	//	getWidget().ensureIndexIsVisible(args.asInt(-1));
	//	return;
	//}
	if (request.equals("setTrigger")) {
		if (args.isNull())
			fTrigger = null;
		else
			fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args);
		return;
	}
	super.handleRequest(conn, request, args);
}
protected void handleSetItemList(ORBConnection conn, Anything args, boolean notify) {
	if (getItemList() != null)
		getItemList().removeListener(this);
	setItemList(conn, args, notify);
}
/**
 * Answer boolean whether the new selection is indeed different
 */
private boolean hasChanged(int[] newSelection, int[] oldSelection) {
	if (newSelection == oldSelection) return false;
	if (newSelection.length != oldSelection.length) return true;
	for (int i= 0; i < newSelection.length; i++) {
		if (newSelection[i] != oldSelection[i]) return true;
	}
	return false;
}
public void mouseReleased(MouseEvent e) {
	// workaround: mouseClicks are not delivered reliably in Swing 0.5.1; moved code to mouseReleased
	selectionChanged(e.getClickCount() == 2 ? "action" : "select");
	super.mouseReleased(e); //will handle the popup if necessary
}
// IItemListListener interface
public void notify(int type, String[] attributeNames, int start, int end) {
	switch (type) {
		case 99 : //TABLE_MODEL_CONTENTS_CLEARED
		case TABLE_MODEL_CONTENTS_CHANGED :
			clearSelection();
			selectionChanged("select");
		case TABLE_MODEL_ROWS_CHANGED :
			updatePendingSelection(true);
			break;
		case TABLE_MODEL_ROWS_ADDED :
			updatePendingSelection(true);
			break;
		case TABLE_MODEL_CELL_CHANGED :
			break;
		case TABLE_MODEL_ROWS_REMOVED :
			removeFromSelection(start, end);
			break;
	}
	getWidget().notify(type, attributeNames, start, end);
}
protected int oidToIndex(int oid) {
	return fItemList.oidToIndex(oid);
}
private Anything prepareCurrentSelection() {
	Anything selection = new Anything();
	Anything indices = new Anything();
	Anything rowIds = new Anything();
	for (int i = 0; i < fSelectedRowIds.length; i++) {
		int oid = fSelectedRowIds[i];
		if (oid != -1)
			rowIds.append(new Anything(oid));
		int index = oidToIndex(oid);
		indices.append(new Anything(index));
	}
	selection.put("s", indices);
	selection.put("rowids", rowIds);
	return selection;
}
protected void refreshCurrentSelection() {
	IRow[] rows = new IRow[fSelectedRowIds.length];
	for (int i = 0; i < rows.length; i++) {
		int nextOid = fSelectedRowIds[i];
		IRow nextRow = fItemList.getRow(nextOid);
		rows[i] = nextRow;
	}
	refreshSelectedItems(rows);
}
void refreshSelectedItems(IRow[] rows) {
	setSelectedItems(rows);
}
protected void registerRowModels(boolean register) {
	if (getItemList() == null)
		return;
	UIRowModel[] rowModels = getRowModels();
	UiItemCache cache = getItemList().getItemCache();
	if (register) {
		for (int i = 0; i < rowModels.length; i++) {
			cache.addListener(rowModels[i]);
		}
	}
	else {
		for (int i = 0; i < rowModels.length; i++) {
			cache.removeListener(rowModels[i]);
		}
	}
}
/**
 * remove the given component as listener from being informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void removeEnableListener(IEnableListenerTarget component) {
	Assert.isNotNull(component);
	if (fEmptyStateListeners == null)
		return;
	fEmptyStateListeners.removeElement(component);
}
protected void removeFromSelection(int from, int to) {
	int[] selectedRowIds = fSelectedRowIds;
	Vector toBeSelected = new Vector(selectedRowIds.length);
	for (int i = 0; i < selectedRowIds.length; i++) {
		int index = fItemList.oidToIndex(selectedRowIds[i]);
		if (index > -1)
			toBeSelected.addElement(new Integer(selectedRowIds[i]));
	}
	if (toBeSelected.size() > 0) {
		IRow[] newSelection = new IRow[toBeSelected.size()];
		for (int i = 0; i < toBeSelected.size(); i++) {
			newSelection[i] = fItemList.getRow(((Integer) toBeSelected.elementAt(i)).intValue());
		}
		setSelectedItems(newSelection);
	}
	else {
		clearSelection();
	}
}
/*
  * 	In order to have rowModels keep track of tableModel changes in the receiver
  *	the receiver needs to keep track of them to notify when a setModel happens.
  */
protected void removeRowModel(com.ibm.ulc.ui.lists.UIRowModel rowModel) {
	removeSelectionListener(rowModel);
	fRowModels.removeElement(rowModel);
	if (getItemList() != null) {
		getItemList().getItemCache().removeListener(rowModel);
	}
}
/**
 * remove the given component as listener from being informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void removeSelectionListener(ISelectionListener component) {
	Assert.isNotNull(component);
	if (fSelectionListeners == null)
		return;
	fSelectionListeners.removeElement(component);
	if (fSelectionListeners.size() == 0) {
		fSelectionListeners = null;
	}
}
public void requestFocus() {
	if (getWidget() != null) {
		getWidget().requestFocus();
	}
}
protected UIItemList restoreItemList(ORBConnection conn, Anything args) {
	return (UIItemList) getManaged(UIItemList.class, conn, args);
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	setItemList(conn, args, false);
	setComponent(conn, args);
	setSelectionMode(args.get("selectionMode", -1));
	setDimensions(args);
	setInitialSelection(args);
	//fItemList.stopRequest();
	setDecoration(args, true);
	fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args.get("trigger"));
	super.restoreState(conn, args);
}
public void selectionChanged(String event) {
	if (hasChanged(getSelectedRowIds(), fSelectedRowIds) || ("action".equals(event)))
		basicSelectionChanged(event);
}
abstract protected void setComponent(ORBConnection conn, Anything args);
abstract protected void setDimensions(Anything args);
protected void setInitialSelection(Anything args) {
	if (args.isDefined("selectedIndices"))
		setSelectedItems(args.get("selectedIndices"));
}
protected void setItemList(ORBConnection conn, Anything args, boolean notify) {
	UITableModel model = (UITableModel) getManaged(UITableModel.class, conn, args.get("model"));
	UIItemList itemList = restoreItemList(conn, args.get("itemList"));
	if (itemList == null) {
		itemList = handleMissingItemList(conn, model);
	}
	registerRowModels(false);
	setItemList(itemList);
	registerRowModels(true);
	if (notify)
		notify(TABLE_MODEL_CONTENTS_CHANGED, null, -1, -1);
}
protected void setItemList(UIItemList itemList) {
	fItemList = itemList;
	fItemList.addListener(this);
}
public void setRowHeight(int rh) {
	if (rh >= 0)
		getWidget().setRowHeight(rh);
	return;
}
/**
 * This method was created in VisualAge.
 */
public void setSelectedItem(IRow row) {
	setSelectedItems(new IRow[] {row});
}
public void setSelectedItems(IRow[] rows) {
	Vector missingOids = new Vector();
	int[] indices = new int[rows.length];
	for (int i = 0; i < rows.length; i++) {
		indices[i] = fItemList.oidToIndex(rows[i].getOid());
		if (indices[i] == -1) {
			int rowId = rows[i].getOid();
			missingOids.addElement(new Integer(rowId));
		}
	}
	if (missingOids.size() > 0) {
		int[] rowsToRequest = new int[missingOids.size()];
		for (int i = 0; i < missingOids.size(); i++) {
			rowsToRequest[i] = ((Integer) missingOids.elementAt(i)).intValue();
		}
		fItemList.requestOidIndices(rowsToRequest);
		fPendingSelection = rows;
	} else {
		int[][] intervals = new UlcSorter().sortToIntervals(indices);
		getWidget().setSelectionInvervals(intervals, indices);
		selectionChanged("");
		fPendingSelection = new IRow[0];
	}
}
/**
 * This method was created in VisualAge.
 */
public void setSelectedItems(Anything args) {
	Anything rowIds = args.get("rowids");
	IRow[] rows = new IRow[rowIds.size()];
	for (int i = 0; i < rowIds.size(); i++) {
		int nextOid = rowIds.get(i).asInt(-1);
		IRow nextRow = fItemList.getRow(nextOid);
		rows[i] = nextRow;
	}
	setSelectedItems(rows);
}
/**
 * Set the background of all selected items in the receiver 
 * to the color defined by the RGB int values <code>red, green,
 * and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setSelectionBackgroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JScrollPane sp = (JScrollPane) getComponent();
	if (sp != null) {
		JViewport vp = sp.getViewport();
		if (vp != null) {
			IJList jl = (IJList) vp.getView();
			if (jl != null) {
				if (red == -1 || green == -1 || blue == -1) {
					color = UIManager.getColor("List.selectionBackground");
				
				}
				else {
					color = new Color(red, green, blue);
				}
				jl.setSelectionBackground(color);
				if (refresh) {
					refreshComponentColor(vp);
				}
			}
		}
	}
}
/**
 * Set the foreground of all selected items in the receiver 
 * to the color defined by the RGB int values <code>red, green,
 * and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */

public void setSelectionForegroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JScrollPane sp = (JScrollPane) getComponent();
	if (sp != null) {
		JViewport vp = sp.getViewport();
		if (vp != null) {
			IJList jl = (IJList) vp.getView();
			if (jl != null) {
				if (red == -1 || green == -1 || blue == -1) {
					color = UIManager.getColor("List.selectionForeground");
				}
				else {
					color = new Color(red, green, blue);
				}
				jl.setSelectionForeground(color);
				if (refresh) {
					refreshComponentColor(vp);
				}
			}
		}
	}
}
public void setSelectionMode(int mode) {
	getWidget().setSelectionMode(mode);
	refreshCurrentSelection();
}
/*
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
public boolean shouldEnableListener() {
	return getSelectedRowIds().length > 0;
}
public void updatePendingSelection(boolean tryRefreshCurrentSelection) {
	if (fPendingSelection.length > 0) {
		IRow[] pendingSelection = fPendingSelection;
		fPendingSelection = new IRow[0];
		setSelectedItems(pendingSelection);
	}
	else {
		if (tryRefreshCurrentSelection) {
			refreshCurrentSelection();
		}
	}
}
public void updateSelectionListeners(int[] selectedRows) {
	if (fSelectionListeners == null) {
		return;
	}
	for (int i = 0; i < fSelectionListeners.size(); i++) {
		((ISelectionListener) fSelectionListeners.elementAt(i)).updateSelection(selectedRows);
	}
}
}
