package com.ibm.vap.Transactions;


/**
 * TransactionView
 *
 * Licensed Material - Property of IBM 
 * IBM(R) VisualAge(TM) for Java(TM) Version 2.0 
 * (C) Copyright IBM Corp. 1997, 1998 - All Rights Reserved. 
 * US Government Users Restricted Rights - Use, duplication or disclosure 
 * restricted by GSA ADP Schedule Contract with IBM Corp. 
 *
 */
import java.util.Hashtable;
import java.util.Enumeration;
import java.rmi.RemoteException;
import com.ibm.vap.Isolation.TransactionIsolationPolicy;
import com.ibm.vap.common.VapObjectLockedException;
import com.ibm.vap.common.VapMergeFailureException;
import com.ibm.vap.common.VapAccessException;

public class TransactionView extends java.lang.Object
{
	protected Transaction transaction;
	protected Hashtable versions = new Hashtable(30);
	private static final java.lang.String copyright  = "(c) Copyright International Business Machines Corporation, 1998";
/**
 * TransactionView constructor
 */
public TransactionView (Transaction aTransaction)
{
	this.setTransaction(aTransaction);
}
/**
 * Add a new version into the view and into the parent view
 */
public Version addVersion(Version aVersion) throws RemoteException, VapObjectLockedException
{

	this.getIsolationPolicy().notifyRegistrationFor(aVersion);	
	this.primAddVersion(aVersion);
	if (aVersion.isOld())
		this.getParent().addVersionCopy(aVersion);

	return aVersion;
}
/**
 * Add a copy of a new version into the view and into the parent view
 */
protected Version addVersionCopy(Version aVersion)
{
	Version newVersion;
	
	newVersion = this.createReadOnlyVersion();
	newVersion.shallowInitializeAsIsFrom(aVersion);
	this.primAddVersion(newVersion);
	this.getParent().addVersionCopy(newVersion);
	
	return newVersion;
}
/*
 * Create a new read only version
 */
public Version createReadOnlyVersion()
{
	return new ReadOnlyVersion(this);
}
/*
 * Create a new version for the passed BO
 */
public Version createVersion()
{
	return new Version(this);
}
/**
 * Decrement reference count for a version
 */
protected void decrementReferenceCountFor(Version aVersion)
{
}
/**
 * Discard all the resources needed by the view
 */
protected void discard()
{
	this.discardVersions();
	this.setTransaction(null);
}
/**
 * Discard versions
 */
protected void discardVersions()
{
	Enumeration versions;
	Version version;
	
	versions = this.getVersions().elements();
	while (versions.hasMoreElements())
	{
		version = (Version)versions.nextElement();
		if (this.isParentShared())
			this.getParent().decrementReferenceCountFor(version);
		version.discard();
	}
	
	this.setVersions(null);
}
/**
 * Answer transaction's isolation policy
 */
public TransactionIsolationPolicy getIsolationPolicy()
{
	return this.getTransaction().getIsolationPolicy();
}
/**
 * Return view's name
 */
public String getName()
{
	return this.getTransaction().getName();	
}
/**
 * Answer the parent view
 */
protected TransactionView getParent()
{
	return this.getParentTransaction().getView();
}
/**
 * Answer the parent transaction
 */
protected Transaction getParentTransaction()
{
	return this.getTransaction().getParent();
}
/**
 * Answer the transaction of the view
 */
protected Transaction getTransaction()
{
	return transaction;
}
/**
 * Answer a version for notifying or adding dependants. If not found in this view, 
 * copy if from the parent's view
 */
public Version getUnrealizedVersionForRead(OID anOid) throws RemoteException
{
	Version version, parentVersion;

	if ((version = this.primGetVersion(anOid)) == null)
		if ((parentVersion = this.getParent().getUnrealizedVersionForRead(anOid)) != null)
		{
			version = this.createReadOnlyVersion();
			this.getIsolationPolicy().initializeVersion(version,parentVersion);
			version.beUnrealized();
			this.primAddVersion(version);
		}

	return version;
}
/**
 * Answer a version. If not found in this view,
 * copy if from the parent's view
 */
public Version getUnrealizedVersionForUpdate(OID anOid) throws RemoteException
{
	Version version=null, oldVersion;

	if ((oldVersion = this.primGetVersion(anOid)) != null)
		if (!oldVersion.isReadOnlyVersion())
			version = oldVersion;
		else
		{
			version = this.createVersion();
			version.initializeAsIsFrom(oldVersion);
			oldVersion.transferLockTo(version);
			this.primAddVersion(version);
		}
	else
		if((oldVersion = this.getParent().getUnrealizedVersionForUpdate(anOid)) != null)
		{
			version = this.createVersion();
			version.initializeAsSynchronizedFrom(oldVersion);
			version.beUnrealized();
			this.primAddVersion(version);
		}

	return version;
}
/**
 * Answer a version. If not found in this view,
 * copy if from the parent's view
 */
protected Version getVersionForRead(OID anOid) throws RemoteException,VapObjectLockedException
{
	Version version, parentVersion;

	if ((version = this.primGetVersion(anOid)) != null)
	{
		if (version.isUnrealized())
			this.getIsolationPolicy().notifyFirstReadFor(version);
	}
	else
		if ((parentVersion = this.getParent().primGetVersionForRead(anOid)) != null)
		{
			version = this.createReadOnlyVersion();
			this.getIsolationPolicy().initializeVersion(version, parentVersion);
			this.getIsolationPolicy().notifyFirstReadFor(version);
			this.primAddVersion(version);
		}

	if ((version != null) && (version.isUnrealized()))
	{
		version.beRealized();
		this.getParent().markVersionRealized(anOid);
	}

	return version;
}
/**
 * Answer a version. If not found in this view,
 * copy if from the parent's view
 */
protected Version getVersionForUpdate(OID anOid) throws RemoteException,VapObjectLockedException
{
	Version version=null, oldVersion;

	if ((oldVersion = this.primGetVersion(anOid)) != null)
	{
		if (!oldVersion.isReadOnlyVersion())
		{
			version = oldVersion;
			if (version.isUnrealized())
				this.getIsolationPolicy().notifyFirstUpdateFor(version);
		}
		else
		{
			version = this.createVersion();
			version.initializeAsIsFrom(oldVersion);
			oldVersion.transferLockTo(version);
			this.getIsolationPolicy().notifyFirstUpdateFor(version);
			this.primAddVersion(version);
		}
	}
	else
		if ((oldVersion = this.getParent().primGetVersionForRead(anOid)) != null)
		{
			version = this.createVersion();
			version.initializeAsSynchronizedFrom(oldVersion);
			this.getIsolationPolicy().notifyFirstUpdateFor(version);
			this.primAddVersion(version);
		}

	if ((version != null) && version.isUnrealized())
	{
		version.beRealized();
		this.getParent().markVersionRealized(anOid);
	}
	
	return version;
}
/**
 * Answer the versions
 */
public Hashtable getVersions()
{
	return versions;
}
/**
 * Answer true if a view has a version
 */
public boolean hasVersionAt(OID anOID)
{
	return this.primGetVersion(anOID) != null;
}
/**
 * Test if the view's parent is a shared view
 */
public boolean isParentShared( )
{
	return this.getTransaction().isParentShared();
}
/**
 * Test if the view is the top level view
 */
public boolean isTopLevel( )
{
	return this.getTransaction().isTopLevel();
}
/**
 * Mark a version realized
 */
public void markVersionRealized(OID anOid)
{
	Version version;

	if ((version = this.primGetVersion(anOid)) != null)
		if (!version.isRealized())
		{
			version.beRealized();
			this.getParent().markVersionRealized(anOid);
		}
}
/**
 * Merge the view and its child view
 */
protected void merge(TransactionView aChildView)
{
	Enumeration childVersions;
	Version version, childVersion;
	
	childVersions = aChildView.getVersions().elements();
	while (childVersions.hasMoreElements())
	{
		childVersion = (Version) childVersions.nextElement();
		version = this.primGetVersion(childVersion.getVersionIdentifier());
		if (version != null)
			this.mergeOldVersion(version,childVersion);
		else
			this.mergeNewVersion(childVersion);
	}

	this.getTransaction().abtMerged();
}
/**
 * Merge a new child version
 */
protected void mergeNewVersion(Version childVersion)
{
	Version version;

	if (!(childVersion.isRemoved() || childVersion.isInternallySynchronized()))
	{
		version = this.createVersion();
		version.merge(childVersion);
		this.primAddVersion(version);
	}	
}
/**
 * Merge the version and its child version
 */
protected void mergeOldVersion(Version version, Version childVersion)
{
	version.merge(childVersion);
}
/**
 * Merge the view to its parent
 */
protected void mergeToParent()
{
	this.getParent().merge(this);
}
/**
 * Add a new version into the view
 */
protected Version primAddVersion(Version aVersion)
{
	this.getVersions().put(aVersion.getVersionIdentifier(), aVersion);
	return aVersion;
}
/**
 * Answer a version
 */
protected Version primGetVersion(OID anOid)
{
	return (Version) this.getVersions().get(anOid);
}
/**
 * Answer a version. If not found in this view, copy if from the parent's view
 */
public Version primGetVersionForRead(OID anOid) throws RemoteException
{
	Version version, parentVersion;

	if ((version = this.primGetVersion(anOid)) == null)
		if ((parentVersion = this.getParent().primGetVersionForRead(anOid)) != null)
		{
			version = this.createReadOnlyVersion();
			this.getIsolationPolicy().initializeVersion(version,parentVersion);
			this.primAddVersion(version);
		}

	if ((version != null) && (version.isUnrealized()))
	{
		version.beRealized();
		this.getParent().markVersionRealized(anOid);
	}

	return version;
}
/**
 * Answer a version. If not found in this view, copy if from the parent's view
 */
public Version primGetVersionForUpdate(OID anOid) throws RemoteException
{
	Version version=null, oldVersion;

	if ((oldVersion = this.primGetVersion(anOid)) != null)
		if (!oldVersion.isReadOnlyVersion())
			version = oldVersion;
		else
		{
			version = this.createVersion();
			version.initializeAsIsFrom(oldVersion);
			oldVersion.transferLockTo(version);
			this.primAddVersion(version);
		}
	else
		if ((oldVersion = this.getParent().primGetVersionForRead(anOid)) != null)
		{
			version = this.createVersion();
			version.initializeAsSynchronizedFrom(oldVersion);
			this.primAddVersion(version);
		}

	if ((version != null) && (version.isUnrealized()))
	{
		version.beRealized();
		this.getParent().markVersionRealized(anOid);
	}

	return version;
}
/**
 * Remove a version from the view
 */
protected void primRemoveVersion(Version aVersion)
{
	this.getVersions().remove(aVersion.getVersionIdentifier());
	aVersion.discard();
}
/**
 * Release locks acquired in the view
 */
protected void releaseLocks()
{
	Enumeration versions;
	Version version;
	
	versions = this.getVersions().elements();
	while (versions.hasMoreElements())
	{
		version = (Version) versions.nextElement();
		version.releaseLock();
	}	
}
/**
 * Remove a version from the view
 */
protected void removeVersion(OID anOid)
{
	Version version;

	if ((version = this.primGetVersion(anOid)) != null)
		this.primRemoveVersion(version);
}
/**
 * Set the transaction of the view
 */
protected void setTransaction(Transaction aTransaction)
{
	transaction = aTransaction;
}
/**
 * Set the versions
 */
protected void setVersions(Hashtable theVersions)
{
	versions = theVersions;
}
/**
 * Answer the size of the view
 */
public int size()
{
	return this.getVersions().size();
}
/**
 * Test if the view and its child view can merged
 */
public void testMerge(TransactionView aChildView) throws VapMergeFailureException
{
	Enumeration childVersions;
	Version version, childVersion;

	childVersions = aChildView.getVersions().elements();
	while (childVersions.hasMoreElements())
	{
		childVersion = (Version)childVersions.nextElement();
		if ((version = this.primGetVersion(childVersion.getVersionIdentifier())) != null)
			if (!version.canMerge(childVersion))
				throw new VapMergeFailureException(version.getEjbObject());
	}
}
/**
 * Test if the view can be merged to its parent
 */
public void testMergeToParent() throws VapMergeFailureException
{
	this.getParent().testMerge(this);
}
/**
 * Return string representation of the object
 */
public String toString()
{
	return this.getClass().getName() + ":" + this.getName();
}
}