/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * The contents of this file are subject to the Netscape Public License
 * Version 1.0 (the "NPL"); you may not use this file except in
 * compliance with the NPL.  You may obtain a copy of the NPL at
 * http://www.mozilla.org/NPL/
 *
 * Software distributed under the NPL is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the NPL
 * for the specific language governing rights and limitations under the
 * NPL.
 *
 * The Initial Developer of this code under the NPL is Netscape
 * Communications Corporation.  Portions created by Netscape are
 * Copyright (C) 1998 Netscape Communications Corporation.  All Rights
 * Reserved.
 *
 * Contributor(s): 
 *   Pierre Phaneuf <pp@ludusdesign.com>
 */
  #ifdef XP_OS2
#define INCL_WINSHELLDATA
#endif

#include "nsProfileAccess.h"
#include "nsProfile.h"

#include "pratom.h"
#include "prmem.h"
#include "plstr.h"
#include "prenv.h"

#include "nsIEnumerator.h"
#include "prprf.h"
#include "nsSpecialSystemDirectory.h"
#include "nsCOMPtr.h"
#include "nsIComponentManager.h"
#include "nsFileStream.h"
#include "nsEscape.h"
#include "nsDirectoryServiceDefs.h"
#include "nsAppDirectoryServiceDefs.h"
#include "nsILocalFile.h"

#define NS_IMPL_IDS
#include "nsICharsetConverterManager.h"
#include "nsIPlatformCharset.h"
#undef NS_IMPL_IDS

#define MAX_PERSISTENT_DATA_SIZE  1000
#define NUM_HEX_BYTES             8
#define ISHEX(c) ( ((c) >= '0' && (c) <= '9') || ((c) >= 'a' && (c) <= 'f') || ((c) >= 'A' && (c) <= 'F') )

#if defined (XP_UNIX)
#define USER_ENVIRONMENT_VARIABLE "USER"
#define HOME_ENVIRONMENT_VARIABLE "HOME"
#define PROFILE_NAME_ENVIRONMENT_VARIABLE "PROFILE_NAME"
#define PROFILE_HOME_ENVIRONMENT_VARIABLE "PROFILE_HOME"
#elif defined (XP_BEOS)
#endif

#ifdef XP_OS2
#include "nsTextFormatter.h"

#define max_name_length_OS2 32
#define max_directory_length_OS2 128
#define NETSCAPE_VERSION_OS2 "4.61"
#define KEY_PROFILE_NAMES_OS2 "Users"
#define KEY_VERSION "CurrentVersion"
#define DEFAULT_RESULT_OS2 "NOTHING"
#define BUFFER_MAX_LENGTH_OS2 200
#define NETSCAPE_APP "Netscape Navigator"

int NumberProfiles40x(HINI handler);
void ProfilesNames(HINI handler,char** profiles);
void DirectoryNames(HINI handler,int numberprofiles,char** profiles,char** directories);
nsresult ConvertTo (char* name, PRUnichar** result);
nsresult Copy(char* directory, char** result);
#endif

#if defined(XP_PC)
#define WIN_MOZ_REG "mozregistry.dat"
#elif defined(XP_MAC)
#define MAC_MOZ_REG "Mozilla Registry"
#elif defined(XP_UNIX)
#define UNIX_MOZ_REG_FOLDER ".mozilla"
#define UNIX_MOZ_REG_FILE   "registry"
#endif

// IID and CIDs of all the services needed
static NS_DEFINE_CID(kRegistryCID, NS_REGISTRY_CID);
static NS_DEFINE_CID(kCharsetConverterManagerCID, NS_ICHARSETCONVERTERMANAGER_CID);

/*
 * Constructor/Destructor
 * FillProfileInfo reads the registry and fills profileStructs
 */
nsProfileAccess::nsProfileAccess()
{
    m_registry           =  null_nsCOMPtr();
    mCount               =  0;
    mNumProfiles         =  0; 
    mNumOldProfiles      =  0;
    m4xCount             =  0;
    mFixRegEntries       =  PR_FALSE;
    mProfileDataChanged	 =  PR_FALSE;
    mForgetProfileCalled =  PR_FALSE;
    mProfiles            =  new nsVoidArray();
    m4xProfiles          =  new nsVoidArray();

    // Get the profile registry path
    NS_GetSpecialDirectory(NS_XPCOM_APPLICATION_REGISTRY_FILE, getter_AddRefs(mNewRegFile));

    PRBool regDataMoved = PR_FALSE;
    PRBool oldMozRegFileExists = PR_FALSE;

    // Get the old moz registry
    nsCOMPtr<nsIFile> mozRegFile;

#if defined(XP_OS2)
    NS_GetSpecialDirectory(NS_OS2_DIR, getter_AddRefs(mozRegFile));
    if (mozRegFile)
        mozRegFile->Append(WIN_MOZ_REG);
#elif defined(XP_PC)
    NS_GetSpecialDirectory(NS_WIN_WINDOWS_DIR, getter_AddRefs(mozRegFile));
    if (mozRegFile)
        mozRegFile->Append(WIN_MOZ_REG);
#elif defined(XP_MAC)
    NS_GetSpecialDirectory(NS_MAC_PREFS_DIR, getter_AddRefs(mozRegFile));
    if (mozRegFile)
        mozRegFile->Append(MAC_MOZ_REG);
#elif defined(XP_UNIX)
    NS_GetSpecialDirectory(NS_UNIX_HOME_DIR, getter_AddRefs(mozRegFile));
    if (mozRegFile)
    { 
        mozRegFile->Append(UNIX_MOZ_REG_FOLDER);
        mozRegFile->Append(UNIX_MOZ_REG_FILE);
    }
#endif

    // Check if the old profile registry exists, before we decide 
    // on transfering the data.
    if (mozRegFile)
        mozRegFile->Exists(&oldMozRegFileExists);

    if (oldMozRegFileExists) {
        // Check to see if there is a requirement to move the registry data..
        GetMozRegDataMovedFlag(&regDataMoved);

        // If we have not transfered the data from old registry,
        // do it now....
        if (!regDataMoved)
        {
            // Get the data from old moz registry
            FillProfileInfo(mozRegFile);
            // Flush the registry to be on safe side.
            CloseRegistry();

            // Internal data structure now has all the data from old
            // registry. Update the Flush the new registry with this info.
            mProfileDataChanged = PR_TRUE;
            UpdateRegistry(mNewRegFile);
            CloseRegistry();

            // Set the flag in the new registry to indicate that we have 
            // transfered the data from the old registry
            SetMozRegDataMovedFlag(mNewRegFile);

            // Time to clean the internal data structure and make it
            // ready for reading values from the new registry.
            ResetProfileMembers();
        }
    } 
    // Now the new registry is the one with all profile information
    // Read the data into internal data structure....
    FillProfileInfo(mNewRegFile);
}

// On the way out, close the registry if it is 
// still opened and free up the resources.
nsProfileAccess::~nsProfileAccess() 
{
    // Release all resources.
    mNewRegFile = nsnull;
    FreeProfileMembers(mProfiles, mCount);
    FreeProfileMembers(m4xProfiles, m4xCount);
}

// A wrapper function to call the interface to get a platform file charset.
static
nsresult 
GetPlatformCharset(nsAutoString& aCharset)
{
    nsresult rv;

    // we may cache it since the platform charset will not change through application life
    nsCOMPtr <nsIPlatformCharset> platformCharset = do_GetService(NS_PLATFORMCHARSET_CONTRACTID, &rv);
    if (NS_SUCCEEDED(rv) && platformCharset) {
        rv = platformCharset->GetCharset(kPlatformCharsetSel_FileName, aCharset);
    }
    if (NS_FAILED(rv)) {
        aCharset.AssignWithConversion("ISO-8859-1");  // use ISO-8859-1 in case of any error
    }
    return rv;
}

// Apply a charset conversion from the given charset to Unicode for input C string.
static
nsresult 
ConvertStringToUnicode(nsAutoString& aCharset, const char* inString, nsAutoString& outString)
{
    nsresult rv;
    // convert result to unicode
    NS_WITH_SERVICE(nsICharsetConverterManager, ccm, kCharsetConverterManagerCID, &rv);

    if(NS_SUCCEEDED(rv)) {
        nsCOMPtr <nsIUnicodeDecoder> decoder; // this may be cached
        rv = ccm->GetUnicodeDecoder(&aCharset, getter_AddRefs(decoder));

        if(NS_SUCCEEDED(rv) && decoder) {
            PRInt32 uniLength = 0;
            PRInt32 srcLength = nsCRT::strlen(inString);
            rv = decoder->GetMaxLength(inString, srcLength, &uniLength);

            if (NS_SUCCEEDED(rv)) {
                PRUnichar *unichars = new PRUnichar [uniLength];

                if (nsnull != unichars) {
                    // convert to unicode
                    rv = decoder->Convert(inString, &srcLength, unichars, &uniLength);

                    if (NS_SUCCEEDED(rv)) {
                        // Pass back the unicode string
                        outString.Assign(unichars, uniLength);
                    }
                    delete [] unichars;
                }
                else {
                    rv = NS_ERROR_OUT_OF_MEMORY;
                }
            }
        }    
    }
    return rv;
}

// Free up the member profile structs
void
nsProfileAccess::FreeProfileMembers(nsVoidArray *profiles, PRInt32 numElems)
{
    NS_ASSERTION(profiles, "Invalid profiles");

    PRInt32 index = 0;

    ProfileStruct* aProfile;
    if (profiles) {
        for (index = 0; index < numElems; index++)
        {
            aProfile = (ProfileStruct *) profiles->ElementAt(index);

            delete aProfile;
        }

        delete profiles;
    }
}

// Close the registry.
nsresult
nsProfileAccess::CloseRegistry()
{
    m_registry = 0;
    return NS_OK;
}

// Open the registry.
// If already opened, just use it.
nsresult
nsProfileAccess::OpenRegistry(const char* regName)
{
    nsresult rv;
    PRBool openalready = PR_FALSE;

    if (!regName)
        return NS_ERROR_FAILURE;

    if (!m_registry) {
        rv = nsComponentManager::CreateInstance(kRegistryCID,
                                                nsnull,
                                                NS_GET_IID(nsIRegistry),
                                                getter_AddRefs(m_registry));
        if (NS_FAILED(rv)) return rv;
        if (!m_registry) return NS_ERROR_FAILURE;
    }

    // Open the registry
    rv = m_registry->IsOpen( &openalready);
    if (NS_FAILED(rv)) return rv;

    if (!openalready)
        rv = m_registry->Open(regName);   

    return rv;
}

// Given the name of the profile, the structure that
// contains the relavant profile information will be filled.
// Caller must free up the profile struct.
nsresult	
nsProfileAccess::GetValue(const PRUnichar* profileName, ProfileStruct** aProfile)
{
    NS_ASSERTION(profileName, "Invalid profile name");
    NS_ASSERTION(aProfile, "Invalid profile pointer");

    PRInt32 index = 0;
    index = FindProfileIndex(profileName);

    if (index < 0) 
        return NS_ERROR_FAILURE; 

    *aProfile = new ProfileStruct();
    if (!*aProfile)
        return NS_ERROR_OUT_OF_MEMORY;

    ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

    (*aProfile)->profileName        = profileItem->profileName;
    (*aProfile)->profileLocation    = profileItem->profileLocation;
    (*aProfile)->isMigrated         = profileItem->isMigrated;

    if (!profileItem->NCProfileName.IsEmpty())
        (*aProfile)->NCProfileName	= profileItem->NCProfileName;

    if (!profileItem->NCDeniedService.IsEmpty())
        (*aProfile)->NCDeniedService	= profileItem->NCDeniedService;

    if (!profileItem->NCEmailAddress.IsEmpty())
        (*aProfile)->NCEmailAddress	= profileItem->NCEmailAddress;

    if (!profileItem->NCHavePregInfo.IsEmpty())
        (*aProfile)->NCHavePregInfo	= profileItem->NCHavePregInfo;

    return NS_OK;
}

// This method writes all changes to the array of the 
// profile structs. If it is an existing profile, it
// will be updated. If it is a new profile, it gets added
// to the list. 
nsresult
nsProfileAccess::SetValue(ProfileStruct* aProfile)
{
    NS_ASSERTION(aProfile, "Invalid profile");

    PRInt32	index = 0;
    PRBool isNewProfile = PR_FALSE;
    ProfileStruct* profileItem;

    index = FindProfileIndex(aProfile->profileName.GetUnicode());

    if (index >= 0)
    {
        profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));
    }
    else
    {
        isNewProfile = PR_TRUE;

        profileItem	= new ProfileStruct();
        if (!profileItem)
            return NS_ERROR_OUT_OF_MEMORY;

        profileItem->profileName        = aProfile->profileName;
    }

    profileItem->profileLocation = aProfile->profileLocation;

    profileItem->isMigrated = aProfile->isMigrated;

    profileItem->updateProfileEntry = PR_TRUE;

    if (!aProfile->NCProfileName.IsEmpty())
        profileItem->NCProfileName = aProfile->NCProfileName;

    if (!aProfile->NCDeniedService.IsEmpty())
        profileItem->NCDeniedService = aProfile->NCDeniedService;

    if (!aProfile->NCEmailAddress.IsEmpty())
        profileItem->NCEmailAddress = aProfile->NCEmailAddress;

    if (!aProfile->NCHavePregInfo.IsEmpty())
        profileItem->NCHavePregInfo = aProfile->NCHavePregInfo;


    if (isNewProfile) {
        if (!mProfiles)
            mProfiles = new nsVoidArray();

        mProfiles->AppendElement((void*)profileItem);
        mCount++;
    }

    return NS_OK;
}

// Enumerates through the registry for profile
// information. Reads in the data into the array 
// of profile structs. After this, all the callers
// requesting profile info will get thier data from
// profiles array. All the udates will be done to this
// data structure to reflect the latest status.
// Data will be flushed at the end.
nsresult 
nsProfileAccess::FillProfileInfo(nsIFile* regName)
{
    nsresult rv = NS_OK;
    nsXPIDLCString regFile;

    if (regName)
        regName->GetPath(getter_Copies(regFile));

    rv = OpenRegistry(regFile);
    if (NS_FAILED(rv)) return rv;       

    // Enumerate all subkeys (immediately) under the given node.
    nsCOMPtr<nsIEnumerator> enumKeys;
    nsRegistryKey profilesTreeKey;

    nsAutoString registryProfileSubtreeString;
    registryProfileSubtreeString.AssignWithConversion(REGISTRY_PROFILE_SUBTREE_STRING);

    rv = m_registry->GetKey(nsIRegistry::Common, 
                            registryProfileSubtreeString.GetUnicode(), 
                            &profilesTreeKey);

    if (NS_FAILED(rv)) 
    {
        rv = m_registry->AddKey(nsIRegistry::Common, 
                                registryProfileSubtreeString.GetUnicode(), 
                                &profilesTreeKey);
        if (NS_FAILED(rv)) return rv;
    }


    // introducing these tmp variables as nsString variables cannot be passed to
    // the resgitry methods
    nsXPIDLString tmpCurrentProfile;
    nsXPIDLString tmpVersion;
    nsXPIDLString tmpPREGInfo;


    // For the following variables, we do not check for the rv value
    // but check for the variable instead, because it is valid to proceed
    // without the variables having a value. That's why there are no returns 
    // for invalid rv values.

    // Get the current profile
    nsAutoString registryCurrentProfileString;
    registryCurrentProfileString.AssignWithConversion(REGISTRY_CURRENT_PROFILE_STRING);
    rv = m_registry->GetString(profilesTreeKey, 
                               registryCurrentProfileString.GetUnicode(), 
                               getter_Copies(tmpCurrentProfile));

    if (tmpCurrentProfile)
    {
        // If current profile does not exist, mCurrentProfile will not be set
        // This is not harmful, as GetCurrentProfile method needs to return this value
        // And GetCurrentProfile returns:
        //    the current profile if set
        //    the first profile if profiles exist but no current profile is set
        //    an empty string if no profiles exist.

        mCurrentProfile = NS_STATIC_CAST(const PRUnichar*, tmpCurrentProfile);
    }

    // Get the profile version
    {
      nsAutoString registryVersionString;
      registryVersionString.AssignWithConversion(REGISTRY_VERSION_STRING);
      rv = m_registry->GetString(profilesTreeKey, 
                                 registryVersionString.GetUnicode(), 
                                 getter_Copies(tmpVersion));
    }

    if (tmpVersion == nsnull)
    {
        mFixRegEntries = PR_TRUE;
        mVersion.AssignWithConversion(REGISTRY_VERSION_1_0);

        mProfileDataChanged = PR_TRUE;
    }

    // Get the preg info
    {
      nsAutoString registryHavePRegInfoString;
      registryHavePRegInfoString.AssignWithConversion(REGISTRY_HAVE_PREG_INFO_STRING);
      rv = m_registry->GetString(profilesTreeKey, 
                                 registryHavePRegInfoString.GetUnicode(), 
                                 getter_Copies(tmpPREGInfo));
    }

    if (tmpPREGInfo == nsnull)
    {
        mHavePREGInfo.AssignWithConversion(REGISTRY_NO_STRING);

        mProfileDataChanged = PR_TRUE;
    }

    rv = m_registry->EnumerateSubtrees( profilesTreeKey, getter_AddRefs(enumKeys));
    if (NS_FAILED(rv)) return rv;

    rv = enumKeys->First();
    if (NS_FAILED(rv)) return rv;

    mCount = 0;
    mNumProfiles = 0;
    mNumOldProfiles = 0;
    PRBool currentProfileValid = mCurrentProfile.IsEmpty();
    nsAutoString profilePath; 
    
    while( (NS_OK != enumKeys->IsDone()) ) 
    {
        nsCOMPtr<nsISupports> base;

        rv = enumKeys->CurrentItem( getter_AddRefs(base) );
        if (NS_FAILED(rv)) return rv;

        // Get specific interface.
        nsCOMPtr <nsIRegistryNode> node;
        nsIID nodeIID = NS_IREGISTRYNODE_IID;

        rv = base->QueryInterface( nodeIID, getter_AddRefs(node));
        if (NS_FAILED(rv)) return rv;

        // Get node name.
        nsXPIDLString profile;
        nsXPIDLString isMigrated;
        nsXPIDLString NCProfileName;
        nsXPIDLString NCDeniedService;
        nsXPIDLString NCEmailAddress;
        nsXPIDLString NCHavePregInfo;
        nsXPIDLString directory;

        rv = node->GetName(getter_Copies(profile));
        if (NS_FAILED(rv)) return rv;

        nsRegistryKey profKey;								
        rv = node->GetKey(&profKey);
        if (NS_FAILED(rv)) return rv;

        {
          nsAutoString registryDirectoryString;
          registryDirectoryString.AssignWithConversion(REGISTRY_DIRECTORY_STRING);
          rv = m_registry->GetString(profKey, 
                               registryDirectoryString.GetUnicode(), 
                               getter_Copies(directory));
          if (NS_FAILED(rv)) return rv;
        }

        if (mFixRegEntries)
            FixRegEntry(getter_Copies(directory));

#if defined (XP_MAC)
        // Need to store persistent strings on mac..
        PRBool pathChanged = PR_FALSE;
        rv = GetProfilePathString(directory.get(), profile.get(), profilePath, &pathChanged);
        NS_ENSURE_SUCCESS(rv,rv); 

        if (pathChanged)
        {
            *((PRUnichar **)getter_Copies(directory)) = 
                                      nsXPIDLString::Copy(profilePath.GetUnicode());
        }
#endif

        {
          nsAutoString registryMigratedString;
          registryMigratedString.AssignWithConversion(REGISTRY_MIGRATED_STRING);
          rv = m_registry->GetString(profKey, 
                               registryMigratedString.GetUnicode(), 
                               getter_Copies(isMigrated));
          if (NS_FAILED(rv)) return rv;
        }


        // Not checking the return values of these variables as they
        // are for activation, they are optional and their values 
        // do not call for a return
        {
          nsAutoString registryNCProfileNameString;
          registryNCProfileNameString.AssignWithConversion(REGISTRY_NC_PROFILE_NAME_STRING);
          m_registry->GetString(profKey, 
                                registryNCProfileNameString.GetUnicode(), 
                                getter_Copies(NCProfileName));
        }

        {
          nsAutoString registryNCServiceDenialString;
          registryNCServiceDenialString.AssignWithConversion(REGISTRY_NC_SERVICE_DENIAL_STRING);
          m_registry->GetString(profKey, 
                                registryNCServiceDenialString.GetUnicode(), 
                                getter_Copies(NCDeniedService));
        }

        {
          nsAutoString registryNCUserEmailString;
          registryNCUserEmailString.AssignWithConversion(REGISTRY_NC_USER_EMAIL_STRING);
          m_registry->GetString(profKey, 
                                registryNCUserEmailString.GetUnicode(), 
                                getter_Copies(NCEmailAddress));
        }

        {
          nsAutoString registryNCHavePRegInfoString;
          registryNCHavePRegInfoString.AssignWithConversion(REGISTRY_NC_HAVE_PREG_INFO_STRING);
          m_registry->GetString(profKey, 
                                registryNCHavePRegInfoString.GetUnicode(), 
                                getter_Copies(NCHavePregInfo));
        }

        // Make sure that mCurrentProfile is valid
        if (!mCurrentProfile.IsEmpty() && mCurrentProfile.Equals(profile))
          currentProfileValid = PR_TRUE;

        ProfileStruct*  profileItem	= new ProfileStruct();
        if (!profileItem)
            return NS_ERROR_OUT_OF_MEMORY;

        profileItem->updateProfileEntry     = PR_TRUE;

        profileItem->profileName      = NS_STATIC_CAST(const PRUnichar*, profile);
        profileItem->profileLocation  = NS_STATIC_CAST(const PRUnichar*, directory);
        profileItem->isMigrated       = NS_STATIC_CAST(const PRUnichar*, isMigrated);

        if (NCProfileName)
            profileItem->NCProfileName = NS_STATIC_CAST(const PRUnichar*, NCProfileName);

        if (NCDeniedService)
            profileItem->NCDeniedService = NS_STATIC_CAST(const PRUnichar*, NCDeniedService);

        if (NCEmailAddress)
            profileItem->NCEmailAddress = NS_STATIC_CAST(const PRUnichar*, NCEmailAddress);

        if (NCHavePregInfo)
            profileItem->NCHavePregInfo = NS_STATIC_CAST(const PRUnichar*, NCHavePregInfo);


        {
          nsAutoString isMigratedAutoString(isMigrated);
          if (isMigratedAutoString.EqualsWithConversion(REGISTRY_YES_STRING))
              mNumProfiles++;
          else if (isMigratedAutoString.EqualsWithConversion(REGISTRY_NO_STRING))
              mNumOldProfiles++;
        }

        if (!mProfiles) {
            mProfiles = new nsVoidArray();

            if (!mProfiles)
                return NS_ERROR_OUT_OF_MEMORY;
        }

        mProfiles->AppendElement((void*)profileItem);			

        mCount++;

        rv = enumKeys->Next();
    }

    if (!currentProfileValid)
      mCurrentProfile.SetLength(0);
      
    mFixRegEntries = PR_FALSE;
    rv = CloseRegistry();
    return rv;
}

// Return the number of 5x profiles.
// A member variable mNumProfiles is used
// to keep track of 5x profiles. 
void
nsProfileAccess::GetNumProfiles(PRInt32 *numProfiles)
{
    NS_ASSERTION(numProfiles, "Invalid numProfiles");

    *numProfiles = mNumProfiles;
}

// Return the number of 4x (>=4.5 & < 5.0) profiles.
// A member variable mNumOldProfiles is used
// to keep track of 4x profiles. 
void
nsProfileAccess::GetNum4xProfiles(PRInt32 *numProfiles)
{
    NS_ASSERTION(numProfiles, "Invalid numProfiles");

    PRInt32 index = 0;

    *numProfiles = 0;

    for(index = 0; index < mCount; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if (profileItem->isMigrated.EqualsWithConversion(REGISTRY_NO_STRING))
        {
            (*numProfiles)++;
        }
    }

    // ******** This is a HACK -- to be changed later ********
    // When we run mozilla -installer for the second time, mNumOldProfiles is set to 0
    // This happens because MigrateProfileInfo realizes that the old profiles info 
    // already exists in mozRegistry (from the first run of -installer)
    // and does not fill m4xProfiles leaving it empty.

    // A default profile is created if there are 0 number of 4x and 5x profiles.
    // Setting mNumOldProfiles to 0 can result in this side effect if there are no
    // 5x profiles, although there are >0 number of 4x profiles.
    // This side effect would happen in nsProfile::ProcessArgs -- INSTALLER option.

    // So we query the mProfiles array for the latest numOfOldProfiles 
    // This returns the right value and we set mNumOldProfiles to this value
    // This results in the correct behaviour.

    mNumOldProfiles = *numProfiles;
}

// If the application can't find the current profile,
// the first profile will be used as the current profile.
// This routine returns the first 5x profile.
// Caller must free up the string (firstProfile).
void 
nsProfileAccess::GetFirstProfile(PRUnichar **firstProfile)
{
    NS_ASSERTION(firstProfile, "Invalid firstProfile pointer");

    PRInt32 index = 0;

    *firstProfile = nsnull;

    for(index = 0; index < mCount; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if (profileItem->isMigrated.EqualsWithConversion(REGISTRY_YES_STRING))
        {
            *firstProfile = profileItem->profileName.ToNewUnicode();
            break;
        }
    }
}

// Set the current profile. Opearting directly on the tree.
// A separate struct should be maintained for the top level info.
// That way we can eliminate additional registry access. For
// now, we depend on registry operations.
// Capture the current profile information into mCurrentProfile.
void
nsProfileAccess::SetCurrentProfile(const PRUnichar *profileName)
{
    NS_ASSERTION(profileName, "Invalid profile name");

    mCurrentProfile = profileName;
    mProfileDataChanged = PR_TRUE;
}

// Return the current profile value.
// If mCurrent profile is already set, that value is returned.
// If there is only one profile that value is set to CurrentProfile.
void 
nsProfileAccess::GetCurrentProfile(PRUnichar **profileName)
{
    *profileName = nsnull;

    if (!mCurrentProfile.IsEmpty() || mForgetProfileCalled)
    {
        *profileName = mCurrentProfile.ToNewUnicode();
    }

    // If there are profiles and profileName is not
    // set yet. Get the first one and set it as Current Profile.
    if (mNumProfiles > 0 && (*profileName == nsnull))
    {
        GetFirstProfile(profileName);
        SetCurrentProfile(*profileName);
    }
}

// Delete a profile from profile structs
void
nsProfileAccess::RemoveSubTree(const PRUnichar* profileName)
{
    NS_ASSERTION(profileName, "Invalid profile name");

    // delete this entry from the mProfiles array
    // by moving the pointers with something like memmove
    // decrement mCount if it works.
    PRInt32	index = 0;
    PRBool  isOldProfile = PR_FALSE;

    index = FindProfileIndex(profileName);

    if (index >= 0)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if (profileItem->isMigrated.EqualsWithConversion(REGISTRY_NO_STRING))
            isOldProfile = PR_TRUE;

        mProfiles->RemoveElementAt(index);

        mCount--;

        if (isOldProfile)
            mNumOldProfiles--;
        else
            mNumProfiles--;

        if (mCurrentProfile.EqualsWithConversion(profileName))
        {
            mCurrentProfile.SetLength(0);
        }
    }
}

// Fix registry incompatabilities with previous builds
void 
nsProfileAccess::FixRegEntry(PRUnichar** dirName) 
{
    NS_ASSERTION(dirName, "Invalid dirName pointer");

    nsSimpleCharString decodedDirName;
    PRBool haveHexBytes = PR_TRUE;

    // Decode the directory name to return the ordinary string
    nsCAutoString dirNameCString; dirNameCString.AssignWithConversion(*dirName);
    nsInputStringStream stream(dirNameCString);
//  nsInputStringStream stream(NS_ConvertUCS2toUTF8(*dirName));
    nsPersistentFileDescriptor descriptor;

    char bigBuffer[MAX_PERSISTENT_DATA_SIZE + 1];
    // The first 8 bytes of the data should be a hex version of the data size to follow.
    PRInt32 bytesRead = NUM_HEX_BYTES;
    bytesRead = stream.read(bigBuffer, bytesRead);

    if (bytesRead != NUM_HEX_BYTES)
        haveHexBytes = PR_FALSE;

    if (haveHexBytes)
    {
        bigBuffer[NUM_HEX_BYTES] = '\0';
        
        for (int i = 0; i < NUM_HEX_BYTES; i++)
        {
            if (!(ISHEX(bigBuffer[i])))
            {
                haveHexBytes = PR_FALSE;
                break;
            }
        }
    }

    if (haveHexBytes)
    {
        //stream(dirName);
        PR_sscanf(bigBuffer, "%x", (PRUint32*)&bytesRead);
        if (bytesRead > MAX_PERSISTENT_DATA_SIZE)
        {
            // Try to tolerate encoded values with no length header
            bytesRead = NUM_HEX_BYTES + 
                        stream.read(bigBuffer + NUM_HEX_BYTES, 
                             MAX_PERSISTENT_DATA_SIZE - NUM_HEX_BYTES);
        }
        else
        {
            // Now we know how many bytes to read, do it.
            bytesRead = stream.read(bigBuffer, bytesRead);
        }

        // Make sure we are null terminated
        bigBuffer[bytesRead]='\0';
        descriptor.SetData(bigBuffer, bytesRead);				
        descriptor.GetData(decodedDirName);

        nsAutoString dirNameString;
        dirNameString.AssignWithConversion(decodedDirName);
        *dirName = dirNameString.ToNewUnicode();
    }
}   
    
// Return the index of a given profiel from the arraf of profile structs.
PRInt32
nsProfileAccess::FindProfileIndex(const PRUnichar* profileName)
{    
    NS_ASSERTION(profileName, "Invalid profile name");

    PRInt32 retval = -1;
    PRInt32 index = 0;

    for (index=0; index < mCount; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if(profileItem->profileName.EqualsWithConversion(profileName))
        {
            retval = index;
            break;
        }
    }
    return retval;
}

// Flush profile information from the data structure to the registry.
nsresult 
nsProfileAccess::UpdateRegistry(nsIFile* regName)
{
    nsresult rv;
    nsXPIDLCString regFile;

    if (!mProfileDataChanged)
    {
        return NS_OK;
    }

    if (!regName)
    {
        if (mNewRegFile)
            mNewRegFile->GetPath(getter_Copies(regFile));   
    }
    else
    {
        regName->GetPath(getter_Copies(regFile));   
    }

    rv = OpenRegistry(regFile);
    if (NS_FAILED(rv)) return rv;

    // Enumerate all subkeys (immediately) under the given node.
    nsCOMPtr<nsIEnumerator> enumKeys;
    nsRegistryKey profilesTreeKey;

    // Get the major subtree
    {
      nsAutoString registryProfileSubtreeString;
      registryProfileSubtreeString.AssignWithConversion(REGISTRY_PROFILE_SUBTREE_STRING);
      rv = m_registry->GetKey(nsIRegistry::Common, 
                              registryProfileSubtreeString.GetUnicode(), 
                              &profilesTreeKey);
      if (NS_FAILED(rv)) return rv;
    }

    // Set the current profile
    if (!mCurrentProfile.IsEmpty()) {

        nsAutoString registry_current_profile_string;
        registry_current_profile_string.AssignWithConversion(REGISTRY_CURRENT_PROFILE_STRING);
        rv = m_registry->SetString(profilesTreeKey, 
                                   registry_current_profile_string.GetUnicode(), 
                                   mCurrentProfile.GetUnicode());
        if (NS_FAILED(rv)) return rv;
    }

    // Set the registry version
    {
      nsAutoString registry_version_string;
      registry_version_string.AssignWithConversion(REGISTRY_VERSION_STRING);
      rv = m_registry->SetString(profilesTreeKey, 
                                 registry_version_string.GetUnicode(), 
                                 mVersion.GetUnicode());
      if (NS_FAILED(rv)) return rv;
    }

    // Set preg info
    {
      nsAutoString registry_have_preg_info_string;
      registry_have_preg_info_string.AssignWithConversion(REGISTRY_HAVE_PREG_INFO_STRING);
      rv = m_registry->SetString(profilesTreeKey, 
                                 registry_have_preg_info_string.GetUnicode(), 
                                 mHavePREGInfo.GetUnicode());
      if (NS_FAILED(rv)) return rv;
    }

    rv = m_registry->EnumerateSubtrees(profilesTreeKey, getter_AddRefs(enumKeys));
    if (NS_FAILED(rv)) return rv;

    rv = enumKeys->First();
    if (NS_FAILED(rv)) return rv;
    
    nsAutoString persistentPath;
    while( (NS_OK != enumKeys->IsDone()) ) 
    {
        nsCOMPtr<nsISupports> base;

        rv = enumKeys->CurrentItem( getter_AddRefs(base) );
        if (NS_FAILED(rv)) return rv;

        // Get specific interface.
        nsCOMPtr <nsIRegistryNode> node;
        nsIID nodeIID = NS_IREGISTRYNODE_IID;

        rv = base->QueryInterface( nodeIID, getter_AddRefs(node));
        if (NS_FAILED(rv)) return rv;

        // Get node name.
        nsXPIDLString profile;
        nsXPIDLString isMigrated;
        nsXPIDLString directory;

        rv = node->GetName( getter_Copies(profile) );
        if (NS_FAILED(rv)) return rv;

        PRInt32 index = 0;

        index = FindProfileIndex(profile);

        if (index < 0)
        {
            // This profile is deleted.
            rv = m_registry->RemoveKey(profilesTreeKey, profile);
            if (NS_FAILED(rv)) return rv;
        }
        else
        {
            nsRegistryKey profKey;								

            ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

            rv = node->GetKey(&profKey);
            if (NS_FAILED(rv)) return rv;

            {
              nsAutoString registry_directory_string;
              registry_directory_string.AssignWithConversion(REGISTRY_DIRECTORY_STRING);

#if defined (XP_MAC)
              rv = SetProfilePathString(profileItem->profileLocation, profileItem->profileName, persistentPath);  
              NS_ENSURE_SUCCESS(rv,rv); 
              rv = m_registry->SetString(profKey, 
                                  registry_directory_string.GetUnicode(), 
                                  persistentPath.GetUnicode());
              if (NS_FAILED(rv)) return rv;
#else
              rv = m_registry->SetString(profKey, 
                                  registry_directory_string.GetUnicode(), 
                                  profileItem->profileLocation.GetUnicode());
              if (NS_FAILED(rv)) return rv;
#endif
            }

            {
              nsAutoString registry_migrated_string;
              registry_migrated_string.AssignWithConversion(REGISTRY_MIGRATED_STRING);
              rv = m_registry->SetString(profKey, 
                                  registry_migrated_string.GetUnicode(), 
                                  profileItem->isMigrated.GetUnicode());
              if (NS_FAILED(rv)) return rv;
            }

            {
              nsAutoString registry_nc_profile_name_string;
              registry_nc_profile_name_string.AssignWithConversion(REGISTRY_NC_PROFILE_NAME_STRING);
              m_registry->SetString(profKey, 
                                registry_nc_profile_name_string.GetUnicode(),  
                                profileItem->NCProfileName.GetUnicode());
            }

            {
              nsAutoString registry_nc_service_denial_string;
              registry_nc_service_denial_string.AssignWithConversion(REGISTRY_NC_SERVICE_DENIAL_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_service_denial_string.GetUnicode(), 
                                    profileItem->NCDeniedService.GetUnicode());
            }

            {
              nsAutoString registry_nc_user_email_string;
              registry_nc_user_email_string.AssignWithConversion(REGISTRY_NC_USER_EMAIL_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_user_email_string.GetUnicode(), 
                                    profileItem->NCEmailAddress.GetUnicode());
            }

            {
              nsAutoString registry_nc_have_preg_info_string;
              registry_nc_have_preg_info_string.AssignWithConversion(REGISTRY_NC_HAVE_PREG_INFO_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_have_preg_info_string.GetUnicode(), 
                                    profileItem->NCHavePregInfo.GetUnicode());
            }

            profileItem->updateProfileEntry = PR_FALSE;
        }
        rv = enumKeys->Next();
    }

    // Take care of new nodes
    for (int i = 0; i < mCount; i++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(i));

        if (profileItem->updateProfileEntry)
        {
            nsRegistryKey profKey;								

            rv = m_registry->AddKey(profilesTreeKey, 
                                    profileItem->profileName.GetUnicode(), 
                                    &profKey);
            if (NS_FAILED(rv)) return rv;

            {
              nsAutoString registry_directory_string;
              registry_directory_string.AssignWithConversion(REGISTRY_DIRECTORY_STRING);

#if defined (XP_MAC)
              rv = SetProfilePathString(profileItem->profileLocation, profileItem->profileName, persistentPath);  
              NS_ENSURE_SUCCESS(rv,rv); 
              rv = m_registry->SetString(profKey, 
                                  registry_directory_string.GetUnicode(), 
                                  persistentPath.GetUnicode());
              if (NS_FAILED(rv)) return rv;
#else
              rv = m_registry->SetString(profKey, 
                                    registry_directory_string.GetUnicode(), 
                                    profileItem->profileLocation.GetUnicode());
              if (NS_FAILED(rv)) return rv;
#endif
            }

            {
              nsAutoString registry_migrated_string;
              registry_migrated_string.AssignWithConversion(REGISTRY_MIGRATED_STRING);
              rv = m_registry->SetString(profKey, 
                                         registry_migrated_string.GetUnicode(), 
                                         profileItem->isMigrated.GetUnicode());
              if (NS_FAILED(rv)) return rv;
            }

            {
              nsAutoString registry_nc_profile_name_string;
              registry_nc_profile_name_string.AssignWithConversion(REGISTRY_NC_PROFILE_NAME_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_profile_name_string.GetUnicode(), 
                                    profileItem->NCProfileName.GetUnicode());
            }

            {
              nsAutoString registry_nc_service_denial_string;
              registry_nc_service_denial_string.AssignWithConversion(REGISTRY_NC_SERVICE_DENIAL_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_service_denial_string.GetUnicode(), 
                                    profileItem->NCDeniedService.GetUnicode());
            }

            {
              nsAutoString registry_nc_user_email_string;
              registry_nc_user_email_string.AssignWithConversion(REGISTRY_NC_USER_EMAIL_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_user_email_string.GetUnicode(), 
                                    profileItem->NCEmailAddress.GetUnicode());
            }

            {
              nsAutoString registry_nc_have_preg_info_string;
              registry_nc_have_preg_info_string.AssignWithConversion(REGISTRY_NC_HAVE_PREG_INFO_STRING);
              m_registry->SetString(profKey, 
                                    registry_nc_have_preg_info_string.GetUnicode(), 
                                    profileItem->NCHavePregInfo.GetUnicode());
            }

            profileItem->updateProfileEntry = PR_FALSE;
        }
    }

    rv = CloseRegistry();
    mProfileDataChanged = PR_FALSE;

    return rv;
}

// Return the list of profiles, 4x and 5x.
// For 4x profiles text "- migrate" is appended
// to inform the JavaScript about the migration status.
void
nsProfileAccess::GetProfileList(PRUnichar **profileListStr)
{
    NS_ASSERTION(profileListStr, "Invalid profileListStr pointer");

    nsAutoString profileList;

    for (PRInt32 index=0; index < mCount; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if (index != 0)
        {
            profileList.AppendWithConversion(",");
        }
        profileList += profileItem->profileName;

        if (profileItem->isMigrated.EqualsWithConversion(REGISTRY_NO_STRING))
            profileList.AppendWithConversion(" - migrate");
    }

    *profileListStr = profileList.ToNewUnicode();
}

// Return a boolean based on the profile existence.
PRBool
nsProfileAccess::ProfileExists(const PRUnichar *profileName)
{
    NS_ASSERTION(profileName, "Invalid profile name");

    PRBool exists = PR_FALSE;

    for (PRInt32 index=0; index < mCount; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));
        if (profileItem->profileName.EqualsWithConversion(profileName))
        {
            exists = PR_TRUE;
            break;
        }
    }
    return exists;
}

// Capture the 4x profile information from the old registry (4x)
nsresult
nsProfileAccess::Get4xProfileInfo(const char *registryName)
{
    NS_ASSERTION(registryName, "Invalid registryName");

    nsresult rv = NS_OK;
    mNumOldProfiles = 0;

    nsAutoString charSet;
    rv = GetPlatformCharset(charSet);
    if (NS_FAILED(rv)) return rv;

#if defined(XP_PC) || defined(XP_MAC)
    nsCOMPtr <nsIRegistry> oldReg;
    rv = nsComponentManager::CreateInstance(kRegistryCID,
                                            nsnull,
                                            NS_GET_IID(nsIRegistry),
                                            getter_AddRefs(oldReg));
    if (NS_FAILED(rv)) return rv;

    rv = oldReg->Open(registryName);
    if (NS_FAILED(rv)) return rv;

    // Enumerate 4x tree and create an array of that information.
    // Enumerate all subkeys (immediately) under the given node.
    nsCOMPtr<nsIEnumerator> enumKeys;

    rv = oldReg->EnumerateSubtrees(nsIRegistry::Users,
                                   getter_AddRefs(enumKeys));
    if (NS_FAILED(rv)) return rv;

    rv = enumKeys->First();
    if (NS_FAILED(rv)) return rv;

    // Enumerate subkeys till done.
    while( (NS_OK != enumKeys->IsDone())) 
    {
        nsCOMPtr<nsISupports> base;
        rv = enumKeys->CurrentItem(getter_AddRefs(base));
        if (NS_FAILED(rv)) return rv;

        // Get specific interface.
        nsCOMPtr <nsIRegistryNode> node;
        nsIID nodeIID = NS_IREGISTRYNODE_IID;
        rv = base->QueryInterface( nodeIID, getter_AddRefs(node));
        if (NS_FAILED(rv)) return rv;

        nsXPIDLString profile;
        rv = node->GetName(getter_Copies(profile));
        if (NS_FAILED(rv)) return rv;

        PRBool exists = PR_FALSE;;
        exists = ProfileExists(profile);
        if (exists)
        {		
            rv = enumKeys->Next();
            if (NS_FAILED(rv)) return rv;

            continue;
        }

        nsRegistryKey key;								
        rv = node->GetKey(&key);
        if (NS_FAILED(rv)) return rv;

        nsXPIDLString profLoc;
        nsAutoString locString;
        locString.AssignWithConversion("ProfileLocation");
        
        rv = oldReg->GetString( key, locString.GetUnicode(), getter_Copies(profLoc));
        if (NS_FAILED(rv)) return rv;
		
        ProfileStruct*	profileItem  = new ProfileStruct();
        if (!profileItem)
            return NS_ERROR_OUT_OF_MEMORY;

        profileItem->updateProfileEntry    = PR_TRUE;

        // Unescape is done on the profileName to interpret special characters like %, _ etc.
        // For example something like %20 would probably be interpreted as a space
        // There is some problem I guess in sending a space as itself

#if defined(XP_MAC)
        // 4.x profiles coming from japanese machine are already in unicode.
        // So, there is no need to decode into unicode further.
        
        // Unescape profile name.
        nsCAutoString temp; 
        temp = (const char*) NS_ConvertUCS2toUTF8(profile);
        nsCAutoString profileName(nsUnescape( NS_CONST_CAST(char*, temp.GetBuffer())));
        nsAutoString convertedProfName((const PRUnichar*) NS_ConvertUTF8toUCS2(profileName));

        // Unescape profile location
        nsCAutoString tempLoc; 
        tempLoc = (const char*) NS_ConvertUCS2toUTF8(profLoc);
        nsCAutoString profileLocation(nsUnescape( NS_CONST_CAST(char*, tempLoc.GetBuffer())));
        nsAutoString convertedProfLoc((const PRUnichar*) NS_ConvertUTF8toUCS2(profileLocation));
#else
        nsCAutoString temp; temp.AssignWithConversion(profile);

        nsCAutoString profileName(nsUnescape( NS_CONST_CAST(char*, temp.GetBuffer())));
        nsAutoString convertedProfName;
        ConvertStringToUnicode(charSet, profileName.GetBuffer(), convertedProfName);

        // Unescape profile location and convert it to the right format
        nsCAutoString tempLoc; tempLoc.AssignWithConversion(profLoc);

        nsCAutoString profileLocation(nsUnescape( NS_CONST_CAST(char*, tempLoc.GetBuffer())));
        nsAutoString convertedProfLoc;
        ConvertStringToUnicode(charSet, profileLocation.GetBuffer(), convertedProfLoc);
#endif

        profileItem->profileName  = convertedProfName;
        profileItem->profileLocation = convertedProfLoc;

        profileItem->isMigrated.AssignWithConversion(REGISTRY_NO_STRING);

        if (!m4xProfiles)
            m4xProfiles = new nsVoidArray();

        m4xProfiles->AppendElement((void*)profileItem);

        mNumOldProfiles++;

        //delete profileItem;

        rv = enumKeys->Next();
        if (NS_FAILED(rv)) return rv;
    }

#elif defined (XP_BEOS)
#else
/* XP_UNIX */
        char *unixProfileName = PR_GetEnv(PROFILE_NAME_ENVIRONMENT_VARIABLE);
        char *unixProfileDirectory = PR_GetEnv(PROFILE_HOME_ENVIRONMENT_VARIABLE);

        if (!unixProfileName || 
            !unixProfileDirectory || 
            (PL_strlen(unixProfileName) == 0) || 
            (PL_strlen(unixProfileDirectory) == 0)) 
        {
            unixProfileName = PR_GetEnv(USER_ENVIRONMENT_VARIABLE);
            unixProfileDirectory = PR_GetEnv(HOME_ENVIRONMENT_VARIABLE);
        }

        PRBool exists = PR_FALSE;;
        exists = ProfileExists(NS_ConvertASCIItoUCS2(unixProfileName).GetUnicode());
        if (exists)
        {		
            return NS_OK;
        }

        if (unixProfileName && unixProfileDirectory) {
            nsCAutoString profileLocation(unixProfileDirectory);
            profileLocation += "/.netscape";
            nsCOMPtr<nsIFileSpec> users4xDotNetscapeDirectory;

            rv = NS_NewFileSpec(getter_AddRefs(users4xDotNetscapeDirectory));
            if (NS_FAILED(rv)) return rv;

            rv = users4xDotNetscapeDirectory->SetNativePath((const char *)profileLocation);
            if (NS_FAILED(rv)) return rv;
            rv = users4xDotNetscapeDirectory->Exists(&exists);

            if (NS_FAILED(rv)) return rv;

#ifdef DEBUG
            printf("%s exists:  %d\n",profileLocation.GetBuffer(), exists);
#endif
            if (exists) {
                ProfileStruct*  profileItem     = new ProfileStruct();
                if (!profileItem)
                    return NS_ERROR_OUT_OF_MEMORY;

                profileItem->updateProfileEntry = PR_TRUE;

                profileItem->profileName = NS_ConvertASCIItoUCS2(nsUnescape(unixProfileName)).ToNewUnicode();
                profileItem->profileLocation = NS_ConvertASCIItoUCS2(profileLocation).ToNewUnicode();
                profileItem->isMigrated = NS_ConvertASCIItoUCS2(REGISTRY_NO_STRING).ToNewUnicode();

                if (!m4xProfiles)
                    m4xProfiles = new nsVoidArray();

                m4xProfiles->AppendElement((void*)profileItem);

                mNumOldProfiles++;

				//delete profileItem;
            }
            else {
#ifdef DEBUG
                printf("no 4.x profile\n");
#endif
            }
        }
#endif /* XP_UNIX */

    m4xCount = mNumOldProfiles;

    if (m4xCount > 0) {
        UpdateProfileArray();
    }

    return rv;
}
 



//------------------------------------>Code for migration of Netscapa 4.04 profiles on OS/2
#ifdef XP_OS2   //To migrate nestcape 4.04 profiles this feature is for OS/2 only
// Capture the 4.0x profile information from nscp.ini (4.0x)
nsresult
nsProfileAccess::Get40xProfileInfo(const char *iniFile)
{
    NS_ASSERTION(iniFile, "Invalid registryName");

    char pszFileName[max_directory_length_OS2] = {'\0'};
    strcpy(pszFileName,iniFile);

    HAB hab;
    HINI handler = PrfOpenProfile(hab,pszFileName);

    nsresult rv = NS_OK;
    mNumOldProfiles = 0;
    int numberprofiles = NumberProfiles40x(handler);
    if(!numberprofiles) return rv;
    int counter = 0,j = 0;

    char** profiles;
    char** directories;

    profiles = new char*[numberprofiles];
    if(!profiles) return NS_ERROR_OUT_OF_MEMORY;
    directories = new char*[numberprofiles];
    if(!directories) return NS_ERROR_OUT_OF_MEMORY;
    for( j=0 ; j<numberprofiles ; j++)
    {
        profiles[j] = new char[max_name_length_OS2];
        directories[j] = new char[max_directory_length_OS2];
        if(!profiles[j] || !directories[j]) return NS_ERROR_OUT_OF_MEMORY;
    }

    ProfilesNames(handler,profiles);
    DirectoryNames(handler,numberprofiles,profiles,directories);

    PrfCloseProfile(handler);

    // Enumerate subkeys till done.
    while( counter<numberprofiles )
    {
        nsXPIDLString profile;
        rv = ConvertTo(profiles[counter],getter_Copies(profile));
        if(NS_FAILED(rv))
        {
           for( j=0 ; j<numberprofiles ; j++)
           {
                delete profiles[j];
                delete directories[j];
           }
           delete profiles;
           delete directories;
           return rv;
        }

        PRBool exists = PR_FALSE;;
        exists = ProfileExists(profile);
        if (exists)
        {
            counter++;
            continue;
        }

        nsXPIDLCString profLoc;
        rv = Copy(directories[counter],getter_Copies(profLoc));
        if(NS_FAILED(rv)) return rv;

#if defined(DEBUG_profile)
    printf("oldProflie Location = %s\n", profLoc);
#endif

        ProfileStruct*	profileItem  = new ProfileStruct();
        if (!profileItem)
            return NS_ERROR_OUT_OF_MEMORY;

        profileItem->updateProfileEntry    = PR_TRUE;

        // Unescape is done on the profileName to interpret special characters like %, _ etc.
        // For example something like %20 would probably be interpreted as a space
        // There is some problem I guess in sending a space as itself

        nsCAutoString temp; temp.AssignWithConversion(profile);
        nsAutoString unescpPrfName;
        unescpPrfName.AssignWithConversion(nsUnescape( NS_CONST_CAST(char*, temp.GetBuffer()) ));

        profileItem->profileName  = unescpPrfName;

        profileItem->profileLocation.AssignWithConversion(profLoc);
        profileItem->isMigrated.AssignWithConversion(REGISTRY_NO_STRING);

        if (!m4xProfiles)
            m4xProfiles = new nsVoidArray();

        m4xProfiles->AppendElement((void*)profileItem);

        mNumOldProfiles++;
        counter++;

        //delete profileItem;

    }


    m4xCount = mNumOldProfiles;

    if (m4xCount > 0) {
        UpdateProfileArray();
    }

    for( j=0 ; j<numberprofiles ; j++)
    {
        delete profiles[j];
        delete directories[j];
    }
    delete profiles;
    delete directories;

    return rv;
}


//---------------------------->Look for the latest version of netscape
PRBool
nsProfileAccess::GetVersion(const char *iniFile)
{
char pszFileName[max_directory_length_OS2] = {'\0'};
strcpy(pszFileName,iniFile);
HAB hab;
HINI handler = PrfOpenProfile(hab,pszFileName);
char pBuffer[_MAX_LENGTH] = {'\0'};
PrfQueryProfileString(handler,(PSZ)NETSCAPE_APP,(PSZ)KEY_VERSION,(PSZ)DEFAULT_RESULT_OS2,(PVOID)pBuffer,(LONG)sizeof(pBuffer));
char temp [_MAX_LENGTH] = {'\0'};
strncpy(temp,pBuffer,strlen(NETSCAPE_VERSION_OS2));
temp[strlen(NETSCAPE_VERSION_OS2)] = '\0';
PrfCloseProfile(handler);
if(!strcmp(temp,NETSCAPE_VERSION_OS2) || !handler) {return PR_TRUE;}
return PR_FALSE;
}


//------------------------>Convert the char* name to PRUnichar
nsresult ConvertTo (char* name, PRUnichar** result)
{
    PRUnichar widestrFormat[] = {PRUnichar('%'),PRUnichar('s'),PRUnichar(0)};

    if(result==nsnull) return NS_ERROR_NULL_POINTER;

    *result = nsTextFormatter::smprintf(widestrFormat,name);

    if(!*result) return NS_ERROR_OUT_OF_MEMORY;

    return NS_OK;
}


//----------------------->Look for the number of old profiles
int NumberProfiles40x(HINI handler)
{
     char pBuffer[BUFFER_MAX_LENGTH_OS2] = {'\0'};
     PrfQueryProfileString(handler,(PSZ)KEY_PROFILE_NAMES_OS2,NULL,(PSZ)DEFAULT_RESULT_OS2,(PVOID)pBuffer,(LONG)sizeof(pBuffer));
     int numberprofiles = 0,j = 0,m = 0;
     for( j=0; j<BUFFER_MAX_LENGTH_OS2-1 ; j++){
        if(pBuffer[j]==pBuffer[j+1]&&pBuffer[j]==NULL)
                break;
        while(pBuffer[j]!=NULL){
             j++;
        }
        numberprofiles++;
     }
     if(!strcmp(pBuffer,DEFAULT_RESULT_OS2))
        {
        return 0;
        }
     return numberprofiles;
}


//----------------------->Look for the names of all the old profiles
void ProfilesNames(HINI handler,char** profiles)
{
     char pBuffer[BUFFER_MAX_LENGTH_OS2] = {'\0'};
     PrfQueryProfileString(handler,(PSZ)KEY_PROFILE_NAMES_OS2,NULL,(PSZ)DEFAULT_RESULT_OS2,(PVOID)pBuffer,(LONG)sizeof(pBuffer));
     int j = 0,i = 0,m = 0;
     for( j=0; j<BUFFER_MAX_LENGTH_OS2-1 ; j++)
     {
        if(pBuffer[j]==pBuffer[j+1]&&pBuffer[j]==NULL)
                break;
        m = 0;
        while(pBuffer[j]!=NULL){
                profiles[i][m] = pBuffer[j];
                j++;
                m++;
        }
        profiles[i][m] = '\0';
        i++;
     }
}


//--------------------------->Look for the name of the directories of the old profiles
void DirectoryNames(HINI handler,int numberprofiles,char** profiles,char** directories)
{
     char* pszKey = NULL;
     char pBuffer[BUFFER_MAX_LENGTH_OS2] = {'\0'};
     int i = 0,j = 0;
     for (j=0 ; j<numberprofiles ; j++)
     {
        pszKey = new char[strlen(profiles[j])+1];
        strcpy(pszKey,profiles[j]);
        pszKey[strlen(profiles[j])] = '\0';
        PrfQueryProfileString(handler,(PSZ)KEY_PROFILE_NAMES_OS2,(PSZ)pszKey,(PSZ)DEFAULT_RESULT_OS2,(PVOID)pBuffer,(LONG)sizeof(pBuffer));
        i = 0;
        while(pBuffer[i]!=NULL){
                directories[j][i] = pBuffer[i];
                i++;
        }
        directories[j][i] = '\0';
        delete pszKey;
     }
}


//-------------------------------->Copy the directory in the XPIDLCString
nsresult Copy(char* directory, char** result)
{
     *result = nsCRT::strdup(directory);
     if(!*result) return NS_ERROR_OUT_OF_MEMORY;
     return NS_OK;
}

#endif
//---------------------------------------------------->End of code for 4.04 migration






// Update the mozregistry with the 4x profile names
// and thier locations. Entry REGISTRY_MIGRATED_STRING is set to REGISTRY_NO_STRING
// to differentiate these profiles from 5x profiles.
nsresult
nsProfileAccess::UpdateProfileArray()
{
    nsresult rv = NS_OK;

    for (PRInt32 idx = 0; idx < m4xCount; idx++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (m4xProfiles->ElementAt(idx));		


        PRBool exists;
        exists = ProfileExists(profileItem->profileName.GetUnicode());
        if (NS_FAILED(rv)) return rv;

        // That profile already exists...
        // move on.....
        if (exists) {
            continue;
        }

        nsCOMPtr<nsILocalFile> locProfileDir (do_CreateInstance(NS_LOCAL_FILE_CONTRACTID));
        rv = locProfileDir->InitWithUnicodePath((profileItem->profileLocation).GetUnicode());
        if (NS_FAILED(rv)) return rv;
        
        nsXPIDLString profileDirString;        
        rv = locProfileDir->GetUnicodePath(getter_Copies(profileDirString));
        if (NS_FAILED(rv)) return rv;

        if (NS_SUCCEEDED(rv) && profileDirString)
        {
            profileItem->profileLocation = profileDirString;
            SetValue(profileItem);
        }
    }
    mProfileDataChanged = PR_TRUE;
    return rv;
}

// Set the PREG flag to indicate if that info exists
void
nsProfileAccess::SetPREGInfo(const char* pregInfo)
{
    NS_ASSERTION(pregInfo, "Invalid pregInfo");

    // This is always going to be just a yes/no string
    mHavePREGInfo.AssignWithConversion(pregInfo);
}

//Get the for PREG info.
void 
nsProfileAccess::CheckRegString(const PRUnichar *profileName, char **info)
{
    NS_ASSERTION(profileName, "Invalid profile name");
    NS_ASSERTION(info, "Invalid info pointer");

    *info = nsnull;
    PRInt32 index = 0;

    index = FindProfileIndex(profileName);

    if (index >= 0 )
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));

        if (!profileItem->NCHavePregInfo.IsEmpty()) {
            nsCAutoString pregC;
            pregC.AssignWithConversion(profileItem->NCHavePregInfo);
            *info = nsCRT::strdup(NS_STATIC_CAST(const char*, pregC));
        }
        else
            *info = nsCRT::strdup(REGISTRY_NO_STRING);
    }
}

// Get the flag that from the new reigstry which indicates that it  
// got the transfered data from old mozilla registry
nsresult
nsProfileAccess::GetMozRegDataMovedFlag(PRBool *isDataMoved)
{
    nsresult rv = NS_OK;
    nsXPIDLCString regFile;

    nsRegistryKey profilesTreeKey;
    nsXPIDLString tmpRegDataMoved;
        
    nsAutoString mozRegDataMovedString;
    mozRegDataMovedString.AssignWithConversion(REGISTRY_MOZREG_DATA_MOVED_STRING);
    nsAutoString registryProfileSubtreeString;
    registryProfileSubtreeString.AssignWithConversion(REGISTRY_PROFILE_SUBTREE_STRING);

    if (mNewRegFile)
        mNewRegFile->GetPath(getter_Copies(regFile));   

    rv = OpenRegistry(regFile);
    if (NS_FAILED(rv)) return rv;

    rv = m_registry->GetKey(nsIRegistry::Common, 
                            registryProfileSubtreeString.GetUnicode(), 
                            &profilesTreeKey);

    if (NS_SUCCEEDED(rv)) 
    {
        rv = m_registry->GetString(profilesTreeKey, 
                         mozRegDataMovedString.GetUnicode(), 
                         getter_Copies(tmpRegDataMoved));
         
        nsAutoString isDataMovedString(tmpRegDataMoved);
        if (isDataMovedString.EqualsWithConversion(REGISTRY_YES_STRING))
            *isDataMoved = PR_TRUE;
    }
    else
    {
        rv = m_registry->AddKey(nsIRegistry::Common, 
                                registryProfileSubtreeString.GetUnicode(), 
                                &profilesTreeKey);
    }
    CloseRegistry();
    return rv;        
}

// Set the flag in the new reigstry which indicates that it  
// got the transfered data from old mozilla registry
nsresult
nsProfileAccess::SetMozRegDataMovedFlag(nsIFile* regName)
{
    nsresult rv = NS_OK;
    nsXPIDLCString regFile;

    if (regName)
        regName->GetPath(getter_Copies(regFile));   

    nsRegistryKey profilesTreeKey;
    nsXPIDLString tmpRegDataMoved;
        
    nsAutoString mozRegDataMovedString;
    mozRegDataMovedString.AssignWithConversion(REGISTRY_MOZREG_DATA_MOVED_STRING);
    nsAutoString registryProfileSubtreeString;
    registryProfileSubtreeString.AssignWithConversion(REGISTRY_PROFILE_SUBTREE_STRING);
    nsAutoString regYesString;
    regYesString.AssignWithConversion(REGISTRY_YES_STRING);

    rv = OpenRegistry(regFile);
    if (NS_FAILED(rv)) return rv;

    rv = m_registry->GetKey(nsIRegistry::Common, 
                            registryProfileSubtreeString.GetUnicode(), 
                            &profilesTreeKey);

    if (NS_SUCCEEDED(rv)) 
    {
        rv = m_registry->SetString(profilesTreeKey, 
                         mozRegDataMovedString.GetUnicode(), 
                         regYesString.GetUnicode());
    }
    CloseRegistry();
    return rv;        
}

// Clear the profile member data structure 
// We need to fill in the data from the new registry
nsresult
nsProfileAccess::ResetProfileMembers()
{
    FreeProfileMembers(mProfiles, mCount);
    mProfiles = new nsVoidArray();
    mCount = 0;
    return NS_OK;
}

nsresult
nsProfileAccess::DetermineForceMigration(PRBool *forceMigration)
{
	if (!forceMigration) return NS_ERROR_NULL_POINTER;

	*forceMigration = PR_FALSE;

	if (mNumProfiles > 0) {
		// we have some 6.0 profiles, don't force migration:
		return NS_OK;
	}

	// even if we don't any 4.x profiles, running -installer is safe.  so do it
	*forceMigration = PR_TRUE;
	return NS_OK;
}

// Get the unicode path from the persistent descriptor string
nsresult
nsProfileAccess::GetProfilePathString(const PRUnichar *dirString, const PRUnichar *profileName, nsString& dirPath, PRBool *pathChanged)
{
    nsresult rv = NS_OK;

    // Initialize the boolean
    *pathChanged = PR_FALSE;

    nsLiteralString path(dirString);

    PRInt32 firstColon = path.FindChar(PRUnichar(':'));
    // If there is no colon in the path, on Mac, we have a persistent descriptor 
    // string. Get the path out of it.
    if ( firstColon == -1 )
    {
        // Get localfile and get unicode path 
        nsCOMPtr<nsILocalFile> localFile( do_CreateInstance(NS_LOCAL_FILE_CONTRACTID, &rv));
        NS_ENSURE_SUCCESS(rv, rv);
        if ( localFile )
        {
            rv = localFile->SetPersistentDescriptor(NS_ConvertUCS2toUTF8(path));

            // If persistent opeartion to set the profile dir failed,
            // We need to build a default path here..
            if (NS_FAILED(rv))
            { 
                rv = CreateDefaultProfileFolder(profileName, getter_AddRefs(localFile));
                NS_ENSURE_SUCCESS(rv, rv);
            }

            nsXPIDLString convertedPath;
            rv = localFile->GetUnicodePath(getter_Copies(convertedPath));
            NS_ENSURE_SUCCESS(rv, rv);

            dirPath = convertedPath.get();
            *pathChanged = PR_TRUE;
        }
    }
    return rv; 
}

// Set the persistent descriptor string from unicode path 
nsresult
nsProfileAccess::SetProfilePathString(const nsString& dirPath, const nsString& profileName, nsString& persistentPath)
{
    nsresult rv = NS_OK;

    // Get localfile and set unicode path 
    nsCOMPtr<nsILocalFile> localFile( do_CreateInstance(NS_LOCAL_FILE_CONTRACTID, &rv));
    NS_ENSURE_SUCCESS(rv, rv);
    if ( localFile )
    {
        rv = localFile->InitWithUnicodePath(dirPath.GetUnicode());
        NS_ENSURE_SUCCESS(rv, rv);

        // If the user deletes the directory on the disk for some reason,
        // we need to create it at this point from the registry entry as
        // GetPresistent Descriptor expects the directory to exist...
        PRBool exists;
        rv = localFile->Exists(&exists);
        NS_ENSURE_SUCCESS(rv, rv);
        
        // we will  be dumping all default contents in the getProfileDir()
        // after detecting that this is an empty folder...
        if (!exists)
        {
            rv = localFile->Create(nsIFile::DIRECTORY_TYPE, 0775);

            // If create fails because user has removed the volume
            // or some other reason, create a unique dir in default location 
            // NS_APP_USER_PROFILES_ROOT_DIR always returns something valid.
            // If the volume that had Documents folder removed, then we get 
            // the Documents of the startup volume.
            if (NS_FAILED(rv))
            {
                // Create a profile folder in default profile location..
                rv = CreateDefaultProfileFolder(profileName.GetUnicode(), getter_AddRefs(localFile));
                NS_ENSURE_SUCCESS(rv, rv);
            }
        }
                
        nsXPIDLCString convertedPath;
        rv = localFile->GetPersistentDescriptor(getter_Copies(convertedPath));
        NS_ENSURE_SUCCESS(rv, rv);

        persistentPath = NS_ConvertUTF8toUCS2(convertedPath);
    }
    return rv; 
}

// We need to create a default folder with given profile name here...
nsresult
nsProfileAccess::CreateDefaultProfileFolder(const PRUnichar* profileName, nsILocalFile** newProfileDir)
{
    NS_ASSERTION(profileName, "Invalid Profile name");
    NS_ENSURE_ARG_POINTER(newProfileDir);

    nsresult rv;
    
    // Get default user profiles location
    nsCOMPtr<nsIFile> defaultDir;    
    rv = NS_GetSpecialDirectory(NS_APP_USER_PROFILES_ROOT_DIR, getter_AddRefs(defaultDir));
    NS_ENSURE_SUCCESS(rv, rv);

    // append profile name
    rv = defaultDir->AppendUnicode(profileName);
    NS_ENSURE_SUCCESS(rv, rv);

    // Create unique dir
    rv = defaultDir->CreateUnique(NS_ConvertUCS2toUTF8(profileName), 
                                         nsIFile::DIRECTORY_TYPE, 0775);
    NS_ENSURE_SUCCESS(rv, rv);
    
    rv = defaultDir->QueryInterface(NS_GET_IID(nsILocalFile), (void **) newProfileDir);
    NS_ENSURE_SUCCESS(rv, rv);
    
    return NS_OK;
}

#ifdef XP_OS2_FIX
// Return the list of 4x profiles
nsresult
nsProfileAccess::Get4xProfileList(PRUint32 *length, PRUnichar ***result)
{
    NS_ENSURE_ARG_POINTER(length);
    *length = 0;
    NS_ENSURE_ARG_POINTER(result);
    *result = nsnull;

    nsresult rv = NS_OK;    
    PRInt32 count, localLength = 0;
    PRUnichar **outArray, **next;
    
    GetNum4xProfiles(&count);
    
    next = outArray = (PRUnichar **)nsMemory::Alloc(count * sizeof(PRUnichar *));
    if (!outArray)
        return NS_ERROR_OUT_OF_MEMORY;
        
    for (PRInt32 index=0; index < mCount && localLength < count; index++)
    {
        ProfileStruct* profileItem = (ProfileStruct *) (mProfiles->ElementAt(index));
        
        if (profileItem->isMigrated.EqualsWithConversion(REGISTRY_YES_STRING))
            continue;

        *next = profileItem->profileName.ToNewUnicode();
        if (*next == nsnull)
        {
            rv = NS_ERROR_OUT_OF_MEMORY;
            break;
        }
        next++;
        localLength++;
    }
    
    if (NS_SUCCEEDED(rv))
    {
        *result = outArray;
        *length = localLength;
    }
    else
    {
        while (--next >= outArray)
            nsMemory::Free(*next);
        nsMemory::Free(outArray);
    }
    
    return rv;
}
#endif /* XP_OS2_FIX */
