;;;
;;;   ekn.lisp
;;;   Convert output of astl into EKN 
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;        Copyright (c) 1993  Alan W Black (awb@ed.ac.uk)
;;;
;;; This file is part of an implementation of ASTL a computational
;;; situation theoretic language.  This implementation of ASTL is free
;;; software; you can redistribute it and/or modify it under the terms
;;; of the GNU General Public License as published by the Free
;;; Software Foundation; either version 1, or (at your option) any
;;; later version.
;;;
;;; ASTL is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts
;;; responsibility to anyone for the consequences of using it or for
;;; whether it serves any particular purpose or works at all, unless
;;; he says so in writing.  Refer to the GNU Genral Public License for
;;; full details.
;;;
;;; Everyone is granted permission to copy, modify and redistribute
;;; this file, but only under the conditions described in the
;;; GNU General Public License.   A copy of this license is
;;; supposed to have been given to you along with this program so you
;;; can know your rights and responsibilities.  It should be in a
;;; file named COPYING.  Among other things, the copyright notice
;;; and this notice must be preserved on all copies.
;;;
;;; The author gratefully acknowledges financial support for the
;;; research this implementation is based on from an SERC studnetship
;;; (number 89313458) and Esprit Basic Research Action Project 6852
;;; (DYANA-2).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Alan W Black
;;;   Dept of AI Edinburgh
;;;   (awb@ed.ac.uk)
;;;   June 1991
;;;
;;;   Takes the standard output from a chart proof and converts
;;;   its to a string in EKN.  EKN is a box type notation for
;;;   situation theoretic objects.
;;;
;;;   I hate to say this but I should probably make this work in
;;;   emacs lisp too.
;;;
;;;   29th Oct 92  Make it deal with type terms, this is an absolute hack
;;;     as there isn't really a way to distinguish types from some fact
;;;     terms.
;;;

(lisp:in-package 'user)

;;;
;;; Basic EKN-object structure
;;;
(defstruct eknobj height width printform)

;;;
;;; Top function
;;;
(defun ekn (term)
  "Takes a term and changes it to string containing a printed representation
of the ekn boxes"
  (ekn-realise
   (ekn-convert term)))

(defun ekn-convert (term &optional donelist)
  "Converts the given term to a box-object which has dimensions,
ekn-realise actually prints this as boxes.  I had hoped to do this in
two passes, this one doing the gets the dimensions and the realising
changing it to strings with boxes in them but I don't think that helps."
  (cond ((null donelist) (setq donelist (list (cons (list t) t)))))
  (cond
    ((assoc term donelist)
     (cdr (assoc term donelist)))
    ((and (situation-p term)
	  (assoc (situation-name term) donelist))
     ;; sometime the situation structure isn't preserved
     ;; properly so depend on the name
     (cdr (assoc (situation-name term) donelist)))
    ((assoc term scp-sit-points)
     ;; A new situation not expanded yet
     ;; push the name of this on the done list so we only expand it once
     (let ((simpleform (make-eknobj
			:width (length (string term))
			:height 1 :printform (list (string term)))))
       (si-push-onto (cons term simpleform) donelist)
       (ekn-sit (string term)
		(mapcar
		 #'(lambda (fact)
		     (ekn-convert fact donelist))
		 (reverse
		  (ekn-remove-hushed
		   (sit-point-facts
		    (cdr (assoc term scp-sit-points)))))))))
    ((situation-p term)
     ;; A new situation not expanded yet
     ;; push the name of this on the done list so we only expand it once
     (let ((simpleform (make-eknobj
			:width (length (string (situation-name term)))
			:height 1
			:printform (list (string (situation-name term))))))
       (si-push-onto (cons term simpleform) donelist)
       (si-push-onto (cons (situation-name term)
			   simpleform) donelist)
       (ekn-sit (string (situation-name term))
		(mapcar
		 #'(lambda (fact)
		     (ekn-convert fact donelist))
		 (reverse
		  (or
		    (and (assoc (situation-name term) scp-sit-points)
			 (ekn-remove-hushed
			  (sit-point-facts
			   (cdr (assoc (situation-name term)
				       scp-sit-points)))))
		    (ekn-remove-hushed (situation-tableau term))))))))
    ((si-sitvarp term)
     (let ((simpleform (make-eknobj
			:width (length (format nil "~A" (variable-name term)))
			:height 1
			:printform
			(list (format nil "~A" (variable-name term))))))
       (si-push-onto (cons term simpleform) donelist)
       (si-push-onto (cons (variable-name term)
			   simpleform) donelist)
       (ekn-sit (car (eknobj-printform simpleform))
		(mapcar
		 #'(lambda (fact)
		     (ekn-convert fact donelist))
		 (ekn-remove-hushed
		  (variable-facts term))))))
    ((variable-p term)
     (let ((sform (format nil "~A" (variable-name term))))
       (make-eknobj :width (length sform)
		    :height 1
		    :printform (list sform))))
    ((and (listp term)            ;; hackily check if this is a type
	  (not (equal 1 (car (last term))))
	  (not (equal 0 (car (last term)))))
       (ekn-type
	(string-upcase (format nil "~S" (car term)))     ;; parameter
	(mapcar
	 #'(lambda (fact)
	     (ekn-convert fact donelist))
	 (ekn-remove-hushed
	  (cadr term)))))
    ((atom term)
     ;; individual of some sort
     (let (sform)
       (cond
	((member term st.parameters)
	 ;; output in upper case
	 (setq sform (string-upcase (format nil "~S" term))))
	(t   ;; not a parameter
	 ;; output in its own case (lower)
	 (setq sform (format nil "~S" term))))
       (make-eknobj :width (length sform)
		    :height 1
		    :printform (list sform))))
    ((listp term)
     ;; a fact (I hope)
     (let ((subforms (mapcar
		      #'(lambda (thingy) 
			  (ekn-convert thingy donelist))
		      term)))
       (make-eknobj :width (ekn-fact-width subforms)
		    :height (ekn-fact-height subforms)
		    :printform (ekn-fact-printform subforms))))
    (t
     (let ((sform (format nil "???~A???" term)))
       (make-eknobj :width (length sform)
		    :height 1
		    :printform (list sform))))))

(defun ekn-sit (sitname subforms)
  "Returns an ekn object for this situation.  Which is a boxed
form with the situation name in the corners and the facts one under
each other."
  (let ((sit-width (+ 2
		      (max
		       (apply #'max (cons 0 (mapcar #'eknobj-width subforms)))
		       (+ 6 (length sitname)))))
	(sit-height
	 (+ 4 (apply #'+ (cons 0 (mapcar #'eknobj-height subforms))))))
    (make-eknobj
     :width sit-width
     :height sit-height
     :printform
     (append
      (list (make-string sit-width :initial-element #\=))
      (list (concatenate 'string
			 "| " sitname " |"
			 (make-string (- sit-width
					 (+ 5 (length sitname)))
				      :initial-element #\ )
			 "|"))
      (list (concatenate 'string
			 "|"
			 (make-string (+ 3 (length sitname))
				      :initial-element #\-)
			 (make-string (- sit-width
					 (+ 5 (length sitname)))
				      :initial-element #\ )
			 "|"))
      (mapcan
       #'(lambda (subform)
	   (mapcar
	    #'(lambda (line)
		(concatenate 'string
			     "|" line
			     (make-string (- sit-width (+ 2 (length line)))
					  :initial-element #\ )
			     "|"))
	    (eknobj-printform subform)))
       subforms)
      (list (make-string sit-width :initial-element #\=))))))

(defun ekn-type (param subforms)
  "Returns an ekn object for this situation type.  Which is a boxed form
with the parameter in the corner and the facts one under each other,
and a little box at the top for the parameter."
  (let* ((sit-width (+ 2
		      (max
		       (apply #'max (cons 0 (mapcar #'eknobj-width subforms)))
		       (+ 6 (length param)))))
	(sit-height
	 (+ 6 (apply #'+ (cons 0 (mapcar #'eknobj-height subforms)))))
	(indent (round (/ (- sit-width
			     (+ (length param) 4)) 2))))
    (make-eknobj
     :width sit-width
     :height sit-height
     :printform
     (append
      (list (make-string sit-width :initial-element #\-))
      (list (concatenate 'string
			 "| "
			 (make-string indent
				      :initial-element #\ )
			 param
			 (make-string (- sit-width
					 (+ 4 (length param)
					    indent))
				      :initial-element #\ )
			 " |"))
      (list  (concatenate 'string
			  "|"
			  (make-string (- sit-width 2)
				       :initial-element #\-)
			  "|"))
      (list (concatenate 'string
			 "| " param " |"
			 (make-string (- sit-width
					 (+ 5 (length param)))
				      :initial-element #\ )
			 "|"))
      (list (concatenate 'string
			 "|"
			 (make-string (+ 3 (length param))
				      :initial-element #\-)
			 (make-string (- sit-width
					 (+ 5 (length param)))
				      :initial-element #\ )
			 "|"))
      (mapcan
       #'(lambda (subform)
	   (mapcar
	    #'(lambda (line)
		(concatenate 'string
			     "|" line
			     (make-string (- sit-width (+ 2 (length line)))
					  :initial-element #\ )
			     "|"))
	    (eknobj-printform subform)))
       subforms)
      (list (make-string sit-width :initial-element #\-))))))
  
(defun ekn-fact-width (subforms)
  "Returns the width of the ekn object that will be build up from
the given subforms.  The size is their width plus a left and right
paren, commas between the arguments and a space at the start and end."
  (+ 2 2                           ;  parens, spaces
     (- (length subforms) 3)       ;  commas between arguments
     (apply #'+ (mapcar #'eknobj-width subforms))))

(defun ekn-fact-height (subforms)
  "Returns the height of the ekn object.  This is the highest object."
  (apply #'max (mapcar #'eknobj-height subforms)))

(defun ekn-fact-printform (subforms)
  "Build the print form for the given fact.  A print form is a
list of strings which when printed with each string underneath each other
will look cute."
  (ekn-build-printform
   (ekn-fact-width subforms)
   (ekn-fact-height subforms)
   (append
    (list " ")
    (if (string-equal "1" (car (eknobj-printform (car (last subforms)))))
	(list " ")
	(list "~"))
    (list (car subforms))          ; relation
    (list "(")
    (list (cadr subforms))         ; first argument
    (mapcan                        ; the rest of the arguments
     #'(lambda (arg)
	 (list "," arg)) (butlast (cddr subforms)))
    (list ") "))))

(defun ekn-build-printform (width height printforms)
  "Return a list of strings which when printed will look good."
  (let ((printframe
	 (mapcar #'(lambda (i)
		     i
		     (make-string width :initial-element #\ ))
		 (make-list height))))
    (ekn-print printframe 0 0 printforms)))

(defun ekn-print (printframe x y printforms)
  "Print each printform into the given printframe."
  (cond
   ((null printforms)
    printframe)
   ((stringp (car printforms))
    (ekn-print-in-frame printframe x y (list (car printforms)))
    (ekn-print printframe (+ x (length (car printforms))) y (cdr printforms)))
   ((eknobj-p (car printforms))
    (ekn-print-in-frame
     printframe x y (eknobj-printform (car printforms)))
    (ekn-print
     printframe (+ x (eknobj-width (car printforms))) y (cdr printforms)))
   (t
    (error "Unkown printform"))))

(defun ekn-print-in-frame (printframe x y printform)
  "Prints the printform, list of strings into printframe at x y,
counting from top left."
  (cond
   ((null printform)
    printframe)
   (t               ; there is probably a better way to do this 
    (rplaca (nthcdr y printframe)
	    (concatenate
	     'string
	     (subseq (nth y printframe) 0 x)
	     (car printform)
	     (subseq (nth y printframe) (+ x (length (car printform))))))
    (ekn-print-in-frame printframe x (1+ y) (cdr printform)))))

(defun ekn-realise (eknobj)
  "Print out each part with a newline at the end."
  (terpri)
  (mapc
   #'(lambda (line)
       (princ line) (terpri))
   (eknobj-printform eknobj))
  t)

(defun ekn-remove-hushed (facts)
  "Removes facts which have a relation name that is declared hushed."
  (remove-if
   #'(lambda (fact)
       (member (car fact) st.hush))
   facts))

