;;;
;;;   File: si-chart.lisp
;;;   Chart theorem prover
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;        Copyright (c) 1993  Alan W Black (awb@ed.ac.uk)
;;;
;;; This file is part of an implementation of ASTL a computational
;;; situation theoretic language.  This implementation of ASTL is free
;;; software; you can redistribute it and/or modify it under the terms
;;; of the GNU General Public License as published by the Free
;;; Software Foundation; either version 1, or (at your option) any
;;; later version.
;;;
;;; ASTL is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts
;;; responsibility to anyone for the consequences of using it or for
;;; whether it serves any particular purpose or works at all, unless
;;; he says so in writing.  Refer to the GNU Genral Public License for
;;; full details.
;;;
;;; Everyone is granted permission to copy, modify and redistribute
;;; this file, but only under the conditions described in the
;;; GNU General Public License.   A copy of this license is
;;; supposed to have been given to you along with this program so you
;;; can know your rights and responsibilities.  It should be in a
;;; file named COPYING.  Among other things, the copyright notice
;;; and this notice must be preserved on all copies.
;;;
;;; The author gratefully acknowledges financial support for the
;;; research this implementation is based on from an SERC studnetship
;;; (number 89313458) and Esprit Basic Research Action Project 6852
;;; (DYANA-2).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Alan W Black
;;;   Dept of AI Edinburgh
;;;   (awb@ed.ac.uk)
;;;   May 1991
;;;
;;;   A chart which holds situated facts (propositions) and uses
;;;   a set of constraints over these facts to prove queries about
;;;   the whole.  Basically this is a chart parser used for theorem
;;;   proving (I believe this is called the tableau method).
;;;
;;;   A standard chart (with vertices) is also included to deal
;;;   with grammar rules.  
;;;
;;;   Bugs:
;;;    directly cyclic constraints cause loops
;;;

(lisp:in-package 'user)

(defvar debug.scp nil)   ; set to non-nil makes the chart stop at each edge
(defvar scp-agenda nil)
(defvar scp-chart nil)
(defvar scp-sit-points nil)
(defvar scp-prop-vertex nil)      ; vertex for all non-uttsits
(defvar scp-complete-queries nil)
(defvar st.paramcount 0)

;;;
;;; Each edge represents a proposition.  Complete edges (null
;;; conditions) represent proved propositions.  Sit-points are used to
;;; collect propositions of the same situation together.  The type is
;;; the original type specified for the situation.  Facts may contain
;;; facts other than those that are in the original type i.e those
;;; that have been added by constraints.
;;;
;;; Start and end are set to the special vertex scp-prop-vertex normal
;;; proposition edges but set in sentence edges
;;;
(defstruct edge start end situation fact conditions bindings info)
(defstruct vertex name active-in rules finalp inactive-out)
(defstruct sit-point name type facts constraints start end)
(defstruct proposition situation start end fact)

;;;
;;;  Some macros
;;;
(defmacro scp-proved-edgep (edge) `(null (edge-conditions ,edge)))
(defmacro scp-proving-edgep (edge) `(edge-conditions ,edge))
(defmacro scp-sentence-edgep (edge)
  `(null (eq scp-prop-vertex (edge-start ,edge))))
(defmacro scp-empty-index ()
  `(list nil (cons nil nil) (cons nil nil) (cons nil nil) ))

;;;
;;;   Main function
;;;
(defun scp-prove (query)
;;;
;;; Takes a query (typed situation -- list of propositions) and tries
;;; to prove it with restpect to the current basesituations and
;;; constraints.
;;;
  (let (n-query)
    (scp-initchart)   ; reset all the globals
    (setq n-query (si-unrefsits query))
    (scp-buildinitchart)
    (scp-buildinitagenda n-query scp-prop-vertex)
    (scp-chartprove)
    (scp-findsolutions)
    ))

(defun scp-parse (sentence query)
;;;
;;; Takes a sentence and tries to parse it with respect to the current
;;; grammar.  This looks up the words in st.lexentries and adds basic
;;; propostion edges.  What about the topcat ???
;;;
  (let (initvertex n-query)
    (scp-initchart)
    (setq n-query (si-unrefsits query))
    (scp-buildinitchart)
    (setq initvertex (scp-buildinitsentence sentence 0))
    (scp-buildinitagenda n-query initvertex)
    (scp-chartprove)
    (scp-find-parses initvertex)
    ))

(defun scp-findsolutions ()
;;;
;;; Find all answers to the given query.  These are defined to be all
;;; situations in proved edges with info query.  Returns a list of
;;; situations/bindings paits of all edges that are completed queries.
;;; Duplicate situations are removed from the list.
;;;
  (remove-duplicates
   (mapcar
    #'(lambda (edge)
	(situation-name (edge-situation edge)))
    scp-complete-queries)))

(defun scp-find-parses (initvertex)
;;;
;;; Find all situations of all proved edges that start at initvertex
;;; and go to an edge marked as final.
;;;
  (remove-duplicates
   (mapcar
    #'(lambda (edge)
	(situation-name (edge-situation edge)))
    scp-complete-queries)))

(defun scp-initchart ()
;;;
;;; Initialise global structures.
;;;
  (setq scp-sit-points nil)
  (setq scp-chart nil)
  (setq scp-agenda nil)
  (setq scp-complete-queries nil)
  (setq st.paramcount 0)
)

(defun scp-buildinitsentence (sentence v-name)
;;;
;;; Build the initial well-formed substring table for the given sentence.
;;;
  (cond
   ((null sentence)
    ;; last one
    (make-vertex :name v-name
		 :active-in (scp-empty-index)
		 :inactive-out (scp-empty-index)
		 :finalp t))
   (t
    (let ((start (make-vertex
		  :name v-name
		  :active-in (scp-empty-index)
		  :inactive-out (scp-empty-index)))
	  (end (scp-buildinitsentence (cdr sentence) (1+ v-name))))
      (mapc
       #'(lambda (entry)
	   (let* ((newentry (st_build_situation
			     (intern (string (gensym "SIT")))
			     (situation-type entry)))
		  (basefacts (si-realiseterm
			      (situation-tableau newentry)
			      (si-emptybinds)
			      ;;; ??? what ???
			      scp-prop-vertex scp-prop-vertex
			      (si-emptybinds)))
		  (newsit (scp-make-sit-point
			   (situation-name newentry)
			   basefacts nil
			   (situation-type newentry)
			   start end)))
	     ;; don't use make-facts so that these never go on the agenda
	     (mapc
	      #'(lambda (fact)
		  (scp-addedge
		   (make-edge
		    :start start
		    :end end
		    :situation newentry
		    :fact fact
		    :conditions nil
		    :bindings (si-emptybinds)
		    :info 'word-entry)))
	      basefacts)))
       (cdr (assoc (car sentence) st.lexentries)))
      start))))

(defun scp-buildinitagenda (query vertex)
;;;
;;; Build the initial goals on the agenda from the query
;;;
  (setq scp-agenda
	(list (scp-buildinitedge query vertex))))

(defun scp-buildinitedge (situation vertex)
;;;
;;; Build an active edge requiring the given proposition.
;;;
  (let ((newedge (make-edge)))
    (setf (edge-situation newedge) situation)
    (setf (edge-conditions newedge)
	  (mapcar #'(lambda (fact)
		      (make-proposition
		       :situation (situation-name situation)
		       :start vertex
		       :end (make-variable :name 'query-end)
		       :fact fact))
		  (situation-tableau situation)))
    (setf (edge-start newedge) vertex)
    (setf (edge-end newedge) vertex)
    (setf (edge-bindings newedge) (si-emptybinds))
    (setf (edge-info newedge) 'query)
    newedge))

(defun scp-buildinitchart ()
;;;
;;; Add basic situations to the chart as complete edges.
;;;
  (setq scp-prop-vertex
	(make-vertex
	 :name 'prop-vertex
	 :active-in (scp-empty-index)
	 :inactive-out (scp-empty-index)))
  (mapc
   #'(lambda (basesit)
       (let* ((basefacts (si-realiseterm
			  (situation-tableau basesit)
			  (si-emptybinds)
			  scp-prop-vertex scp-prop-vertex
			  (si-emptybinds)))
	      (newsit (scp-make-sit-point
		       (situation-name basesit)
		       basefacts nil
		       (situation-type basesit)
		       scp-prop-vertex scp-prop-vertex)))
	 (mapcar
	  #'(lambda (fact)
	      (let ((newedge (make-edge)))
		(setf (edge-conditions newedge) nil)
		(setf (edge-bindings newedge) (si-emptybinds))
		(setf (edge-situation newedge) basesit)
		(setf (edge-fact newedge) fact)
		(setf (edge-start newedge) scp-prop-vertex)
		(setf (edge-end newedge) scp-prop-vertex)
		(setf (edge-info newedge) 'basesit)
		(scp-addedge newedge)))
	  basefacts)))
   st.basesits))

(defun scp-chartprove ()
;;;
;;; This is the main loop.  Loops until the agenda is empty.  This
;;; could be changed to loop until the first answer is found.
;;;
  (let (current)
    (loop 
     (if (setq current (scp-selectedge))
	 (scp-combine current)
	 (return t)))))

(defun scp-selectedge ()
;;;
;;; Returns an edge from the agenda and removes it from the agenda.
;;;
   (let ((new (car scp-agenda)))
     (setq scp-agenda (cdr scp-agenda))
     new))

(defun scp-combine (agendaedge)
;;;
;;; Combine this edge with appropriate ones in the chart
;;;
  (if debug.scp (break))
  (cond
   ((scp-proving-edgep agendaedge)
    ;; proving edge looking for proved edge
    (let ((search-vertex
	   ;; ??? probably wrong ???
	   (si-deref (proposition-start
		      (car (edge-conditions agendaedge)))
		     (edge-bindings agendaedge))))
      (mapc
	#'(lambda (edge)
	    (scp-checkcombine edge agendaedge))
	(scp-find-index-no-var
	 (vertex-inactive-out search-vertex)
	 (si-deref
	  (proposition-situation (car (edge-conditions agendaedge)))
	  (edge-bindings agendaedge))
	 (car (proposition-fact (car (edge-conditions agendaedge))))))
      (scp-propose (car (edge-conditions agendaedge))
		   (edge-bindings agendaedge)
		   (scp-sentence-edgep agendaedge))
      (scp-addedge agendaedge)))
   ((scp-proved-edgep agendaedge)
    ;; proved edge looking for proving edge
     (scp-find-index
      (vertex-active-in (or scp-prop-vertex (edge-start agendaedge)))
      (situation-name (edge-situation agendaedge))
      (car (edge-fact agendaedge))
      #'(lambda (edge)
	  (scp-checkcombine agendaedge edge)))
    (scp-addedge agendaedge))
   (t (error "Unknown type of agenda edge"))))

(defun scp-addedge (edge)
;;;
;;; Adds the given edge to the chart updating any indexes as
;;; appropriate.
;;;
  (cond
   ((scp-proved-edgep edge)
    (setf (vertex-inactive-out (edge-start edge))
	  (scp-add-to-index
	   (vertex-inactive-out (edge-start edge))
	   (situation-name (edge-situation edge))
	   (car (edge-fact edge))
	   edge))
    (if (not (eq (edge-start edge) scp-prop-vertex))
	(setf (vertex-inactive-out scp-prop-vertex)
	      (scp-add-to-index
               (vertex-inactive-out scp-prop-vertex)
	       (situation-name (edge-situation edge))
	       (car (edge-fact edge))
	       edge))))
   ((scp-proving-edgep edge)
    (let ((index-vertex
	   (si-deref (proposition-start (car (edge-conditions edge)))
		     (edge-bindings edge)))
	  (sitname
	   (si-deref
	    (proposition-situation (car (edge-conditions edge)))
	    (edge-bindings edge)))
	  (relation (car (proposition-fact (car (edge-conditions edge))))))
      (setf (vertex-active-in index-vertex)
	    (scp-add-to-index
	     (vertex-active-in index-vertex)
	     sitname relation edge))
      (if (not (eq index-vertex scp-prop-vertex))
	  (setf (vertex-active-in scp-prop-vertex)
		(scp-add-to-index
		 (vertex-active-in scp-prop-vertex)
		 sitname relation edge))))))
  (setq scp-chart (cons edge scp-chart)))

(defun scp-propose (proposition bindings sentence)
;;;
;;; Search the constraints for some constraints that might make this
;;; true.  Finds all ways of matching with all constraints and adds
;;; new provingedges to the agenda for these constraints.
;;;
  (let ((pstart (si-deref (proposition-start proposition) bindings)))
   (mapc
    #'(lambda (sit-constraint)
	(let* ((ps (car sit-constraint))
	       (newsit (cadr sit-constraint))
	       (constraint (caddr sit-constraint))
	       (instance (scp-instance))
	       (bindings (si-emptybinds))
	       (newedge (make-edge))
	       (pend (make-variable :name (gensym "VERTEX")))
	       (required-props
		(scp-buildpropositions
		 (cdr constraint)
		 ps pend)))
	  (setf (edge-conditions newedge) required-props)
	  (setf (edge-info newedge) instance)
	  (setf (edge-bindings newedge) bindings)
	  (setf (edge-start newedge) pstart)  
	  (setf (edge-end newedge)
		(if (eq ps pstart)
		    pend
		    (make-variable :name (gensym "VERTEX"))))
	  (setf (edge-fact newedge) (situation-tableau (car constraint)))
	  (setf (edge-situation newedge) newsit)
	  (scp-addtoagenda newedge)))
    (scp-select-constraints proposition pstart sentence))))

(defun scp-buildpropositions (situations start end)
;;;
;;; Build list of propositions for these situations, linked from start
;;; to end.
;;;
  (cond
   ((null (cdr situations))
    (mapcar
     #'(lambda (fact)
	 (make-proposition
	  :situation (situation-name (car situations))
	  :start start
	  :end end
	  :fact fact))
     (situation-tableau (car situations))))
   (t
    (let ((newvertex (make-variable :name (gensym "VERTEX"))))
      (append
       (mapcar
	#'(lambda (fact)
	    (make-proposition
	     :situation (situation-name (car situations))
	     :start start
	     :end newvertex
	     :fact fact))
	(situation-tableau (car situations)))
       (scp-buildpropositions
	(cdr situations) newvertex end))))))
	 
(defun scp-select-constraints (proposition pstart sentence)
;;;
;;; Checks each constraints to see if it could make this
;;; proposition true.  Returns a list of sit-point and constraint
;;; for each matching one.
;;;
  (append
   (if (and sentence
	    (null (eq pstart scp-prop-vertex)))
       (mapcan
	#'(lambda (constraint)
	    (let* ((result (scp-quick-check proposition (car constraint))))
	      (cond
		((null result) nil)
		((member constraint (vertex-rules pstart))
		 ;; already had this one
		 nil)
		(t
		 (let ((newconstraint (si-instance-constraint constraint)))
		   (setf (vertex-rules pstart)
			 (cons constraint (vertex-rules pstart)))
		   (list
		    (list
		     pstart
		     (car newconstraint)
		     newconstraint)))))))
	st.grules)
       nil)
   (mapcan
    #'(lambda (constraint)
	(let* ((result (scp-quick-check proposition (car constraint))))
	  (cond
	    ((null result) nil)
	    ((member constraint (vertex-rules scp-prop-vertex))
	     ;; already had this one
	     nil)
	    (t
	     (let ((newconstraint (si-instance-constraint constraint)))
	       (setf (vertex-rules scp-prop-vertex)
		     (cons constraint (vertex-rules scp-prop-vertex)))
	       (list
		(list
		 scp-prop-vertex
		 (car newconstraint)
		 newconstraint)))))))
    st.constraints)))

(defun scp-quick-check (proposition constraint-head)
;;;
;;; Tests if propsotion might be proved with this constraint.  It
;;; checks to see in the situations can be combined and if so it then
;;; checks to see if the relation in the propositions fact appears
;;; somewhere in the constraint-head.  This is probably too
;;; restrictive for the time being (once properties are introduced or
;;; parametric relation names or variables as relations names this
;;; will exclude those constraints).
;;;
  (let ((result (si-prove-term
		 (list (proposition-situation proposition))
		 (list (situation-name constraint-head))
		 (si-emptybinds))))
    (cond
      ((and result
	    (member (car (proposition-fact proposition))
		    (mapcar #'car (situation-tableau constraint-head))))
       result)
      (t nil))))

(defun scp-checkcombine (provededge provingedge)
;;;
;;; Checks to see if the next required proposition in proving edge can
;;; be provided by the proposition in provededge.
;;;
  (let (result)
    (cond
      ((eq scp-prop-vertex
	   (proposition-start (car (edge-conditions provingedge))))
       (setq result
	     (si-match-proposition
	      (situation-name (edge-situation provededge))
	      (list
	       scp-prop-vertex scp-prop-vertex
	       (edge-fact provededge))
	      (list
	       (proposition-situation
		(car (edge-conditions provingedge)))
	       (proposition-start (car (edge-conditions provingedge)))
	       (proposition-end (car (edge-conditions provingedge)))
	       (proposition-fact (car (edge-conditions provingedge))))
	      (edge-bindings provingedge))))
      (t    ;; a grammar rule is active
       (setq result
	     (si-match-proposition
	      (situation-name (edge-situation provededge))
	      (list
	       (edge-start provededge)
	       (edge-end provededge)
	       (edge-fact provededge))
	      (list
	       (proposition-situation (car (edge-conditions provingedge)))
	       (proposition-start (car (edge-conditions provingedge)))
	       (proposition-end (car (edge-conditions provingedge)))
	       (proposition-fact (car (edge-conditions provingedge))))
	      (edge-bindings provingedge)))))
    (if result
	(scp-makeedge result provededge provingedge))))

(defun scp-makeedge (match-result provededge provingedge)
;;;
;;; Build a new edge after a successful match.  Apply any special
;;; functions if the newly created edge is now complete (proved).
;;;
   (let ((newedge (make-edge)))
     (setf (edge-conditions newedge)
	   (append
	    (mapcar
	     #'(lambda (sitfact)
		 (make-proposition
		  :situation (car sitfact)
		  :start scp-prop-vertex
		  :end scp-prop-vertex
		  :fact (cdr sitfact)))
	     (cdr match-result))
	    (cdr (edge-conditions provingedge))))
     (setf (edge-bindings newedge)      ; was inefficient hack
	   (car match-result))
     (setf (edge-info newedge) (edge-info provingedge))
     (setf (edge-situation newedge) (edge-situation provingedge))
     (setf (edge-fact newedge) (edge-fact provingedge))
     ;; these might be nil anyway
     (setf (edge-start newedge) (edge-start provingedge))
     (setf (edge-end newedge)
	   (si-deref (edge-end provingedge) (edge-bindings newedge)))
     (cond
       ((scp-proving-edgep newedge)
	(scp-addtoagenda newedge))
       ((and (scp-proved-edgep newedge) (eq (edge-info newedge) 'query))
	;; query is a special case
	;; ??? not sure about starts and ends ???
	(let* ((newsit (scp-make-sit-point
			(si-realiseterm
			 (situation-name (edge-situation newedge))
			 (edge-bindings newedge)
			 (edge-start newedge) (edge-end newedge)
			 (si-emptybinds) "SIT")
			nil nil
			(situation-type (edge-situation newedge))
			(edge-start newedge) (edge-end newedge)))
	       (facts (si-realiseterm
		       (edge-fact newedge)
		       (cons (cons (situation-name (edge-situation newedge))
				   (sit-point-name newsit))
			     (edge-bindings newedge))
		       (edge-start newedge)
		       (edge-end newedge)
		       (si-emptybinds))))
	  (setf (edge-situation newedge)
		(make-situation
		 :name (sit-point-name newsit)
		 :type (situation-type (edge-situation newedge))))
	  (setq scp-complete-queries (cons newedge scp-complete-queries))
	  (scp-addtoagenda newedge)))
       (t
	;; its a new proved edge so has to be split into
	;; a bunch of proved facts
	(let* ((newsit (scp-make-sit-point
			(si-realiseterm
			 (situation-name (edge-situation newedge))
			 (edge-bindings newedge)
			 (edge-start newedge) (edge-end newedge)
			 (si-emptybinds) "SIT")
			nil nil
			(situation-type (edge-situation newedge))
			(edge-start newedge) (edge-end newedge)))
	       (facts (si-realiseterm
		       (edge-fact newedge)
		       (cons (cons (situation-name (edge-situation newedge))
				   (sit-point-name newsit))
			     (edge-bindings newedge))
		       (edge-start newedge)
		       (edge-end newedge)
		       (si-emptybinds))))
	  (scp-make-facts
	   (make-situation
	    :name (sit-point-name newsit)
	    :type (situation-type (edge-situation newedge)))
	   facts                      ; list of new facts
	   (sit-point-start newsit)
	   (sit-point-end newsit)
	   (edge-info provingedge)
	   (edge-bindings newedge)))))))

(defun scp-make-facts (situation facts start end info bindings)
;;;
;;; Make edges for these facts and add them to the chart.
;;;
  (mapc
   #'(lambda (fact)
       (let ((factedge (make-edge)))
	 (setf (edge-situation factedge) situation)
	 (setf (edge-bindings factedge) bindings)
	 (setf (edge-info factedge) info)
	 (setf (edge-start factedge) start)
	 (setf (edge-end factedge) end)
	 (setf (edge-fact factedge) fact)
	 ;; Only add this new edge to the agenda if we have not
	 ;; had it before
	 (if (scp-update-situation
	      (scp-make-sit-point
	       (situation-name situation) nil nil nil nil nil)
	      fact)
	     (scp-addtoagenda factedge))))
   facts))

(defun scp-addtoagenda (edge)
;;;
;;; Add the new edge to the agenda
;;;
   (setq scp-agenda (cons edge scp-agenda)))

(defun scp-instance ()
;;;
;;; Returns a symbol for this rule instance
;;;
  (let ((newvar (intern (string (gensym "RULE")))))
    newvar))

(defun scp-update-situation (situation fact)
;;;
;;; Add proved fact to situation. Returns t if this fact
;;; is not already in this situation.
;;;
    (cond
      ((null fact) t)
      ((member fact (sit-point-facts situation) :test #'equal)
       ;; this probably puts the code in a loop 
       nil)
      (t
       (setf (sit-point-facts situation)
	     (cons fact (sit-point-facts situation)))
       t)))

(defun scp-make-sit-point (name facts constraints type start end)
;;;
;;; Find or make a sitpoint.  Name is the atomic name of the situation
;;; -- this may be a variable.  facts are a list of facts this will
;;; grow as new things are proved about this situation.  Constraints
;;; is a list of constraints that are already 'active' on this
;;; situation -- i.e.  have been applied and can be applied again with
;;; no further action -- this is used for the 'left recursion check'.
;;; Type is a situation type as it was first specified it can be used
;;; to generate a new set of basic facts for a situation.
;;;
;;; If a sit-point of that name already exists the existing one is
;;; returned in that case the rest of the arguments are ignored.
;;;
  (cond
    ((assoc name scp-sit-points)
     (cdr (assoc name scp-sit-points)))
    (t
     (let ((newsit (make-sit-point :name name
				   :type type
				   :facts facts
				   :constraints constraints
				   :start start
				   :end end)))
       (setq scp-sit-points (cons (cons name newsit) scp-sit-points))
       newsit))))

(defun scp-add-to-index (index situation relation edge)
;;;
;;; Destructively add edge to this index.  This is based on the
;;; siutation and relation (name) of the edge.
;;;
  ;; add to master index -- access when both situation and relation are var
  (setf (car index) (cons edge (car index)))
  (cond
    ((null (variable-p relation))
     (let ((place (assoc relation (car (cadr index)))))
       (cond
	 ((null place)  ;; first one of this relation
	  (setf (car (cadr index))
		(cons (list relation edge) (car (cadr index)))))
	 (t
	  (setf (cdr place) (cons edge (cdr place)))))))
    (t
     (cond
       ((variable-p situation)
	;; put on the var var list
	(setf (cdr (nth 3 index))
	      (cons edge (cdr (nth 3 index)))))
       (t
	(let ((place (assoc situation (cdr (cadr index)))))
	  (cond
	    ((null place)
	     (setf (cdr (cadr index))
		   (cons (list situation edge) (cdr (cadr index)))))
	    (t
	     (setf (cdr place) (cons edge (cdr place))))))))))
  (cond
    ((null (variable-p situation))
     (let ((place (assoc situation (car (caddr index)))))
       (cond
	 ((null place)  ;; first one of this situation
	  (setf (car (caddr index))
		(cons (list situation edge) (car (caddr index)))))
	 (t
	  (setf (cdr place) (cons edge (cdr place)))))))
    (t
     (let ((place (assoc relation (cdr (caddr index)))))
       (cond
	 ((null place)
	  (setf (cdr (caddr index))
		(cons (list relation edge) (cdr (caddr index)))))
	 (t
	  (setf (cdr place) (cons edge (cdr place))))))))
  (cond
    ((null (or (variable-p situation) (variable-p relation)))
     ;; both are nonvar so we can really index them
     ;; I reckon there will be more situation than relation names
     (let ((relplace (assoc relation (car (nth 3 index)))))
       (cond
	 ((null relplace)
	  ;; first time with this relation name
	  (setf (car (nth 3 index))
		(cons (list relation (list situation edge))
		      (car (nth 3 index)))))
	 (t
	  (let ((sitplace (assoc situation (cdr relplace))))
	    (cond
	      ((null sitplace)
	       ;; first time for this situation for this relation
	       (setf (cdr relplace)
		     (cons (list situation edge) (cdr relplace))))
	      (t
	       (setf (cdr sitplace)
		     (cons edge (cdr sitplace)))))))))))
  index)

(defun scp-find-index (index situation relation func)
;;;
;;; Applies the func to each of the edges it finds in the index
;;; Returns a list of edges for the given situation and/or relation
;;;
  (cond
    ((variable-p situation)
     (cond
       ((variable-p relation)
	;; no indexing possible -- so return all edges
	(mapc func (car index)))
       (t
	(mapc func (cdr (assoc relation (car (cadr index)))))
	(mapc func (cdr (assoc relation (cdr (caddr index)))))
	(mapc func (cdr (nth 3 index))))))
    ((variable-p relation)
     (mapc func (cdr (assoc situation (car (caddr index)))))
     (mapc func (cdr (assoc situation (cdr (cadr index)))))
     (mapc func (cdr (nth 3 index))))
    (t
     (mapc func
      (cdr (assoc situation
		  (cdr (assoc relation (car (nth 3 index)))))))
     (mapc func
      (cdr (assoc situation (cdr (cadr index)))))
     (mapc func
      (cdr (assoc relation (cdr (caddr index)))))
     (mapc func
      (cdr (nth 3 index))))))

(defun scp-find-index-no-var (index situation relation)
;;;
;;; Returns a list of edges for the given situation and/or relation
;;; --- this only works if no var edges were inserted in the index ---
;;; i.e. ir is of inactive edges.
;;;
  (cond
    ((variable-p situation)
     (cond
       ((variable-p relation)
	;; no indexing possible -- so return all edges
	(car index))
       (t
	(cdr (assoc relation (car (cadr index)))))))
    ((variable-p relation)
     (cdr (assoc situation (car (caddr index)))))
    (t
     (cdr (assoc situation
		 (cdr (assoc relation (car (nth 3 index)))))))))

