;;;
;;;   File: si-match.lisp
;;;   Situated inference for the situation theoretic based language ASTL
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;        Copyright (c) 1993  Alan W Black (awb@ed.ac.uk)
;;;
;;; This file is part of an implementation of ASTL a computational
;;; situation theoretic language.  This implementation of ASTL is free
;;; software; you can redistribute it and/or modify it under the terms
;;; of the GNU General Public License as published by the Free
;;; Software Foundation; either version 1, or (at your option) any
;;; later version.
;;;
;;; ASTL is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts
;;; responsibility to anyone for the consequences of using it or for
;;; whether it serves any particular purpose or works at all, unless
;;; he says so in writing.  Refer to the GNU Genral Public License for
;;; full details.
;;;
;;; Everyone is granted permission to copy, modify and redistribute
;;; this file, but only under the conditions described in the
;;; GNU General Public License.   A copy of this license is
;;; supposed to have been given to you along with this program so you
;;; can know your rights and responsibilities.  It should be in a
;;; file named COPYING.  Among other things, the copyright notice
;;; and this notice must be preserved on all copies.
;;;
;;; The author gratefully acknowledges financial support for the
;;; research this implementation is based on from an SERC studnetship
;;; (number 89313458) and Esprit Basic Research Action Project 6852
;;; (DYANA-2).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;   Author:  Alan W Black (awb@ed.ac.uk)
;;;   Date: 24 May 1991
;;;
;;;   This is effectively a replacement for unification which finds
;;;   matches.  It only matches non-situation terms any situation
;;;   terms embedded in a term will be assumed to be matched but
;;;   returned as conditions which then are proved by the
;;;   chart-prover.
;;;
;;;   This is re-written for the n+1 th time
;;;   

(lisp:in-package 'user)

;;;
;;;  Some macros
;;;
(defmacro si-situationp (sit) `(situation-p ,sit))
(defmacro si-varp (var) `(variable-p ,var))
(defmacro si-emptybinds () `(list (cons (list 'bd) (list 'bd))))

;;;
;;;  Globals used
;;;
(defvar st.constraints nil)
(defvar st.basesits nil)
(defvar st.variables nil)
(defvar st-paramcount 0)

;;;
;;;  Main function
;;;
(defun si-match-proposition (situation fact proposition bindings)
  "Find out if this proposition can be matched to this situation
and fact."
  (let ((matchsit-name (si-prove-term
			(list (si-derefsitname (car proposition) bindings))
			(list (si-derefsitname situation bindings))
			(si-emptybinds))))
    (cond
     ((null matchsit-name)
      ;; these situations don't match
      nil)
     (t
      (si-prove-term (cdr proposition) fact
		     (append
		      (car matchsit-name)
		      bindings))))))

(defun si-prove-term (qterm bterm bindings &optional situation-conds)
  "See if qterm is provable with bterm, this basically means see if
qterm is subsumed by bterm.  Returns nil if not successful or
bindings and list of sub-situations which must also be matched if the
match is successful."
  (cond
    ((or (null qterm) (null bterm))
     (cond
       ((and (null qterm) (null bterm)) ; no more to match
	(cons bindings situation-conds))
       (t nil)))
    ((eq qterm bterm)
     (cons bindings nil))
    (t
       (let ((real-qterm (si-deref (car qterm) bindings))
	     (real-bterm (si-deref (car bterm) bindings)))
	 ;; dereference the terms with respect to the current bindings
	 (cond
	   ((eq real-qterm real-bterm)
	    (si-prove-term
	     (cdr qterm) (cdr bterm)
	     bindings situation-conds))
	   ((si-varp real-qterm)            
	    (cond
	      ((si-varp real-bterm)        ; both are variables
	       ;; can't happen anymore ! (I think)
	       (let ((newvar (si-newvar    ; 'union' the types
			      nil
			      (append (variable-facts real-qterm)
				      (variable-facts real-bterm)))))
		 (si-prove-term 	   ; bind them both to a new variable
		  (cdr qterm)              ; and continue
		  (cdr bterm)
		  (cons			   ; new bindings
		   (cons real-qterm newvar)
		   (cons
		    (cons real-bterm newvar)
		    bindings))
		  situation-conds)))
	      (t
	       (si-match-var
		real-qterm real-bterm
		(cdr qterm)
		(cdr bterm)
		bindings
		situation-conds))))
	   ((si-varp real-bterm)	; only bterm is a variable
	    (cond                       ; we get here from quick-check !!
	     ((or t (si-checkbind real-bterm real-qterm))
	      (si-prove-term              
	       (cdr qterm)
	       (cdr bterm)
	       (cons
		(cons real-bterm real-qterm)
		bindings)
		(si-maybe-add-conds
		 real-bterm real-qterm
		 situation-conds bindings)))
	     (t                         ; incompatible variable
	      nil)))
	   ((si-situationp real-qterm)
	    (cond
	      ((or (si-situationp real-bterm)
		   (assoc real-bterm scp-sit-points))
	       ;; move these situations to conditions
	       (si-prove-sit
	        real-qterm real-bterm
		(cdr qterm) (cdr bterm)
		bindings
		situation-conds))
	      (t			; failed
	       nil)))
	   ((atom real-qterm)
	    (cond
	      ((eq real-qterm real-bterm)
	       (si-prove-term
		(cdr qterm) (cdr bterm)
		bindings situation-conds))
	      (t nil)))
	   ((or (atom real-qterm) (atom real-bterm))
	    nil)
	   (t				; both are conses so reduce them
	    (si-prove-term
	     (append real-qterm (cdr qterm))
	     (append real-bterm (cdr bterm))
	     bindings situation-conds)))))))

(defun si-maybe-add-conds (variable value conds bindings)
  "Add situation conditions to conds if the variable has any
conditions in its type."
  (cond
   ((null (si-sitvarp variable))
    ;; no new conditions
    conds)
   (t
    (append
     (si-buildconditions
      (or (and (situation-p value) (situation-name value))
	  value)
      (variable-facts variable)
      bindings)
     conds))))

(defun si-match-var (var value q-rest b-rest bindings situation-conditions)
  "Try to match var to value taking into account any typing of the
variable."
  (cond
    ((null (si-sitvarp var))
     ;; a simple variable
     (si-prove-term
      q-rest b-rest
      (cons
       (cons var value) bindings)
      situation-conditions))
    ((consp value)
     ;; value is a fact so this can't work -- typed vars only match sits
     nil)
    ((variable-type var)
     ;; a variable type
     ;;; Grrr ! if its already bound ! I'll miss its value
     (si-prove-term
      q-rest b-rest
      (cons
	(cons (variable-type var)
	      (si-find-type value))
	(cons
	 (cons var value) bindings))
      (append
       (si-buildconditions
	(or (and (situation-p value) (situation-name value))
	    value)
	(variable-facts var)
	bindings)
       situation-conditions))
     )
    (t
     ;; no variable type so can continue but add the conditions in
     ;; the type to the general conditions
     (si-prove-term
      q-rest b-rest
      (cons
       (cons var value) bindings)
      (append
       (si-buildconditions
	(or (and (situation-p value) (situation-name value))
	    value)
	(variable-facts var)
	bindings)
       situation-conditions)))))

(defun si-find-type (sit)
  "Find the type of this situation."
  (let ((sit-point (cdr (assoc sit scp-sit-points))))
    (cond
      ((null sit-point)
       (error (format nil "No sit-point for ~A" sit)))
      (t
       (sit-point-type sit-point)))))

(defun si-prove-sit (qsit bsit qterm bterm bindings situation-conds)
  "See if bsit might allow qsit to be true.  If so assume so and
continue matching addind list of conditions (propositions) to
situation-conds."
  (let ((result (si-prove-term (list (si-derefsitname qsit bindings))
			       (list (si-derefsitname bsit bindings))
			       bindings situation-conds))
	(rbsit-name (or (and (situation-p bsit) (situation-name bsit))
			bsit)))
    (cond
      ((null result)
       nil)
      (t
       ;; this assumes that bsit is already grounded -- is it ???
       (si-prove-term
	qterm bterm
	(cons
	 (cons qsit rbsit-name)   ; assume these are now one
	 (car result))
	(append
	 (si-buildconditions
	  (situation-name qsit)
	  (situation-tableau qsit)
	  (car result))
	 situation-conds))))))

(defun si-buildconditions (sitname conditions bindings)
  "Returns a new list of conditions."
  (mapcar
   #'(lambda (fact)
       (cons
	(si-deref sitname bindings)
	fact))
   conditions))

(defun si-uniquifyterm (term &optional beenthere)
  "Replace all variables in term with new variables"
  (cond
    ((null beenthere)
     (si-uniquifyterm term (list (cons (list t) t))))
    ((assoc term beenthere)
     (cdr (assoc term beenthere)))      ;; done that
    ((null term) nil)
    ((member term st.variables)
     ; a declared variable
     (let ((newvar (si-newvar nil nil)))
       (si-push-onto (cons term newvar) beenthere)
       newvar))
    ((si-varp term)
     (cond
       ((si-sitvarp term)
	(let ((newvar (si-newvar nil nil)))
	  (si-push-onto (cons term newvar) beenthere)
	  (setf (variable-type newvar)
		(si-uniquifyterm (variable-type term) beenthere))
	  (setf (variable-facts newvar)
		(si-uniquifyterm (variable-facts term) beenthere))
	  newvar))
       (t
	;; simple variable
	(let ((newvar (si-newvar nil nil)))
	  (si-push-onto (cons term newvar) beenthere)
	  newvar))))
    ((si-situationp term)
     (let ((newsit (make-situation)))
       (si-push-onto (cons term newsit) beenthere)
       (setf (situation-name newsit) 
	     (si-uniquifyterm (situation-name term) beenthere))
       (setf (situation-type newsit)
	     (si-uniquifyterm (situation-type term) beenthere))
       (setf (situation-tableau newsit) 
	     (si-uniquifyterm (situation-tableau term) beenthere))
       newsit))
    ((atom term) term)
    (t
     (let* ((newcons (cons nil nil)))
       (si-push-onto (cons term newcons) beenthere)
       (setf (car newcons) (si-uniquifyterm (car term) beenthere))
       (setf (cdr newcons) (si-uniquifyterm (cdr term) beenthere))
       newcons))))
  
(defun si-instance-constraint (constraint)
  "Returns an instantiated constraint by taking a uniquified form
of the constraint and the head situations name to a name."
  (si-uniquifyterm constraint))

(defun si-newvar (type facts)
  "Return a new variable"
  (make-variable :name (gensym "*VAR") :type type :facts facts))

(defun si-deref (var bindings)
  "Dereference the (possible) var"
  (let ((newval (assoc var bindings)))
    (cond
      ((null newval) var)
      (t
       (si-deref (cdr newval) bindings)))))

(defun si-derefsitname (sit bindings &optional beenthere)
  "Dereference the situtation right down to its name."
  (cond
    ((member sit beenthere)
     (if (situation-p sit)
	 (situation-name sit)
	 sit))
    (t
     (let ((newval (assoc sit bindings)))
       (cond
	 ((null newval)
	  (if (situation-p sit)
	      (si-derefsitname
	       (situation-name sit) bindings (cons sit beenthere))
	      sit))
	 (t
	  (si-derefsitname (cdr newval) bindings
			   (cons sit beenthere))))))))

(defun si-unrefsits (term &optional scopesits)
  "Replace all situations in name with the name of the situation rather
than its structure except where the situation does not appear in st.basesits."
  (cond
    ((situation-p term)
     (cond
       ((or (assoc (situation-name term) scp-sit-points)
	    (member term scopesits))
	(situation-name term))
       (t
	(setf (situation-tableau term)
	      (mapcar
	       #'(lambda (fact)
		   (si-unrefsits fact (cons term scopesits)))
	       (situation-tableau term)))
	term)))
    ((atom term) term)
    (t
     (cons
      (si-unrefsits (car term) scopesits)
      (si-unrefsits (cdr term) scopesits)))))

(defun si-realiseterm (term bindings start end beenthere &optional prefix)
  "Go through this term changing all variables to constants.  If the
variable is a situation variable a new situation is created and the
facts are asserted to it and to the chart agenda.  The new term is
returned"
  (cond
    ((assoc term beenthere)
     (cdr (assoc term beenthere)))       ; done that
    ((null term) nil)
    (t
     (let ((real-term (si-deref term bindings)))
       (cond
	((si-varp real-term)
	 (cond
	  ((si-sitvarp real-term)
	   (cond
	    ((null (variable-type real-term))
	     ;; simple situation variable
	     (let* ((newvar (si-newvar nil nil))
		    facts
		    (newsit (make-situation
			     :name (intern (string (gensym "SIT")))))
		    (newsitname (situation-name newsit))
		    (sit-point (scp-make-sit-point
				newsitname
				nil nil nil
				start end)))
	       (si-push-onto (cons term newsitname) beenthere)
	       (si-push-onto (cons real-term newsitname) beenthere)
	       (setf (situation-type newsit)
		     (list real-term
			   (si-realiseterm
			    (variable-facts real-term)
			    bindings scp-prop-vertex
			    scp-prop-vertex beenthere)))
	       (setq facts (cadr (situation-type newsit)))
	       (setf (situation-type newsit)
		     (st_replace_name
		      (car (situation-type newsit)) newvar
		      (situation-type newsit)))
	       (setf (sit-point-type sit-point)
		     (situation-type newsit))
	       (scp-make-facts
		newsit facts
		start end 'realised bindings)
	       newsitname))
	    (t        ; a variable type
	     (let* ((newvar (si-newvar nil nil))
		    (newsit
		     (make-situation
		      :name (intern (string (gensym "SIT")))))
		    (newsitname (situation-name newsit))
		    (sit-point (scp-make-sit-point
				newsitname nil nil nil
				start end)))
	       (si-push-onto (cons term newsitname) beenthere)
	       (si-push-onto (cons real-term newsitname) beenthere)
	       (setf (situation-type newsit)
		     (si-build-type
		      newvar
		      (si-deref (variable-type real-term) bindings)
		      (si-realiseterm
		       (variable-facts real-term) bindings
		       scp-prop-vertex scp-prop-vertex beenthere)))
	       (setf (sit-point-type sit-point)
		     (situation-type newsit))
	       (scp-make-facts
		newsit
		(cadr (situation-type newsit))
		start end 
		'realised
		bindings)
	       newsitname))))
	  (t
	   (let ((newconst (intern (string (gensym (or prefix "C"))))))
	     (si-push-onto (cons term newconst) beenthere)
	     (si-push-onto (cons real-term newconst) beenthere)
	     newconst))))
	((si-situationp real-term)
	 (let ((sit-point (scp-make-sit-point (situation-name real-term)
					      nil nil nil
					      start end)))
	   ; previously made situations will be treated funny *BUG*
	   (si-push-onto (cons term (sit-point-name sit-point)) beenthere)
	   (si-push-onto (cons real-term (sit-point-name sit-point)) beenthere)
	   (scp-make-facts
	    real-term
	    (si-realiseterm
	     (situation-tableau real-term) bindings
	     scp-prop-vertex scp-prop-vertex beenthere)
	    (sit-point-start sit-point)
	    (sit-point-end sit-point)
	    'realised
	    bindings)
	   (sit-point-name sit-point)))
	((si-typevarp real-term)      ;; special case -- reduce type var
	 (let ((newtype (list nil nil))
	       (derefedtype
		(si-deref (cadr (car real-term)) bindings))
	       (newparam (si-newparam)))
	   (si-push-onto (cons term newtype) beenthere)
	   (si-push-onto (cons real-term newtype) beenthere)
	   (setf (car newtype)
		 (si-realiseterm (cadr real-term) bindings
				 scp-prop-vertex scp-prop-vertex beenthere))
	   (si-push-onto (cons (car derefedtype) newparam) beenthere)
	   (si-push-onto (cons (car newtype) newparam) beenthere)
	   (setf (car newtype) newparam)
	   (setf (cadr newtype)
		 (si-realiseterm
		  (append (cadr derefedtype)
			  (caddr real-term))
		  bindings
		  scp-prop-vertex scp-prop-vertex beenthere))
	   newtype))
	((si-typep real-term)       ;; special case -- sit type
	 (let ((newtype (list nil nil))
	       (newparam (si-newparam)))
	   (si-push-onto (cons term newtype) beenthere)
	   (si-push-onto (cons real-term newtype) beenthere)
	   (setf (car newtype) newparam)
	   (si-push-onto (cons (car real-term) newparam) beenthere)
	   (setf (cadr newtype)
		 (si-realiseterm
		  (cadr real-term)
		  bindings
		  scp-prop-vertex scp-prop-vertex beenthere))
	   newtype))
	((atom real-term) real-term)
	(t           ; cons cell
	 (let* ((newcons (cons nil nil)))
	   (si-push-onto (cons term newcons) beenthere)
	   (si-push-onto (cons real-term newcons) beenthere)
	   (setf (car newcons)
		 (si-realiseterm (car real-term) bindings
				 scp-prop-vertex scp-prop-vertex beenthere))
	   (setf (cdr newcons)
		 (si-realiseterm (cdr real-term) bindings
				 scp-prop-vertex scp-prop-vertex beenthere))
	   newcons)))))))
       
(defun si-push-onto (newitem list)
  "Destructive adds newitem onto the front of the list."
  (rplacd list (cons (car list) (cdr list)))
  (rplaca list newitem)
  list)

(defun si-build-type (name oldtype newfacts)
  "Build a new type from these parts."
  (let ((newname (si-newvar nil nil)))
    (if (or (variable-p oldtype) (null oldtype))
	(setq oldtype (list t nil)))
    (list
     newname
     (append newfacts
	     (st_replace_name
	      (car oldtype)
	      newname
	      (st_replace_name name newname (cadr oldtype)))))))

(defun si-typevarp (term)
  "Returns true if this is a variable type."
  (and (listp term)
       (listp (car term))
       (eq (caar term) 'typevar)))

(defun si-typep (term)
  "Returns true if this is a situation type."
  (and (listp term)
       (eql 2 (length term))
       (variable-p (car term))  ;; a var
       (listp (cadr term))
       (null (cddr term))))

(defun si-sitvarp (variable)
  "Returns true if this is a situation variable."
  (and (variable-p variable)
       (or (variable-type variable)
	   (variable-facts variable))))

(defun si-newparam ()
  "Returns a symbol of the form Pn where n is hopefully small."
    (values (intern (format nil "P~A" (setq st.paramcount
					    (1+ st.paramcount))))))
