;;;
;;;  File: stparse.lisp
;;;
;;;  This decribes the basic syntax for the ASTL language
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;        Copyright (c) 1993  Alan W Black (awb@ed.ac.uk)
;;;
;;; This file is part of an implementation of ASTL a computational
;;; situation theoretic language.  This implementation of ASTL is free
;;; software; you can redistribute it and/or modify it under the terms
;;; of the GNU General Public License as published by the Free
;;; Software Foundation; either version 1, or (at your option) any
;;; later version.
;;;
;;; ASTL is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts
;;; responsibility to anyone for the consequences of using it or for
;;; whether it serves any particular purpose or works at all, unless
;;; he says so in writing.  Refer to the GNU Genral Public License for
;;; full details.
;;;
;;; Everyone is granted permission to copy, modify and redistribute
;;; this file, but only under the conditions described in the
;;; GNU General Public License.   A copy of this license is
;;; supposed to have been given to you along with this program so you
;;; can know your rights and responsibilities.  It should be in a
;;; file named COPYING.  Among other things, the copyright notice
;;; and this notice must be preserved on all copies.
;;;
;;; The author gratefully acknowledges financial support for the
;;; research this implementation is based on from an SERC studnetship
;;; (number 89313458) and Esprit Basic Research Action Project 6852
;;; (DYANA-2).
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  Alan W Black (awb@ed.ac.uk) Dec 1990
;;;
;;;  It uses the my prule stuff to define a recursive decent parser
;;;  for the language.  This parser also defines translations into the
;;;  internal form suitable for the theorem prover/parser
;;;
;;;  19 Jan 92 - awb (0.4)
;;;  Add general abstractions to the syntax.
;;;  30 Oct 92 - awb (0.3)
;;;  Tidy up the use of types as terms.
;;;  14 Oct 91 - awb (0.3)
;;;  Change the use of colons to be closer to standard situation
;;;  theory.
;;;  24 Apr 91 - awb (0.2)
;;;  Changed the internal format of terms to the one suitable for
;;;  the situated inference module.  This requires minor changes
;;;  throughout the whole file.
;;;  15 Apr 91 - awb (0.2)
;;;  Add a top level which can use pruleparse to get input from the 
;;;  command line and prove/parse it
;;;
;;;

;;;
;;;  Structure definitions (don't read much into the names)
;;;
(defstruct situation name type tableau)
(defstruct variable name type facts)

(defmacro astl-version () "ASTL-0.4-beta Jan 93")

(setq gc.singlecharsymbols
   '( #\( #\) #\[ #\] #\{ #\} #\, #\. #\< #\> #\: #\\ #\/ #\^ #\! #\= #\@))

(defvar st.constraints nil)
(defvar st.basesits nil)
(defvar st.parameters nil)
(defvar st.lexentries nil)
(defvar st.grules nil)
(defvar st.featrels nil)
(defvar st.relnames nil)
(defvar st.variables nil)
(defvar st.situations nil)
(defvar st.topcat nil)
(defvar st.scope nil)
(defvar st.hush nil)

(defvar scp-sit-points nil)
(defvar st.ekn-mode t)
(defvar st.time nil)

;;;
;;;  Initialise everything (well only the bits that are necessary :-)
;;;

(pruleinit)

(defprule stinit
;;;
;;;  This clause is called before we start parsing so initialisation
;;;  of global structures is done here.  It recognises nothing but has
;;;  a number of actions for initialising structures.  This is a bit
;;;  of hack in the prule stuff but sorry, that's life.
;;;
   ( () ;; recognise nothing but can be used for initialisation
     (setq st.lexentries nil)
     (setq st.grules nil)
     (setq st.constraints nil)
     (setq st.scope nil)
     (setq st.featrels nil)
     (setq st.relnames nil)
     (setq st.variables nil)
     (setq st.situations nil)
   ))

;;;
;;;  Syntax for command line
;;;
(defprule stcommand
   (('quit) (list $1))
   (('ekn) (list $1))
   (('load 'symbol) (list $1 $2))
   (('parse simplelist) (list $1 $2))
   (('goalprop proposition) (list $1 $2))
   (('prove proposition) (list $1 $2))
   (('hush list-or-atom) (list $1 $2))
   (('unhush list-or-atom) (list $1 $2))
   (('debug 'symbol) (list $1 $2))
   (('undebug 'symbol) (list $1 $2))
   (('help)(list $1))
   (( '\( '\) ) nil)
   )

;;;
;;;  Syntax for an ASTL file
;;;

(defprule stgram
   ((stinit rstgram)
    (setq st.basesits
	  (mapcar #'si-uniquifyterm st.basesits))
    (setq st.constraints
	  (mapcar #'si-uniquifyterm st.constraints))
    (setq st.grules
	  (mapcar #'si-uniquifyterm st.grules))
    t))

(defprule rstgram
   ((individuals relations parameters sit-vars
     topcat constraint-defs)))
(defprule sit-vars
    ;; for compatibility I need to allow these to be in both orders
    (('Situations situationslist 'Variables var-simplelist)
     (setq st.situations
	    (mapcar #'(lambda (sit) (situation-name sit)) $2))
     (setq st.basesits
	   (mapcar
	    #'(lambda (sit)
		(mapc
		 #'(lambda (sit2)
		     (st_replace_name
		      (situation-name sit) sit
		      sit2 (list sit)))
		 $2) sit)
	    $2)))
    (('Variables var-simplelist 'Situations situationslist)
     (setq st.situations
	    (mapcar #'(lambda (sit) (situation-name sit)) $4))
     (setq st.basesits
	   (mapcar
	    #'(lambda (sit)
		(mapc
		 #'(lambda (sit2)
		     (st_replace_name
		      (situation-name sit) sit
		      sit2 (list sit)))
		 $4) sit)
	    $4))))
(defprule var-simplelist
    (( simplelist )
     (setq st.variables $1)))
(defprule individuals
   (('Individuals simplelist)
     (setq st.names $2)))
(defprule parameters
   (('Parameters simplelist)
     (setq st.parameters $2)))
(defprule relations
    (('F-relations f-relations relations) t)
    (('Relations n-relations hushlist) t))
(defprule n-relations
   ((rellist)
     (setq st.relnames (append st.relnames $1))))
(defprule f-relations
   ((rellist)
     (setq st.featrels (mapcar #'car $1))
     (setq st.relnames $1)))
(defprule rellist
   (( '\( relrestlist ) $2))
(defprule relrestlist
   (( '\) ) nil)
   (( 'symbol '/ 'symbol relrestlist )
       (cons (cons $1 $3) $4)))
(defprule hushlist
   (( 'Hush simplelist )
    (setq st.hush $2))
   (() (setq st.hush nil)))
(defprule list-or-atom
  (( '\( restlist )
    $2)
  (( '\{ restbracelist )
    $2)
  (( 'symbol )
    (list $1)))
(defprule situationslist
   (( '\( restsituations ) $2))
(defprule restsituations
   (( '\) ) nil)
   (( situation-name typed restsituations )
    (cons
     (st_build_situation $1 $2)
     $3)))
(defprule situation-name
   (( 'symbol )
    (setq st.situations (cons $1 st.situations))
    $1))
;;(defprule variables
;;   (('Variables simplelist)
;;     (setq st.variables $2)))
(defprule simplelist
   (( '\( restlist )
     $2)
   (( '\{ restbracelist )
     $2))
(defprule restlist
   (( '\) ) nil)
   (( 'symbol restlist) (cons $1 $2)))
(defprule restbracelist
   (( '\} ) nil)
   (( '\, 'symbol restbracelist)
       (cons $2 $3))
   (( 'symbol restbracelist )
      ;; this isn't right as it doesn't enforce commas
      (cons $1 $2)))
(defprule topcat
   (('GoalProp proposition)
    (setq st.topcat $2))
   (() t))
(defprule constraint-defs
    (('Constraints constraints))
    (('Grammar 'Rules rules))
    (('Lexical 'Entries entries))
    (('eof)))
(defprule constraints
    (('Grammar 'Rules rules))
    (('Lexical 'Entries entries))
    (('eof))
    ((constraint constraints)))
(defprule rules
   (('eof))
   (('Lexical 'Entries entries))
   ((rule rules)))
(defprule constraint
  ((c_st_type '\< '\= c_st_type otherdaughters)
    (setq st.constraints
	  (cons
	    (st_build_rule (cons $1 (cons $4 $5))) st.constraints))))
(defprule rule
  ((c_st_type '\- '\> c_st_type otherdaughters)
    (setq st.grules 
	  (cons
	    (st_build_rule (cons $1 (cons $4 $5))) st.grules))))
(defprule otherdaughters
  (( '\, c_st_type otherdaughters)
   (cons $2 $3))
  (( '\. )
   nil))
(defprule entries
  (('eof ) nil)
  (( entry entries)
    (setq st.lexentries (cons $1 st.lexentries))))
(defprule entry
    (( 'symbol '- st_type )
     ;; these are like rules so make them situations
     (let* ((newname (intern (string (gensym "SIT"))))
	    (this (st_build_situation newname $3)))
       (list $1
	     (st_build_entry this)))))
;;;
;;;  Situation types follow, and allow their possible
;;;  arguments (i.e all terms)
;;;
(defprule proposition
   (( 'symbol '\: st_type )
    (let ((this (st_build_situation $1 $3)))
      (setq st.scope                 
	    (cons (cons $1 this) st.scope))
      this)))
	      
(defprule c_st_type                  ;; These are the same as situations
   (('\[ 'symbol '\! conditions )    ;; I suppose its the way rules are
    (let* ((newvar (st_newvar))      ;; interpreted that makes this the case
	   (this (st_build_situation newvar (st_build_type $2 $4))))
      (setq st.scope
	    (append (list
		     (cons newvar this)
		     (cons $2 this))
		    st.scope))
      this))
   (( 'symbol '\: st_type )
    (let ((this (st_build_situation $1 $3)))
      (setq st.scope                 
	    (cons (cons $1 this) st.scope))
      this)))

(defprule st_type
   (('\[ 'symbol '\! conditions )
      (st_build_type $2 $4))
   (( 'symbol typelist )
      (st_build_type (car $2) (cdr $2) $1)))

(defprule rest_type
   (('symbol rest_type_or_abstract )
    (cond
      ((eq (car $2) '@)        ;; an abstraction
       (list
	(cons (list $1 (cadr $2))
	      (caddr $2))
	(car (cdddr $2))))
      ((eq (car $2) '\!)      ;; a simple situation type
       (st_build_type $1 (cadr $2)))
      (t
       (pruleperror "failed to parse type or abstract"))))
   )

(defprule rest_type_or_abstract
  (('\! conditions)
   (list $1 $2))
  (('@ 'symbol rest_labels term '\])
   (list $1 $2 $3 $4)))

(defprule rest_labels
  (('\!) nil)
  (('symbol '@ 'symbol rest_labels)
   (cons
    (list $1 $3) $4)))

(defprule typelist
  (('\& '\[ 'symbol '\! conditions )
   (cons $3 $5))
  (()
   (cons
    (intern (string (gensym "P")))
    nil)))
  
(defprule conditions
   (( '\] ) nil)
   (( 'symbol '\! '\= fact conditions)
    (cons $4 $5)))

(defprule fact
   (( '\< '\< 'symbol factrest )
     (cons $3 $4))
   (( var )           ;; variable
     $1))

(defprule factrest
   (( '\> '\> )
    nil)
   (( '\, term factrest)
    (cons $2 $3)))

;;; requires look ahead so I'll use rrules (reversible rules) for this
(defrrule term
  (( '\( term '\) )
   $2)
  (('symbol)
   $1
   (and (member $$ st.names)  t))
  (('symbol) $1
   (and (assoc $$ st.relnames)  t))
  (('symbol) $1
   (and (member $$ st.parameters)  t))
  ((var_term)
   $1
   (and (member $$ st.variables)  t))
  (('symbol) $1
   (and (or (eql $$ 1) (eql $$ 0))  t))
  (('symbol)
   $1
   (and (stringp $$)  t))
  (('symbol rest_type)
   $2
   (and (eq $$ '\[)  t))
  ((fact)
   $1
   (and (eq $$ '\<)  t))
  ((sit_term)
   $1
   (and (member $$ st.situations)  t))
  (('symbol)
   $1
   (and (pruleperror (format t "Unknown term: ~s" $$)) t)))

(defprule sit_term
    (('symbol typed)
     (cond
       ((null $2)
	$1)
       (t
	(let ((this (st_build_situation $1 $2)))
	  (setq st.scope                 
		(cons (cons $1 this) st.scope))
	  this)))))

(defprule var_term
    ;; a variable that might be a situation so might be followed
    ;; by a type (or followed by an anchoring environment)
    ((var typed_or_tlist)
     (cond
       ((null $2)			; no type so possibly not a situation
	$1)
       ((eq 'sit (car $2))              ; it has a type so it is a situation
	(cond
	  ((consp (car (cdr $2)))             ; its a variable type
	   (let ((this (st_newvar)))
	     (setf (variable-type this) (cadar (cdr $2)))
	     (setf (variable-facts this)
		 (st_replace_name (cadr (cdr $2))
				  this ; (cadar (cdr $2))
				  (caddr (cdr $2))))
	     (setq st.scope                 
		   (cons (cons $1 this) st.scope))
	     this))
	  (t                            ; its a simple type
	   (let ((this (st_newvar)))
	     (setf (variable-facts this)
		   (st_replace_name (car (cdr $2)) this (cadr (cdr $2))))
	     (setq st.scope                 
		   (cons (cons $1 this) st.scope))
	     this))))
       ((eq 'abstract (car $2))
	(st_build_reduce
	  $1 (cdr $2)))
       ((eq 'type (car $2))             ; followed by an & type
	(st_build_type
	 (car (cdr $2))    ;; the parameter
	 (cdr (cdr $2))    ;; facts
	 $1)))))           ;; the variable

(defprule typed_or_tlist
    ;; may be variable situation name (or abstraction)
    (('\: '\: st_type) (cons 'sit $3))
    ;; or may be a variable type with a type following
    (('\& '\[ 'symbol '\! conditions )
     (cons 'type (cons $3 $5)))
    (('\/ '\/ sit_term)
     (cons 'abstract $3))
    (() nil))
      
(defrrule var
    (('symbol)
     $1
     (and (member $$ st.variables)  t)))

(defprule typed
  ;; type for a typed situation (optional)
  (('\: '\: st_type) $3)
  (() nil))

;;;
;;;   Related support functions
;;;

(defun st_featuralp (relname)
;;;
;;;  Returns yes is this relation name is declared as feature
;;;  and no otherwise
;;;
  (cond
   ((member relname st.featrels) 'yes)
   (t 'no)))
  
(defun st_build_situation (name type)
;;;
;;;   Return a situation the original type is saved and the basic
;;;   list of facts is found by 'applying' the type to rhe given name
;;;
  (let ((newsit (make-situation)))
    (if (consp (car type))
	(pruleperror (format nil "Situation ~A cannot have variable type ~A"
		       name type)))
    (setf (situation-name newsit) name)
    (setf (situation-type newsit) type)
    (if type           ; there is a type specified
	(setf (situation-tableau newsit)
	      (st_replace_name (car type) name (cadr type))))
    newsit))
	   
(defun st_build_rule (rule)
;;;
;;;  Build rule getting cross references of variables right
;;;  The list of variables to deal with are in st.scope
;;;
  (or t
   (mapcar
    #'(lambda (varvalue)
	(mapcar #'(lambda (category)
		    (st_replace_name (car varvalue) (cdr varvalue)
				     category))
		rule))
    st.scope))
   (setq st.scope nil)    ;; reset the scope ready for next rule
   rule)

(defun st_build_entry (entry)
;;;
;;;  Build lexical entry cross indexing any variables
;;;
   (mapcar
    #'(lambda (varvalue)
	(st_replace_name (car varvalue) (cdr varvalue) entry))
    st.scope)
   (setq st.scope nil)    ;; reset the scope ready for next rule
   entry)

(defun st_build_reduce (abstract anchor)
;;;
;;;  Build a reduce object
;;;
  (list
   'reduce abstract anchor))

(defun st_build_type (abstraction conditions &optional variable_type)
;;;
;;;  Build a type for the given conditions (facts) and abstraction (a
;;;  parameter).  Returns a pair of the abstraction (a variable) and
;;;  the conditions or 'typevar abstraction conditions if a variable
;;;  type is specifieed.  Variable types are specified when the
;;;  original syntax specified the type with a variable.
;;;
  (let ((name (st_newvar)))
    (cond
      ((member variable_type st.variables)
       (list
	(list 'typevar variable_type)
	name
	(st_replace_name abstraction name conditions)))
      (t
       (list
	name
	(st_replace_name abstraction name conditions))))))

(defun st_replace_name (name value term &optional terms)
;;;
;;; replace name with value throughout term
;;; terms is a list of terms already processed
;;;
  (cond
    ((member term terms)
     term)
    ((eq name term) value)
    ((situation-p term)
     (setf (situation-name term) (situation-name term))
     (setf (situation-tableau term)
	   (st_replace_name name value (situation-tableau term) 
			    (cons term terms)))
     term)
    ((variable-p term)
     (if (si-sitvarp term)
	 (progn
	   (setf (variable-facts term)
		 (st_replace_name name value (variable-facts term) 
				  (cons term terms)))
	   (setf (variable-type term)
		 (st_replace_name name value (variable-type term) 
				  (cons term terms)))))
     term)
    ((atom term)
     term)
    (t
     (mapcar
      #'(lambda (subterm)
	  (cond
	    ((eq subterm name) value)
	    (t 
	     (st_replace_name name value subterm (cons term terms)))))
      term))))

(defun st_newvar ()
;;;
;;; Returns a new variables name
;;;
  (let ((newvar (intern (format nil "~A" (gensym "*")))))
    (setq st.variables (cons newvar st.variables))
    (make-variable :name newvar :type nil)))

(defun st_uninternalise (term &optional donelist)
;;;
;;;  Takes an internalised form and changes it into astl input syntax
;;;  This is pretty hacky, pity I can't get
;;;  rrules to do this.  Returns a lisp-ish format for time being
;;;
  (let (param)
  (cond ((null donelist) (setq donelist (list (cons (list t) t)))))
  (cond
    ((assoc term donelist)
     (cdr (assoc term donelist)))
    ((and (situation-p term)
	  (member (situation-name term) donelist
		  :test #'(lambda (a b) (eq a (cdr b)))))
     ;; sometime the situation structure isn't preserved
     ;; propoerly so depend on the name
     (situation-name term))
    ((situation-p term)
     ;; A new situation not expanded yet
     ;; push the name of this on the done list so we only expand it once
     (si-push-onto (cons term (situation-name term)) donelist)
     (si-push-onto (cons (situation-name term)
			 (situation-name term)) donelist)
     (setq param (intern (string (gensym "P"))))
     (list
      (situation-name term)
      '|::|
      (cons param
	    (cons
	     '!
	    (mapcar #'(lambda (fact) 
			(list param
			      '!=
			      (st_uninternalise fact donelist)))
		    (reverse
		     (st-remove-hushed
		      (sit-point-facts
		       (cdr (assoc (situation-name term)
				   scp-sit-points))))))))))
    ((assoc term scp-sit-points)
     ;; A new situation not expanded yet
     ;; push the name of this on the done list so we only expand it once
     (si-push-onto (cons term term) donelist)
     (setq param (intern (string (gensym "P"))))
     (list
      term
      '|:|
      (cons param
	    (cons
	     '!
	    (mapcar #'(lambda (fact) 
			(list param
			      '!=
			      (st_uninternalise fact donelist)))
		    (reverse
		     (st-remove-hushed
		      (sit-point-facts
		       (cdr (assoc term scp-sit-points))))))))))
    ((variable-p term)
     (cond
       ((null (si-sitvarp term))
	(variable-name term))
       (t
	(si-push-onto (cons term (variable-name term)) donelist)
	(setq param (intern (string (gensym "P"))))
	(list
	 (variable-name term)
	 '|:|
	 (cons param
	       (cons
		'!
		(mapcar #'(lambda (fact) 
			    (list param
				  '!=
				  (st_uninternalise fact donelist)))
			(st-remove-hushed
			 (variable-facts term)))))))))
    ((atom term)
     ;; individual of some sort
     term)
    ((listp term)
     ;; a fact (I hope)
     (mapcar
      #'(lambda (thingy) 
	  (st_uninternalise thingy donelist)) term))
    (t
     (cons 'DUNNO term)))))

(defun st-remove-hushed (facts)
  "Removes facts which have a relation name that is declared hushed."
  (remove-if
   #'(lambda (fact)
       (member (car fact) st.hush))
   facts))

(defun st_gensitname ()
;;;
;;;  Generate a new name
  (values (intern (string (gensym "S")))))

;;;
;;;  The ASTL top level loop 
;;;
;;;  accepts a number of different commands such as load, parse,
;;;  and prove and quit
;;;
(defun astl ()
;;;
;;;  Start astl top level reader.  read commands from standard input
;;;  and processes them
;;;
  (let (command)
    (do ()
	((eq (car command) 'quit)
	 t)
	(terpri) (princ "astl> ")   ; prompt
        #+:cmu(terpri)
        (setq command (st_readcommand))
	(cond
	 ((null command) t)
	 ((eq (car command) 'load)
	  (st_load (cadr command)))
	 ((eq (car command) 'parse)
	  (st_parse (cadr command)))
	 ((eq (car command) 'goalprop)
	  (setq st.topcat (cadr command))
	  (princ "GoalProp reset."))
	 ((eq (car command) 'prove)
	  (st_prove (cadr command)))
	 ((eq (car command) 'hush)
	  (st_hush_add (cadr command)))
	 ((eq (car command) 'unhush)
	  (st_hush_remove (cadr command)))
	 ((eq (car command) 'ekn)
	  (setq st.ekn-mode (null st.ekn-mode))
	  (princ (format nil "EKN mode is ~A." (if st.ekn-mode 'on 'off))))
	 ((eq (car command) 'debug)
	  (st_debug (cadr command)))
	 ((eq (car command) 'undebug)
	  (st_undebug (cadr command)))
	 ((eq (car command) 'help)
	  (st_help))
	 ((eq (car command) 'quit)
	  t)
	 (t t)))))

(defun st_readcommand ()
;;;
;;;  Read a command from the command line
;;;
  (pruleparse 'stcommand 't))

(defun st_load (filename)
;;;
;;;  Loads and parses an st file
;;;
  (if (probe-file filename)
      (let ()
	(princ (format nil "loading [~A] ... " filename))
	(finish-output)
	(pruleparse 'stgram filename)
	(princ "done."))
      (if (probe-file (concatenate 'string filename ".st"))
	  (let ((fullfilename (concatenate 'string filename ".st")))
	    (princ (format nil "loading [~A] ... " fullfilename))
	    (finish-output)
	    (pruleparse 'stgram fullfilename)
	    (princ "done."))
	  (princ (format nil "~A and ~A.st not found."
			 filename filename)))))

(defun st_parse (sentence)
;;;
;;;  parse a sentence using the chart parser
;;;
  (cond
    ((situation-p st.topcat)
     (cond
       (st.time
	(time
	 (st-print-solutions
	  (scp-parse sentence
		     (si-uniquifyterm st.topcat)) 1)))
       (t
	(st-print-solutions
	 (scp-parse sentence
		    (si-uniquifyterm st.topcat)) 1))))
    (t
     (terpri)
     (princ "GoalProp not set to a valid query."))))

(defun st_prove (situation)
;;;
;;;  Try to prove if there is a situation of this type
;;;
  (cond
    (st.time
     (time
      (let ((u-sit (si-uniquifyterm situation)))
	(st-print-solutions
	 (scp-prove u-sit) 1))))
    (t
      (let ((u-sit (si-uniquifyterm situation)))
	(st-print-solutions
	 (scp-prove u-sit) 1)))))

(defun st_debug (symbol)
;;;
;;; set debug flag
;;; Currently choices are rules or proof
;;;
  (cond
    ((eq symbol 'rules)
     (setq prule.debug t)
     (princ "rules debug mode switched ON"))
    ((eq symbol 'proof)
     (setq debug.scp t)
     (princ "proof debug mode switched ON"))
    ((eq symbol 'time)
     (setq st.time t)
     (princ "timing switched ON"))
    (t
     (princ "Unknown debug option.  Options are rules time or proof"))))

(defun st_undebug (symbol)
;;;
;;; unset debug flag
;;; Currently choices are rules or proof
;;;
  (cond
    ((eq symbol 'rules)
     (setq prule.debug nil)
     (princ "rules debug mode switched OFF"))
    ((eq symbol 'proof)
     (setq debug.scp nil)
     (princ "proof debug mode switched OFF"))
    ((eq symbol 'time)
     (setq st.time nil)
     (princ "timing switched OFF"))
    (t
     (princ "Unknown debug option.  Options are rules or proof"))))

(defun st-print-solutions (results sol-num)
;;;
;;; Print out the solutions
;;;
  (cond
    ((null results)
     (if (= sol-num 1)
	 (progn
	   (terpri) (princ "Dunno for sure")))
     nil)
    (t
     (terpri)
     (princ (format nil "Solution ~A" sol-num))
     (if st.ekn-mode
	 (ekn (car results))
	 (print (st_uninternalise (car results))))
     (st-print-solutions
      (cdr results) (1+ sol-num)))))
  

(defun st_hush_add (addlist)
  "Add the give relations to the hush list"
  (setq st.hush
	(remove-duplicates
	 (append addlist st.hush)))
  (princ (format nil "Hush list is: ~A" st.hush)))

(defun st_hush_remove (removelist)
  "Remove the given list from st.hush.  If addlist is nil remove
all relations from the hush list."
  (cond
   ((null removelist)
    (setq st.hush nil))
   (t
    (mapc
     #'(lambda (relation)
	 (setq st.hush (remove relation st.hush)))
     removelist)))
  (princ (format nil "Hush list is: ~A" st.hush)))

(defun st_help ()
;;;
;;;  Prints out simple help for available commands
;;;
  (princ (format nil "Command options for : ~a" (astl-version)))
  (terpri) ;;(terpri)
  (princ "  load <string>.             : load ASTL description file")
  (terpri)
  (princ "  parse (w1 w2 ...).         : parse utterance")
  (terpri)
  (princ "  prove <proposition>.       : try to prove proposition")
  (terpri)
  (princ "  goalprop <proposition>.    : set goal proposition")
  (terpri)
  (princ "  hush [r1 | (r1 r2 ...)].   : do not display relations in output")
  (terpri)
  (princ "  unhush [r1 | (r1 r2 ...)]. : (re)display relations in output")
  (terpri)
  (princ "  ekn.                       : toggle EKN output display mode")
  (terpri)
  (princ "  debug [rules|proof|time].  : switch debug mode for rules or proof on")
  (terpri)
  (princ "  undebug [rules|proof|time].: switch debug mode for rules or proof off")
  (terpri)
  (princ "  quit.                      : exit from ASTL to lisp")
  (terpri)
  (princ "  help.                      : this information")
  (terpri)
  (princ "  ().                        : nothing")
  )
