/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic morris
 *
 *   $Id: menu.c,v 1.4 2003/11/18 23:04:59 dom Exp $
 *   $Date: 2003/11/18 23:04:59 $
 *
 *
 *   Constructs a menu and all such things
 *
 */

#include "mediamvp.h"
#include "keys.h"


#define BLACK   0x00, 0x00, 0x00
#define RED     0xFC, 0x14, 0x14
#define GREEN   0x24, 0xFC, 0x24
#define YELLOW  0xFC, 0xC0, 0x24
#define BLUE    0x00, 0x00, 0xFC
#define CYAN    0x00, 0xFC, 0xFC
#define MAGENTA 0xB0, 0x00, 0xFC
#define WHITE   0xFC, 0xFC, 0xFC


typedef struct {
    char    *name;
    int     (*callback_func)(menu_t *menu,app_t *app, int sel);
} option_t;


struct _menu {
    render_t     *render;
    app_t        *app;
    gui_t        *gui;
    int           numoptions;
    option_t     *options;
    char         *title;
    int           curline;
    int           flags;
    menu_t        *parent;
    menu_t        *child;
};

static int        menu_keys(void *ptr, int code);



menu_t *new_menu(render_t *render, menu_t *parent, app_t *app, gui_t *gui, int flags)
{
    menu_t       *menu = malloc(sizeof(*menu));

    menu->render = render;
    menu->app    = app;
    menu->gui    = gui;
    menu->title  = NULL;
    menu->numoptions = 0;
    menu->options = NULL;
    menu->curline = 0;
    menu->flags   = flags;

    if ( parent ) {
	menu->parent = parent;
	parent->child = menu;
    } else {
	menu->parent = NULL;
    }
    menu->child = NULL;

    return menu;
}

void delete_menu_tree(menu_t *menu)
{
    menu_t   *temp;

    while ( menu->child ) {
	menu = menu->child;
    }

    while ( menu->parent ) {
	temp = menu->parent;
	delete_menu(temp);
	menu = temp;
    }

}

void delete_menu(menu_t *menu)
{
    int     i;

    if ( menu->title ) {
        free(menu->title);
	menu->title = NULL;
    }

    for ( i = 0; i < menu->numoptions; i++ ) {
        free(menu->options[i].name);
    }
    if ( menu->options ) {
        free(menu->options);
    }
}

void menu_set_title(menu_t *menu, char *text)
{
    if ( menu->title ) {
        free(menu->title);
    }

    menu->title = strdup(text);
}

void menu_add(menu_t *menu, char *name, int (*callback_func)(menu_t *menu,app_t *app, int sel))
{
    int     i = menu->numoptions++;

    menu->options = realloc(menu->options,menu->numoptions * sizeof(option_t));

    menu->options[i].name = strdup(name);
    menu->options[i].callback_func = callback_func;
}




void menu_display(menu_t *menu)
{
    char          buf[256];
    render_t     *render = menu->render;
    int           row;
    int           i;

    while ( menu->child ) {
	menu = menu->child;
    }

    render_set_bgcol(render,BLACK);
    render_clear(render);

    snprintf(buf,sizeof(buf)," %s\n",menu->title ? menu->title : "");
    render_set_bgcol(render,CYAN);
    render_set_fgcol(render,BLACK);
    render_printrow(render,0,buf);

    for ( row = 2, i = 0; i < menu->numoptions; i++, row++ ) {
        if ( i < 9 && menu->flags & MENU_NUMBERS ) {
            snprintf(buf,sizeof(buf)," %d. %s\n",i+1,menu->options[i].name);
        } else {
            snprintf(buf,sizeof(buf),"   %s\n",menu->options[i].name);
        }

        if ( i == menu->curline ) {
            render_set_bgcol(render,CYAN);
            render_set_fgcol(render,BLACK);
        } else {
            render_set_bgcol(render,BLACK);
            render_set_fgcol(render,WHITE);
        }
        render_printrow(render,row,buf);
    }
    gui_register_keys(menu->gui,KEYS_MENU,menu_keys,menu);
}


static int menu_keys(void *ptr, int code)
{
    menu_t   *menu = ptr;
    int       pos;
    int     (*func)(menu_t *,app_t *app, int sel);


    switch ( code ) {
    case kChanUp:
        if ( menu->curline > 0 ) {
            menu->curline--;
        } else {
            menu->curline = menu->numoptions - 1;
        }
        menu_display(menu);
        break;
    case kChanDn:
        if ( menu->curline < menu->numoptions - 1 ) {
            menu->curline++;
        } else {
            menu->curline = 0;
        }
        menu_display(menu);
        break;
    case kBack:
	if ( menu->parent ) {
	    menu_t *parent = menu->parent;
	    delete_menu(menu);
	    parent->child = NULL;
	    menu_display(parent);
	}
        break;
    case kGo:
	while ( menu->parent ) {
	    menu_t *parent = menu->parent;
	    delete_menu(menu);
	    menu = parent;
	}
	menu_display(menu);
	break;
    case kOk:
	if ( menu->options != NULL ) {
	    func = menu->options[menu->curline].callback_func;
	    if ( func && func(menu,menu->app,menu->curline) == 0 ) {
		delete_menu(menu);
	    }        
	}
        break;
    case k1 ... k9:
	if ( menu->flags & MENU_NUMBERS ) {
	    pos = code - k1;
	    if ( pos < menu->numoptions && menu->options ) {
		func = menu->options[pos].callback_func;
		if ( func && func(menu,menu->app,pos) == 0 ) {
		    delete_menu(menu);
		}
	    }
	}
        break;
    default:
        printf("Received IR code %d - not used here\n",code);
    }
    return 0;
}
 
