/*
 *   MediaMVP Server Library
 *
 *   (C) 2004 Dave Kelly
 *
 *   $Id: mp3tags.c,v 1.1 2004/05/11 19:26:06 dom Exp $
 *   $Date: 2004/05/11 19:26:06 $
 *
 */


#include "id3tag.h"
#include "libmvp.h"

struct _mp3info {
	unsigned char *title;
	unsigned char *artist;
	unsigned char *album;
	unsigned char *track;
	unsigned char *year;
};

mp3info_t *new_mp3info()
{
	mp3info_t *info;

	info = (mp3info_t *)malloc(sizeof(mp3info_t));

	info->title = NULL;
	info->artist = NULL;
	info->album = NULL;
	info->track = NULL;
	info->year = NULL;

	return info;
}

unsigned char *mp3_get_title(mp3info_t *info)
{
	if ( !info ) {
		return NULL;
	}
	return info->title;
}

unsigned char *mp3_get_artist(mp3info_t *info)
{
	if ( !info ) {
		return NULL;
	}
	return info->artist;
}

unsigned char *mp3_get_album(mp3info_t *info)
{
	if ( ! info ) {
		return NULL;
	}
	return info->album;
}

unsigned char *mp3_get_track(mp3info_t *info)
{
	if ( !info ) {
		return NULL;
	}
	return info->track;
}

unsigned char *mp3_get_year(mp3info_t *info)
{
	if ( !info ) {
		return NULL;
	}
	return info->year;
}

static id3_latin1_t *mp3_get_tag(struct id3_tag *tag, char *id)
{
	union id3_field const *field;
	struct id3_frame const *frame;
	id3_latin1_t *latin1 = NULL;

	frame = id3_tag_findframe(tag, id, 0);

	if ( frame == 0 ) {
		return NULL;
	}

	field = id3_frame_field(frame, 1);
	latin1 = id3_ucs4_latin1duplicate(id3_field_getstrings(field, 0));

	return latin1;
}

mp3info_t *mp3_get_info(struct id3_file *mp3_file)
{
	unsigned int i;
	struct id3_tag *tag;
	mp3info_t *info = NULL;
	id3_latin1_t *value;

	char *taglist[] = {
		ID3_FRAME_TITLE,
	 	ID3_FRAME_ARTIST,
		ID3_FRAME_ALBUM,
		ID3_FRAME_TRACK,
		ID3_FRAME_YEAR,
	};

	if ( mp3_file == NULL ) {
		return NULL;
	}

	tag = id3_file_tag(mp3_file);

	if ( tag == NULL ) {
		return NULL;
	}

	info = new_mp3info();

	for ( i = 0; i < sizeof(taglist) / sizeof(taglist[0]); ++i ) {
		value = mp3_get_tag(tag, taglist[i]);

		if(value) {
			if(strcmp(taglist[i], ID3_FRAME_TITLE) == 0) {
				info->title = strdup(value);
			}
			if(strcmp(taglist[i], ID3_FRAME_ARTIST) == 0) {
				info->artist = strdup(value);
			}
			if(strcmp(taglist[i], ID3_FRAME_ALBUM) == 0) {
				info->album = strdup(value);
			}
			if(strcmp(taglist[i], ID3_FRAME_TRACK) == 0) {
				info->track = strdup(value);
			}
			if(strcmp(taglist[i], ID3_FRAME_YEAR) == 0) {
				info->year = strdup(value);
			}
		}

		free(value);
	}

	return info;
}

mp3info_t *mp3_get_info_fd(int mp3_fd)
{
	struct id3_file *mp3_file;
	mp3info_t *info = NULL;
	int dup_fd;

	if ( mp3_fd < 0 ) {
		return NULL;
	}

	dup_fd = dup(mp3_fd);

	if ( dup_fd < 0 ) {
		return NULL;
	}

	mp3_file = id3_file_fdopen(dup_fd, ID3_FILE_MODE_READONLY);

	if ( mp3_file ) {
		info = mp3_get_info(mp3_file);
		id3_file_close(mp3_file);
	}

	close(dup_fd);

	return info;
}

mp3info_t *mp3_get_info_file(char *mp3_filename)
{
	struct id3_file *mp3_file = NULL;
	mp3info_t *info = NULL;

	if ( !mp3_filename ) {
		return NULL;
	}

	mp3_file = id3_file_open(mp3_filename, ID3_FILE_MODE_READONLY);

	if ( mp3_file ) {
		info = mp3_get_info(mp3_file);
		id3_file_close(mp3_file);
	}

	return info;
}

void mp3_info_delete(mp3info_t *info)
{
	if ( !info ) {
		return;
	}

	free(info->title);
	free(info->artist);
	free(info->album);
	free(info->track);
	free(info->year);

	free(info);
}


/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
