/*  XMMS - ALSA output plugin
 *    Copyright (C) 2001-2003 Matthieu Sozeau <mattam@altern.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "alsa.h"
#include <stdio.h>

static GtkWidget *configure_win = NULL;
static GtkWidget *buffer_size_spin, *period_time_spin;
static GtkWidget *use_user_device_check, *user_device_entry, *mmap_button;
static GtkWidget *mixer_device_entry, *mixer_card_spin;

static int audio_device;
static int my_index = 0;

struct device_item {
    int numcard, numdevice;
    char label[128];
};

static struct device_item dev[5];


static void configure_win_ok_cb(GtkWidget * w, gpointer data)
{
    ConfigFile *cfgfile;

    alsa_cfg.use_user_device =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
				     (use_user_device_check));
    g_free(alsa_cfg.user_device);
    alsa_cfg.user_device =
	gtk_editable_get_chars(GTK_EDITABLE(user_device_entry), 0, -1);
    g_free(alsa_cfg.mixer_device);
    alsa_cfg.mixer_card =
	gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(mixer_card_spin));
    alsa_cfg.mixer_device =
	gtk_editable_get_chars(GTK_EDITABLE(mixer_device_entry), 0, -1);
    alsa_cfg.audio_device = dev[audio_device].numdevice;
    alsa_cfg.audio_card = dev[audio_device].numcard;
    alsa_cfg.buffer_time =
	gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					 (buffer_size_spin));
    alsa_cfg.period_time =
	gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON
					 (period_time_spin));
    alsa_cfg.mmap =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(mmap_button));

    cfgfile = xmms_cfg_open_default_file();

    xmms_cfg_write_int(cfgfile, "ALSA", "audio_card", alsa_cfg.audio_card);
    xmms_cfg_write_int(cfgfile, "ALSA", "audio_device",
		       alsa_cfg.audio_device);
    xmms_cfg_write_int(cfgfile, "ALSA", "buffer_time",
		       alsa_cfg.buffer_time);
    xmms_cfg_write_int(cfgfile, "ALSA", "period_time",
		       alsa_cfg.period_time);
    xmms_cfg_write_boolean(cfgfile, "ALSA", "use_user_device",
			   alsa_cfg.use_user_device);
    xmms_cfg_write_boolean(cfgfile, "ALSA", "mmap", alsa_cfg.mmap);
    xmms_cfg_write_string(cfgfile, "ALSA", "user_device",
			  alsa_cfg.user_device);
    xmms_cfg_write_int(cfgfile, "ALSA", "mixer_card", alsa_cfg.mixer_card);
    xmms_cfg_write_string(cfgfile, "ALSA", "mixer_device",
			  alsa_cfg.mixer_device);
    xmms_cfg_write_default_file(cfgfile);
    xmms_cfg_free(cfgfile);

}

static void configure_win_audio_dev_cb(GtkWidget * widget, int device)
{
    audio_device = device;
}

static void user_device_toggled(GtkToggleButton * widget, gpointer data)
{
    gboolean use_user_device = gtk_toggle_button_get_active(widget);
    gtk_widget_set_sensitive(GTK_WIDGET(data), !use_user_device);
    gtk_widget_set_sensitive(user_device_entry, use_user_device);
}

static void get_alsa_name(char *b, struct device_item *dev)
{
    int i = 0, cpt = 0;
    char tmp;
    int len;
    strncpy(&tmp, b + 1, 1);
    dev->numcard = (int) tmp - 48;
    strncpy(&tmp, b + 4, 1);
    dev->numdevice = (int) tmp - 48;

    do {
	b = index(b, ':') + 1;
	cpt++;
    } while (cpt < 2);
    (char *) i = index(b, ':');
    len = ((int) i - (int) b);

    strncpy(dev->label, b + 1, len - 1);
    dev->label[len - 2] = '\0';
}



static void scan_devices(char *type, GtkWidget * option_menu,
			 GtkSignalFunc sigfunc)
{
    GtkWidget *menu, *item;
    FILE *file;
    char buffer[256];
    int index = 0;

    menu = gtk_menu_new();

    if (!strcmp(type, "Audio devices:")) {
	my_index = 0;
	if ((file = fopen("/proc/asound/pcm", "r"))) {
	    while (fgets(buffer, 255, file)) {
		if (strstr(buffer, "playback")) {
		    get_alsa_name(buffer, &dev[my_index]);
		    if (alsa_cfg.debug)
			printf("Label: %s Card: %i Device: %i\n",
			       dev[my_index].label, dev[my_index].numcard,
			       dev[my_index].numdevice);
		    if (dev[my_index].numcard == alsa_cfg.audio_card
			&& dev[my_index].numdevice ==
			alsa_cfg.audio_device)
			audio_device = my_index;

		    item =
			gtk_menu_item_new_with_label(dev[my_index].label);
		    gtk_signal_connect(GTK_OBJECT(item), "activate",
				       sigfunc, (gpointer) index++);
		    gtk_widget_show(item);
		    gtk_menu_append(GTK_MENU(menu), item);
		    my_index++;
		}
	    }
	    fclose(file);
	}
    }

    gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), menu);
    gtk_menu_set_active((GtkMenu *) menu, audio_device);
}

void alsa_configure(void)
{
    GtkWidget *vbox, *notebook;
    GtkWidget *dev_vbox, *adevice_frame, *adevice_box, *adevice;
    GtkWidget *mixer_card_box;
    GtkWidget *buffer_frame, *buffer_vbox;
    GtkWidget *period_time_box;
    GtkWidget *buffer_size_box, *buffer_size_label;
    GtkObject *buffer_size_adj;
    GtkWidget *b;
    GtkWidget *bbox, *ok, *cancel;

    if (configure_win) {
	gtk_window_present(GTK_WINDOW(configure_win));
	return;
    }

    configure_win = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    gtk_signal_connect(GTK_OBJECT(configure_win), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed),
		       &configure_win);
    gtk_window_set_title(GTK_WINDOW(configure_win),
			 _("ALSA Driver configuration"));
    gtk_window_set_policy(GTK_WINDOW(configure_win), FALSE, FALSE, FALSE);
    gtk_window_set_position(GTK_WINDOW(configure_win), GTK_WIN_POS_MOUSE);
    gtk_container_border_width(GTK_CONTAINER(configure_win), 10);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(configure_win), vbox);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

    dev_vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(dev_vbox), 5);

    adevice_frame = gtk_frame_new(_("Audio device:"));
    gtk_box_pack_start(GTK_BOX(dev_vbox), adevice_frame, FALSE, FALSE, 0);

    adevice_box = gtk_vbox_new(FALSE, 5);
    gtk_container_set_border_width(GTK_CONTAINER(adevice_box), 5);
    gtk_container_add(GTK_CONTAINER(adevice_frame), adevice_box);

    adevice = gtk_option_menu_new();
    gtk_box_pack_start(GTK_BOX(adevice_box), adevice, TRUE, TRUE, 0);
    scan_devices("Audio devices:", adevice, configure_win_audio_dev_cb);

    gtk_option_menu_set_history(GTK_OPTION_MENU(adevice),
				alsa_cfg.audio_device);

    use_user_device_check =
	gtk_check_button_new_with_label(_("User defined:"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(use_user_device_check),
				 alsa_cfg.use_user_device);
    gtk_signal_connect(GTK_OBJECT(use_user_device_check), "toggled",
		       G_CALLBACK(user_device_toggled), adevice);
    gtk_box_pack_start(GTK_BOX(adevice_box), use_user_device_check, FALSE,
		       FALSE, 0);

    user_device_entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(user_device_entry), alsa_cfg.user_device);
    gtk_box_pack_start(GTK_BOX(adevice_box), user_device_entry, TRUE, TRUE,
		       0);

    mixer_card_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(mixer_card_box),
		       gtk_label_new(_("Mixer card:")), FALSE, FALSE, 0);

    mixer_card_spin =
	gtk_spin_button_new(GTK_ADJUSTMENT
			    (gtk_adjustment_new
			     (alsa_cfg.mixer_card, 0, 50, 1, 100, 100)), 8,
			    0);

    gtk_box_pack_start(GTK_BOX(mixer_card_box), mixer_card_spin, FALSE,
		       FALSE, 0);

    b = gtk_hbox_new(FALSE, 5);

    gtk_box_pack_start(GTK_BOX(b), mixer_card_box, TRUE, TRUE, 0);

    gtk_box_pack_start(GTK_BOX(b), gtk_label_new(_("Mixer device")), FALSE,
		       FALSE, 0);

    mixer_device_entry = gtk_entry_new();
    gtk_entry_set_text(GTK_ENTRY(mixer_device_entry),
		       alsa_cfg.mixer_device);

    gtk_box_pack_start(GTK_BOX(b), mixer_device_entry, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(adevice_box), b, TRUE, TRUE, 0);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dev_vbox,
			     gtk_label_new(_("Device settings")));

    buffer_frame = gtk_frame_new(_("Advanced settings:"));
    gtk_container_set_border_width(GTK_CONTAINER(buffer_frame), 5);

    buffer_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(buffer_frame), buffer_vbox);

    gtk_container_set_border_width(GTK_CONTAINER(buffer_vbox), 5);

    buffer_size_box = gtk_hbox_new(FALSE, 5);

    buffer_size_label = gtk_label_new(_("Buffer time (ms):"));
    gtk_box_pack_start(GTK_BOX(buffer_size_box), buffer_size_label, FALSE,
		       FALSE, 0);

    buffer_size_adj =
	gtk_adjustment_new(alsa_cfg.buffer_time, 200, 1000000, 100, 100,
			   100);
    buffer_size_spin =
	gtk_spin_button_new(GTK_ADJUSTMENT(buffer_size_adj), 8, 0);
    gtk_widget_set_usize(buffer_size_spin, 60, -1);
    gtk_box_pack_start(GTK_BOX(buffer_size_box), buffer_size_spin, FALSE,
		       FALSE, 0);

    gtk_box_pack_start(GTK_BOX(buffer_vbox), buffer_size_box, FALSE, FALSE,
		       0);

    period_time_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(period_time_box),
		       gtk_label_new(_("Period time (ms):")), FALSE, FALSE,
		       0);

    period_time_spin =
	gtk_spin_button_new(GTK_ADJUSTMENT
			    (gtk_adjustment_new
			     (alsa_cfg.period_time, 1, 500, 1, 100, 100)),
			    8, 0);

    gtk_widget_set_usize(period_time_spin, 60, -1);
    gtk_box_pack_start(GTK_BOX(period_time_box), period_time_spin, FALSE,
		       FALSE, 0);

    gtk_box_pack_start(GTK_BOX(buffer_vbox), period_time_box, FALSE, FALSE,
		       0);


    mmap_button = gtk_check_button_new_with_label(_("Mmap mode"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mmap_button),
				 alsa_cfg.mmap);
    gtk_box_pack_start(GTK_BOX(buffer_vbox), mmap_button, FALSE, FALSE, 0);

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), buffer_frame,
			     gtk_label_new(_("Advanced settings")));


    bbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

    ok = gtk_button_new_from_stock(GTK_STOCK_APPLY);
    gtk_signal_connect(GTK_OBJECT(ok), "clicked",
		       GTK_SIGNAL_FUNC(configure_win_ok_cb), NULL);
    GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(bbox), ok, TRUE, TRUE, 0);
    gtk_widget_grab_default(ok);

    cancel = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked",
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(configure_win));
    GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(bbox), cancel, TRUE, TRUE, 0);

    gtk_widget_realize(configure_win);
    gtk_window_set_position(GTK_WINDOW(configure_win), GTK_WIN_POS_CENTER);
    gtk_widget_show_all(configure_win);

    user_device_toggled(GTK_TOGGLE_BUTTON(use_user_device_check), adevice);
}
