
#include "config.h"
#include "i18n.h"

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>
#include <glib/gprintf.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <pthread.h>

#include <X11/Xlib.h>

#include "textbox.h"
#include "mainwin.h"
#include "pixmaps.h"
#include "beep_mini.xpm"

#include "main.h"

#include "controlsocket.h"
#include "pluginenum.h"
#include "sm.h"

#include "about.h"
#include "dnd.h"
#include "dock.h"
#include "dirbrowser.h"
#include "equalizer.h"
#include "hints.h"
#include "input.h"
#include "playlistwin.h"
#include "prefswin.h"
#include "skinwin.h"

#include "hslider.h"
#include "menurow.h"
#include "monostereo.h"
#include "pbutton.h"
#include "playlist.h"
#include "playlist_list.h"
#include "playstatus.h"
#include "sbutton.h"
#include "svis.h"
#include "textbox.h"
#include "urldecode.h"
#include "util.h"
#include "vis.h"
#include "visualization.h"
#include "libbeep/configfile.h"
#include "libbeep/util.h"

enum {
    MAINWIN_SONGNAME_FILEINFO, MAINWIN_SONGNAME_JTF, MAINWIN_SONGNAME_JTT,
    MAINWIN_SONGNAME_SCROLL
};

enum {
    MAINWIN_OPT_PREFS, MAINWIN_OPT_SKIN, MAINWIN_OPT_RELOADSKIN,
    MAINWIN_OPT_REPEAT, MAINWIN_OPT_SHUFFLE, MAINWIN_OPT_NPA,
    MAINWIN_OPT_TELAPSED, MAINWIN_OPT_TREMAINING, MAINWIN_OPT_ALWAYS,
    MAINWIN_OPT_STICKY, MAINWIN_OPT_WS, MAINWIN_OPT_PWS,
    MAINWIN_OPT_EQWS, MAINWIN_OPT_DOUBLESIZE, MAINWIN_OPT_EASY_MOVE
};

enum {
    MAINWIN_VIS_ANALYZER, MAINWIN_VIS_SCOPE, MAINWIN_VIS_OFF,
    MAINWIN_VIS_ANALYZER_NORMAL, MAINWIN_VIS_ANALYZER_FIRE,
    MAINWIN_VIS_ANALYZER_VLINES,
    MAINWIN_VIS_ANALYZER_LINES, MAINWIN_VIS_ANALYZER_BARS,
    MAINWIN_VIS_ANALYZER_PEAKS,
    MAINWIN_VIS_SCOPE_DOT, MAINWIN_VIS_SCOPE_LINE, MAINWIN_VIS_SCOPE_SOLID,
    MAINWIN_VIS_VU_NORMAL, MAINWIN_VIS_VU_SMOOTH,
    MAINWIN_VIS_REFRESH_FULL, MAINWIN_VIS_REFRESH_HALF,
    MAINWIN_VIS_REFRESH_QUARTER, MAINWIN_VIS_REFRESH_EIGHTH,
    MAINWIN_VIS_AFALLOFF_SLOWEST, MAINWIN_VIS_AFALLOFF_SLOW,
    MAINWIN_VIS_AFALLOFF_MEDIUM, MAINWIN_VIS_AFALLOFF_FAST,
    MAINWIN_VIS_AFALLOFF_FASTEST,
    MAINWIN_VIS_PFALLOFF_SLOWEST, MAINWIN_VIS_PFALLOFF_SLOW,
    MAINWIN_VIS_PFALLOFF_MEDIUM, MAINWIN_VIS_PFALLOFF_FAST,
    MAINWIN_VIS_PFALLOFF_FASTEST,
    MAINWIN_VIS_PLUGINS
};

/*
 * If you change the menu above change these defines also
 */

#define MAINWIN_VIS_MENU_VIS_MODE		1
#define MAINWIN_VIS_MENU_NUM_VIS_MODE		3
#define MAINWIN_VIS_MENU_ANALYZER_MODE		5
#define MAINWIN_VIS_MENU_NUM_ANALYZER_MODE	3
#define MAINWIN_VIS_MENU_ANALYZER_TYPE	    	9
#define MAINWIN_VIS_MENU_NUM_ANALYZER_TYPE  	2
#define MAINWIN_VIS_MENU_ANALYZER_PEAKS		12
#define MAINWIN_VIS_MENU_SCOPE_MODE		14
#define MAINWIN_VIS_MENU_NUM_SCOPE_MODE		3
#define MAINWIN_VIS_MENU_WSHADEVU_MODE		18
#define MAINWIN_VIS_MENU_NUM_WSHADEVU_MODE	2
#define MAINWIN_VIS_MENU_REFRESH_RATE		21
#define MAINWIN_VIS_MENU_NUM_REFRESH_RATE	4
#define MAINWIN_VIS_MENU_AFALLOFF		26
#define MAINWIN_VIS_MENU_NUM_AFALLOFF		5
#define MAINWIN_VIS_MENU_PFALLOFF		32
#define MAINWIN_VIS_MENU_NUM_PFALLOFF		5


GtkWidget *mainwin = NULL;

static GdkBitmap *nullmask;
static gint balance;

static pthread_mutex_t title_mutex = PTHREAD_MUTEX_INITIALIZER;

static GtkWidget *mainwin_url_window = NULL, *mainwin_dir_browser = NULL;
GtkWidget *mainwin_jtt = NULL, *mainwin_jtf = NULL;
GtkItemFactory *mainwin_options_menu, *mainwin_songname_menu,
    *mainwin_vis_menu;
GtkItemFactory *mainwin_general_menu, *mainwin_play_menu;
static GdkPixmap *mainwin_bg = NULL, *mainwin_bg_dblsize;
GdkGC *mainwin_gc;
static GtkAccelGroup *mainwin_accel;

PButton *mainwin_menubtn, *mainwin_minimize, *mainwin_shade,
    *mainwin_close;
PButton *mainwin_rew, *mainwin_play, *mainwin_pause, *mainwin_stop,
    *mainwin_fwd, *mainwin_eject;
TButton *mainwin_shuffle, *mainwin_repeat, *mainwin_eq, *mainwin_pl;
TextBox *mainwin_info, *mainwin_rate_text, *mainwin_freq_text,
    *mainwin_stime_min, *mainwin_stime_sec;

PlayStatus *mainwin_playstatus;

Number *mainwin_minus_num, *mainwin_10min_num, *mainwin_min_num,
    *mainwin_10sec_num, *mainwin_sec_num;

gint bitrate = 0, frequency = 0, numchannels = 0;
gboolean setting_volume = FALSE;

Vis *active_vis;
Vis *mainwin_vis;
SVis *mainwin_svis;

static MenuRow *mainwin_menurow;
static HSlider *mainwin_volume, *mainwin_balance, *mainwin_position,
    *mainwin_sposition = NULL;
static MonoStereo *mainwin_monostereo;
static SButton *mainwin_srew, *mainwin_splay, *mainwin_spause,
    *mainwin_sstop, *mainwin_sfwd, *mainwin_seject, *mainwin_about;

static GList *mainwin_wlist = NULL;

gboolean mainwin_focus = FALSE;

gint mainwin_timeout_tag;

static gboolean mainwin_force_redraw = FALSE;
static gchar *mainwin_title_text = NULL;
static gboolean mainwin_info_text_locked = FALSE;

void mainwin_songname_menu_callback(gpointer cb_data, guint action,
				    GtkWidget * w);
void mainwin_general_menu_callback(gpointer cb_data, guint action,
				   GtkWidget * w);
void mainwin_vis_menu_callback(gpointer cb_data, guint action,
			       GtkWidget * w);
void mainwin_options_menu_callback(gpointer cb_data, guint action,
				   GtkWidget * w);

static GtkItemFactoryEntry mainwin_songname_menu_entries[] = {
    {N_("/View Track Details"), "<alt>i", mainwin_general_menu_callback,
     MAINWIN_GENERAL_FILEINFO, "<ImageItem>", my_pixbuf},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Autoscroll Songname"), NULL, mainwin_songname_menu_callback,
     MAINWIN_SONGNAME_SCROLL, "<ToggleItem>"},
};

static gint mainwin_songname_menu_entries_num =
    sizeof(mainwin_songname_menu_entries) /
    sizeof(mainwin_songname_menu_entries[0]);

static GtkItemFactoryEntry mainwin_vis_menu_entries[] = {
    {N_("/Visualization Mode"), NULL, NULL, 0, "<Branch>"},
    {N_("/Visualization Mode/Analyzer"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER, "<RadioItem>"},
    {N_("/Visualization Mode/Scope"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_SCOPE, "/Visualization Mode/Analyzer"},
    {N_("/Visualization Mode/Off"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_OFF, "/Visualization Mode/Analyzer"},
    {N_("/Analyzer Mode"), NULL, NULL, 0, "<Branch>"},
    {N_("/Analyzer Mode/Normal"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_NORMAL, "<RadioItem>"},
    {N_("/Analyzer Mode/Fire"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_FIRE, "/Analyzer Mode/Normal"},
    {N_("/Analyzer Mode/Vertical Lines"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_VLINES, "/Analyzer Mode/Normal"},
    {N_("/Analyzer Mode/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Analyzer Mode/Lines"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_LINES, "<RadioItem>"},
    {N_("/Analyzer Mode/Bars"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_BARS, "/Analyzer Mode/Lines"},
    {N_("/Analyzer Mode/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Analyzer Mode/Peaks"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_ANALYZER_PEAKS, "<ToggleItem>"},
    {N_("/Scope Mode"), NULL, NULL, 0, "<Branch>"},
    {N_("/Scope Mode/Dot Scope"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_SCOPE_DOT, "<RadioItem>"},
    {N_("/Scope Mode/Line Scope"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_SCOPE_LINE, "/Scope Mode/Dot Scope"},
    {N_("/Scope Mode/Solid Scope"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_SCOPE_SOLID, "/Scope Mode/Dot Scope"},
    {N_("/WindowShade VU Mode"), NULL, NULL, 0, "<Branch>"},
    {N_("/WindowShade VU Mode/Normal"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_VU_NORMAL, "<RadioItem>"},
    {N_("/WindowShade VU Mode/Smooth"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_VU_SMOOTH, "/WindowShade VU Mode/Normal"},
    {N_("/Refresh Rate"), NULL, NULL, 0, "<Branch>"},
    {N_("/Refresh Rate/Full (~50 fps)"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_REFRESH_FULL, "<RadioItem>"},
    {N_("/Refresh Rate/Half (~25 fps)"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_REFRESH_HALF, "/Refresh Rate/Full (~50 fps)"},
    {N_("/Refresh Rate/Quarter (~13 fps)"), NULL,
     mainwin_vis_menu_callback, MAINWIN_VIS_REFRESH_QUARTER,
     "/Refresh Rate/Full (~50 fps)"},
    {N_("/Refresh Rate/Eighth (~6 fps)"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_REFRESH_EIGHTH, "/Refresh Rate/Full (~50 fps)"},
    {N_("/Analyzer Falloff"), NULL, NULL, 0, "<Branch>"},
    {N_("/Analyzer Falloff/Slowest"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_AFALLOFF_SLOWEST, "<RadioItem>"},
    {N_("/Analyzer Falloff/Slow"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_AFALLOFF_SLOW, "/Analyzer Falloff/Slowest"},
    {N_("/Analyzer Falloff/Medium"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_AFALLOFF_MEDIUM, "/Analyzer Falloff/Slowest"},
    {N_("/Analyzer Falloff/Fast"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_AFALLOFF_FAST, "/Analyzer Falloff/Slowest"},
    {N_("/Analyzer Falloff/Fastest"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_AFALLOFF_FASTEST, "/Analyzer Falloff/Slowest"},
    {N_("/Peaks Falloff"), NULL, NULL, 0, "<Branch>"},
    {N_("/Peaks Falloff/Slowest"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_PFALLOFF_SLOWEST, "<RadioItem>"},
    {N_("/Peaks Falloff/Slow"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_PFALLOFF_SLOW, "/Peaks Falloff/Slowest"},
    {N_("/Peaks Falloff/Medium"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_PFALLOFF_MEDIUM, "/Peaks Falloff/Slowest"},
    {N_("/Peaks Falloff/Fast"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_PFALLOFF_FAST, "/Peaks Falloff/Slowest"},
    {N_("/Peaks Falloff/Fastest"), NULL, mainwin_vis_menu_callback,
     MAINWIN_VIS_PFALLOFF_FASTEST, "/Peaks Falloff/Slowest"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Visualization plugins"), "<control>V", mainwin_vis_menu_callback,
     MAINWIN_VIS_PLUGINS, "<StockItem>", GTK_STOCK_PROPERTIES}
};

static gint mainwin_vis_menu_entries_num =
    sizeof(mainwin_vis_menu_entries) / sizeof(mainwin_vis_menu_entries[0]);

static GtkItemFactoryEntry mainwin_play_menu_entries[] = {
    {N_("/Jump to File"), "J", mainwin_general_menu_callback,
     MAINWIN_GENERAL_JTF, "<StockItem>", GTK_STOCK_JUMP_TO},
    {N_("/Jump to Time"), "<control>J", mainwin_general_menu_callback,
     MAINWIN_GENERAL_JTT, "<StockItem>", GTK_STOCK_JUMP_TO},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Previous"), "z", mainwin_general_menu_callback,
     MAINWIN_GENERAL_PREV, "<Item>"},
    {N_("/Play"), "x", mainwin_general_menu_callback, MAINWIN_GENERAL_PLAY,
     "<Item>"},
    {N_("/Pause"), "c", mainwin_general_menu_callback,
     MAINWIN_GENERAL_PAUSE, "<Item>"},
    {N_("/Stop"), "v", mainwin_general_menu_callback, MAINWIN_GENERAL_STOP,
     "<Item>"},
    {N_("/Next"), "b", mainwin_general_menu_callback, MAINWIN_GENERAL_NEXT,
     "<Item>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/5 Seconds REW"), "<alt>1", mainwin_general_menu_callback,
     MAINWIN_GENERAL_BACK5SEC, "<Item>"},
    {N_("/5 Seconds FFWD"), "<alt>2", mainwin_general_menu_callback,
     MAINWIN_GENERAL_FWD5SEC, "<Item>"},
    {N_("/Jump to Playlist Start"), "<control>Z",
     mainwin_general_menu_callback, MAINWIN_GENERAL_START, "<Item>"},
};


static const int mainwin_play_menu_entries_num =
    sizeof(mainwin_play_menu_entries) /
    sizeof(mainwin_play_menu_entries[0]);


static GtkItemFactoryEntry mainwin_general_menu_entries[] = {
    {N_("/View Track details"), "<alt>I", mainwin_general_menu_callback,
     MAINWIN_GENERAL_FILEINFO, "<ImageItem>", my_pixbuf},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Add Files..."), "<control>f", mainwin_general_menu_callback,
     MAINWIN_GENERAL_PLAYDIRECTORY, "<StockItem>", GTK_STOCK_OPEN},
    {N_("/Add URL..."), "<control>h", mainwin_general_menu_callback,
     MAINWIN_GENERAL_PLAYLOCATION, "<Item>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Play CD"), "<control>D", mainwin_general_menu_callback,
     MAINWIN_GENERAL_PLAYCD, "<StockItem>", GTK_STOCK_CDROM},
    {N_("/Playback"), NULL, NULL, 0, "<ImageItem>", play_pixbuf},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Preferences"), "<control>P", mainwin_options_menu_callback,
     MAINWIN_OPT_PREFS, "<StockItem>", GTK_STOCK_PREFERENCES},
    {N_("/Skin Browser"), "<alt>S", mainwin_options_menu_callback,
     MAINWIN_OPT_SKIN, "<Item>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/_Options"), NULL, NULL, 0, "<StockItem>", GTK_STOCK_PROPERTIES},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/_Quit"), NULL, mainwin_general_menu_callback,
     MAINWIN_GENERAL_EXIT, "<StockItem>", GTK_STOCK_QUIT}
};

static const int mainwin_general_menu_entries_num =
    sizeof(mainwin_general_menu_entries) /
    sizeof(mainwin_general_menu_entries[0]);

static GtkItemFactoryEntry mainwin_options_menu_entries[] = {
    {N_("/Show Playlist Editor"), "<alt>E", mainwin_general_menu_callback,
     MAINWIN_GENERAL_SHOWPLWIN, "<ToggleItem>"},
    {N_("/Show Graphical EQ"), "<alt>G", mainwin_general_menu_callback,
     MAINWIN_GENERAL_SHOWEQWIN, "<ToggleItem>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Reload skin"), "F5", mainwin_options_menu_callback,
     MAINWIN_OPT_RELOADSKIN, "<Item>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Repeat"), "R", mainwin_options_menu_callback, MAINWIN_OPT_REPEAT,
     "<ToggleItem>"},
    {N_("/Shuffle"), "S", mainwin_options_menu_callback,
     MAINWIN_OPT_SHUFFLE, "<ToggleItem>"},
    {N_("/No Playlist Advance"), "<control>N",
     mainwin_options_menu_callback, MAINWIN_OPT_NPA, "<ToggleItem>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Time Elapsed"), "<control>E", mainwin_options_menu_callback,
     MAINWIN_OPT_TELAPSED, "<RadioItem>"},
    {N_("/Time Remaining"), "<control>R", mainwin_options_menu_callback,
     MAINWIN_OPT_TREMAINING, "/Time Elapsed"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Always On Top"), "<control>A", mainwin_options_menu_callback,
     MAINWIN_OPT_ALWAYS, "<ToggleItem>"},
    {N_("/Show on all desktops"), "<control>S",
     mainwin_options_menu_callback, MAINWIN_OPT_STICKY, "<ToggleItem>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/WindowShade Mode"), "<control>W", mainwin_options_menu_callback,
     MAINWIN_OPT_WS, "<ToggleItem>"},
    {N_("/Playlist WindowShade Mode"), "<control><shift>W",
     mainwin_options_menu_callback, MAINWIN_OPT_PWS, "<ToggleItem>"},
    {N_("/Equalizer WindowShade Mode"), "<control><alt>W",
     mainwin_options_menu_callback, MAINWIN_OPT_EQWS, "<ToggleItem>"},
    {N_("/DoubleSize"), "<control>D", mainwin_options_menu_callback,
     MAINWIN_OPT_DOUBLESIZE, "<ToggleItem>"},
    {N_("/Easy Move"), "<control>E", mainwin_options_menu_callback,
     MAINWIN_OPT_EASY_MOVE, "<ToggleItem>"},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/About bmp"), NULL, mainwin_general_menu_callback,
     MAINWIN_GENERAL_ABOUT, "<StockItem>", GTK_STOCK_DIALOG_INFO},
};

static const int mainwin_options_menu_entries_num =
    sizeof(mainwin_options_menu_entries) /
    sizeof(mainwin_options_menu_entries[0]);


static void set_timer_mode_menu_cb(TimerMode mode);

void mainwin_set_always_on_top(gboolean always)
{
    GtkWidget *widget = gtk_item_factory_get_widget(mainwin_options_menu,
						    "/Always On Top");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget),
				   mainwin_menurow->mr_always_selected);
}

void mainwin_set_shape_mask(void)
{
    if (!cfg.player_visible || cfg.show_wm_decorations)
	return;

    gtk_widget_shape_combine_mask(mainwin,
				  skin_get_mask(SKIN_MASK_MAIN,
						cfg.doublesize,
						cfg.player_shaded), 0, 0);
}

void mainwin_set_shade(gboolean shaded)
{
    GtkWidget *widget;
    widget = gtk_item_factory_get_widget(mainwin_options_menu,
					 "/WindowShade Mode");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), shaded);
}

void mainwin_set_shade_menu_cb(gboolean shaded)
{
    cfg.player_shaded = shaded;

    mainwin_set_shape_mask();
    if (shaded) {
	dock_shade(dock_window_list, mainwin,
		   MAINWIN_SHADED_HEIGHT * (cfg.doublesize + 1));

	show_widget(mainwin_svis);
	vis_clear_data(mainwin_vis);

	show_widget(mainwin_srew);
	show_widget(mainwin_splay);
	show_widget(mainwin_spause);
	show_widget(mainwin_sstop);
	show_widget(mainwin_sfwd);
	show_widget(mainwin_seject);

	show_widget(mainwin_stime_min);
	show_widget(mainwin_stime_sec);

	if (get_input_playing() && playlist_get_current_length() != -1)
	    show_widget(mainwin_sposition);

	mainwin_shade->pb_ny = mainwin_shade->pb_py = 27;

    } else {
	dock_shade(dock_window_list, mainwin,
		   MAINWIN_HEIGHT * (cfg.doublesize + 1));

	hide_widget(mainwin_svis);
	svis_clear_data(mainwin_svis);

	hide_widget(mainwin_srew);
	hide_widget(mainwin_splay);
	hide_widget(mainwin_spause);
	hide_widget(mainwin_sstop);
	hide_widget(mainwin_sfwd);
	hide_widget(mainwin_seject);

	hide_widget(mainwin_stime_min);
	hide_widget(mainwin_stime_sec);
	hide_widget(mainwin_sposition);

	mainwin_shade->pb_ny = mainwin_shade->pb_py = 18;
    }

    draw_main_window(TRUE);
}

enum {
    MAINWIN_VIS_ACTIVE_MAINWIN, MAINWIN_VIS_ACTIVE_PLAYLISTWIN
};

void mainwin_vis_set_active_vis(gint new_vis)
{
    switch (new_vis) {
    case MAINWIN_VIS_ACTIVE_MAINWIN:
	playlistwin_vis_disable();
	active_vis = mainwin_vis;
	break;
    case MAINWIN_VIS_ACTIVE_PLAYLISTWIN:
	playlistwin_vis_enable();
	active_vis = playlistwin_vis;
	break;
    }
}

void mainwin_vis_set_refresh(RefreshRate rate)
{
    cfg.vis_refresh = rate;
}

void mainwin_vis_set_afalloff(FalloffSpeed speed)
{
    cfg.analyzer_falloff = speed;
}

void mainwin_vis_set_pfalloff(FalloffSpeed speed)
{
    cfg.peaks_falloff = speed;
}

void mainwin_vis_set_analyzer_mode(AnalyzerMode mode)
{
    cfg.analyzer_mode = mode;
}

void mainwin_vis_set_analyzer_type(AnalyzerType mode)
{
    cfg.analyzer_type = mode;
}

void mainwin_vis_set_type(VisType mode)
{
    char *path =
	mainwin_vis_menu_entries[MAINWIN_VIS_MENU_VIS_MODE + mode].path;
    GtkWidget *widget =
	gtk_item_factory_get_widget(mainwin_vis_menu, path);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), TRUE);
}

static void mainwin_vis_set_type_menu_cb(VisType mode)
{
    cfg.vis_type = mode;

    if (mode == VIS_OFF) {
	if (cfg.player_shaded && cfg.player_visible)
	    svis_clear(mainwin_svis);
	else
	    vis_clear(active_vis);
    }
    if (mode == VIS_ANALYZER) {
	vis_clear_data(active_vis);
	svis_clear_data(mainwin_svis);
    }
}

void mainwin_menubtn_cb(void)
{
    gint x, y;
    dock_get_widget_pos(mainwin, &x, &y);
    util_item_factory_popup(mainwin_general_menu,
			    x + 6 * (1 + cfg.doublesize),
			    y + MAINWIN_SHADED_HEIGHT * (1 +
							 cfg.doublesize),
			    1, GDK_CURRENT_TIME);
}

void mainwin_minimize_cb(void)
{
    if (!mainwin->window)
	return;

    gtk_window_iconify(GTK_WINDOW(mainwin));
}

void mainwin_shade_toggle(void)
{
    mainwin_set_shade(!cfg.player_shaded);
}

void mainwin_quit_cb(void)
{
    gtk_widget_hide(equalizerwin);
    gtk_widget_hide(playlistwin);
    gtk_widget_hide(mainwin);
    gdk_flush();
    util_dump_menu_rc();
    gtk_timeout_remove(mainwin_timeout_tag);
    util_set_cursor(NULL);
    save_config();
    cleanup_ctrlsocket();
    playlist_stop_get_info_thread();
    playlist_clear();
    cleanup_plugins();
    sm_cleanup();
    gtk_exit(0);
}

void mainwin_destroy(GtkWidget * widget, gpointer data)
{
    mainwin_quit_cb();
}

void draw_mainwin_titlebar(int focus)
{
    if (focus || !cfg.dim_titlebar)
	skin_draw_pixmap(mainwin_bg, mainwin_gc, SKIN_TITLEBAR,
			 27, 29 * cfg.player_shaded, 0, 0, MAINWIN_WIDTH,
			 MAINWIN_TITLEBAR_HEIGHT);
    else
	skin_draw_pixmap(mainwin_bg, mainwin_gc, SKIN_TITLEBAR,
			 27, (27 * cfg.player_shaded) + 15, 0, 0,
			 MAINWIN_WIDTH, MAINWIN_TITLEBAR_HEIGHT);
}

void draw_main_window(gboolean force)
{
    GdkImage *img, *img2;
    GList *wl;
    Widget *w;
    gboolean redraw;

    if (!cfg.player_visible)
	return;
    lock_widget_list(mainwin_wlist);
    if (force) {
	skin_draw_pixmap(mainwin_bg, mainwin_gc, SKIN_MAIN,
			 0, 0, 0, 0, MAINWIN_WIDTH,
			 cfg.
			 player_shaded ? MAINWIN_SHADED_HEIGHT :
			 MAINWIN_HEIGHT);
	draw_mainwin_titlebar(mainwin_focus);
	draw_widget_list(mainwin_wlist, &redraw, TRUE);
    } else
	draw_widget_list(mainwin_wlist, &redraw, FALSE);

    if (redraw || force) {

	if (force) {
	    if (cfg.doublesize) {
		img = gdk_image_get(mainwin_bg, 0, 0, MAINWIN_WIDTH,
				    cfg.
				    player_shaded ? MAINWIN_SHADED_HEIGHT :
				    MAINWIN_HEIGHT);
		img2 = create_dblsize_image(img);
		gdk_draw_image(mainwin_bg_dblsize, mainwin_gc, img2, 0, 0,
			       0, 0, MAINWIN_WIDTH * 2,
			       cfg.player_shaded ? MAINWIN_SHADED_HEIGHT *
			       2 : MAINWIN_HEIGHT * 2);
		gdk_image_destroy(img2);
		gdk_image_destroy(img);
	    }
	} else {
	    wl = mainwin_wlist;
	    while (wl) {
		w = (Widget *) wl->data;
		if (w->redraw && w->visible) {
		    if (cfg.doublesize) {
			img =
			    gdk_image_get(mainwin_bg, w->x, w->y, w->width,
					  w->height);
			img2 = create_dblsize_image(img);
			gdk_draw_image(mainwin_bg_dblsize, mainwin_gc,
				       img2, 0, 0, w->x << 1, w->y << 1,
				       w->width << 1, w->height << 1);
			gdk_image_destroy(img2);
			gdk_image_destroy(img);
			gdk_window_clear_area(mainwin->window, w->x << 1,
					      w->y << 1, w->width << 1,
					      w->height << 1);
		    } else
			gdk_window_clear_area(mainwin->window, w->x, w->y,
					      w->width, w->height);
		    w->redraw = FALSE;

		}
		wl = wl->next;
	    }
	}
	if (force)
	    gdk_window_clear(mainwin->window);
	gdk_flush();
    }
    unlock_widget_list(mainwin_wlist);
}

void mainwin_set_title(const gchar * text)
{
    gchar *title_win;

    if (!text)
	title_win = g_strdup("bmp");
    else
	title_win = g_strdup_printf("bmp: %s", text);

    gdk_window_set_icon_name(mainwin->window, title_win);
    gdk_window_set_title(mainwin->window, title_win);
    gtk_window_set_title(GTK_WINDOW(mainwin), title_win);

    g_free(title_win);
}

void mainwin_set_info_text(void)
{
    gchar *text;

    if (mainwin_info_text_locked)
	return;

    if ((text = input_get_info_text()) != NULL) {
	textbox_set_text(mainwin_info, text);
    } else if ((text = playlist_get_info_text()) != NULL) {
	textbox_set_text(mainwin_info, text);
    } else
	text = g_strdup_printf("%s %s", PACKAGE, VERSION);

    /* memory leak after g_strdup_print() above! - Des */
    text = g_locale_to_utf8(text, -1, NULL, NULL, NULL);
/* 	mainwin_set_title(text); */
    g_free(text);
}

void mainwin_lock_info_text(gchar * text)
{
    mainwin_info_text_locked = TRUE;
    textbox_set_text(mainwin_info, text);
}

void mainwin_release_info_text(void)
{
    mainwin_info_text_locked = FALSE;
    mainwin_set_info_text();
}

void mainwin_set_song_info(gint rate, gint freq, gint nch)
{
    gchar text[10], *tmp;

    bitrate = rate;
    frequency = freq;
    numchannels = nch;

    if (rate != -1) {
	rate /= 1000;
	if (rate < 1000) {
	    g_sprintf(text, "%3d", rate);
	    textbox_set_text(mainwin_rate_text, text);
	} else {
	    rate /= 100;
	    g_sprintf(text, "%2dH", rate);
	    textbox_set_text(mainwin_rate_text, text);
	}
    } else
	textbox_set_text(mainwin_rate_text, "VBR");

    g_sprintf(text, "%2d", freq / 1000);
    textbox_set_text(mainwin_freq_text, text);
    monostereo_set_num_channels(mainwin_monostereo, nch);

    show_widget(mainwin_minus_num);
    show_widget(mainwin_10min_num);
    show_widget(mainwin_min_num);
    show_widget(mainwin_10sec_num);
    show_widget(mainwin_sec_num);
    if (!get_input_paused())
	playstatus_set_status(mainwin_playstatus, STATUS_PLAY);
    if (playlist_get_current_length() != -1) {
	if (cfg.player_shaded)
	    show_widget(mainwin_sposition);
	show_widget(mainwin_position);
    } else {
	hide_widget(mainwin_position);
	hide_widget(mainwin_sposition);
	mainwin_force_redraw = TRUE;
    }
    if ((tmp = playlist_get_info_text()) != NULL) {
	pthread_mutex_lock(&title_mutex);
	if (mainwin_title_text)
	    g_free(mainwin_title_text);

	// Without the app name prefixed it's much easier to see what's going on
	mainwin_title_text =
	    g_strdup_printf("%s",
			    g_locale_to_utf8(tmp, -1, NULL, NULL, NULL));
	pthread_mutex_unlock(&title_mutex);
	g_free(tmp);
    }
}

void mainwin_clear_song_info(void)
{
    bitrate = 0;
    frequency = 0;
    numchannels = 0;

    mainwin_position->hs_pressed = FALSE;
    mainwin_sposition->hs_pressed = FALSE;
    textbox_set_text(mainwin_rate_text, "   ");
    textbox_set_text(mainwin_freq_text, "  ");
    monostereo_set_num_channels(mainwin_monostereo, 0);
    playstatus_set_status(mainwin_playstatus, STATUS_STOP);
    hide_widget(mainwin_minus_num);
    hide_widget(mainwin_10min_num);
    hide_widget(mainwin_min_num);
    hide_widget(mainwin_10sec_num);
    hide_widget(mainwin_sec_num);
    textbox_set_text(mainwin_stime_min, "   ");
    textbox_set_text(mainwin_stime_sec, "  ");
    hide_widget(mainwin_position);
    hide_widget(mainwin_sposition);
    playlistwin_hide_timer();
    draw_main_window(TRUE);
    vis_clear(active_vis);
    gtk_window_set_title(GTK_WINDOW(mainwin), _("bmp"));
}

void mainwin_disable_seekbar(void)
{
    /*
     * We dont call draw_main_window() here so this will not
     * remove them visually.  It will only prevent us from sending
     * any seek calls to the input plugin before the input plugin
     * calls ->set_info().
     */
    hide_widget(mainwin_position);
    hide_widget(mainwin_sposition);
}

void mainwin_release(GtkWidget * widget, GdkEventButton * event,
		     gpointer callback_data)
{
    gdk_pointer_ungrab(GDK_CURRENT_TIME);

    /*
     * The gdk_flush() is just for making sure that the pointer really
     * gets ungrabbed before calling any button callbacks
     *
     */

    gdk_flush();

    if (dock_is_moving(mainwin)) {
	dock_move_release(mainwin);
    }
    if (mainwin_menurow->mr_doublesize_selected) {
	event->x /= 2;
	event->y /= 2;
    }
    handle_release_cb(mainwin_wlist, widget, event);

    draw_main_window(FALSE);

}

void mainwin_motion(GtkWidget * widget, GdkEventMotion * event,
		    gpointer callback_data)
{
    XEvent ev;
    gint i = 0;

    XSync(GDK_DISPLAY(), False);

    while (XCheckTypedEvent(GDK_DISPLAY(), MotionNotify, &ev)) {
	event->x = ev.xmotion.x;
	event->y = ev.xmotion.y;
	i++;
    }
    if (cfg.doublesize) {
	event->x /= 2;
	event->y /= 2;
    }
    if (dock_is_moving(mainwin)) {
	dock_move_motion(mainwin, event);
    } else {
	handle_motion_cb(mainwin_wlist, widget, event);
	draw_main_window(FALSE);
    }
    gdk_flush();

}

static gboolean inside_sensitive_widgets(gint x, gint y)
{
    return (inside_widget(x, y, mainwin_menubtn)
	    || inside_widget(x, y, mainwin_minimize)
	    || inside_widget(x, y, mainwin_shade)
	    || inside_widget(x, y, mainwin_close)
	    || inside_widget(x, y, mainwin_rew)
	    || inside_widget(x, y, mainwin_play)
	    || inside_widget(x, y, mainwin_pause)
	    || inside_widget(x, y, mainwin_stop)
	    || inside_widget(x, y, mainwin_fwd)
	    || inside_widget(x, y, mainwin_eject)
	    || inside_widget(x, y, mainwin_shuffle)
	    || inside_widget(x, y, mainwin_repeat)
	    || inside_widget(x, y, mainwin_pl)
	    || inside_widget(x, y, mainwin_eq)
	    || inside_widget(x, y, mainwin_info)
	    || inside_widget(x, y, mainwin_menurow)
	    || inside_widget(x, y, mainwin_volume)
	    || inside_widget(x, y, mainwin_balance)
	    || (inside_widget(x, y, mainwin_position)
		&& ((Widget *) mainwin_position)->visible)
	    || inside_widget(x, y, mainwin_minus_num)
	    || inside_widget(x, y, mainwin_10min_num)
	    || inside_widget(x, y, mainwin_min_num)
	    || inside_widget(x, y, mainwin_10sec_num)
	    || inside_widget(x, y, mainwin_sec_num)
	    || inside_widget(x, y, mainwin_vis)
	    || inside_widget(x, y, mainwin_minimize)
	    || inside_widget(x, y, mainwin_shade)
	    || inside_widget(x, y, mainwin_close)
	    || inside_widget(x, y, mainwin_menubtn)
	    || inside_widget(x, y, mainwin_sposition)
	    || inside_widget(x, y, mainwin_stime_min)
	    || inside_widget(x, y, mainwin_stime_sec)
	    || inside_widget(x, y, mainwin_srew)
	    || inside_widget(x, y, mainwin_splay)
	    || inside_widget(x, y, mainwin_spause)
	    || inside_widget(x, y, mainwin_sstop)
	    || inside_widget(x, y, mainwin_sfwd)
	    || inside_widget(x, y, mainwin_seject)
	    || inside_widget(x, y, mainwin_svis)
	    || inside_widget(x, y, mainwin_about));
}

void mainwin_scrolled(GtkWidget * widget, GdkEventScroll * event,
		      gpointer callback_data)
{
    gint d = cfg.mouse_change;
    if (event->direction == GDK_SCROLL_DOWN)
	d *= -1;
    mainwin_set_volume_diff(d);
}


void mainwin_press(GtkWidget * widget, GdkEventButton * event,
		   gpointer callback_data)
{

    gboolean grab = TRUE;

    if (cfg.doublesize) {
	/*
	 * A hack to make doublesize transparent to callbacks.
	 * We should make a copy of this data instead of
	 * tampering with the data we get from gtk+
	 */
	event->x /= 2;
	event->y /= 2;
    }

    if (event->button == 1 && event->type == GDK_BUTTON_PRESS &&
	!inside_sensitive_widgets(event->x, event->y) &&
	(cfg.easy_move || event->y < 14)) {
	if (0 && hint_move_resize_available()) {
	    hint_move_resize(mainwin, event->x_root, event->y_root, TRUE);
	    grab = FALSE;
	} else {
	    gdk_window_raise(mainwin->window);
	    equalizerwin_raise();
	    playlistwin_raise();
	    dock_move_press(dock_window_list, mainwin, event, TRUE);
	}
    } else if (event->button == 1 && event->type == GDK_2BUTTON_PRESS &&
	       event->y < 14
	       && !inside_sensitive_widgets(event->x, event->y)) {
	mainwin_set_shade(!cfg.player_shaded);
	if (dock_is_moving(mainwin))
	    dock_move_release(mainwin);
    } else if (event->button == 1 && event->type == GDK_2BUTTON_PRESS &&
	       inside_widget(event->x, event->y, mainwin_info)) {
	playlist_fileinfo_current();
    } else {
	handle_press_cb(mainwin_wlist, widget, event);
	draw_main_window(FALSE);
    }

    if ((event->button == 1) && event->type != GDK_2BUTTON_PRESS &&
	(inside_widget(event->x, event->y, mainwin_vis) ||
	 inside_widget(event->x, event->y, mainwin_svis))) {
	cfg.vis_type++;
	if (cfg.vis_type > VIS_OFF)
	    cfg.vis_type = VIS_ANALYZER;
	mainwin_vis_set_type(cfg.vis_type);
    }
    if (event->button == 3) {
	if (inside_widget(event->x, event->y, mainwin_info)) {
	    util_item_factory_popup(mainwin_songname_menu,
				    event->x_root, event->y_root,
				    3, event->time);
	    grab = FALSE;
	} else if (inside_widget(event->x, event->y, mainwin_vis) ||
		   inside_widget(event->x, event->y, mainwin_svis)) {
	    util_item_factory_popup(mainwin_vis_menu, event->x_root,
				    event->y_root, 3, event->time);
	    grab = FALSE;
	} else {
	    /*
	     * Pop up the main menu a few pixels down.
	     * This will avoid that anything is selected
	     * if one right-clicks to focus the window
	     * without raising it.
	     */
	    util_item_factory_popup(mainwin_general_menu,
				    event->x_root,
				    event->y_root + 2, 3, event->time);
	    grab = FALSE;
	}
    }
    if (event->button == 1) {
	if ((event->x > 35 && event->x < 100 &&
	     event->y > 25 && event->y < 40) ||
	    inside_widget(event->x, event->y, mainwin_stime_min) ||
	    inside_widget(event->x, event->y, mainwin_stime_sec)) {
	    if (cfg.timer_mode == TIMER_ELAPSED)
		set_timer_mode(TIMER_REMAINING);
	    else
		set_timer_mode(TIMER_ELAPSED);
	}

    }
    if (grab)
	gdk_pointer_grab(mainwin->window, FALSE,
			 GDK_BUTTON_MOTION_MASK |
			 GDK_BUTTON_RELEASE_MASK,
			 GDK_WINDOW(GDK_NONE), NULL, GDK_CURRENT_TIME);
}

static void mainwin_focus_in(GtkWindow * window, GtkWidget * widget,
			     gpointer callback_data)
{
    mainwin_focus = 1;

    mainwin_menubtn->pb_allow_draw = TRUE;
    mainwin_minimize->pb_allow_draw = TRUE;
    mainwin_shade->pb_allow_draw = TRUE;
    mainwin_close->pb_allow_draw = TRUE;
    draw_main_window(TRUE);

    if (cfg.playlist_visible) {
	gdk_window_raise(mainwin->window);
    }

}


void mainwin_focus_out(GtkWidget * widget, GdkEventButton * event,
		       gpointer callback_data)
{
    mainwin_focus = 0;

    mainwin_menubtn->pb_allow_draw = FALSE;
    mainwin_minimize->pb_allow_draw = FALSE;
    mainwin_shade->pb_allow_draw = FALSE;
    mainwin_close->pb_allow_draw = FALSE;
    draw_main_window(TRUE);

}

gint mainwin_keypress(GtkWidget * grab_widget, GdkEventKey * event,
		      gpointer data)
{

    switch (event->keyval) {

    case GDK_Up:
    case GDK_KP_Up:
	mainwin_set_volume_diff(2);
	break;
    case GDK_Down:
    case GDK_KP_Down:
	mainwin_set_volume_diff(-2);
	break;
    case GDK_Left:
    case GDK_KP_Left:
	if (playlist_get_current_length() != -1)
	    input_seek(CLAMP
		       (input_get_time() - 5000, 0,
			playlist_get_current_length()) / 1000);
	break;
    case GDK_Right:
    case GDK_KP_Right:
	if (playlist_get_current_length() != -1)
	    input_seek(CLAMP
		       (input_get_time() + 5000, 0,
			playlist_get_current_length()) / 1000);
	break;
    case GDK_Escape:
	mainwin_minimize_cb();
	break;
    default:
	break;
    }

    return 0;
}

void mainwin_jump_to_time_cb(GtkWidget * widget, GtkWidget * entry)
{
    guint min = 0, sec = 0, params, time;
    gchar *timestr;

    timestr = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));

    params = sscanf(timestr, "%u:%u", &min, &sec);
    if (params == 2)
	time = (min * 60) + sec;
    else if (params == 1)
	time = min;
    else
	return;

    if (playlist_get_current_length() > -1 &&
	time <= (playlist_get_current_length() / 1000)) {
	input_seek(time);
	gtk_widget_destroy(mainwin_jtt);
    }
}

void mainwin_jump_to_time(void)
{
    GtkWidget *vbox, *frame, *vbox_inside, *hbox_new, *hbox_total;
    GtkWidget *time_entry, *label, *bbox, *jump, *cancel;
    guint len, tindex;
    gchar timestr[10];

    if (!get_input_playing())
	return;

    mainwin_jtt = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(mainwin_jtt), _("Jump to time"));
    gtk_window_set_position(GTK_WINDOW(mainwin_jtt), GTK_WIN_POS_CENTER);
    gtk_window_set_policy(GTK_WINDOW(mainwin_jtt), FALSE, FALSE, FALSE);
    gtk_window_set_transient_for(GTK_WINDOW(mainwin_jtt),
				 GTK_WINDOW(mainwin));
    g_signal_connect(G_OBJECT(mainwin_jtt), "destroy",
		     G_CALLBACK(gtk_widget_destroyed), &mainwin_jtt);
    gtk_container_border_width(GTK_CONTAINER(mainwin_jtt), 10);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(mainwin_jtt), vbox);
    gtk_widget_show(vbox);
    frame = gtk_frame_new(_("Jump to:"));
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
    gtk_widget_set_usize(frame, 250, -1);
    gtk_widget_show(frame);
    vbox_inside = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), vbox_inside);
    gtk_widget_show(vbox_inside);

    hbox_new = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox_inside), hbox_new, TRUE, TRUE, 5);
    gtk_widget_show(hbox_new);
    time_entry = gtk_entry_new();
    gtk_box_pack_start(GTK_BOX(hbox_new), time_entry, FALSE, FALSE, 5);
    g_signal_connect(G_OBJECT(time_entry), "activate",
		     G_CALLBACK(mainwin_jump_to_time_cb), time_entry);
    gtk_widget_show(time_entry);
    gtk_widget_set_usize(time_entry, 70, -1);
    label = gtk_label_new(_("minutes:seconds"));
    gtk_box_pack_start(GTK_BOX(hbox_new), label, FALSE, FALSE, 5);
    gtk_widget_show(label);

    hbox_total = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox_inside), hbox_total, TRUE, TRUE, 5);
    gtk_widget_show(hbox_total);
    label = gtk_label_new(_("Track length:"));
    gtk_box_pack_start(GTK_BOX(hbox_total), label, FALSE, FALSE, 5);
    gtk_widget_show(label);
    len = playlist_get_current_length() / 1000;
    g_sprintf(timestr, "%u:%2.2u", len / 60, len % 60);
    label = gtk_label_new(timestr);
    gtk_box_pack_start(GTK_BOX(hbox_total), label, FALSE, FALSE, 10);
    gtk_widget_show(label);

    bbox = gtk_hbutton_box_new();
    gtk_box_pack_start(GTK_BOX(vbox), bbox, TRUE, TRUE, 0);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
    gtk_widget_show(bbox);
    jump = gtk_button_new_with_label(_("Jump"));
    GTK_WIDGET_SET_FLAGS(jump, GTK_CAN_DEFAULT);
    gtk_container_add(GTK_CONTAINER(bbox), jump);
    g_signal_connect(G_OBJECT(jump), "clicked",
		     G_CALLBACK(mainwin_jump_to_time_cb), time_entry);
    gtk_widget_show(jump);
    cancel = gtk_button_new_with_label(_("Cancel"));
    GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
    gtk_container_add(GTK_CONTAINER(bbox), cancel);
    g_signal_connect_swapped(G_OBJECT(cancel), "clicked",
			     G_CALLBACK(gtk_widget_destroy),
			     G_OBJECT(mainwin_jtt));
    gtk_widget_show(cancel);

    tindex = input_get_time() / 1000;
    g_sprintf(timestr, "%u:%2.2u", tindex / 60, tindex % 60);
    gtk_entry_set_text(GTK_ENTRY(time_entry), timestr);

    gtk_entry_select_region(GTK_ENTRY(time_entry), 0, strlen(timestr));

    gtk_window_set_modal(GTK_WINDOW(mainwin_jtt), 1);
    gtk_widget_show(mainwin_jtt);
    gtk_widget_grab_focus(time_entry);
    gtk_widget_grab_default(jump);
}

static void mainwin_jump_to_file_real_cb(GtkWidget * w,
					 GtkTreeView * treeview)
{

    GtkTreeSelection *selection = NULL;
    GtkTreeIter iter;
    gchar *pos;
    guint64 pos_guint;

    GtkTreeModel *model;

    model = gtk_tree_view_get_model(treeview);
    if (gtk_tree_model_get_iter_first(model, &iter)) {

	gtk_tree_model_get(model, &iter, 0, &pos, -1);

	// This *is* kinda retarded, yeah..
	// FIXME: The whole column should be integer not text
	pos_guint = g_ascii_strtoull(pos, NULL, 10);

	if (selection) {
	    if (get_input_playing()) {
		input_stop();
	    }
	    playlist_set_position(pos_guint - 1);
	    playlist_play();
	}
    }
}

static void mainwin_jump_to_file_select_row_cb(GtkTreeSelection * widget,
					       GtkTreePath * arg1,
					       GtkTreeViewColumn * column,
					       gpointer data)
{
    gchar *pos;
    GtkTreeIter iter;
    guint64 pos_guint;
    GtkTreeSelection *selection;

    GtkTreeModel *model;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));

    gtk_tree_selection_get_selected(selection, NULL, &iter);
    gtk_tree_model_get(model, &iter, 0, &pos, -1);
    pos_guint = g_ascii_strtoull(pos, NULL, 10);

    if (get_input_playing())
	input_stop();
    playlist_set_position(pos_guint - 1);
    playlist_play();

/* FIXME: Queuing code:
//		pos = gtk_treeview_get_row_data(widget, GPOINTER_TO_INT(widget->selection->data));
		if (pos)
			gtk_widget_set(GTK_WIDGET(cb_data), "label", playlist_is_position_queued(*pos) ? "Unqueue" : "Queue", NULL);
	}
*/
}

static gboolean mainwin_jump_to_file_keypress_cb(GtkWidget * widget,
						 GdkEventKey * event,
						 gpointer userdata)
{
    if (event && (event->keyval == GDK_Escape)) {
	gtk_widget_destroy(mainwin_jtf);
	return TRUE;
    } else
	return FALSE;
}

/*
static gboolean mainwin_jump_to_file_entry_keypress_cb(GtkWidget * widget, GdkEventKey * event, gpointer userdata)
{
	if (!event)
		return FALSE;
	
	switch (event->keyval)
	{
		case GDK_Escape:
			gtk_widget_destroy(mainwin_jtf);
			break;
		default:
			return FALSE;
	}

	return TRUE;
}
*/

static void mainwin_jump_to_file_jump_cb(GtkButton * widget,
					 gpointer userdata)
{
    mainwin_jump_to_file_real_cb(NULL, GTK_TREE_VIEW(userdata));
}

/*
static void mainwin_jump_to_file_queue_cb(GtkButton * widget, gpointer userdata)
{
	GtkCList *treeview = GTK_CLIST(userdata);
	int *pos;

	if (treeview->selection)
	{
		pos = gtk_treeview_get_row_data(treeview, GPOINTER_TO_INT(treeview->selection->data));
		playlist_queue_position(*pos);
		gtk_widget_set(GTK_WIDGET(widget), "label", playlist_is_position_queued(*pos) ? "Unqueue" : "Queue", NULL);
	}	
}
*/

static gboolean mainwin_jump_to_file_match(gchar * song, gchar ** keys)
{
    gint i = 0;
    gchar *key;
    gchar *song_lc;

    while (keys[i] != NULL) {
	key = g_ascii_strdown(keys[i], strlen(keys[i]));
	song_lc = g_ascii_strdown(song, strlen(song));

	if (!g_strrstr(song_lc, key)) {
	    return FALSE;
	}

	i++;
    }

    return TRUE;
}

// FIXME: Clear the entry when the list gets updated

void mainwin_update_jtf(GtkWidget * widget, gpointer userdata)
{


    // FIXME: Is not in sync with playlist due to delayed extinfo reading
    gint row;
    GList *playlist;
    gchar *desc_buf;
    GtkTreeIter iter;
    GtkTreeSelection *selection;

    GtkListStore *store;

    store =
	(GtkListStore *) gtk_tree_view_get_model(GTK_TREE_VIEW(userdata));


    if (GTK_IS_WIDGET(mainwin_jtf)) {

	playlist = get_playlist();

	gtk_list_store_clear(store);
	if (!playlist)
	    return;

	row = 1;
	while (playlist) {
	    if (((PlaylistEntry *) playlist->data)->title)
		desc_buf = ((PlaylistEntry *) playlist->data)->title;
	    else if (strchr
		     (((PlaylistEntry *) playlist->data)->filename, '/'))
		desc_buf =
		    strrchr(((PlaylistEntry *) playlist->data)->filename,
			    '/') + 1;
	    else
		desc_buf = ((PlaylistEntry *) playlist->data)->filename;

	    gtk_list_store_append(GTK_LIST_STORE(store), &iter);
	    gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			       0, g_strdup_printf("%d", row++),
			       1, desc_buf, -1);
	    playlist = playlist->next;
	}

	gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(userdata));
	gtk_tree_selection_select_iter(selection, &iter);
    }
}

static void mainwin_jump_to_file_edit_cb(GtkWidget * widget,
					 gpointer userdata)
{

    gint songnr = 0;
    gchar *desc_buf;
    gchar **words;
    GList *playlist;
    gchar *key;
    GtkTreeIter iter;
    GtkTreeSelection *selection;
    GtkListStore *store;
    gboolean match = FALSE;

    PLAYLIST_LOCK();
    playlist = get_playlist();
    key = g_strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
    store =
	(GtkListStore *) gtk_tree_view_get_model(GTK_TREE_VIEW(userdata));

    /* Chop the key string into ' '-separeted key words */
    words = g_strsplit(key, " ", 0);

    /* FIXME: Remove the connected signals before clearing (row-selected will still eventually arrive once) */
    gtk_list_store_clear(store);

    /* lowercase the key string */

    while (playlist) {

	char *title, *filename;

	title = ((PlaylistEntry *) playlist->data)->title;
	filename = ((PlaylistEntry *) playlist->data)->filename;

	if (title)
	    desc_buf = title;
	else if (strchr(filename, '/'))
	    desc_buf = strrchr(filename, '/') + 1;
	else
	    desc_buf = filename;

	// words[0] = NULL equiv nw = 0         
	// words[1] = NULL equiv nw = 1

	if (words[0] == NULL)
	    match = TRUE;
	else {
	    /*
	     * FIXME: The search string should be adapted to the current display setting,
	     * e.g. if the user has set it to "%p - %t" then build the match string like that too,
	     * or even better, search for each of the tags seperatly.
	     *
	     * In any case the string to match should _never_ contain something the user can't actually
	     * see in the playlist.
	     */

	    /* I've moved the code to do lowercasing and stuff because IMO
	       that should be up to the implementation of the match algorithm. */

	    /* Compare the key words to the string - if
	       all the words match, add to the ListStore */

	    /* FIXME: This shouldn't be handled here, but rather in the playlist functions,
	       which should return the filename instead of the title if title == NULL. */
	    if (!title)
		title = filename;
	    match = mainwin_jump_to_file_match(title, words);
	}

	if (match) {
	    gtk_list_store_append(GTK_LIST_STORE(store), &iter);
	    gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			       0, g_strdup_printf("%d", songnr + 1),
			       1, desc_buf, -1);

	}

	songnr++;
	playlist = playlist->next;
    }

    if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter)) {
	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(userdata));
	gtk_tree_selection_select_iter(selection, &iter);
    }

    g_strfreev(words);
    PLAYLIST_UNLOCK();
    g_free(key);

}

static void mainwin_jump_to_file(void)
{
    GtkWidget *vbox, *scrollwin, *sep, *bbox, *jump, /**queue,*/ *cancel,
	*rescan, *edit, *search_label, *hbox;
    GList *playlist;
    gchar *desc_buf;
    gint row;

    GtkListStore *jtf_store;
    GtkTreeModel *jtf_model;
    GtkTreeView *treeview;

    GtkTreeIter iter;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    PLAYLIST_LOCK();

    playlist = get_playlist();

/*	if (!playlist)
	{
		PLAYLIST_UNLOCK();
		return;
	}
*/
    mainwin_jtf = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(mainwin_jtf), _("Jump to track"));

    gtk_window_set_position(GTK_WINDOW(mainwin_jtf), GTK_WIN_POS_CENTER);
    g_signal_connect(G_OBJECT(mainwin_jtf), "destroy",
		     G_CALLBACK(gtk_widget_destroyed), &mainwin_jtf);

    gtk_container_border_width(GTK_CONTAINER(mainwin_jtf), 10);
    gtk_window_set_default_size(GTK_WINDOW(mainwin_jtf), 550, 350);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(mainwin_jtf), vbox);
    gtk_widget_show(vbox);

    jtf_store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
    jtf_model = GTK_TREE_MODEL(jtf_store);
    treeview = (GtkTreeView *)
	gtk_tree_view_new_with_model(GTK_TREE_MODEL(jtf_model));
    g_object_unref(G_OBJECT(jtf_model));

    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);

    column = gtk_tree_view_column_new();
    gtk_tree_view_set_headers_visible(treeview, FALSE);
    gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);

    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_attributes(column, renderer, "text", 0, NULL);
    gtk_tree_view_column_set_spacing(column, 4);

    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_attributes(column, renderer, "text", 1, NULL);
    gtk_tree_view_column_set_spacing(column, 4);
    gtk_tree_view_append_column(treeview, column);

    g_signal_connect(G_OBJECT(treeview), "row-activated",
		     G_CALLBACK(mainwin_jump_to_file_select_row_cb),
		     treeview);


    hbox = gtk_hbox_new(FALSE, 3);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 3);
    gtk_widget_show(hbox);

    search_label = gtk_label_new(_("Filter: "));
    gtk_label_set_markup_with_mnemonic(GTK_LABEL(search_label),
				       "_Filter:");
    gtk_box_pack_start(GTK_BOX(hbox), search_label, FALSE, FALSE, 0);
    gtk_widget_show(search_label);

    edit = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(edit), TRUE);
    gtk_label_set_mnemonic_widget(GTK_LABEL(search_label), edit);
    g_signal_connect(G_OBJECT(edit), "changed",
		     G_CALLBACK(mainwin_jump_to_file_edit_cb), treeview);
    g_signal_connect(G_OBJECT(edit), "activate",
		     G_CALLBACK(mainwin_jump_to_file_real_cb), treeview);
    g_signal_connect(G_OBJECT(mainwin_jtf), "key_press_event",
		     G_CALLBACK(mainwin_jump_to_file_keypress_cb), NULL);

    gtk_box_pack_start(GTK_BOX(hbox), edit, TRUE, TRUE, 3);
    gtk_widget_show(edit);

    scrollwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(scrollwin), GTK_WIDGET(treeview));
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_ALWAYS);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrollwin),
					GTK_SHADOW_IN);
    gtk_box_pack_start(GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
    gtk_widget_set_usize(scrollwin, 330, 200);
    gtk_widget_show(GTK_WIDGET(treeview));
    gtk_widget_show(scrollwin);

    sep = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 0);
    gtk_widget_show(sep);

    bbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

/*	queue = gtk_button_new_with_label(_("Queue"));
	gtk_box_pack_start(GTK_BOX(bbox), queue, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(queue), "clicked", G_CALLBACK(mainwin_jump_to_file_queue_cb), treeview);
	GTK_WIDGET_SET_FLAGS(queue, GTK_CAN_DEFAULT);
	gtk_widget_show(queue);
*/


    rescan = gtk_button_new_from_stock(GTK_STOCK_REFRESH);
    gtk_box_pack_start(GTK_BOX(bbox), rescan, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(rescan), "clicked",
		     G_CALLBACK(mainwin_update_jtf), treeview);
    GTK_WIDGET_SET_FLAGS(rescan, GTK_CAN_DEFAULT);
    gtk_widget_show(rescan);
    gtk_widget_grab_default(rescan);

    jump = gtk_button_new_from_stock(GTK_STOCK_JUMP_TO);
    gtk_box_pack_start(GTK_BOX(bbox), jump, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(jump), "clicked",
		     G_CALLBACK(mainwin_jump_to_file_jump_cb), treeview);
    GTK_WIDGET_SET_FLAGS(jump, GTK_CAN_DEFAULT);
    gtk_widget_show(jump);
    gtk_widget_grab_default(jump);

    cancel = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_box_pack_start(GTK_BOX(bbox), cancel, FALSE, FALSE, 0);
    g_signal_connect_swapped(G_OBJECT(cancel), "clicked",
			     G_CALLBACK(gtk_widget_destroy),
			     G_OBJECT(mainwin_jtf));
    GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
    gtk_widget_show(cancel);
    gtk_widget_show(bbox);

    gtk_list_store_clear(jtf_store);

    row = 1;
    while (playlist) {
	if (((PlaylistEntry *) playlist->data)->title)
	    desc_buf = ((PlaylistEntry *) playlist->data)->title;
	else if (strchr(((PlaylistEntry *) playlist->data)->filename, '/'))
	    desc_buf =
		strrchr(((PlaylistEntry *) playlist->data)->filename,
			'/') + 1;
	else
	    desc_buf = ((PlaylistEntry *) playlist->data)->filename;

	gtk_list_store_append(GTK_LIST_STORE(jtf_store), &iter);
	gtk_list_store_set(GTK_LIST_STORE(jtf_store), &iter,
			   0, g_strdup_printf("%d", row++),
			   1, desc_buf, -1);

	playlist = playlist->next;
    }

    PLAYLIST_UNLOCK();

/*	if (cfg.sort_jump_to_file)
	{
		gtk_treeview_set_sort_column(GTK_CLIST(treeview), 0);
		gtk_treeview_set_sort_type(GTK_CLIST(treeview), GTK_SORT_ASCENDING);
		gtk_treeview_sort(GTK_CLIST(treeview));
	}
*/
    gtk_widget_show(mainwin_jtf);
    gtk_widget_grab_focus(edit);
}

static gboolean mainwin_configure(GtkWidget * window,
				  GdkEventConfigure * event, gpointer data)
{
    if (!GTK_WIDGET_VISIBLE(window))
	return FALSE;

    if (cfg.show_wm_decorations)
	gdk_window_get_root_origin(window->window,
				   &cfg.player_x, &cfg.player_y);
    else
	gdk_window_get_deskrelative_origin(window->window,
					   &cfg.player_x, &cfg.player_y);
    return FALSE;
}

void mainwin_set_back_pixmap(void)
{
    if (cfg.doublesize)
	gdk_window_set_back_pixmap(mainwin->window, mainwin_bg_dblsize, 0);
    else
	gdk_window_set_back_pixmap(mainwin->window, mainwin_bg, 0);
    gdk_window_clear(mainwin->window);
}

gint mainwin_client_event(GtkWidget * w, GdkEventClient * event,
			  gpointer data)
{
    static GdkAtom atom_rcfiles = GDK_NONE;

    if (!atom_rcfiles)
	atom_rcfiles = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);
    if (event->message_type == atom_rcfiles) {
	mainwin_set_back_pixmap();
	equalizerwin_set_back_pixmap();
	playlistwin_set_back_pixmap();
	return TRUE;

    }
    return FALSE;
}

static void mainwin_drag_data_received(GtkWidget * widget,
				       GdkDragContext * context,
				       gint x,
				       gint y,
				       GtkSelectionData * selection_data,
				       guint info,
				       guint time, gpointer user_data)
{
    ConfigFile *cfgfile;
    gchar *ending;
    gchar *path;
    gchar *filename;
    gchar *decoded;

    if (selection_data->data) {

	path = selection_data->data;

	if (!strncasecmp(path, "fonts:///", 9)) {
	    path[strlen(path) - 2] = 0;	// Why the hell a CR&LF?
	    path = path + 8;

	    // plain, since we already stripped the first URI part
	    decoded = xmms_urldecode_plain(path);

	    // Get the old font's size, and add it to the dropped font's name
	    cfg.playlist_font =
		g_strconcat(decoded + 1, strrchr(cfg.playlist_font, ' '),
			    NULL);
	    playlist_list_set_font(cfg.playlist_font);
	    playlistwin_update_list();

	    g_free(decoded);
	} else {

	    if (!strncasecmp(path, "file:///", 8)) {
		path[strlen(path) - 2] = 0;	// Why the hell a CR&LF?
		path = path + 7;
	    } else if (!strncasecmp(path, "file:", 5)) {
		path = path + 5;
	    }

	    ending = strrchr(path, '.');

	    if (ending) {
		if (!strcasecmp(ending, ".wsz")
		    || !strcasecmp(ending, ".zip")
		    || !strcasecmp(ending, ".bz2")
		    || !strcasecmp(ending, ".gz")
		    || !strcasecmp(ending, ".tgz")
		    || !strcasecmp(ending, ".tbz2")

		    ) {
		    load_skin(path);	/* Load... */
		    skin_install_skin(path);	/* ...and install the skin */

		    /* Change skin name in the config file */
		    filename =
			g_build_filename(g_get_home_dir(), BMP_RCPATH,
					 "config", NULL);
		    cfgfile = xmms_cfg_open_file(filename);
		    xmms_cfg_write_string(cfgfile, "xmms", "skin", path);
		    xmms_cfg_free(cfgfile);
		} else {
		    if (input_check_file
			((gchar *) selection_data->data, FALSE)) {
			playlist_clear();
			g_message((gchar *) selection_data->data);
			playlist_add_url_string((gchar *) selection_data->
						data);
			playlist_play();
		    }
		}
	    } else {
		playlist_clear();
		g_message((gchar *) selection_data->data);
		playlist_add_url_string((gchar *) selection_data->data);
		playlist_play();
	    }

	}
    }

}



void mainwin_add_dir_handler(gchar * dir)
{
    if (cfg.filesel_path)
	g_free(cfg.filesel_path);
    cfg.filesel_path = g_strdup(dir);
    playlist_clear();
    playlist_add_dir(dir);
    playlist_play();
}

void mainwin_show_dirbrowser(void)
{
    if (!mainwin_dir_browser) {
	mainwin_dir_browser = bmp_db_create(cfg.filesel_path);
	gtk_widget_show_all(mainwin_dir_browser);
    } else
	gtk_window_present(GTK_WINDOW(mainwin_dir_browser));

}

void mainwin_url_enqueue_clicked(GtkWidget * w, GtkWidget * entry)
{
    gchar *text;

    text = (char *) gtk_entry_get_text(GTK_ENTRY(entry));
    if (text && *text)
	playlist_add_url_string(text);
}

void mainwin_show_add_url_window(void)
{
    if (!mainwin_url_window) {
	mainwin_url_window =
	    util_create_add_url_window(_("Add URL..."),
				       G_CALLBACK
				       (mainwin_url_enqueue_clicked));
	gtk_window_set_transient_for(GTK_WINDOW(mainwin_url_window),
				     GTK_WINDOW(mainwin));
	g_signal_connect(G_OBJECT(mainwin_url_window), "destroy",
			 G_CALLBACK(gtk_widget_destroyed),
			 &mainwin_url_window);
	gtk_window_present(GTK_WINDOW(mainwin_url_window));
    }
}

void mainwin_eject_pushed(void)
{
    util_run_filebrowser(PLAY_BUTTON);
}

void mainwin_play_pushed(void)
{
    if (get_input_paused()) {
	input_pause();
	return;
    }
    if (get_playlist_length())
	playlist_play();
    else
	mainwin_eject_pushed();
}

void mainwin_stop_pushed(void)
{
    mainwin_clear_song_info();
    input_stop();
}

void mainwin_shuffle_pushed(gboolean toggled)
{
    GtkWidget *widget;
    widget = gtk_item_factory_get_widget(mainwin_options_menu, "/Shuffle");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), toggled);
}

void mainwin_repeat_pushed(gboolean toggled)
{
    GtkWidget *widget;
    widget = gtk_item_factory_get_widget(mainwin_options_menu, "/Repeat");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), toggled);
}

void mainwin_pl_pushed(gboolean toggled)
{
    playlistwin_show(toggled);
}

gint mainwin_spos_frame_cb(gint pos)
{
    if (mainwin_sposition) {
	if (pos < 6)
	    mainwin_sposition->hs_knob_nx = mainwin_sposition->hs_knob_px =
		17;
	else if (pos < 9)
	    mainwin_sposition->hs_knob_nx = mainwin_sposition->hs_knob_px =
		20;
	else
	    mainwin_sposition->hs_knob_nx = mainwin_sposition->hs_knob_px =
		23;
    }
    return 1;
}

void mainwin_spos_motion_cb(gint pos)
{
    gint time;
    gchar *tmp;

    pos--;

    time = ((playlist_get_current_length() / 1000) * pos) / 12;
    if (cfg.timer_mode == TIMER_REMAINING) {
	time = (playlist_get_current_length() / 1000) - time;
	tmp = g_strdup_printf("-%2.2d", time / 60);
	textbox_set_text(mainwin_stime_min, tmp);
	g_free(tmp);
    } else {
	tmp = g_strdup_printf(" %2.2d", time / 60);
	textbox_set_text(mainwin_stime_min, tmp);
	g_free(tmp);
    }
    tmp = g_strdup_printf("%2.2d", time % 60);
    textbox_set_text(mainwin_stime_sec, tmp);
    g_free(tmp);
}

void mainwin_spos_release_cb(gint pos)
{
    input_seek(((playlist_get_current_length() / 1000) * (pos - 1)) / 12);
}

void mainwin_position_motioncb(gint pos)
{
    gint length, time;
    gchar *buf;

    length = playlist_get_current_length() / 1000;
    time = (length * pos) / 219;
    buf =
	g_strdup_printf(_("SEEK TO: %d:%-2.2d/%d:%-2.2d (%d%%)"),
			time / 60, time % 60, length / 60, length % 60,
			(length != 0) ? (time * 100) / length : 0);
    mainwin_lock_info_text(buf);
    g_free(buf);
}

void mainwin_position_releasecb(gint pos)
{
    int length, time;

    length = playlist_get_current_length() / 1000;
    time = (length * pos) / 219;
    input_seek(time);
    mainwin_release_info_text();
}

gint mainwin_volume_framecb(gint pos)
{
    return (gint) rint((pos / 52.0) * 28);
}

void mainwin_adjust_volume_motion(gint v)
{
    gchar *tmp;

    setting_volume = TRUE;
    tmp = g_strdup_printf(_("VOLUME: %d%%"), v);
    mainwin_lock_info_text(tmp);
    g_free(tmp);
    if (balance < 0)
	input_set_volume(v, (v * (100 - abs(balance))) / 100);
    else if (balance > 0)
	input_set_volume((v * (100 - abs(balance))) / 100, v);
    else
	input_set_volume(v, v);
}

void mainwin_adjust_volume_release(void)
{
    mainwin_release_info_text();
    setting_volume = FALSE;
    read_volume(VOLUME_ADJUSTED);
}

void mainwin_adjust_balance_motion(gint b)
{
    char *tmp;
    gint v, pvl, pvr;

    setting_volume = TRUE;
    balance = b;
    input_get_volume(&pvl, &pvr);
    v = MAX(pvl, pvr);
    if (b < 0) {
	tmp = g_strdup_printf(_("BALANCE: %d%% LEFT"), -b);
	input_set_volume(v, (gint) rint(((100 + b) / 100.0) * v));
    } else if (b == 0) {
	tmp = g_strdup_printf(_("BALANCE: CENTER"));
	input_set_volume(v, v);
    } else {			/* (b > 0) */

	tmp = g_strdup_printf(_("BALANCE: %d%% RIGHT"), b);
	input_set_volume((gint) rint(((100 - b) / 100.0) * v), v);
    }
    mainwin_lock_info_text(tmp);
    g_free(tmp);
}

void mainwin_adjust_balance_release(void)
{
    mainwin_release_info_text();
    setting_volume = FALSE;
    read_volume(VOLUME_ADJUSTED);
}

void mainwin_set_volume_slider(gint percent)
{
    hslider_set_position(mainwin_volume,
			 (gint) rint((percent * 51) / 100.0));
}

void mainwin_set_balance_slider(gint percent)
{
    hslider_set_position(mainwin_balance,
			 (gint) rint(((percent * 12) / 100.0) + 12));
}

void mainwin_volume_motioncb(gint pos)
{
    gint vol = (pos * 100) / 51;
    mainwin_adjust_volume_motion(vol);
    equalizerwin_set_volume_slider(vol);
}

void mainwin_volume_releasecb(gint pos)
{
    mainwin_adjust_volume_release();
}

gint mainwin_balance_framecb(gint pos)
{
    return ((abs(pos - 12) * 28) / 13);
}

void mainwin_balance_motioncb(gint pos)
{
    gint bal = ((pos - 12) * 100) / 12;
    mainwin_adjust_balance_motion(bal);
    equalizerwin_set_balance_slider(bal);
}

void mainwin_balance_releasecb(gint pos)
{
    mainwin_adjust_volume_release();
}

void mainwin_set_volume_diff(gint diff)
{
    gint vl, vr, vol;
    input_get_volume(&vl, &vr);
    vol = MAX(vl, vr);
    vol = CLAMP(vol + diff, 0, 100);
    mainwin_adjust_volume_motion(vol);
    setting_volume = FALSE;
    mainwin_set_volume_slider(vol);
    equalizerwin_set_volume_slider(vol);
    read_volume(VOLUME_SET);
}

void mainwin_set_balance_diff(gint diff)
{
    gint b;
    b = CLAMP(balance + diff, -100, 100);
    mainwin_adjust_balance_motion(b);
    setting_volume = FALSE;
    mainwin_set_balance_slider(b);
    equalizerwin_set_balance_slider(b);
    read_volume(VOLUME_SET);
}

void mainwin_show(gboolean show)
{
    /*  Set again the Always On Top flag - probably a bug in Metacity 2.4.xx   */
    hint_set_always(cfg.always_on_top);
    if (show)
	mainwin_real_show();
}

void mainwin_real_show(void)
{
    cfg.player_visible = TRUE;

    if (cfg.player_shaded)
	vis_clear_data(active_vis);

    mainwin_vis_set_active_vis(MAINWIN_VIS_ACTIVE_MAINWIN);
    mainwin_set_shape_mask();
    if (cfg.show_wm_decorations) {
	if (!pposition_broken && cfg.player_x != -1
	    && cfg.save_window_position)
	    dock_set_uposition(mainwin, cfg.player_x, cfg.player_y);
	gtk_widget_show(mainwin);
	if (pposition_broken && cfg.player_x != -1
	    && cfg.save_window_position)
	    dock_set_uposition(mainwin, cfg.player_x, cfg.player_y);
	return;
    }
    if (!nullmask)
	/* Startup */
	return;
    gdk_pixmap_unref(nullmask);
    nullmask = NULL;
    gdk_window_set_hints(mainwin->window, 0, 0,
			 PLAYER_WIDTH, PLAYER_HEIGHT,
			 PLAYER_WIDTH, PLAYER_HEIGHT,
			 GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
    gdk_window_resize(mainwin->window, PLAYER_WIDTH, PLAYER_HEIGHT);
    draw_main_window(TRUE);
    gdk_window_raise(mainwin->window);
}

void mainwin_real_hide(void)
{
    GdkGC *gc;
    GdkColor pattern;

/*  	if (!cfg.player_visible) */
/*  		return; */

    if (cfg.player_shaded) {
	svis_clear_data(mainwin_svis);
	vis_clear_data(playlistwin_vis);
    }
    if (cfg.show_wm_decorations)
	gtk_widget_hide(mainwin);
    else {
	nullmask = gdk_pixmap_new(mainwin->window, 20, 20, 1);
	gc = gdk_gc_new(nullmask);
	pattern.pixel = 0;
	gdk_gc_set_foreground(gc, &pattern);
	gdk_draw_rectangle(nullmask, gc, TRUE, 0, 0, 20, 20);
	gdk_gc_destroy(gc);
	gtk_widget_shape_combine_mask(mainwin, nullmask, 0, 0);

	gdk_window_set_hints(mainwin->window, 0, 0, 0, 0, 0, 0,
			     GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	gdk_window_resize(mainwin->window, 0, 0);

    }
    mainwin_vis_set_active_vis(MAINWIN_VIS_ACTIVE_PLAYLISTWIN);
    cfg.player_visible = FALSE;
}

void mainwin_songname_menu_callback(gpointer cb_data, guint action,
				    GtkWidget * w)
{
    switch (action) {
    case MAINWIN_SONGNAME_FILEINFO:
	playlist_fileinfo_current();
	break;
    case MAINWIN_SONGNAME_JTF:
	mainwin_jump_to_file();
	break;
    case MAINWIN_SONGNAME_JTT:
	mainwin_jump_to_time();
	break;
    case MAINWIN_SONGNAME_SCROLL:
	cfg.autoscroll =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_songname_menu,
				 "/Autoscroll Songname"))->active;
	textbox_set_scroll(mainwin_info, cfg.autoscroll);
	break;
    }
}

void set_doublesize(gboolean ds)
{
    gint height;

    cfg.doublesize = ds;

    if (cfg.player_shaded)
	height = MAINWIN_HEIGHT;
    else
	height = MAINWIN_SHADED_HEIGHT;

    mainwin_set_shape_mask();
    if (cfg.doublesize) {
	dock_resize(dock_window_list, mainwin, MAINWIN_WIDTH * 2,
		    height * 2);
	gdk_window_set_back_pixmap(mainwin->window, mainwin_bg_dblsize, 0);
    } else {
	dock_resize(dock_window_list, mainwin, MAINWIN_WIDTH, height);
	gdk_window_set_back_pixmap(mainwin->window, mainwin_bg, 0);
    }
    draw_main_window(TRUE);
    vis_set_doublesize(mainwin_vis, ds);

    if (cfg.eq_doublesize_linked)
	equalizerwin_set_doublesize(ds);
}

void mainwin_options_menu_callback(gpointer cb_data, guint action,
				   GtkWidget * w)
{
    switch (action) {
    case MAINWIN_OPT_PREFS:
	show_prefs_window();
	break;
    case MAINWIN_OPT_SKIN:
	GDK_THREADS_ENTER();
	show_skin_window();
	GDK_THREADS_LEAVE();
	break;
    case MAINWIN_OPT_RELOADSKIN:
	reload_skin();
	break;
    case MAINWIN_OPT_SHUFFLE:
	{
	    gboolean shuffle;
	    GtkWidget *w =
		gtk_item_factory_get_widget(mainwin_options_menu,
					    "/Shuffle");
	    shuffle = GTK_CHECK_MENU_ITEM(w)->active;
	    tbutton_set_toggled(mainwin_shuffle, shuffle);
	    playlist_set_shuffle(shuffle);
	    break;
	}
    case MAINWIN_OPT_REPEAT:
	cfg.repeat =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_options_menu, "/Repeat"))->active;
	tbutton_set_toggled(mainwin_repeat, cfg.repeat);
	break;
    case MAINWIN_OPT_TELAPSED:
	set_timer_mode_menu_cb(TIMER_ELAPSED);
	break;
    case MAINWIN_OPT_TREMAINING:
	set_timer_mode_menu_cb(TIMER_REMAINING);
	break;
    case MAINWIN_OPT_ALWAYS:
	mainwin_menurow->mr_always_selected =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_options_menu,
				 "/Always On Top"))->active;
	cfg.always_on_top = mainwin_menurow->mr_always_selected;
	draw_widget(mainwin_menurow);
	hint_set_always(cfg.always_on_top);
	break;
    case MAINWIN_OPT_STICKY:
	{
	    GtkWidget *w;
	    w = gtk_item_factory_get_widget(mainwin_options_menu,
					    "/Show on all desktops");
	    cfg.sticky = GTK_CHECK_MENU_ITEM(w)->active;
	    hint_set_sticky(cfg.sticky);
	    break;
	}
    case MAINWIN_OPT_WS:
	mainwin_set_shade_menu_cb(GTK_CHECK_MENU_ITEM
				  (gtk_item_factory_get_widget
				   (mainwin_options_menu,
				    "/WindowShade Mode"))->active);
	break;
    case MAINWIN_OPT_PWS:
	playlistwin_set_shade_menu_cb(GTK_CHECK_MENU_ITEM
				      (gtk_item_factory_get_widget
				       (mainwin_options_menu,
					"/Playlist WindowShade Mode"))->
				      active);
	break;
    case MAINWIN_OPT_EQWS:
	equalizerwin_set_shade_menu_cb(GTK_CHECK_MENU_ITEM
				       (gtk_item_factory_get_widget
					(mainwin_options_menu,
					 "/Equalizer WindowShade Mode"))->
				       active);
	break;
    case MAINWIN_OPT_DOUBLESIZE:
	mainwin_menurow->mr_doublesize_selected =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_options_menu,
				 "/DoubleSize"))->active;
	draw_widget(mainwin_menurow);
	set_doublesize(mainwin_menurow->mr_doublesize_selected);
	gdk_flush();
	break;
    case MAINWIN_OPT_EASY_MOVE:
	cfg.easy_move =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_options_menu,
				 "/Easy Move"))->active;
	break;
    case MAINWIN_OPT_NPA:
	cfg.no_playlist_advance =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_options_menu,
				 "/No Playlist Advance"))->active;
    }
}

void mainwin_vis_menu_callback(gpointer cb_data, guint action,
			       GtkWidget * w)
{
    switch (action) {
    case MAINWIN_VIS_ANALYZER:
    case MAINWIN_VIS_SCOPE:
    case MAINWIN_VIS_OFF:
	mainwin_vis_set_type_menu_cb(action - MAINWIN_VIS_ANALYZER);
	break;
    case MAINWIN_VIS_ANALYZER_NORMAL:
    case MAINWIN_VIS_ANALYZER_FIRE:
    case MAINWIN_VIS_ANALYZER_VLINES:
	mainwin_vis_set_analyzer_mode(action -
				      MAINWIN_VIS_ANALYZER_NORMAL);
	break;
    case MAINWIN_VIS_ANALYZER_LINES:
    case MAINWIN_VIS_ANALYZER_BARS:
	mainwin_vis_set_analyzer_type(action - MAINWIN_VIS_ANALYZER_LINES);
	break;
    case MAINWIN_VIS_ANALYZER_PEAKS:
	cfg.analyzer_peaks =
	    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget
				(mainwin_vis_menu,
				 mainwin_vis_menu_entries
				 [MAINWIN_VIS_MENU_ANALYZER_PEAKS].path))->
	    active;
	break;
    case MAINWIN_VIS_SCOPE_DOT:
    case MAINWIN_VIS_SCOPE_LINE:
    case MAINWIN_VIS_SCOPE_SOLID:
	cfg.scope_mode = action - MAINWIN_VIS_SCOPE_DOT;
	break;
    case MAINWIN_VIS_VU_NORMAL:
    case MAINWIN_VIS_VU_SMOOTH:
	cfg.vu_mode = action - MAINWIN_VIS_VU_NORMAL;
	break;
    case MAINWIN_VIS_REFRESH_FULL:
    case MAINWIN_VIS_REFRESH_HALF:
    case MAINWIN_VIS_REFRESH_QUARTER:
    case MAINWIN_VIS_REFRESH_EIGHTH:
	mainwin_vis_set_refresh(action - MAINWIN_VIS_REFRESH_FULL);
	break;
    case MAINWIN_VIS_AFALLOFF_SLOWEST:
    case MAINWIN_VIS_AFALLOFF_SLOW:
    case MAINWIN_VIS_AFALLOFF_MEDIUM:
    case MAINWIN_VIS_AFALLOFF_FAST:
    case MAINWIN_VIS_AFALLOFF_FASTEST:
	mainwin_vis_set_afalloff(action - MAINWIN_VIS_AFALLOFF_SLOWEST);
	break;
    case MAINWIN_VIS_PFALLOFF_SLOWEST:
    case MAINWIN_VIS_PFALLOFF_SLOW:
    case MAINWIN_VIS_PFALLOFF_MEDIUM:
    case MAINWIN_VIS_PFALLOFF_FAST:
    case MAINWIN_VIS_PFALLOFF_FASTEST:
	mainwin_vis_set_pfalloff(action - MAINWIN_VIS_PFALLOFF_SLOWEST);
	break;
    case MAINWIN_VIS_PLUGINS:
	//show_prefs_window();
	prefswin_show_vis_plugins_page();
	break;
    }
}

void mainwin_general_menu_callback(gpointer cb_data, guint action,
				   GtkWidget * w)
{
    switch (action) {
    case MAINWIN_GENERAL_ABOUT:
	show_about_window();
	break;
    case MAINWIN_GENERAL_PLAYFILE:
	mainwin_eject_pushed();
	break;
    case MAINWIN_GENERAL_PLAYDIRECTORY:
	mainwin_show_dirbrowser();
	break;
    case MAINWIN_GENERAL_PLAYCD:
	play_medium();
	break;
    case MAINWIN_GENERAL_PLAYLOCATION:
	mainwin_show_add_url_window();
	break;
    case MAINWIN_GENERAL_FILEINFO:
	playlist_fileinfo_current();
	break;
	break;
    case MAINWIN_GENERAL_SHOWPLWIN:
	if (GTK_CHECK_MENU_ITEM
	    (gtk_item_factory_get_widget
	     (mainwin_options_menu, "/Show Playlist Editor"))->active)
	    playlistwin_real_show();
	else
	    playlistwin_real_hide();
	break;
    case MAINWIN_GENERAL_SHOWEQWIN:
	if (GTK_CHECK_MENU_ITEM
	    (gtk_item_factory_get_widget
	     (mainwin_options_menu, "/Show Graphical EQ"))->active)
	    equalizerwin_real_show();
	else
	    equalizerwin_real_hide();
	break;
    case MAINWIN_GENERAL_PREV:
	playlist_prev();
	break;
    case MAINWIN_GENERAL_PLAY:
	mainwin_play_pushed();
	break;
    case MAINWIN_GENERAL_PAUSE:
	input_pause();
	break;
    case MAINWIN_GENERAL_STOP:
	mainwin_stop_pushed();
	break;
    case MAINWIN_GENERAL_NEXT:
	playlist_next();
	break;
/*	case MAINWIN_GENERAL_STOPFADE:
		break;
*/
    case MAINWIN_GENERAL_BACK5SEC:
	if (get_input_playing() && playlist_get_current_length() != -1)
	    input_seek((((input_get_time() / 1000) - 5 >=
			 0) ? (input_get_time() / 1000) - 5 : 0));
	break;
    case MAINWIN_GENERAL_FWD5SEC:
	if (get_input_playing() && playlist_get_current_length() != -1)
	    input_seek(((((input_get_time() / 1000) + 5) <
			 (playlist_get_current_length() /
			  1000)) ? ((input_get_time() / 1000) +
				    5) : ((playlist_get_current_length() /
					   1000) - 1)));
	break;
    case MAINWIN_GENERAL_START:
	playlist_set_position(0);
	break;
/*	case MAINWIN_GENERAL_BACK10:
		playlist_set_position((((get_playlist_position() - 10) >= 0) ? get_playlist_position() - 10 : 0));
		break;
	case MAINWIN_GENERAL_FWD10:
		playlist_set_position((((get_playlist_position() + 10) < get_playlist_length())? (get_playlist_position() + 10) : (get_playlist_length() - 1)));
		break;
*/
    case MAINWIN_GENERAL_JTT:
	mainwin_jump_to_time();
	break;
    case MAINWIN_GENERAL_JTF:
	mainwin_jump_to_file();
	break;
    case MAINWIN_GENERAL_EXIT:
	mainwin_quit_cb();
	break;
    }
}

void mainwin_mr_change(MenuRowItem i)
{
    switch (i) {
    case MENUROW_NONE:
	mainwin_set_info_text();
	break;
    case MENUROW_OPTIONS:
	mainwin_lock_info_text(_("OPTIONS MENU"));
	break;
    case MENUROW_ALWAYS:
	if (!hint_always_on_top_available()) {
	    if (mainwin_menurow->mr_always_selected)
		mainwin_lock_info_text(_("DISABLE ALWAYS ON TOP (N/A)"));
	    else
		mainwin_lock_info_text(_("ENABLE ALWAYS ON TOP (N/A)"));
	} else if (mainwin_menurow->mr_doublesize_selected)
	    mainwin_lock_info_text(_("DISABLE ALWAYS ON TOP"));
	else
	    mainwin_lock_info_text(_("ENABLE ALWAYS ON TOP"));
	break;
    case MENUROW_FILEINFOBOX:
	mainwin_lock_info_text(_("FILE INFO BOX"));
	break;
    case MENUROW_DOUBLESIZE:
	if (mainwin_menurow->mr_doublesize_selected)
	    mainwin_lock_info_text(_("DISABLE DOUBLESIZE"));
	else
	    mainwin_lock_info_text(_("ENABLE DOUBLESIZE"));
	break;
    case MENUROW_VISUALIZATION:
	mainwin_lock_info_text(_("VISUALIZATION MENU"));
	break;
    }
}

void mainwin_mr_release(MenuRowItem i)
{
    GdkModifierType modmask;
    GtkWidget *widget;
    gint x, y;

    switch (i) {
    case MENUROW_OPTIONS:
	gdk_window_get_pointer(NULL, &x, &y, &modmask);
	util_item_factory_popup(mainwin_options_menu, x, y, 1,
				GDK_CURRENT_TIME);
	break;
    case MENUROW_ALWAYS:
	widget =
	    gtk_item_factory_get_widget(mainwin_options_menu,
					"/Always On Top");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget),
				       mainwin_menurow->
				       mr_always_selected);
	break;
    case MENUROW_FILEINFOBOX:
	playlist_fileinfo_current();
	break;
    case MENUROW_DOUBLESIZE:
	widget =
	    gtk_item_factory_get_widget(mainwin_options_menu,
					"/DoubleSize");
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget),
				       mainwin_menurow->
				       mr_doublesize_selected);
	break;
    case MENUROW_VISUALIZATION:
	gdk_window_get_pointer(NULL, &x, &y, &modmask);
	util_item_factory_popup(mainwin_vis_menu, x, y, 1,
				GDK_CURRENT_TIME);
	break;
    case MENUROW_NONE:
	break;
    }
    mainwin_release_info_text();
}

void run_no_audiocd_dialog(void)
{
    GtkWidget *dialog;
    GtkWidget *label;
    GtkWidget *image;
    GtkWidget *b;
    gint x, y, dx, dy;

    dialog = gtk_dialog_new_with_buttons("CD not Audio",
					 GTK_WINDOW(mainwin),
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);
    b = gtk_hbox_new(FALSE, 8);
    image =
	gtk_image_new_from_stock(GTK_STOCK_DIALOG_ERROR,
				 GTK_ICON_SIZE_DIALOG);
    gtk_box_pack_start(GTK_BOX(b), image, FALSE, FALSE, 8);
    label =
	gtk_label_new
	("The inserted CD is not an Audio CD:\nUnable to start playback.");
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 1);
    gtk_box_pack_start(GTK_BOX(b), label, FALSE, FALSE, 8);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), b, TRUE, TRUE,
		       8);
    gtk_widget_show(b);
    gtk_widget_show(image);
    gtk_widget_show(label);

    gtk_window_set_position(GTK_WINDOW(dialog),
			    GTK_WIN_POS_CENTER_ON_PARENT);
    gtk_window_get_position(GTK_WINDOW(mainwin), &x, &y);
    gtk_window_get_position(GTK_WINDOW(dialog), &dx, &dy);
    y += 130;
    gtk_window_move(GTK_WINDOW(dialog), dx, y);
    gtk_widget_destroy(GTK_DIALOG(dialog)->separator);
    (void) gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}

void play_medium(void)
{


    GList *node;
    gchar *filename;
    gchar *path;
    ConfigFile *cfgfile;

//      filename = g_strconcat(g_get_home_dir(), "/.beep/config", NULL);
//      filename = g_strconcat(g_get_home_dir(), "/",BMP_RCPATH"/config", NULL);
    filename =
	g_build_filename(g_get_home_dir(), BMP_RCPATH, "config", NULL);
    cfgfile = xmms_cfg_open_file(filename);
    xmms_cfg_read_string(cfgfile, "CDDA", "directory", &path);
    xmms_cfg_free(cfgfile);

    node = input_scan_dir(path);

    if (node != NULL) {
	playlist_clear();
    } else {
	run_no_audiocd_dialog();
	return;
    }

    while (node) {
	playlist_add(g_strconcat(path, "/", node->data, NULL));
	g_free(node->data);
	node = g_list_next(node);
    }

    playlist_set_position(0);
    playlist_play();

}


#define VOLSET_DISP_TIMES 5

void read_volume(gint when)
{
    int vl, vr, b, v;
    static int pvl = 0, pvr = 0, times = VOLSET_DISP_TIMES;
    static gboolean changing = FALSE;

    input_get_volume(&vl, &vr);
    if (when == VOLSET_STARTUP) {
	vl = CLAMP(vl, 0, 100);
	vr = CLAMP(vr, 0, 100);
	pvl = vl;
	pvr = vr;
	v = MAX(vl, vr);
	if (vl > vr)
	    b = (gint) rint(((gdouble) vr / vl) * 100) - 100;
	else if (vl < vr)
	    b = 100 - (gint) rint(((gdouble) vl / vr) * 100);
	else
	    b = 0;

	balance = b;
	mainwin_set_volume_slider(v);
	equalizerwin_set_volume_slider(v);
	mainwin_set_balance_slider(b);
	equalizerwin_set_balance_slider(b);
	return;
    }

    else if (when == VOLSET_UPDATE) {
	if (vl == -1 || vr == -1)
	    return;

	if (setting_volume) {
	    pvl = vl;
	    pvr = vr;
	    return;
	} else if (pvr == vr && pvl == vl && changing) {
	    if (times < VOLSET_DISP_TIMES)
		times++;
	    else {
		mainwin_release_info_text();
		changing = FALSE;
	    }
	} else if (pvr != vr || pvl != vl) {
	    gchar *tmp;

	    v = MAX(vl, vr);
	    if (vl > vr)
		b = (gint) rint(((gdouble) vr / vl) * 100) - 100;
	    else if (vl < vr)
		b = 100 - (gint) rint(((gdouble) vl / vr) * 100);
	    else
		b = 0;

	    if (MAX(vl, vr) != MAX(pvl, pvr))
		tmp = g_strdup_printf(_("VOLUME: %d%%"), v);
	    else {
		if (vl > vr) {
		    tmp = g_strdup_printf(_("BALANCE: %d%% LEFT"), -b);
		} else if (vr == vl)
		    tmp = g_strdup_printf(_("BALANCE: CENTER"));
		else {		/* (vl < vr) */

		    tmp = g_strdup_printf(_("BALANCE: %d%% RIGHT"), b);
		}
	    }
	    mainwin_lock_info_text(tmp);
	    g_free(tmp);

	    pvr = vr;
	    pvl = vl;
	    times = 0;
	    changing = TRUE;
	    mainwin_set_volume_slider(v);
	    equalizerwin_set_volume_slider(v);

	    /*
	     * Don't change the balance slider if the volume has been
	     * set to zero.  The balance can be anything, and our best
	     * guess is what is was before.
	     */
	    if (v > 0) {
		balance = b;
		mainwin_set_balance_slider(b);
		equalizerwin_set_balance_slider(b);
	    }
	}
    } else if (when == VOLUME_ADJUSTED) {
	pvl = vl;
	pvr = vr;
    } else if (when == VOLUME_SET) {
	times = 0;
	changing = TRUE;
	pvl = vl;
	pvr = vr;
    }

}

#define CHECK_SET(menu, path, cfg)						\
  gtk_check_menu_item_set_active(						\
    GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(menu, path)), cfg);


gboolean scan_cd_input_plugin(void)
{
    GList *ilist = get_input_list();
    InputPlugin *ip;

    while (ilist) {
	ip = (InputPlugin *) ilist->data;

	if (!g_ascii_strcasecmp(g_basename(ip->filename), "libcdaudio.so")) {
	    return TRUE;
	}

	ilist = ilist->next;
    }
    return FALSE;
}


void create_popups(void)
{
    GtkWidget *widget;
    mainwin_accel = gtk_accel_group_new();

    mainwin_play_menu =
	gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", mainwin_accel);

/* gtk_item_factory_set_translate_func(mainwin_play_menu,
					    util_menu_translate, NULL, NULL);
*/

    gtk_item_factory_create_items(mainwin_play_menu,
				  mainwin_play_menu_entries_num,
				  mainwin_play_menu_entries, NULL);

    mainwin_options_menu =
	gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", mainwin_accel);

/*	gtk_item_factory_set_translate_func(mainwin_options_menu,
					    util_menu_translate, NULL, NULL);
*/
    gtk_item_factory_create_items(mainwin_options_menu,
				  mainwin_options_menu_entries_num,
				  mainwin_options_menu_entries, NULL);
    CHECK_SET(mainwin_options_menu, "/Shuffle", cfg.shuffle);
    CHECK_SET(mainwin_options_menu, "/Repeat", cfg.repeat);
    CHECK_SET(mainwin_options_menu, "/Easy Move", cfg.easy_move);
    CHECK_SET(mainwin_options_menu, "/No Playlist Advance",
	      cfg.no_playlist_advance);
    if (cfg.timer_mode == TIMER_ELAPSED)
	widget =
	    gtk_item_factory_get_widget(mainwin_options_menu,
					"/Time Elapsed");
    else
	widget =
	    gtk_item_factory_get_widget(mainwin_options_menu,
					"/Time Remaining");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(widget), TRUE);

    CHECK_SET(mainwin_options_menu, "/Always On Top", cfg.always_on_top);
    CHECK_SET(mainwin_options_menu, "/Show on all desktops", cfg.sticky);
    CHECK_SET(mainwin_options_menu, "/DoubleSize", cfg.doublesize);
    CHECK_SET(mainwin_options_menu, "/WindowShade Mode",
	      cfg.player_shaded);
    CHECK_SET(mainwin_options_menu, "/Playlist WindowShade Mode",
	      cfg.playlist_shaded);
    CHECK_SET(mainwin_options_menu, "/Equalizer WindowShade Mode",
	      cfg.equalizer_shaded);
    mainwin_songname_menu =
	gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", mainwin_accel);
/*	gtk_item_factory_set_translate_func(mainwin_songname_menu,
					    util_menu_translate, NULL, NULL);
*/
    gtk_item_factory_create_items(mainwin_songname_menu,
				  mainwin_songname_menu_entries_num,
				  mainwin_songname_menu_entries, NULL);

    CHECK_SET(mainwin_songname_menu, "/Autoscroll Songname",
	      cfg.autoscroll);

    mainwin_vis_menu =
	gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", mainwin_accel);
    gtk_item_factory_set_translate_func(mainwin_vis_menu,
					util_menu_translate, NULL, NULL);
    gtk_item_factory_create_items(mainwin_vis_menu,
				  mainwin_vis_menu_entries_num,
				  mainwin_vis_menu_entries, NULL);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_VIS_MODE +
				       cfg.vis_type].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_ANALYZER_MODE +
				       cfg.analyzer_mode].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_ANALYZER_TYPE +
				       cfg.analyzer_type].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_ANALYZER_PEAKS].
	      path, cfg.analyzer_peaks);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_SCOPE_MODE +
				       cfg.scope_mode].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_WSHADEVU_MODE +
				       cfg.vu_mode].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_REFRESH_RATE +
				       cfg.vis_refresh].path, TRUE);
    CHECK_SET(mainwin_vis_menu,
	      mainwin_vis_menu_entries[MAINWIN_VIS_MENU_AFALLOFF +
				       cfg.analyzer_falloff].path, TRUE);
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM
				   (gtk_item_factory_get_widget
				    (mainwin_vis_menu,
				     mainwin_vis_menu_entries
				     [MAINWIN_VIS_MENU_PFALLOFF +
				      cfg.peaks_falloff].path)), TRUE);

    mainwin_general_menu =
	gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", mainwin_accel);

    gtk_item_factory_set_translate_func(mainwin_general_menu,
					util_menu_translate, NULL, NULL);
    gtk_item_factory_create_items(mainwin_general_menu,
				  mainwin_general_menu_entries_num,
				  mainwin_general_menu_entries, NULL);

    gtk_menu_item_set_submenu(GTK_MENU_ITEM
			      (gtk_item_factory_get_widget
			       (mainwin_general_menu, "/Playback")),
			      GTK_WIDGET(gtk_item_factory_get_widget
					 (mainwin_play_menu, "")));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM
			      (gtk_item_factory_get_widget
			       (mainwin_general_menu, "/Options")),
			      GTK_WIDGET(gtk_item_factory_get_widget
					 (mainwin_options_menu, "")));
//      gtk_menu_item_set_submenu(GTK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_general_menu, "/Visualization")), GTK_WIDGET(gtk_item_factory_get_widget(mainwin_vis_menu, "")));

    // Remove "Play CD" in case the cd audio input plugin is n/a
    if (!scan_cd_input_plugin()) {
	gtk_item_factory_delete_item(mainwin_general_menu, "/Play CD");
    }

    gtk_window_add_accel_group(GTK_WINDOW(mainwin), mainwin_accel);

}

static void mainwin_set_icon(GtkWidget * win, gchar * title)
{
    static GdkPixmap *icon;
    static GdkBitmap *mask;
    GdkAtom icon_atom;
    glong data[2];

    if (!icon)
	icon =
	    gdk_pixmap_create_from_xpm_d(win->window, &mask,
					 &win->style->bg[GTK_STATE_NORMAL],
					 beep_mini_xpm);
    data[0] = GDK_WINDOW_XWINDOW(icon);
    data[1] = GDK_WINDOW_XWINDOW(mask);

    icon_atom = gdk_atom_intern((const gchar *) "KWM_WIN_ICON", FALSE);
    gdk_property_change(win->window, icon_atom, icon_atom, 32,
			GDK_PROP_MODE_REPLACE, (guchar *) data, 2);
    gdk_window_set_icon(win->window, NULL, icon, mask);
    gdk_window_set_icon_name(win->window, title);
    gdk_window_set_title(win->window, title);
}

static void mainwin_create_widgets(void)
{
    mainwin_gc = gdk_gc_new(mainwin->window);

    mainwin_menubtn =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 6, 3, 9, 9,
		       0, 0, 0, 9, mainwin_menubtn_cb, SKIN_TITLEBAR);
    mainwin_menubtn->pb_allow_draw = FALSE;
    mainwin_minimize =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 244, 3, 9,
		       9, 9, 0, 9, 9, mainwin_minimize_cb, SKIN_TITLEBAR);
    mainwin_minimize->pb_allow_draw = FALSE;
    mainwin_shade =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 254, 3, 9,
		       9, 0, cfg.player_shaded ? 27 : 18, 9,
		       cfg.player_shaded ? 27 : 18, mainwin_shade_toggle,
		       SKIN_TITLEBAR);
    mainwin_shade->pb_allow_draw = FALSE;
    mainwin_close =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 264, 3, 9,
		       9, 18, 0, 18, 9, mainwin_quit_cb, SKIN_TITLEBAR);
    mainwin_close->pb_allow_draw = FALSE;

    mainwin_rew =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 16, 88, 23,
		       18, 0, 0, 0, 18, playlist_prev, SKIN_CBUTTONS);
    mainwin_play =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 39, 88, 23,
		       18, 23, 0, 23, 18, mainwin_play_pushed,
		       SKIN_CBUTTONS);
    mainwin_pause =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 62, 88, 23,
		       18, 46, 0, 46, 18, input_pause, SKIN_CBUTTONS);
    mainwin_stop =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 85, 88, 23,
		       18, 69, 0, 69, 18, mainwin_stop_pushed,
		       SKIN_CBUTTONS);
    mainwin_fwd =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 108, 88, 22,
		       18, 92, 0, 92, 18, playlist_next, SKIN_CBUTTONS);
    mainwin_eject =
	create_pbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 136, 89, 22,
		       16, 114, 0, 114, 16, mainwin_eject_pushed,
		       SKIN_CBUTTONS);

    mainwin_srew =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 169, 4, 8,
		       7, playlist_prev);
    mainwin_splay =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 177, 4, 10,
		       7, mainwin_play_pushed);
    mainwin_spause =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 187, 4, 10,
		       7, input_pause);
    mainwin_sstop =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 197, 4, 9,
		       7, mainwin_stop_pushed);
    mainwin_sfwd =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 206, 4, 8,
		       7, playlist_next);
    mainwin_seject =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 216, 4, 9,
		       7, mainwin_eject_pushed);

    mainwin_shuffle =
	create_tbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 164, 89, 46,
		       15, 28, 0, 28, 15, 28, 30, 28, 45,
		       mainwin_shuffle_pushed, SKIN_SHUFREP);
    mainwin_repeat =
	create_tbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 210, 89, 28,
		       15, 0, 0, 0, 15, 0, 30, 0, 45,
		       mainwin_repeat_pushed, SKIN_SHUFREP);

    mainwin_eq =
	create_tbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 219, 58, 23,
		       12, 0, 61, 46, 61, 0, 73, 46, 73, equalizerwin_show,
		       SKIN_SHUFREP);
    tbutton_set_toggled(mainwin_eq, cfg.equalizer_visible);
    mainwin_pl =
	create_tbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 242, 58, 23,
		       12, 23, 61, 69, 61, 23, 73, 69, 73,
		       mainwin_pl_pushed, SKIN_SHUFREP);
    tbutton_set_toggled(mainwin_pl, cfg.playlist_visible);

    mainwin_info =
	create_textbox(&mainwin_wlist, mainwin_bg, mainwin_gc, 112, 27,
		       153, 1, SKIN_TEXT);
    textbox_set_scroll(mainwin_info, cfg.autoscroll);
    textbox_set_xfont(mainwin_info, TRUE, cfg.mainwin_font);
    mainwin_rate_text =
	create_textbox(&mainwin_wlist, mainwin_bg, mainwin_gc, 111, 43, 15,
		       0, SKIN_TEXT);
    mainwin_freq_text =
	create_textbox(&mainwin_wlist, mainwin_bg, mainwin_gc, 156, 43, 10,
		       0, SKIN_TEXT);

    mainwin_menurow =
	create_menurow(&mainwin_wlist, mainwin_bg, mainwin_gc, 10, 22, 304,
		       0, 304, 44, mainwin_mr_change, mainwin_mr_release,
		       SKIN_TITLEBAR);
    mainwin_menurow->mr_doublesize_selected = cfg.doublesize;
    mainwin_menurow->mr_always_selected = cfg.always_on_top;

    mainwin_volume =
	create_hslider(&mainwin_wlist, mainwin_bg, mainwin_gc, 107, 57, 68,
		       13, 15, 422, 0, 422, 14, 11, 15, 0, 0, 51,
		       mainwin_volume_framecb, mainwin_volume_motioncb,
		       mainwin_volume_releasecb, SKIN_VOLUME);
    mainwin_balance =
	create_hslider(&mainwin_wlist, mainwin_bg, mainwin_gc, 177, 57, 38,
		       13, 15, 422, 0, 422, 14, 11, 15, 9, 0, 24,
		       mainwin_balance_framecb, mainwin_balance_motioncb,
		       mainwin_balance_releasecb, SKIN_BALANCE);

    mainwin_monostereo =
	create_monostereo(&mainwin_wlist, mainwin_bg, mainwin_gc, 212, 41,
			  SKIN_MONOSTEREO);

    mainwin_playstatus =
	create_playstatus(&mainwin_wlist, mainwin_bg, mainwin_gc, 24, 28);

    mainwin_minus_num =
	create_number(&mainwin_wlist, mainwin_bg, mainwin_gc, 36, 26,
		      SKIN_NUMBERS);
    hide_widget(mainwin_minus_num);
    mainwin_10min_num =
	create_number(&mainwin_wlist, mainwin_bg, mainwin_gc, 48, 26,
		      SKIN_NUMBERS);
    hide_widget(mainwin_10min_num);
    mainwin_min_num =
	create_number(&mainwin_wlist, mainwin_bg, mainwin_gc, 60, 26,
		      SKIN_NUMBERS);
    hide_widget(mainwin_min_num);
    mainwin_10sec_num =
	create_number(&mainwin_wlist, mainwin_bg, mainwin_gc, 78, 26,
		      SKIN_NUMBERS);
    hide_widget(mainwin_10sec_num);
    mainwin_sec_num =
	create_number(&mainwin_wlist, mainwin_bg, mainwin_gc, 90, 26,
		      SKIN_NUMBERS);
    hide_widget(mainwin_sec_num);

    mainwin_about =
	create_sbutton(&mainwin_wlist, mainwin_bg, mainwin_gc, 247, 83, 20,
		       25, show_about_window);

    mainwin_vis =
	create_vis(&mainwin_wlist, mainwin_bg, mainwin->window, mainwin_gc,
		   24, 43, 76, cfg.doublesize);
    mainwin_svis =
	create_svis(&mainwin_wlist, mainwin_bg, mainwin_gc, 79, 5);
    active_vis = mainwin_vis;

    mainwin_position =
	create_hslider(&mainwin_wlist, mainwin_bg, mainwin_gc, 16, 72, 248,
		       10, 248, 0, 278, 0, 29, 10, 10, 0, 0, 219, NULL,
		       mainwin_position_motioncb,
		       mainwin_position_releasecb, SKIN_POSBAR);
    hide_widget(mainwin_position);

    mainwin_sposition =
	create_hslider(&mainwin_wlist, mainwin_bg, mainwin_gc, 226, 4, 17,
		       7, 17, 36, 17, 36, 3, 7, 36, 0, 1, 13,
		       mainwin_spos_frame_cb, mainwin_spos_motion_cb,
		       mainwin_spos_release_cb, SKIN_TITLEBAR);
    hide_widget(mainwin_sposition);

    mainwin_stime_min =
	create_textbox(&mainwin_wlist, mainwin_bg, mainwin_gc, 130, 4, 15,
		       FALSE, SKIN_TEXT);
    mainwin_stime_sec =
	create_textbox(&mainwin_wlist, mainwin_bg, mainwin_gc, 147, 4, 10,
		       FALSE, SKIN_TEXT);

    if (!cfg.player_shaded) {
	hide_widget(mainwin_svis);
	hide_widget(mainwin_srew);
	hide_widget(mainwin_splay);
	hide_widget(mainwin_spause);
	hide_widget(mainwin_sstop);
	hide_widget(mainwin_sfwd);
	hide_widget(mainwin_seject);
	hide_widget(mainwin_stime_min);
	hide_widget(mainwin_stime_sec);
    }

}

static void mainwin_check_state(GtkWidget * widget,
				GdkEvent * event, gpointer user_data)
{

    GdkWindowState state =
	((GdkEventWindowState *) event)->new_window_state;
    GdkWindowState mask = ((GdkEventWindowState *) event)->changed_mask;

    if (state == 0) {
	if (cfg.playlist_visible)
	    gtk_window_deiconify(GTK_WINDOW(playlistwin));

	if (cfg.equalizer_visible)
	    gtk_window_deiconify(GTK_WINDOW(equalizerwin));


	gtk_window_deiconify(GTK_WINDOW(mainwin));
    }

    if ((mask & GDK_WINDOW_STATE_ICONIFIED) == GDK_WINDOW_STATE_ICONIFIED) {
	if ((state & GDK_WINDOW_STATE_ICONIFIED) ==
	    GDK_WINDOW_STATE_ICONIFIED) {
	    gtk_window_iconify(GTK_WINDOW(playlistwin));
	    gtk_window_iconify(GTK_WINDOW(equalizerwin));
	}
    }
}


static void mainwin_raise_other_windows(GtkWidget * widget,
					gpointer user_data)
{

    /* FIXME: Not sure if this is a hack or not, since deiconify doesn't neccesarily sound like
       it would do what it practically does for us here though */

    gtk_window_deiconify(GTK_WINDOW(playlistwin));
    gtk_window_deiconify(GTK_WINDOW(equalizerwin));

}

static void mainwin_create_gtk(void)
{
    mainwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    dock_window_list = dock_add_window(dock_window_list, mainwin);

    /* FIXME: Convert this all to GtkDrawingArea */
    gtk_widget_set_app_paintable(mainwin, TRUE);
    gtk_window_set_title(GTK_WINDOW(mainwin), _("bmp"));
    gtk_window_set_resizable(GTK_WINDOW(mainwin), FALSE);
    gtk_widget_set_events(mainwin,
			  GDK_FOCUS_CHANGE_MASK | GDK_BUTTON_MOTION_MASK |
			  GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			  GDK_SCROLL_MASK | GDK_KEY_PRESS_MASK |
			  GDK_VISIBILITY_NOTIFY_MASK);

    if (cfg.player_x != -1 && cfg.save_window_position)
	dock_set_uposition(mainwin, cfg.player_x, cfg.player_y);
    gtk_widget_realize(mainwin);

    mainwin_set_icon(mainwin, "bmp");
    util_set_cursor(mainwin);

    if (cfg.doublesize)
	gtk_widget_set_usize(mainwin, MAINWIN_WIDTH * 2,
			     cfg.player_shaded ? MAINWIN_SHADED_HEIGHT *
			     2 : MAINWIN_HEIGHT * 2);
    else
	gtk_widget_set_usize(mainwin, MAINWIN_WIDTH,
			     cfg.
			     player_shaded ? MAINWIN_SHADED_HEIGHT :
			     MAINWIN_HEIGHT);
    if (!cfg.show_wm_decorations)
	gdk_window_set_decorations(mainwin->window, 0);

    g_signal_connect(G_OBJECT(mainwin), "destroy",
		     G_CALLBACK(mainwin_destroy), NULL);
    g_signal_connect(G_OBJECT(mainwin), "button_press_event",
		     G_CALLBACK(mainwin_press), NULL);
    g_signal_connect(G_OBJECT(mainwin), "scroll_event",
		     G_CALLBACK(mainwin_scrolled), NULL);
    g_signal_connect(G_OBJECT(mainwin), "button_release_event",
		     G_CALLBACK(mainwin_release), NULL);
    g_signal_connect(G_OBJECT(mainwin), "motion_notify_event",
		     G_CALLBACK(mainwin_motion), NULL);
    g_signal_connect(G_OBJECT(mainwin), "focus_in_event",
		     G_CALLBACK(mainwin_focus_in), NULL);
    g_signal_connect(G_OBJECT(mainwin), "focus_out_event",
		     G_CALLBACK(mainwin_focus_out), NULL);
    g_signal_connect(G_OBJECT(mainwin), "configure_event",
		     G_CALLBACK(mainwin_configure), NULL);

    g_signal_connect(G_OBJECT(mainwin), "client_event",
		     G_CALLBACK(mainwin_client_event), NULL);
    xmms_drag_dest_set(mainwin);
    g_signal_connect(G_OBJECT(mainwin), "drag-data-received",
		     G_CALLBACK(mainwin_drag_data_received), NULL);
    g_signal_connect(G_OBJECT(mainwin), "key_press_event",
		     G_CALLBACK(mainwin_keypress), NULL);
    g_signal_connect(G_OBJECT(mainwin), "show",
		     G_CALLBACK(mainwin_raise_other_windows), NULL);
    g_signal_connect(G_OBJECT(mainwin), "window-state-event",
		     G_CALLBACK(mainwin_check_state), NULL);


    gdk_window_set_group(mainwin->window, mainwin->window);
    mainwin_set_back_pixmap();
}

void mainwin_create(void)
{
    mainwin_bg =
	gdk_pixmap_new(NULL, MAINWIN_WIDTH, MAINWIN_HEIGHT,
		       gdk_rgb_get_visual()->depth);
    mainwin_bg_dblsize =
	gdk_pixmap_new(NULL, MAINWIN_WIDTH * 2, MAINWIN_HEIGHT * 2,
		       gdk_rgb_get_visual()->depth);
    mainwin_create_gtk();
    mainwin_create_widgets();
}

void mainwin_recreate(void)
{
    g_signal_handlers_disconnect_by_func(G_OBJECT(mainwin),
					 mainwin_destroy, NULL);
    dock_window_list = g_list_remove(dock_window_list, mainwin);
    gtk_widget_destroy(mainwin);
    mainwin_create_gtk();
    vis_set_window(mainwin_vis, mainwin->window);
    mainwin_set_shape_mask();
    draw_main_window(TRUE);
}


void set_timer_mode(TimerMode mode)
{
    char *path;

    if (mode == TIMER_ELAPSED)
	path = "/Time Elapsed";
    else
	path = "/Time Remaining";

    CHECK_SET(mainwin_options_menu, path, TRUE);
}

static void set_timer_mode_menu_cb(TimerMode mode)
{
    cfg.timer_mode = mode;
}

gint idle_func(gpointer data)
{
    gint time, t, length;
    gchar stime_prefix, *tmp;
    static gboolean waiting = FALSE;
    static gint count = 0;

    static GTimer *pause_timer = NULL;

    if (get_input_playing()) {
	GDK_THREADS_ENTER();
	vis_playback_start();
	GDK_THREADS_LEAVE();
	time = input_get_time();
	if (time == -1) {
	    if (cfg.pause_between_songs) {
		gint timeleft;
		if (!waiting) {
		    if (!pause_timer)
			pause_timer = g_timer_new();
		    else
			g_timer_reset(pause_timer);
		    waiting = TRUE;
		}
		timeleft = cfg.pause_between_songs_time -
		    (gint) g_timer_elapsed(pause_timer, NULL);
		number_set_number(mainwin_10min_num, timeleft / 600);
		number_set_number(mainwin_min_num, (timeleft / 60) % 10);
		number_set_number(mainwin_10sec_num, (timeleft / 10) % 6);
		number_set_number(mainwin_sec_num, timeleft % 10);
		if (!mainwin_sposition->hs_pressed) {
		    gchar temp[3];

		    g_sprintf(temp, "%2.2d", timeleft / 60);
		    textbox_set_text(mainwin_stime_min, temp);
		    g_sprintf(temp, "%2.2d", timeleft % 60);
		    textbox_set_text(mainwin_stime_sec, temp);
		}

		playlistwin_set_time(timeleft * 1000, 0, TIMER_ELAPSED);
	    }

	    if (!cfg.pause_between_songs ||
		g_timer_elapsed(pause_timer, NULL) >=
		cfg.pause_between_songs_time) {
		GDK_THREADS_ENTER();
		playlist_eof_reached();
		GDK_THREADS_LEAVE();
		waiting = FALSE;
	    }
	} else if (time == -2) {
	    static GtkWidget *infobox;
	    GDK_THREADS_ENTER();
	    if (!infobox) {
		infobox = xmms_show_message(_("Couldn't open audio"),
					    _("Please check that:\n\n"
					      "1. You have the correct output plugin selected\n"
					      "2. No other programs is blocking the soundcard\n"
					      "3. Your soundcard is configured properly"),
					    _("Ok"), FALSE, NULL, NULL);
		g_signal_connect(G_OBJECT(infobox), "destroy",
				 G_CALLBACK(gtk_widget_destroyed),
				 &infobox);
	    } else
		gdk_window_raise(infobox->window);
	    mainwin_stop_pushed();
	    GDK_THREADS_LEAVE();
	} else {
	    length = playlist_get_current_length();
	    playlistwin_set_time(time, length, cfg.timer_mode);
	    input_update_vis(time);

	    if (cfg.timer_mode == TIMER_REMAINING) {
		if (length != -1) {
		    number_set_number(mainwin_minus_num, 11);
		    t = length - time;
		    stime_prefix = '-';
		} else {
		    number_set_number(mainwin_minus_num, 10);
		    t = time;
		    stime_prefix = ' ';
		}
	    } else {
		number_set_number(mainwin_minus_num, 10);
		t = time;
		stime_prefix = ' ';
	    }
	    t /= 1000;

	    /*
	     * Show the time in the format HH:MM when we
	     * have more than 100 minutes.
	     */
	    if (t >= 100 * 60)
		t /= 60;
	    number_set_number(mainwin_10min_num, t / 600);
	    number_set_number(mainwin_min_num, (t / 60) % 10);
	    number_set_number(mainwin_10sec_num, (t / 10) % 6);
	    number_set_number(mainwin_sec_num, t % 10);

	    if (!mainwin_sposition->hs_pressed) {
		tmp = g_strdup_printf("%c%2.2d", stime_prefix, t / 60);
		textbox_set_text(mainwin_stime_min, tmp);
		g_free(tmp);

		tmp = g_strdup_printf("%2.2d", t % 60);
		textbox_set_text(mainwin_stime_sec, tmp);
		g_free(tmp);
	    }

	    time /= 1000;
	    length /= 1000;
	    if (length > 0) {
		if (time > length) {
		    hslider_set_position(mainwin_position, 219);
		    hslider_set_position(mainwin_sposition, 13);
		} else {
		    hslider_set_position(mainwin_position,
					 (time * 219) / length);
		    hslider_set_position(mainwin_sposition,
					 ((time * 12) / length) + 1);
		}
	    } else {
		hslider_set_position(mainwin_position, 0);
		hslider_set_position(mainwin_sposition, 1);
	    }
	}

	if (time != -1)
	    waiting = FALSE;
    } else {
	GDK_THREADS_ENTER();
	vis_playback_stop();
	GDK_THREADS_LEAVE();
    }


    GDK_THREADS_ENTER();
    check_ctrlsocket();

    draw_main_window(mainwin_force_redraw);
    if (!count) {
	read_volume(VOLSET_UPDATE);
	count = 10;
    } else
	count--;
    mainwin_force_redraw = FALSE;
    draw_playlist_window(FALSE);
    draw_equalizer_window(FALSE);

    pthread_mutex_lock(&title_mutex);
    if (mainwin_title_text) {
/* 		gdk_window_set_title(GTK_WINDOW(mainwin), mainwin_title_text); */
	mainwin_set_title(mainwin_title_text);
	g_free(mainwin_title_text);
	mainwin_title_text = NULL;
    }
    pthread_mutex_unlock(&title_mutex);

    GDK_THREADS_LEAVE();

    return TRUE;

}
