/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2001  Haavard Kvaalen <havardk@xmms.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "beep_mini.xpm"
#include "beep.h"
#include "main.h"
#include <gdk/gdkprivate.h>
#include <gtk/gtkimage.h>
#include <X11/Xlib.h>
#include <sys/ipc.h>
#include <ctype.h>
#ifdef HAVE_FTS_H
#include <fts.h>
#endif

static GQuark quark_popup_data;

#include "playlistwin.h"


/*
 * escape_shell_chars()
 *
 * Escapes characters that are special to the shell inside double quotes.
 */

char *escape_shell_chars(const char *string)
{
    const char *special = "$`\"\\";	/* Characters to escape */
    const char *in = string;
    char *out, *escaped;
    int num = 0;

    while (*in != '\0')
	if (strchr(special, *in++))
	    num++;

    escaped = g_malloc(strlen(string) + num + 1);

    in = string;
    out = escaped;

    while (*in != '\0') {
	if (strchr(special, *in))
	    *out++ = '\\';
	*out++ = *in++;
    }
    *out = '\0';

    return escaped;
}


/*
 * find_file_recursively() by Jörg Schuler Wed, 17 Feb 1999 23:50:52
 * +0100 Placed under GPL version 2 or (at your option) any later
 * version
 */
/* find_file_recursively() rewritten by David Lau Saturday, Nov 15 19:02
   combined first and secondary loops, changed everything to use glib functions
   and changed the path combination to g_build_filename which makes more sense 
*/
/*
 * find <file> in directory <dirname> or subdirectories.  return
 * pointer to complete filename which has to be freed by calling
 * "g_free()" after use. Returns NULL if file could not be found.
 */

gchar *find_file_recursively(const gchar * dirname, const gchar * file)
{
    GDir *dir2;
    GError *error = NULL;
    gchar *result, *found = NULL, *retval = NULL;

    const gchar *tmp;
    dir2 = g_dir_open(dirname, 0, &error);

    if (error) {
	g_critical("error!!");
	return retval;
    }
    if (!dir2)
	return retval;
    while ((tmp = g_dir_read_name(dir2)) != NULL) {
	/* We need this in order to find out if file is directory */
	found = g_build_filename(dirname, tmp, NULL);
	if (g_file_test(found, G_FILE_TEST_IS_REGULAR)) {
	    /* Normal file -- maybe just what we are looking for? */
	    if (!strcasecmp(tmp, file)) {
		if (strlen(found) > FILENAME_MAX) {
		    /* No good! File + path too long */
		    g_dir_close(dir2);
		    break;
		} else {
		    retval = found;
		    found = NULL;
		    break;
		}
	    }
	} else if (g_file_test(found, G_FILE_TEST_IS_DIR)) {
	    result = find_file_recursively(found, file);
	    if (result != NULL) {
		retval = result;
		break;
	    }
	}
	g_free(found);
	found = NULL;
    }
    if (found)
	g_free(found);

    g_dir_close(dir2);

    return retval;
}

typedef enum { ARCHIVE_DIR = 0, ARCHIVE_TAR =
	1, ARCHIVE_TGZ, ARCHIVE_ZIP, ARCHIVE_TBZ2
} ArchiveType;

typedef gchar *(*ExtractFunc) (const gchar *, const gchar *);


static gchar *ExtractTar(const gchar * archive, const gchar * dest)
{
    const gchar *tarcmd = getenv("TARCMD") ? getenv("TARCMD") : "tar";
    return g_strdup_printf("%s >/dev/null xf \"%s\" -C %s", tarcmd,
			   archive, dest);
}

static gchar *ExtractZip(const gchar * archive, const gchar * dest)
{
    return g_strdup_printf("%s >/dev/null -o -j \"%s\" -d %s", "unzip",
			   archive, dest);
}

static gchar *ExtractTGZ(const gchar * archive, const gchar * dest)
{
    return g_strdup_printf("%s >/dev/null xzf \"%s\" -C %s", "tar",
			   archive, dest);
}
static gchar *ExtractBZ2(const gchar * archive, const gchar * dest)
{
    return g_strdup_printf("bzip2 -dc \"%s\" | %s >/dev/null xf - -C %s",
			   archive, "tar", dest);
}

ExtractFunc ArchiveExtractFuncs[] = {
    (ExtractFunc) NULL,
    ExtractTar,
    ExtractTGZ,
    ExtractZip,
    ExtractBZ2
};


struct ArchiveTypeInfo {
    ArchiveType type;
    const gchar *ext;
    ExtractFunc extract;

} ArchExtentions[] = {
    {
    ARCHIVE_TAR, ".tar"}, {
    ARCHIVE_ZIP, ".wsz"}, {
    ARCHIVE_ZIP, ".zip"}, {
    ARCHIVE_TGZ, ".tar.gz"}, {
    ARCHIVE_TGZ, ".tgz"}, {
    ARCHIVE_TBZ2, ".tar.bz2"}, {
    ARCHIVE_TBZ2, ".bz2"}, {
    0, NULL}
};


/* should add some ARCHIVE_UNKNOWN type or somethin' */
ArchiveType get_archive_type(const gchar * filename)
{
    gint indx = 0;
    ArchiveType retval = ARCHIVE_DIR;
    if (g_file_test(filename, G_FILE_TEST_IS_DIR))
	return retval;

    while (ArchExtentions[indx].ext) {
	if (g_str_has_suffix(filename, ArchExtentions[indx].ext)) {
	    retval = ArchExtentions[indx].type;
	    break;
	}
	indx++;
    }
    return retval;
}

gchar *archive_basename(const gchar * str)
{
    gint indx = 0;
    gchar *retval = NULL;
    while (ArchExtentions[indx].ext) {
	if (g_str_has_suffix(str, ArchExtentions[indx].ext)) {
	    const gchar *end = g_strrstr(str, ArchExtentions[indx].ext);
	    if (end) {
		retval = g_strndup(str, end - str);
	    }

	    break;
	}
	indx++;

    }
    return retval;

}

gboolean file_is_archive(const gchar * filename)
{
    return (get_archive_type(filename) != ARCHIVE_DIR);
}

/*
   decompress_archive

   Decompresses the archive "filename" to a temporary directory, 
   returns the path to the temp dir, or NULL if failed,
   watch out tho, doesn't actually check if the system command succeeds :-|
*/

gchar *decompress_archive(const gchar * filename)
{
    gchar *tmpdir, *cmd, *escaped, *retval = NULL;
    ArchiveType archtype;
    ExtractFunc extract;
    archtype = get_archive_type(filename);

    if (archtype == ARCHIVE_DIR)
	return retval;

    tmpdir = g_build_filename(g_get_tmp_dir(), "bmp.XXXXXXXX", NULL);
    if (!mkdtemp(tmpdir)) {
	g_free(tmpdir);
	g_message
	    ("Failed to create temporary directory: %s.  Unable to load skin.",
	     strerror(errno));
	return retval;
    }


    extract = ArchiveExtractFuncs[archtype];
    escaped = escape_shell_chars(filename);
    cmd = extract(filename, tmpdir);
    g_free(escaped);

    if (cmd) {
	system(cmd);
	g_free(cmd);
	retval = tmpdir;
    } else {
	g_free(tmpdir);
    }

    return retval;
}


void del_directory(const char *dirname)
{
#ifdef HAVE_FTS_H
    char *const argv[2] = { (char *) dirname, NULL };
    FTS *fts;
    FTSENT *p;

    fts = fts_open(argv, FTS_PHYSICAL, (int (*)()) NULL);
    while ((p = fts_read(fts)) != NULL) {
	switch (p->fts_info) {
	case FTS_D:
	    break;
	case FTS_DNR:
	case FTS_ERR:
	    break;
	case FTS_DP:
	    rmdir(p->fts_accpath);
	    break;
	default:
	    unlink(p->fts_accpath);
	    break;
	}
    }
    fts_close(fts);
#else				/* !HAVE_FTS_H */
    DIR *dir;
    struct dirent *dirent;
    char *file;

    if ((dir = opendir(dirname)) != NULL) {
	while ((dirent = readdir(dir)) != NULL) {
	    if (strcmp(dirent->d_name, ".")
		&& strcmp(dirent->d_name, "..")) {
		file = g_strdup_printf("%s/%s", dirname, dirent->d_name);
		if (unlink(file) == -1)
		    if (errno == EISDIR)
			del_directory(file);
		g_free(file);
	    }
	}
	closedir(dir);
    }
    rmdir(dirname);
#endif				/* !HAVE_FTS_H */
}

GdkImage *create_dblsize_image(GdkImage * img)
{
    GdkImage *dblimg;
    register guint x, y;

    /*
     * This needs to be optimized
     */

    dblimg =
	gdk_image_new(GDK_IMAGE_NORMAL, gdk_visual_get_best(),
		      img->width << 1, img->height << 1);
    if (dblimg->bpp == 1) {
	register guint8 *srcptr, *ptr, *ptr2, pix;

	srcptr = GDK_IMAGE_XIMAGE(img)->data;
	ptr = GDK_IMAGE_XIMAGE(dblimg)->data;
	ptr2 = ptr + dblimg->bpl;

	for (y = 0; y < img->height; y++) {
	    for (x = 0; x < img->width; x++) {
		pix = *srcptr++;
		*ptr++ = pix;
		*ptr++ = pix;
		*ptr2++ = pix;
		*ptr2++ = pix;
	    }
	    srcptr += img->bpl - img->width;
	    ptr += (dblimg->bpl << 1) - dblimg->width;
	    ptr2 += (dblimg->bpl << 1) - dblimg->width;
	}
    }
    if (dblimg->bpp == 2) {
	guint16 *srcptr, *ptr, *ptr2, pix;

	srcptr = (guint16 *) GDK_IMAGE_XIMAGE(img)->data;
	ptr = (guint16 *) GDK_IMAGE_XIMAGE(dblimg)->data;
	ptr2 = ptr + (dblimg->bpl >> 1);

	for (y = 0; y < img->height; y++) {
	    for (x = 0; x < img->width; x++) {
		pix = *srcptr++;
		*ptr++ = pix;
		*ptr++ = pix;
		*ptr2++ = pix;
		*ptr2++ = pix;
	    }
	    srcptr += (img->bpl >> 1) - img->width;
	    ptr += (dblimg->bpl) - dblimg->width;
	    ptr2 += (dblimg->bpl) - dblimg->width;
	}
    }
    if (dblimg->bpp == 3) {
	register guint8 *srcptr, *ptr, *ptr2, pix1, pix2, pix3;

	srcptr = GDK_IMAGE_XIMAGE(img)->data;
	ptr = GDK_IMAGE_XIMAGE(dblimg)->data;
	ptr2 = ptr + dblimg->bpl;

	for (y = 0; y < img->height; y++) {
	    for (x = 0; x < img->width; x++) {
		pix1 = *srcptr++;
		pix2 = *srcptr++;
		pix3 = *srcptr++;
		*ptr++ = pix1;
		*ptr++ = pix2;
		*ptr++ = pix3;
		*ptr++ = pix1;
		*ptr++ = pix2;
		*ptr++ = pix3;
		*ptr2++ = pix1;
		*ptr2++ = pix2;
		*ptr2++ = pix3;
		*ptr2++ = pix1;
		*ptr2++ = pix2;
		*ptr2++ = pix3;

	    }
	    srcptr += img->bpl - (img->width * 3);
	    ptr += (dblimg->bpl << 1) - (dblimg->width * 3);
	    ptr2 += (dblimg->bpl << 1) - (dblimg->width * 3);
	}
    }
    if (dblimg->bpp == 4) {
	register guint32 *srcptr, *ptr, *ptr2, pix;

	srcptr = (guint32 *) GDK_IMAGE_XIMAGE(img)->data;
	ptr = (guint32 *) GDK_IMAGE_XIMAGE(dblimg)->data;
	ptr2 = ptr + (dblimg->bpl >> 2);

	for (y = 0; y < img->height; y++) {
	    for (x = 0; x < img->width; x++) {
		pix = *srcptr++;
		*ptr++ = pix;
		*ptr++ = pix;
		*ptr2++ = pix;
		*ptr2++ = pix;
	    }
	    srcptr += (img->bpl >> 2) - img->width;
	    ptr += (dblimg->bpl >> 1) - dblimg->width;
	    ptr2 += (dblimg->bpl >> 1) - dblimg->width;
	}
    }
    return dblimg;
}

char *read_ini_string(const char *filename, const char *section,
		      const char *key)
{
    FILE *file;
    char *buffer, *ret_buffer = NULL;
    int found_section = 0, off = 0, len = 0;
    struct stat statbuf;

    if (!filename)
	return NULL;

    if ((file = fopen(filename, "r")) == NULL)
	return NULL;

    if (stat(filename, &statbuf) < 0) {
	fclose(file);
	return NULL;
    }

    buffer = g_malloc(statbuf.st_size);
    fread(buffer, 1, statbuf.st_size, file);
    while (!ret_buffer && off < statbuf.st_size) {
	while (off < statbuf.st_size &&
	       (buffer[off] == '\r' || buffer[off] == '\n' ||
		buffer[off] == ' ' || buffer[off] == '\t'))
	    off++;
	if (off >= statbuf.st_size)
	    break;
	if (buffer[off] == '[') {
	    int slen = strlen(section);
	    off++;
	    found_section = 0;
	    if (off + slen + 1 < statbuf.st_size &&
		!strncasecmp(section, &buffer[off], slen)) {
		off += slen;
		if (buffer[off] == ']') {
		    off++;
		    found_section = 1;
		}
	    }
	} else if (found_section && off + strlen(key) < statbuf.st_size &&
		   !strncasecmp(key, &buffer[off], strlen(key))) {
	    off += strlen(key);
	    while (off < statbuf.st_size &&
		   (buffer[off] == ' ' || buffer[off] == '\t'))
		off++;
	    if (off >= statbuf.st_size)
		break;
	    if (buffer[off] == '=') {
		off++;
		while (off < statbuf.st_size &&
		       (buffer[off] == ' ' || buffer[off] == '\t'))
		    off++;
		if (off >= statbuf.st_size)
		    break;
		len = 0;
		while (off + len < statbuf.st_size &&
		       buffer[off + len] != '\r' &&
		       buffer[off + len] != '\n' &&
		       buffer[off + len] != ';')
		    len++;
		ret_buffer = g_strndup(&buffer[off], len);
		off += len;
	    }
	}
	while (off < statbuf.st_size &&
	       buffer[off] != '\r' && buffer[off] != '\n')
	    off++;
    }

    g_free(buffer);
    fclose(file);
    return ret_buffer;
}

GArray *string_to_garray(const gchar * str)
{
    GArray *array;
    gint temp;
    const gchar *ptr = str;
    gchar *endptr;

    array = g_array_new(FALSE, TRUE, sizeof(gint));
    for (;;) {
	temp = strtol(ptr, &endptr, 10);
	if (ptr == endptr)
	    break;
	g_array_append_val(array, temp);
	ptr = endptr;
	while (!isdigit(*ptr) && (*ptr) != '\0')
	    ptr++;
	if (*ptr == '\0')
	    break;
    }
    return (array);
}

GArray *read_ini_array(const gchar * filename, const gchar * section,
		       const gchar * key)
{
    gchar *temp;
    GArray *a;

    if ((temp = read_ini_string(filename, section, key)) == NULL)
	return NULL;
    a = string_to_garray(temp);
    g_free(temp);
    return a;
}

void glist_movedown(GList * list)
{
    gpointer temp;

    if (g_list_next(list)) {
	temp = list->data;
	list->data = list->next->data;
	list->next->data = temp;
    }
}

void glist_moveup(GList * list)
{
    gpointer temp;

    if (g_list_previous(list)) {
	temp = list->data;
	list->data = list->prev->data;
	list->prev->data = temp;
    }
}

struct MenuPos {
    gint x;
    gint y;
};

static void util_menu_position(GtkMenu * menu, gint * x, gint * y,
			       gboolean * push_in, gpointer data)
{
    GtkRequisition requisition;
    gint screen_width;
    gint screen_height;
    struct MenuPos *pos = data;

    gtk_widget_size_request(GTK_WIDGET(menu), &requisition);

    screen_width = gdk_screen_width();
    screen_height = gdk_screen_height();

    *x = CLAMP(pos->x - 2, 0, MAX(0, screen_width - requisition.width));
    *y = CLAMP(pos->y - 2, 0, MAX(0, screen_height - requisition.height));
}

static void util_menu_delete_popup_data(GtkObject * object,
					GtkItemFactory * ifactory)
{
    gtk_signal_disconnect_by_func(object,
				  GTK_SIGNAL_FUNC
				  (util_menu_delete_popup_data), ifactory);
    gtk_object_remove_data_by_id(GTK_OBJECT(ifactory), quark_popup_data);
}


/*
 * util_item_factory_popup[_with_data]() is a replacement for
 * gtk_item_factory_popup[_with_data]().
 *
 * The difference is that the menu is always popped up whithin the
 * screen.  This means it does not neccesarily pop up at (x,y).
 */

void util_item_factory_popup_with_data(GtkItemFactory * ifactory,
				       gpointer data,
				       GtkDestroyNotify destroy, guint x,
				       guint y, guint mouse_button,
				       guint32 time)
{
    static GQuark quark_user_menu_pos = 0;
    struct MenuPos *pos;

    if (!quark_user_menu_pos)
	quark_user_menu_pos = g_quark_from_static_string("user_menu_pos");

    if (!quark_popup_data)
	quark_popup_data =
	    g_quark_from_static_string("GtkItemFactory-popup-data");

    pos = g_object_get_qdata(G_OBJECT(ifactory), quark_user_menu_pos);
    if (!pos) {
	pos = g_malloc0(sizeof(struct MenuPos));

	g_object_set_qdata_full(G_OBJECT(ifactory->widget),
				quark_user_menu_pos, pos, g_free);
    }
    pos->x = x;
    pos->y = y;


    if (data != NULL) {
	g_object_set_qdata_full(G_OBJECT(ifactory),
				quark_popup_data, data, destroy);
	g_signal_connect(G_OBJECT(ifactory->widget),
			 "selection-done",
			 G_CALLBACK(util_menu_delete_popup_data),
			 ifactory);
    }

    gtk_menu_popup(GTK_MENU(ifactory->widget), NULL, NULL,
		   (GtkMenuPositionFunc) util_menu_position,
		   pos, mouse_button, time);
}

void util_item_factory_popup(GtkItemFactory * ifactory, guint x, guint y,
			     guint mouse_button, guint32 time)
{
    util_item_factory_popup_with_data(ifactory, NULL, NULL, x, y,
				      mouse_button, time);
}

static gint util_find_compare_func(gconstpointer a, gconstpointer b)
{
    return strcasecmp(a, b);
}

static void util_add_url_callback(GtkWidget * w, GtkWidget * entry)
{
    gchar *text;
    GList *node;

    text = (char *) gtk_entry_get_text(GTK_ENTRY(entry));
    if (!g_list_find_custom(cfg.url_history, text, util_find_compare_func)) {
	cfg.url_history = g_list_prepend(cfg.url_history, g_strdup(text));
	while (g_list_length(cfg.url_history) > 30) {
	    node = g_list_last(cfg.url_history);
	    g_free(node->data);
	    cfg.url_history = g_list_remove_link(cfg.url_history, node);
	}
    }
}

static void addurl_set_icon(GtkWidget * win, gchar * title)
{
    static GdkPixmap *icon;
    static GdkBitmap *mask;
    GdkAtom icon_atom;
    glong data[2];

    if (!icon)
	icon =
	    gdk_pixmap_create_from_xpm_d(win->window, &mask,
					 &win->style->bg[GTK_STATE_NORMAL],
					 beep_mini_xpm);
    data[0] = GDK_WINDOW_XWINDOW(icon);
    data[1] = GDK_WINDOW_XWINDOW(mask);

    icon_atom = gdk_atom_intern((const gchar *) "KWM_WIN_ICON", FALSE);
    gdk_property_change(win->window, icon_atom, icon_atom, 32,
			GDK_PROP_MODE_REPLACE, (guchar *) data, 2);
    gdk_window_set_icon(win->window, NULL, icon, mask);
    gdk_window_set_icon_name(win->window, title);
    gdk_window_set_title(win->window, title);
    gdk_window_set_group(win->window, mainwin->window);
}

GtkWidget *util_create_add_url_window(gchar * caption,
				      GCallback enqueue_func)
{
    GtkWidget *win, *vbox, *bbox, *enqueue, *cancel, *combo;

    win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), caption);
    gtk_window_set_position(GTK_WINDOW(win), GTK_WIN_POS_CENTER);

    gtk_window_set_default_size(GTK_WINDOW(win), 400, -1);
    gtk_container_set_border_width(GTK_CONTAINER(win), 10);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_add(GTK_CONTAINER(win), vbox);

    combo = gtk_combo_new();
    if (cfg.url_history)
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), cfg.url_history);
    g_signal_connect(G_OBJECT(GTK_COMBO(combo)->entry), "activate",
		     G_CALLBACK(util_add_url_callback),
		     GTK_COMBO(combo)->entry);
    g_signal_connect(G_OBJECT(GTK_COMBO(combo)->entry), "activate",
		     G_CALLBACK(enqueue_func), GTK_COMBO(combo)->entry);
    gtk_box_pack_start(GTK_BOX(vbox), combo, FALSE, FALSE, 0);
    gtk_window_set_focus(GTK_WINDOW(win), GTK_COMBO(combo)->entry);
    gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "");
    gtk_combo_set_use_arrows_always(GTK_COMBO(combo), TRUE);
    gtk_widget_show(combo);

    bbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);

    enqueue = gtk_button_new_from_stock(GTK_STOCK_ADD);
    g_signal_connect(G_OBJECT(enqueue), "clicked",
		     G_CALLBACK(util_add_url_callback),
		     GTK_COMBO(combo)->entry);
    g_signal_connect(G_OBJECT(enqueue), "clicked",
		     G_CALLBACK(enqueue_func), GTK_COMBO(combo)->entry);
    gtk_box_pack_start(GTK_BOX(bbox), enqueue, FALSE, FALSE, 0);
    gtk_widget_show(enqueue);

    cancel = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    g_signal_connect_swapped(G_OBJECT(cancel), "clicked",
			     G_CALLBACK(gtk_widget_destroy),
			     GTK_OBJECT(win));
    gtk_box_pack_start(GTK_BOX(bbox), cancel, FALSE, FALSE, 0);
    gtk_widget_show(cancel);

    gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);
    gtk_widget_show(bbox);
    gtk_widget_show(vbox);

    gtk_widget_realize(win);
    addurl_set_icon(win, "bmp: Add Url...");	// somehow gcc thinks this is a function declaration
    return win;
}

static void filebrowser_changed(GtkFileSelection * filesel)
{
    GList *list;
    GList *node;
    char *filename = (char *)
	gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
    GtkListStore *store;
    GtkTreeIter iter;

    if ((list = input_scan_dir(filename)) != NULL) {
	/*
	 * We enter a directory that has been "hijacked" by an
	 * input-plugin. This is used by the CDDA plugin
	 */
	store =
	    GTK_LIST_STORE(gtk_tree_view_get_model
			   (GTK_TREE_VIEW(filesel->file_list)));
	gtk_list_store_clear(store);

	node = list;
	while (node) {

	    gtk_list_store_append(store, &iter);
	    gtk_list_store_set(store, &iter, 0, node->data, -1);
	    g_free(node->data);
	    node = g_list_next(node);
	}
	g_list_free(list);

    }

}

/*
static int filebrowser_idle_changed(gpointer data)
{
	GDK_THREADS_ENTER();
	filebrowser_changed(GTK_FILE_SELECTION(data));
	GDK_THREADS_LEAVE();

	return FALSE;
}
*/
static void filebrowser_entry_changed(GtkEditable * entry, gpointer data)
{
    filebrowser_changed(GTK_FILE_SELECTION(data));

}

/* static void filebrowser_dir_select(GtkCList *clist, int row, int col, GdkEventButton *event, gpointer data)
{
	if (event->type == GDK_2BUTTON_PRESS)
		gtk_idle_add(filebrowser_idle_changed, data);
}
*/
gboolean util_filebrowser_is_dir(GtkFileSelection * filesel)
{
    char *text;
    struct stat buf;
    gboolean retv = FALSE;

    text = g_strdup(gtk_file_selection_get_filename(filesel));

    if (stat(text, &buf) == 0 && S_ISDIR(buf.st_mode)) {
	/* Selected directory */
	int len = strlen(text);
	if (len > 3 && !strcmp(text + len - 4, "/../")) {
	    if (len == 4)
		/* At the root already */
		*(text + len - 3) = '\0';
	    else {
		char *ptr;
		*(text + len - 4) = '\0';
		ptr = strrchr(text, '/');
		*(ptr + 1) = '\0';
	    }
	} else if (len > 2 && !strcmp(text + len - 3, "/./"))
	    *(text + len - 2) = '\0';
	gtk_file_selection_set_filename(filesel, text);
	retv = TRUE;
    }
    g_free(text);
    return retv;
}

static void filebrowser_add_files(gchar ** files,
				  GtkFileSelection * filesel)
{
    int ctr = 0;
    char *ptr;

    if (GTK_IS_WIDGET(mainwin_jtf))
	gtk_widget_set_sensitive(mainwin_jtf, FALSE);

    while (files[ctr] != NULL) {
	playlist_add(files[ctr++]);
    }
    playlistwin_update_list();

    if (GTK_IS_WIDGET(mainwin_jtf))
	gtk_widget_set_sensitive(mainwin_jtf, TRUE);

    gtk_label_get(GTK_LABEL(GTK_BIN(filesel->history_pulldown)->child),
		  &ptr);

    /* This will give an extra slash if the current dir is the root. */
    cfg.filesel_path = g_strconcat(ptr, "/", NULL);
}

static void filebrowser_ok(GtkWidget * w, GtkWidget * filesel)
{
    gchar **files;

    if (util_filebrowser_is_dir(GTK_FILE_SELECTION(filesel)))
	return;
    files = gtk_file_selection_get_selections(GTK_FILE_SELECTION(filesel));
    filebrowser_add_files(files, GTK_FILE_SELECTION(filesel));
    gtk_widget_destroy(filesel);
}

static void filebrowser_play(GtkWidget * w, GtkWidget * filesel)
{
    gchar **files;

    if (util_filebrowser_is_dir
	(GTK_FILE_SELECTION(GTK_FILE_SELECTION(filesel))))
	return;
    playlist_clear();
    files = gtk_file_selection_get_selections(GTK_FILE_SELECTION(filesel));
    filebrowser_add_files(files, GTK_FILE_SELECTION(filesel));
    gtk_widget_destroy(filesel);
    playlist_play();
}

static void filebrowser_add_selected_files(GtkWidget * w, gpointer data)
{
    gchar **files;

    GtkFileSelection *filesel = GTK_FILE_SELECTION(data);
    files = gtk_file_selection_get_selections(filesel);

    filebrowser_add_files(files, filesel);
    gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
				    (GTK_TREE_VIEW(filesel->file_list)));

     /*HACK*/ gtk_entry_set_text(GTK_ENTRY(filesel->selection_entry), "");
}

static void filebrowser_add_all_files(GtkWidget * w, gpointer data)
{
    gchar **files;
    GtkFileSelection *filesel;

    filesel = data;
    gtk_tree_selection_select_all(gtk_tree_view_get_selection
				  (GTK_TREE_VIEW(filesel->file_list)));
    files = gtk_file_selection_get_selections(filesel);
    filebrowser_add_files(files, filesel);
    gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
				    (GTK_TREE_VIEW(filesel->file_list)));
    gtk_entry_set_text(GTK_ENTRY(filesel->selection_entry), "");
}


gboolean util_run_filebrowser(gboolean play_button)
{
    static GtkWidget *dialog;
    GtkWidget *button_add_selected, *button_add_all, *button_close,
	*button_add;
    char *title;

    if (dialog != NULL) {
	gtk_window_present(GTK_WINDOW(dialog));
	return FALSE;
    }

    if (play_button)
	title = _("Play files");
    else
	title = _("Load files");

    dialog = gtk_file_selection_new(title);

    // I don't think anyone should be even able to delete files from his/her media player, yes i'm making a decision for a plurality here
    // but it's just safer despite the confirmation requests
    gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(dialog));
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);

    gtk_file_selection_set_select_multiple
	(GTK_FILE_SELECTION(dialog), TRUE);

    if (cfg.filesel_path)
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog),
					cfg.filesel_path);

    // if we destroy the OK button we'll never receive a GTK_RESPONSE_OK e.g. trough dbl click on a file
    gtk_widget_hide(GTK_FILE_SELECTION(dialog)->ok_button);
    gtk_widget_destroy(GTK_FILE_SELECTION(dialog)->cancel_button);

    /*
     * The mnemonics are quite unorthodox, but that should guarantee they're unique in any locale
     * plus kinda easy to use
     */
    button_add_selected =
	gtk_dialog_add_button(GTK_DIALOG(dialog), "Add selected",
			      GTK_RESPONSE_NONE);
    gtk_button_set_use_underline(GTK_BUTTON(button_add_selected), TRUE);
    g_signal_connect(G_OBJECT(button_add_selected), "clicked",
		     G_CALLBACK(filebrowser_add_selected_files), dialog);

    button_add_all =
	gtk_dialog_add_button(GTK_DIALOG(dialog), "Add all",
			      GTK_RESPONSE_NONE);
    gtk_button_set_use_underline(GTK_BUTTON(button_add_all), TRUE);
    g_signal_connect(G_OBJECT(button_add_all), "clicked",
		     G_CALLBACK(filebrowser_add_all_files), dialog);

    if (play_button) {

	button_add =
	    gtk_dialog_add_button(GTK_DIALOG(dialog), "Play",
				  GTK_RESPONSE_NONE);
	gtk_button_set_use_underline(GTK_BUTTON(button_add), TRUE);
	g_signal_connect(G_OBJECT(button_add), "clicked",
			 G_CALLBACK(filebrowser_play), dialog);
	g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(dialog)->ok_button),
			 "clicked", G_CALLBACK(filebrowser_play), dialog);
    } else {

	button_add =
	    gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_ADD,
				  GTK_RESPONSE_NONE);
	gtk_button_set_use_stock(GTK_BUTTON(button_add), TRUE);
	g_signal_connect(G_OBJECT(button_add), "clicked",
			 G_CALLBACK(filebrowser_ok), dialog);
	g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(dialog)->ok_button),
			 "clicked", G_CALLBACK(filebrowser_ok), dialog);
    }

    button_close =
	gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CLOSE,
			      GTK_RESPONSE_NONE);
    gtk_button_set_use_stock(GTK_BUTTON(button_close), TRUE);
    g_signal_connect_swapped(G_OBJECT(button_close), "clicked",
			     G_CALLBACK(gtk_widget_destroy),
			     G_OBJECT(dialog));

    gtk_widget_set_size_request(dialog, 600, 450);
    gtk_widget_realize(dialog);

    g_signal_connect(G_OBJECT
		     (GTK_FILE_SELECTION(dialog)->history_pulldown),
		     "changed", G_CALLBACK(filebrowser_entry_changed),
		     dialog);

    g_signal_connect(G_OBJECT(dialog), "destroy",
		     G_CALLBACK(gtk_widget_destroyed), &dialog);

    filebrowser_changed(GTK_FILE_SELECTION(dialog));

    gtk_widget_show(dialog);

    return TRUE;
}

GdkFont *util_font_load(char *name)
{
    GdkFont *font;

    PangoFontDescription *desc;
    desc = pango_font_description_from_string(name);

    font = gdk_font_from_description(desc);

    return font;
}

#ifdef ENABLE_NLS
char *util_menu_translate(const char *path, void *func_data)
{
    char *translation = gettext(path);

    if (!translation || *translation != '/') {
	g_warning("Bad translation for menupath: %s", path);
	translation = (char *) path;
    }

    return translation;
}
#endif

void util_set_cursor(GtkWidget * window)
{
    static GdkCursor *cursor;

    if (!window) {
	if (cursor) {
	    gdk_cursor_destroy(cursor);
	    cursor = NULL;
	}
	return;
    }
    if (!cursor)
	cursor = gdk_cursor_new(GDK_LEFT_PTR);

    gdk_window_set_cursor(window->window, cursor);
}

// text_get_extents() taken from The GIMP (C) Spencer Kimball, Peter Mattis et al
gboolean
text_get_extents(const gchar * fontname,
		 const gchar * text,
		 gint * width,
		 gint * height, gint * ascent, gint * descent)
{
    PangoFontDescription *font_desc;
    PangoLayout *layout;
    PangoRectangle rect;
/*  PangoContext         *context;
  PangoLayout          *layout;
  PangoRectangle        rect;
*/

    g_return_val_if_fail(fontname != NULL, FALSE);
    g_return_val_if_fail(text != NULL, FALSE);

    /* FIXME: resolution */

//  context = pango_ft2_get_context (72.0, 72.0);


//  layout = pango_layout_new(pango_context_new());

    layout = gtk_widget_create_pango_layout(GTK_WIDGET(mainwin), text);

//  g_object_unref (context);  

    font_desc = pango_font_description_from_string(fontname);
    pango_layout_set_font_description(layout, font_desc);
    pango_font_description_free(font_desc);
//  pango_layout_set_text (layout, text, -1);

    pango_layout_get_pixel_extents(layout, NULL, &rect);

    if (width)
	*width = rect.width;
    if (height)
	*height = rect.height;

    if (ascent || descent) {
	PangoLayoutIter *iter;
	PangoLayoutLine *line;

	iter = pango_layout_get_iter(layout);
	line = pango_layout_iter_get_line(iter);
	pango_layout_iter_free(iter);

	pango_layout_line_get_pixel_extents(line, NULL, &rect);

	if (ascent)
	    *ascent = PANGO_ASCENT(rect);
	if (descent)
	    *descent = -PANGO_DESCENT(rect);
    }

    g_object_unref(layout);

    return TRUE;
}

void util_dump_menu_rc(void)
{
//      char *filename = g_strconcat(g_get_home_dir(), "/.beep/menurc", NULL);
//      gtk_item_factory_dump_rc(filename, NULL, FALSE);
//      g_free(filename);
}

void util_read_menu_rc(void)
{
//      char *filename = g_strconcat(g_get_home_dir(), "/.beep/menurc", NULL);
//      gtk_item_factory_parse_rc(filename);
//      g_free(filename);
}


#if defined(USE_DMALLOC)

/*
 * We sometimes need a function pointer for g_free().  If dmalloc is
 * used we must create such a function in place of the g_free function
 * which is expanded as a macro.
 */
void g_free_func(gpointer mem)
{
    g_free(mem);
}

/*
 * When allocating via, say, g_strdup the wrapper macro must free the
 * previously allocated string through the original g_free function.
 */
void g_free_orig(gpointer mem)
{
#undef g_free
    void g_free(gpointer);
    g_free(mem);
}

#endif
