/*
 * gdict Dictionary for UNIX
 * Copyright (C) 2003 Alexander Zapjagaev <frustum@frustum.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dict.h"

/*    koi8r 
 */
static int koi8r[256] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
	0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
	0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
	0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
	0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
	0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
	0xfe, 0xe0, 0xe1, 0xf6, 0xe4, 0xe5, 0xf4, 0xe3, 0xf5, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee,
	0xef, 0xff, 0xf0, 0xf1, 0xf2, 0xf3, 0xe6, 0xe2, 0xfc, 0xd9, 0xe7, 0xf8, 0xfd, 0xfa, 0xf7, 0xfb,
	0xf7, 0xfe, 0xe0, 0xe6, 0xe4, 0xe5, 0xf2, 0xf6, 0xf3, 0xf5, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed,
	0xee, 0xfb, 0xef, 0xff, 0xf0, 0xf1, 0xf4, 0xe1, 0xfd, 0xf9, 0xe3, 0xfc, 0xfa, 0xe7, 0xe2, 0xf8
};

/*   
 */
typedef struct dict_memblock_s {
	void *data;
	void *ptr;
	int size;
	struct dict_memblock_s *next;
} dict_memblock_t;

static dict_memblock_t *root_memblock = NULL;
static int allocated_mem = 0;
static int lost_mem = 0;

/*
 */
void dict_mem_release() {
	dict_memblock_t *memblock = root_memblock;
	while(memblock) {
		free(memblock->data);
		dict_memblock_t *next = memblock->next;
		free(memblock);
		memblock = next;
	}
	root_memblock = NULL;
	allocated_mem = 0;
	lost_mem = 0;
}

/*
 */
void *dict_mem_malloc(size_t size) {
	dict_memblock_t *memblock = root_memblock;
	if(!memblock || memblock->size < size + sizeof(unsigned short)) {
		memblock = calloc(1,sizeof(dict_memblock_t));
		assert(memblock != NULL);
		memblock->size = sizeof(char) * 1024 * 1024;
		memblock->data = malloc(memblock->size);
		assert(memblock->data != NULL);
		memblock->ptr = memblock->data;
		memblock->next = root_memblock;
		root_memblock = memblock;
		
		allocated_mem += memblock->size;
	}
	void *ret = memblock->ptr;
	*((unsigned short*)ret) = size;
	memblock->ptr += size + sizeof(unsigned short);
	memblock->size -= size + sizeof(unsigned short);
	return ret + sizeof(unsigned short);
}

/*
 */
void dict_mem_free(void *ptr) {
	if(ptr == NULL) return;
	lost_mem += *((unsigned short*)(ptr - sizeof(unsigned short)));
}

/*
 */
void *dict_mem_calloc(size_t nmemb,size_t size) {
	void *ret = dict_mem_malloc(nmemb * size);
	memset(ret,0,nmemb * size);
	return ret;
}

/*
 */
void *dict_mem_realloc(void *ptr,size_t size) {
	if(ptr) {
		if(*((unsigned short*)(ptr - sizeof(unsigned short))) > size) {
			return ptr;
		}
	}
	void *ret = dict_mem_malloc(sizeof(unsigned short) + size + 16);
	*((unsigned short*)ret) = size + 16;
	if(ptr) {
		memcpy(ret + sizeof(unsigned short),ptr,*((unsigned short*)(ptr - sizeof(unsigned short))));
		dict_mem_free(ptr);
	}
	return ret + sizeof(unsigned short);
}

/*
 */
char *dict_mem_strdup(const char *str) {
	char *ret = dict_mem_malloc(sizeof(char) * (strlen(str) + 1));
	strcpy(ret,str);
	return ret;
}

/*    
 */
static void dict_word_add_preorder(dict_t *d,char *word,char *trans,int len,int pos) {
	int i;
	if(len == 0) {
		/*     */
		if(d->word) {
			char *old_trans = d->trans;
			d->trans = dict_mem_malloc(strlen(old_trans) + strlen(trans) + 2);
			sprintf(d->trans,"%s\n%s",old_trans,trans);
			dict_mem_free(old_trans);
		} else {
			d->word = dict_mem_strdup(word);
			d->trans = dict_mem_strdup(trans);
		}
		return;
	}
	for(i = 0; i < d->num_childs; i++) {
		if(d->childs[i]->letter == word[pos]) {
			/*      */
			dict_word_add_preorder(d->childs[i],word,trans,len - 1,pos + 1);
			return;
		}
	}
	/*     */
	d->num_childs++;
	d->childs = dict_mem_realloc(d->childs,sizeof(dict_t*) * d->num_childs);
	d->childs[d->num_childs - 1] = dict_mem_calloc(1,sizeof(dict_t));
	d->childs[d->num_childs - 1]->letter = word[pos];
	dict_word_add_preorder(d->childs[d->num_childs - 1],word,trans,len - 1,pos + 1);
}

/*    
 */
static void dict_word_add(dict_t *root,char *word,char *trans) {
	dict_word_add_preorder(root,word,trans,strlen(word),0);
}

/*    
 */
static char *dict_word_find_preorder(dict_t *d,char *word,int len,int *index) {
	int i;
	if(index) *index = d->index;
	/*   */
	if(len == 0) return d->trans;
	for(i = 0; i < d->num_childs; i++) {
		if(d->childs[i]->letter == *word) {
			/*   */
			return dict_word_find_preorder(d->childs[i],word + 1,len - 1,index);
		}
	}
	/*   ... */
	*index = -1;
	return NULL;
}

/*    
 */
char *dict_word_find(dict_t *root,char *word,int *index) {
	if(!root || !word) {
		if(index) *index = -1;
		return NULL;
	}
	return dict_word_find_preorder(root,word,strlen(word),index);
}

/*  
 *      
 *        
 */
static int dict_compare(const void *dict0_ptr,const void *dict1_ptr) {
	const dict_t *dict0 = *(dict_t**)dict0_ptr;
	const dict_t *dict1 = *(dict_t**)dict1_ptr;
	return koi8r[(unsigned char)dict0->letter] - koi8r[(unsigned char)dict1->letter];
}

static void dict_sort(dict_t *d) {
	int i;
	qsort(d->childs,d->num_childs,sizeof(dict_t*),dict_compare);
	for(i = 0; i < d->num_childs; i++) dict_sort(d->childs[i]);
}

/*    
 */
static void dict_index_preorder(dict_t *d,int *num) {
	int i;
	if(d->word) d->index = (*num)++;
	else d->index = -1;
	for(i = 0; i < d->num_childs; i++) dict_index_preorder(d->childs[i],num);
}

/*   
 */
static void dict_index_postorder(dict_t *d,int *num) {
	int i;
	for(i = d->num_childs - 1; i >= 0; i--) dict_index_postorder(d->childs[i],num);
	if(d->word) *num = d->index;
	if(d->index == -1) d->index = *num;
}

/*   
 */
static int dict_add_text(dict_t *root,char *name) {
	FILE *file;
	char buf[2048];
	char *word,*trans,*s;
	file = fopen(name,"rb");
	if(!file) {
		fprintf(stderr,"error open \"%s\" file\n",name);
		return -1;
	}
	word = calloc(1,sizeof(char) * 1024);
	trans = s = calloc(1,sizeof(char) * 1024 * 1024);
	while(fgets(buf,sizeof(buf),file) != NULL) {
		if(*buf == '\n' && *trans != '\0') {
			s = strrchr(word,'\n');
			if(s) *s = '\0';
			s = strrchr(trans,'\n');
			if(s) *s = '\0';
			dict_word_add(root,word,trans);
			*word = '\0';
			*trans = '\0';
			s = trans;
		} else if(*buf != '\n') {
			if(!*word) strcpy(word,buf);
			else {
				strcpy(s,buf);
				s += strlen(buf);
			}
		}
	}
	free(trans);
	free(word);
	fclose(file);
	return 0;
}

/*  gz` 
 */
static int dict_add_gz(dict_t *root,char *name) {
#ifdef HAVE_LIBZ
	gzFile file;
	char buf[2048];
	char *word,*trans,*s;
	file = gzopen(name,"r");
	if(!file) {
		fprintf(stderr,"error open \"%s\" file\n",name);
		return -1;
	}
	word = calloc(1,sizeof(char) * 1024);
	trans = s = calloc(1,sizeof(char) * 1024 * 1024);
	while(gzgets(file,buf,sizeof(buf)) != NULL) {
		if(*buf == '\n' && *trans != '\0') {
			s = strrchr(word,'\n');
			if(s) *s = '\0';
			s = strrchr(trans,'\n');
			if(s) *s = '\0';
			dict_word_add(root,word,trans);
			*word = '\0';
			*trans = '\0';
			s = trans;
		} else if(*buf != '\n') {
			if(!*word) strcpy(word,buf);
			else {
				strcpy(s,buf);
				s += strlen(buf);
			}
		}
	}
	free(trans);
	free(word);
	gzclose(file);
	return 0;
#else
	return -1;
#endif
}

/*  
 */
int dict_add(dict_t *root,char *name) {
	if(!name) return -1;
	if(!strncasecmp(name + strlen(name) - 3,".gz",3)) return dict_add_gz(root,name);
	else return dict_add_text(root,name);
}

/*  
 */
dict_t *dict_load(char *name) {
	dict_t *root;
	root = dict_mem_calloc(1,sizeof(dict_t));
	if(dict_add(root,name) == -1) {
		dict_mem_free(root);
		return NULL;
	}
	return root;
}

/*  
 */
int dict_index(dict_t *root) {
	int num = 0,i = 0;
	/*  */
	dict_sort(root);
	/*  */
	dict_index_preorder(root,&num);
	dict_index_postorder(root,&i);
	return num;
}

/*  
 */
static void dict_free_postorder(dict_t *d) {
	int i;
	for(i = 0; i < d->num_childs; i++) dict_free_postorder(d->childs[i]);
	dict_mem_free(d->childs);
	if(d->word) dict_mem_free(d->word);
	if(d->trans) dict_mem_free(d->trans);
	dict_mem_free(d);
}

/*  
 */
void dict_free(dict_t *root) {
	
	dict_free_postorder(root);
	
	//  
	dict_mem_release();
}
