/*
 * (SLIK) SimpLIstic sKin functions
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif
#include "intl.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "ui_misc.h"


#define DATE_SELECION_KEY "date_selection_data"


typedef struct _DateSelection DateSelection;
struct _DateSelection
{
	GtkWidget *box;

	GtkWidget *spin_d;
	GtkWidget *spin_m;
	GtkWidget *spin_y;

	GtkWidget *button;

	GtkWidget *window;
	GtkWidget *calendar;
};


static void date_selection_popup_hide(DateSelection *ds)
{
	if (!ds->window) return;

	if (GTK_WIDGET_HAS_GRAB(ds->window))
		{
		gtk_grab_remove(ds->window);
		gdk_keyboard_ungrab(GDK_CURRENT_TIME);
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		}

	gtk_widget_hide(ds->window);

	gtk_widget_destroy(ds->window);
	ds->window = NULL;
	ds->calendar = NULL;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ds->button), FALSE);
}

static gint date_selection_popup_release_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	DateSelection *ds = data;

	date_selection_popup_hide(ds);
	return TRUE;
}

static gint date_selection_popup_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	DateSelection *ds = data;
	gint x, y;
	gint w, h;
	gint xr, yr;

	xr = (gint)event->x_root;
	yr = (gint)event->y_root;

	gdk_window_get_origin(ds->window->window, &x, &y);
	gdk_drawable_get_size(ds->window->window, &w, &h);

	if (xr < x || yr < y || xr > x + w || yr > y + h)
		{
		g_signal_connect(G_OBJECT(ds->window), "button_release_event",
				 G_CALLBACK(date_selection_popup_release_cb), ds);
		return TRUE;
		}

	return FALSE;
}

static void date_selection_popup_sync(DateSelection *ds)
{
	gint day, month, year;

	gtk_calendar_get_date(GTK_CALENDAR(ds->calendar), &year, &month, &day);
	date_selection_set(ds->box, day, month + 1, year);
}

static gint date_selection_popup_keypress_cb(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	DateSelection *ds = data;

	switch (event->keyval)
		{
		case GDK_Return:
		case GDK_KP_Enter:
		case GDK_Tab:
		case GDK_ISO_Left_Tab:
			date_selection_popup_sync(ds);
			date_selection_popup_hide(ds);
			break;
		case GDK_Escape:
			date_selection_popup_hide(ds);
			break;
		default:
			break;
		}

	return FALSE;
}

static void date_selection_day_cb(GtkWidget *widget, gpointer data)
{
	DateSelection *ds = data;

	date_selection_popup_sync(ds);
}

static void date_selection_doubleclick_cb(GtkWidget *widget, gpointer data)
{
	DateSelection *ds = data;

	date_selection_popup_hide(ds);
}

static void date_selection_popup(DateSelection *ds)
{
	gint x, y;
	gint wx, wy;
	gint day, month, year;

	if (ds->window) return;

	ds->window = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_resizable(GTK_WINDOW(ds->window), FALSE);
	g_signal_connect(G_OBJECT(ds->window), "button_press_event",
			 G_CALLBACK(date_selection_popup_press_cb), ds);
	g_signal_connect(G_OBJECT(ds->window), "key_press_event",
			 G_CALLBACK(date_selection_popup_keypress_cb), ds);

	ds->calendar = gtk_calendar_new();
	gtk_container_add(GTK_CONTAINER(ds->window), ds->calendar);
	gtk_widget_show(ds->calendar);

	date_selection_get(ds->box, &day, &month, &year);
	gtk_calendar_select_month(GTK_CALENDAR(ds->calendar), month - 1, year);
	gtk_calendar_select_day(GTK_CALENDAR(ds->calendar), day);

	g_signal_connect(G_OBJECT(ds->calendar), "day_selected",
			 G_CALLBACK(date_selection_day_cb), ds);
	g_signal_connect(G_OBJECT(ds->calendar), "day_selected_double_click",
			G_CALLBACK(date_selection_doubleclick_cb), ds);

	gtk_widget_realize(ds->window);

	gdk_window_get_origin(ds->button->window, &wx, &wy);

	x = wx + ds->button->allocation.x + ds->button->allocation.width - ds->window->allocation.width;
	y = wy + ds->button->allocation.y + ds->button->allocation.height;

	if (y + ds->window->allocation.height > gdk_screen_height())
		{
		y = wy + ds->button->allocation.y - ds->window->allocation.height;
		}
	if (x < 0) x = 0;
	if (y < 0) y = 0;

	gtk_window_move(GTK_WINDOW(ds->window), x, y);
	gtk_widget_show(ds->window);

	gtk_widget_grab_focus(ds->calendar);
	gdk_pointer_grab(ds->window->window, TRUE,
			 GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_MOTION_MASK,
			 NULL, NULL, GDK_CURRENT_TIME);
	gdk_keyboard_grab(ds->window->window, TRUE, GDK_CURRENT_TIME);
	gtk_grab_add(ds->window);

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ds->button), TRUE);
}

static void date_selection_button_cb(GtkWidget *widget, gpointer data)
{
	DateSelection *ds = data;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ds->button)) == (!ds->window))
		{
		date_selection_popup(ds);
		}
}

static void button_size_allocate_cb(GtkWidget *button, GtkAllocation *allocation, gpointer data)
{
	GtkWidget *spin = data;

	if (allocation->height > spin->requisition.height)
		{
		GtkAllocation button_allocation;

		button_allocation = button->allocation;
		button_allocation.height = spin->requisition.height;
		button_allocation.y = spin->allocation.y +
			(spin->allocation.height - spin->requisition.height) / 2;
		gtk_widget_size_allocate(button, &button_allocation);
		}
}

static GtkWidget *spin_new(GtkWidget *box, gint min, gint max, gint value,
			   GCallback func, gpointer data)
{
	GtkWidget *spin;

	spin = gtk_spin_button_new_with_range((gdouble)min, (gdouble)max, 1);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), (gdouble)value);

	if (func)
		{
		GtkAdjustment *adj;

		adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
		g_signal_connect(G_OBJECT(adj), "value_changed", func, data);
		}

	gtk_box_pack_start(GTK_BOX(box), spin, FALSE, FALSE, 0);
	gtk_widget_show(spin);

	return spin;
}

static void spin_increase(GtkWidget *spin, gint value)
{
	GtkRequisition req;

	gtk_widget_size_request(spin, &req);
	gtk_widget_set_size_request(spin, req.width + value, -1);
}

static void dest_selection_destroy_cb(GtkWidget *widget, gpointer data)
{
	DateSelection *ds = data;

	date_selection_popup_hide(ds);

	g_free(ds);
}

GtkWidget *date_selection_new(void)
{
	DateSelection *ds;
	GtkWidget *arrow;

	ds = g_new0(DateSelection, 1);

	ds->box = gtk_hbox_new(FALSE, 2);
	g_signal_connect(G_OBJECT(ds->box), "destroy",
			 G_CALLBACK(dest_selection_destroy_cb), ds);

	/* FIXME: use option menu with text format of month instead of a spin button */
	ds->spin_m = spin_new(ds->box, 1, 12, 1, NULL, NULL);
	ds->spin_d = spin_new(ds->box, 1, 31, 1, NULL, NULL);
	ds->spin_y = spin_new(ds->box, 1900, 9999, 1900, NULL, NULL);
	spin_increase(ds->spin_y, 5);

	ds->button = gtk_toggle_button_new();
	g_signal_connect(G_OBJECT(ds->button), "size_allocate",
			 G_CALLBACK(button_size_allocate_cb), ds->spin_y);

	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(ds->button), arrow);
	gtk_widget_show(arrow);

	gtk_box_pack_start(GTK_BOX(ds->box), ds->button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(ds->button), "clicked",
			 G_CALLBACK(date_selection_button_cb), ds);
	gtk_widget_show(ds->button);

	g_object_set_data(G_OBJECT(ds->box), DATE_SELECION_KEY, ds);

	return ds->box;
}
                                                                                                                        
void date_selection_set(GtkWidget *widget, gint day, gint month, gint year)
{
	DateSelection *ds;

	ds = g_object_get_data(G_OBJECT(widget), DATE_SELECION_KEY);
	if (!ds) return;

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(ds->spin_d), (gdouble)day);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(ds->spin_m), (gdouble)month);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(ds->spin_y), (gdouble)year);
}


void date_selection_get(GtkWidget *widget, gint *day, gint *month, gint *year)
{
	DateSelection *ds;

	ds = g_object_get_data(G_OBJECT(widget), DATE_SELECION_KEY);
	if (!ds) return;

	if (day) *day = gtk_spin_button_get_value(GTK_SPIN_BUTTON(ds->spin_d));
	if (month) *month = gtk_spin_button_get_value(GTK_SPIN_BUTTON(ds->spin_m));
	if (year) *year = gtk_spin_button_get_value(GTK_SPIN_BUTTON(ds->spin_y));
}
                                                                                                                        
void date_selection_time_set(GtkWidget *widget, time_t t)
{
	struct tm *lt;

	lt = localtime(&t);
	if (!lt) return;

	date_selection_set(widget, lt->tm_mday, lt->tm_mon + 1, lt->tm_year + 1900);
}

time_t date_selection_time_get(GtkWidget *widget)
{
	struct tm lt;
	gint day = 0;
	gint month = 0;
	gint year = 0;

	date_selection_get(widget, &day, &month ,&year);

	lt.tm_sec = 0;
	lt.tm_min = 0;
	lt.tm_hour = 0;
	lt.tm_mday = day;
	lt.tm_mon = month - 1;
	lt.tm_year = year - 1900;
	lt.tm_isdst = 0;

	return mktime(&lt);
}


/* --- grab sizer --- */

#define SIZER_DATA_KEY "sizer_data"

typedef struct _SizerData SizerData;
struct _SizerData
{
	GtkWidget *sizer;
	GtkWidget *parent;
	GtkWidget *bounding_widget;
	SizerPositionType position;

	gint hsize_min;
	gint hsize_max;
	gint vsize_min;
	gint vsize_max;

	gint in_drag;
	gint press_x;
	gint press_y;
	gint press_width;
	gint press_height;

	gint handle_prelit;
};


static gint sizer_default_handle_size(void)
{
	gint handle_size = 5;
	GtkWidget *paned;
	GtkStyle *style;

	paned = gtk_hpaned_new();

	style = gtk_rc_get_style(paned);
	gtk_widget_set_style(paned, style);
	gtk_widget_style_get(paned, "handle_size", &handle_size, NULL);

	gtk_widget_destroy(paned);

	return handle_size;
}

static gint sizer_motion_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	SizerData *sd = data;
	gint x, y;
	gint w, h;

	if (!sd->in_drag) return FALSE;

	x = sd->press_x - bevent->x_root;
	y = sd->press_y - bevent->y_root;

	w = sd->press_width;
	h = sd->press_height;

	if (sd->position & SIZER_POS_LEFT)
		{
		w += x;
		}
	else if (sd->position & SIZER_POS_RIGHT)
		{
		w -= x;
		}

	if (sd->position & SIZER_POS_TOP)
		{
		h += y;
		}
	else if (sd->position & SIZER_POS_BOTTOM)
		{
		h -= y;
		}

	if (sd->hsize_min >= 0) w = MAX(w, sd->hsize_min);
	if (sd->vsize_min >= 0) h = MAX(h, sd->vsize_min);

	if (sd->bounding_widget)
		{
		w = CLAMP(w, sd->sizer->allocation.width, sd->bounding_widget->allocation.width);
		h = CLAMP(h, sd->sizer->allocation.height, sd->bounding_widget->allocation.height);
		}
	else
		{
		if (w < sd->sizer->allocation.width) w = sd->sizer->allocation.width;
		if (h < sd->sizer->allocation.height) h = sd->sizer->allocation.height;
		}

	if (sd->hsize_max >= 0) w = MIN(w, sd->hsize_max);
	if (sd->vsize_max >= 0) h = MIN(h, sd->vsize_max);

	if (w == sd->parent->allocation.width) w = -1;
	if (h == sd->parent->allocation.height) h = -1;

	if (w > 0 || h > 0) gtk_widget_set_size_request(sd->parent, w, h);

	return TRUE;
}

static gint sizer_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	SizerData *sd = data;

	if (bevent->button != 1) return FALSE;

	sd->in_drag = TRUE;
	sd->press_x = bevent->x_root;
	sd->press_y = bevent->y_root;

	sd->press_width = sd->parent->allocation.width;
	sd->press_height = sd->parent->allocation.height;

	gdk_pointer_grab(sd->sizer->window, FALSE,
			 GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
			 NULL, NULL, bevent->time);
	gtk_grab_add(sd->sizer);

	return TRUE;
}

static gint sizer_release_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	SizerData *sd = data;

	if (bevent->button != 1) return FALSE;

	if (gdk_pointer_is_grabbed() && GTK_WIDGET_HAS_GRAB(sd->sizer))
		{
		gtk_grab_remove(sd->sizer);
		gdk_pointer_ungrab(bevent->time);
		}

	sd->in_drag = FALSE;

	return TRUE;
}

static void sizer_set_prelight(SizerData *sd, gint prelit)
{
	sd->handle_prelit = prelit;
	gtk_widget_queue_draw_area(sd->sizer, 0, 0,
				   sd->sizer->allocation.width, sd->sizer->allocation.height);
}

static gint sizer_enter_cb(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	SizerData *sd = data;

	sizer_set_prelight(sd, TRUE);
	return TRUE;
}

static gint sizer_leave_cb(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
	SizerData *sd = data;

	sizer_set_prelight(sd, FALSE);
	return TRUE;
}

static gint sizer_expose_cb(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
	SizerData *sd = data;
	GdkRectangle clip;
	GtkOrientation orientation;
	GtkStateType state;

	gdk_region_get_clipbox(event->region, &clip);

	if (sd->position & SIZER_POS_LEFT || sd->position & SIZER_POS_RIGHT)
		{
		orientation = GTK_ORIENTATION_VERTICAL;
		}
	else
		{
		orientation = GTK_ORIENTATION_HORIZONTAL;
		}

	if (sd->handle_prelit)
		{
		state = GTK_STATE_PRELIGHT;
		}
	else
		{
		state = widget->state;
		}

	gtk_paint_handle(widget->style, widget->window, state,
			 GTK_SHADOW_NONE, &clip, widget, "paned",
			 0, 0,
			 widget->allocation.width, widget->allocation.height,
			 orientation);

	return TRUE;
}

static void sizer_realize_cb(GtkWidget *widget, gpointer data)
{
	SizerData *sd = data;
	GdkCursorType n;

	n = 0;
	if (sd->position & SIZER_POS_TOP || sd->position & SIZER_POS_BOTTOM)
		{
		n = GDK_SB_V_DOUBLE_ARROW;
		}
	if (sd->position & SIZER_POS_LEFT || sd->position & SIZER_POS_RIGHT)
		{
		n = (n != 0) ? GDK_FLEUR : GDK_SB_H_DOUBLE_ARROW;
		}

	if (n != 0 && widget->window)
		{
		GdkCursor *cursor;
		cursor = gdk_cursor_new(n);
		gdk_window_set_cursor(widget->window, cursor);
		gdk_cursor_unref(cursor);
		}
}

static void sizer_destroy_cb(GtkWidget *widget, gpointer data)
{
	SizerData *sd = data;

	g_free(sd);
}

GtkWidget *sizer_new(GtkWidget *parent, GtkWidget *bounding_widget,
		     SizerPositionType position)
{
	SizerData *sd;
	gint handle_size;

	sd = g_new0(SizerData, 1);

	sd->sizer = gtk_event_box_new();
	sd->parent = parent;
	sd->bounding_widget = bounding_widget;
	sd->position = position;
	sd->hsize_min = -1;
	sd->hsize_max = -1;
	sd->vsize_min = -1;
	sd->vsize_max = -1;

	sd->in_drag = FALSE;
	sd->handle_prelit = FALSE;

	g_signal_connect(G_OBJECT(sd->sizer), "destroy",
			 G_CALLBACK(sizer_destroy_cb), sd);

	g_signal_connect(G_OBJECT(sd->sizer), "motion_notify_event",
			 G_CALLBACK(sizer_motion_cb), sd);
	g_signal_connect(G_OBJECT(sd->sizer), "button_press_event",
			 G_CALLBACK(sizer_press_cb), sd);
	g_signal_connect(G_OBJECT(sd->sizer), "button_release_event",
			 G_CALLBACK(sizer_release_cb), sd);

	g_signal_connect(G_OBJECT(sd->sizer), "enter_notify_event",
			 G_CALLBACK(sizer_enter_cb), sd);
	g_signal_connect(G_OBJECT(sd->sizer), "leave_notify_event",
			 G_CALLBACK(sizer_leave_cb), sd);

	gtk_widget_set_events(sd->sizer, GDK_POINTER_MOTION_MASK |
					 GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK |
					 GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);

	g_signal_connect(sd->sizer, "realize",
			 G_CALLBACK(sizer_realize_cb), sd);
	g_signal_connect(sd->sizer, "expose_event",
			 G_CALLBACK(sizer_expose_cb), sd);

	handle_size = sizer_default_handle_size();

	gtk_widget_set_size_request(sd->sizer, handle_size, handle_size);
#if 0
	/* use this if you add a shadow border to the handle */
	gtk_widget_set_size_request(sd->sizer, handle_size + sd->sizer->style->xthickness * 2,
					       handle_size + sd->sizer->style->ythickness * 2);
#endif

	g_object_set_data(G_OBJECT(sd->sizer), SIZER_DATA_KEY,sd);

	return sd->sizer;
}

void sizer_set_limits(GtkWidget *sizer,
		      gint hsize_min, gint hsize_max,
		      gint vsize_min, gint vsize_max)
{
	SizerData *sd;

	sd = g_object_get_data(G_OBJECT(sizer), SIZER_DATA_KEY);
	if (!sd) return;

	sd->hsize_min = hsize_min;
	sd->hsize_max = hsize_max;
	sd->vsize_min = vsize_min;
	sd->vsize_max = vsize_max;
}

