/*
    This file is part of libkabc.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

/*
Enhanced Version of the file for platform independent KDE tools.
Copyright (c) 2004 Ulf Schenk

$Id: resourcefile.cpp,v 1.12 2004/09/11 23:25:05 zautrix Exp $
*/

#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32_
#include <unistd.h>
#endif

#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qtimer.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
//US #include <ksavefile.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>

#include "formatfactory.h"

#include "resource.h"
#include "resourcefileconfig.h"
#include "stdaddressbook.h"

#include "resourcefile.h"
#include "syncprefwidget.h"

//#define ALLOW_LOCKING

using namespace KABC;

extern "C"
#ifdef _WIN32_
__declspec(dllexport)
#else
{
#endif

//US  void *init_kabc_file()
  void *init_microkabc_file()
  {
    return new KRES::PluginFactory<ResourceFile,ResourceFileConfig, SyncPrefWidgetContainer>();
  }
#ifndef _WIN32_
}
#endif

ResourceFile::ResourceFile( const KConfig *config, bool syncable )
  : Resource( config, syncable ) , mFormat( 0 )
{
  QString fileName, formatName, default_fileName;

  if (syncable == true)
    default_fileName = "/home/polo/kdepim/apps/kabc/localfile.vcf";
  else
    default_fileName = StdAddressBook::fileName();

  KConfig *cfg = (KConfig *)config;
  if ( cfg ) {
    fileName = cfg->readEntry( "FileName", default_fileName );
    formatName = cfg->readEntry( "FileFormat", "vcard" );
  } else {
    fileName = default_fileName;
    formatName = "vcard";
  }

  init( fileName, formatName );
}

ResourceFile::ResourceFile( const QString &fileName, bool syncable ,
                            const QString &formatName )
  : Resource( 0, syncable )
{
//  qDebug("ResourceFile::ResourceFile : 3 %s, %s", fileName.latin1(), formatName.latin1());
  init( fileName, formatName );
}

void ResourceFile::init( const QString &fileName, const QString &formatName )
{
  mFormatName = formatName;

  FormatFactory *factory = FormatFactory::self();
  mFormat = factory->format( mFormatName );

  if ( !mFormat ) {
    mFormatName = "vcard";
    mFormat = factory->format( mFormatName );
  }


  connect( &mDirWatch, SIGNAL( dirty(const QString&) ), SLOT( fileChanged() ) );
  connect( &mDirWatch, SIGNAL( created(const QString&) ), SLOT( fileChanged() ) );
  connect( &mDirWatch, SIGNAL( deleted(const QString&) ), SLOT( fileChanged() ) );

  setFileName( fileName );
}

ResourceFile::~ResourceFile()
{
  delete mFormat;
  mFormat = 0;
}

void ResourceFile::writeConfig( KConfig *config )
{

    config->setGroup( "Resource_" + identifier() );
  Resource::writeConfig( config );

  config->writeEntry( "FileName", mFileName );
  config->writeEntry( "FileFormat", mFormatName );

//    qDebug("ResourceFile::writeConfig format %s, %s", mFileName.latin1(), mFormatName.latin1());

}

Ticket *ResourceFile::requestSaveTicket()
{
  kdDebug(5700) << "ResourceFile::requestSaveTicket()" << endl;

  if ( !addressBook() ) return 0;

#ifdef ALLOW_LOCKING
  if ( !lock( mFileName ) ) {
      qDebug("unablt to lock file ");
    return 0;
  }
#endif
  return createTicket( this );
}


bool ResourceFile::doOpen()
{
  QFile file( mFileName );

  if ( !file.exists() ) {
    // try to create the file
    bool ok = file.open( IO_WriteOnly );
    if ( ok )
      file.close();

    return ok;
  } else {
    if ( !file.open( IO_ReadWrite ) )
      return false;

    if ( file.size() == 0 ) {
      file.close();
      return true;
    }

    bool ok = mFormat->checkFormat( &file );
    file.close();

    return ok;
  }
}

void ResourceFile::doClose()
{
}

bool ResourceFile::load()
{


  QFile file( mFileName );
  if ( !file.open( IO_ReadOnly ) ) {
    addressBook()->error( i18n( "Unable to open file '%1'." ).arg( mFileName ) );
    return false;
  }

//    qDebug("ResourceFile::load format %s, %s", mFileName.latin1(), mFormatName.latin1());

  return mFormat->loadAll( addressBook(), this, &file );
}

bool ResourceFile::save( Ticket *ticket )
{
//  qDebug("ResourceFile::save format %s, %s", mFileName.latin1(), mFormatName.latin1());


  // create backup file
  QString extension = "_" + QString::number( QDate::currentDate().dayOfWeek() );

/*US we use a simpler method to create a backupfile

  (void) KSaveFile::backupFile( mFileName, QString::null
  ,extension );

  KSaveFile saveFile( mFileName );
  bool ok = false;
  if ( saveFile.status() == 0 && saveFile.file() )
  {
    mFormat->saveAll( addressBook(), this, saveFile.file() );
    ok = saveFile.close();
  }
*/

//US ToDo: write backupfile
  mDirWatch.stopScan();

  QFile info;
  info.setName( mFileName );
  bool ok = info.open( IO_WriteOnly );
  if ( ok ) {
    mFormat->saveAll( addressBook(), this, &info );

    info.close();
    ok = true;
  }
  else {

  }

  if ( !ok )
    addressBook()->error( i18n( "Unable to save file '%1'." ).arg( mFileName ) );

  mDirWatch.startScan();

  delete ticket;
#ifdef ALLOW_LOCKING
  unlock( mFileName );
#endif

  return ok;
}

bool ResourceFile::lock( const QString &fileName )
{
#ifdef ALLOW_LOCKING


  QString fn = fileName;

//US change the implementation how the lockfilename is getting created
//US  fn.replace( QRegExp("/"), "_" );
//US  QString lockName = locateLocal( "data", "kabc/lock/" + fn + ".lock" );

  KURL url(fn);
  QString lockName = locateLocal( "data", "kabc/lock/" + url.fileName() + ".lock" );



  if (QFile::exists( lockName )) return false;

  QString lockUniqueName;
  lockUniqueName = fn + KApplication::randomString( 8 );

  url = lockUniqueName;
//US  mLockUniqueName = locateLocal( "data", "kabc/lock/" + lockUniqueName );
  mLockUniqueName = locateLocal( "data", "kabc/lock/" + url.fileName() );
  kdDebug(5700) << "-- lock unique name: " << mLockUniqueName << endl;

  // Create unique file
  QFile file( mLockUniqueName );
  file.open( IO_WriteOnly );
  file.close();

  // Create lock file
  int result = 0;
#ifndef _WIN32_
   result = ::link( QFile::encodeName( mLockUniqueName ),
                       QFile::encodeName( lockName ) );
#endif
  if ( result == 0 ) {
    addressBook()->emitAddressBookLocked();
    return true;
  }

  // TODO: check stat

  return false;
#else
  return true;
#endif
}

void ResourceFile::unlock( const QString &fileName )
{
#ifdef ALLOW_LOCKING
  QString fn = fileName;
//US change the implementation how the lockfilename is getting created
//US  fn.replace( QRegExp( "/" ), "_" );
//US  QString lockName = locateLocal( "data", "kabc/lock/" + fn + ".lock" );
//US  QString lockName = fn + ".lock";
  KURL url(fn);
  QString lockName = locateLocal( "data", "kabc/lock/" + url.fileName() + ".lock" );

  QFile::remove( lockName );
  QFile::remove( mLockUniqueName );
  addressBook()->emitAddressBookUnlocked();
#else
  return;
#endif
}

void ResourceFile::setFileName( const QString &fileName )
{
  mDirWatch.stopScan();
  mDirWatch.removeFile( mFileName );

  mFileName = fileName;


  mDirWatch.addFile( mFileName );
  mDirWatch.startScan();

//US simulate KDirWatch event
//US  fileChanged();
}

QString ResourceFile::fileName() const
{
  return mFileName;
}

void ResourceFile::setFormat( const QString &format )
{
  mFormatName = format;
  delete mFormat;

  FormatFactory *factory = FormatFactory::self();
  mFormat = factory->format( mFormatName );
/*US
//qDebug("ResourceFile::setFormat initialized with format %s ", format.latin1());
  if (mFormatName == "vcard") {
    mFormat = new VCardFormatPlugin2();
//    qDebug("ResourceFile::setFormat format %s", mFormatName.latin1());
  }
  else if (mFormatName == "binary") {
    mFormat = new BinaryFormat();
//    qDebug("ResourceFile::setFormat format %s", mFormatName.latin1());
  }
  else
    qDebug("ResourceFile::setFormat format unknown !!! %s ", format.latin1());
*/

}

QString ResourceFile::format() const
{
  return mFormatName;
}

void ResourceFile::fileChanged()
{
  // There is a small theoretical chance that KDirWatch calls us before
  // we are fully constructed
  if (!addressBook())
    return;


  QString text( i18n( "File resource '%1'<br> has been changed by third party.<br>Do you want to reload?").arg( mFileName ) );
  if ( readOnly() || KMessageBox::questionYesNo( 0, text ) == KMessageBox::Yes ) {
    load();
    addressBook()->emitAddressBookChanged();
  }
}

void ResourceFile::removeAddressee( const Addressee &addr )
{
  QFile::remove( QFile::encodeName( locateLocal( "data", "kabc/photos/" ) + addr.uid() ) );
  QFile::remove( QFile::encodeName( locateLocal( "data", "kabc/logos/" ) + addr.uid() ) );
  QFile::remove( QFile::encodeName( locateLocal( "data", "kabc/sounds/" ) + addr.uid() ) );
}

void ResourceFile::cleanUp()
{
  unlock( mFileName );
}

//US #include "resourcefile.moc"
