/*
  This file is part of KOrganizer.

  Requires the Qt and KDE widget libraries, available at no cost at
  http://www.troll.no and http://www.kde.org respectively

  Copyright (c) 1997, 1998, 1999
  Preston Brown (preston.brown@yale.edu)
  Fester Zigterman (F.J.F.ZigtermanRustenburg@student.utwente.nl)
  Ian Dawes (iadawes@globalserve.net)
  Laszlo Boloni (boloni@cs.purdue.edu)

  Copyright (c) 2000, 2001, 2002
  Cornelius Schumacher <schumacher@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <stdlib.h>

#include <qapplication.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qlayout.h>
#include <qclipboard.h>
#include <qcursor.h>
#include <qmessagebox.h>
#include <qprogressbar.h>
#include <qmultilineedit.h>
#include <qtimer.h>
#include <qwidgetstack.h>
#include <qptrlist.h>
#include <qregexp.h>
#include <qgroupbox.h>
#include <qfile.h>
#include <qdir.h>
#ifndef KORG_NOSPLITTER
#include <qsplitter.h>
#endif

#include <kglobal.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <kconfig.h>

#include "kosyncprefsdialog.h"
#include <krun.h>
#include <kdirwatch.h>
#include <libkdepim/kdatepicker.h>
#include <libkdepim/ksyncprofile.h>

#include <libkcal/vcaldrag.h>
#include <libkcal/icaldrag.h>
#include <libkcal/icalformat.h>
#include <libkcal/vcalformat.h>
#include <libkcal/scheduler.h>
#include <libkcal/calendarlocal.h>
#include <libkcal/journal.h>
#include <libkcal/calfilter.h>
#include <libkcal/attendee.h>
#include <libkcal/dndfactory.h>
#include <libkcal/freebusy.h>
#include <libkcal/filestorage.h>
#include <libkcal/calendarresources.h>
#include <libkcal/qtopiaformat.h>
#include "../kalarmd/alarmdialog.h"

#ifndef DESKTOP_VERSION
#include <libkcal/sharpformat.h>
#endif
#ifndef KORG_NOMAIL
#include "komailclient.h"
#endif
#ifndef KORG_NOPRINTER
#include "calprinter.h"
#endif
#ifndef KORG_NOPLUGINS
#include "kocore.h"
#endif
#include "koeventeditor.h"
#include "kotodoeditor.h"
#include "koprefs.h"
#include "koeventviewerdialog.h"
#include "publishdialog.h"
#include "kofilterview.h"
#include "koglobals.h"
#include "koviewmanager.h"
#include "koagendaview.h"
#include "kodialogmanager.h"
#include "outgoingdialog.h"
#include "incomingdialog.h"
#include "statusdialog.h"
#include "kdatenavigator.h"
#include "kotodoview.h"
#include "datenavigator.h"
#include "resourceview.h"
#include "navigatorbar.h"
#include "searchdialog.h"
#include "mainwindow.h"

#include "calendarview.h"
#ifndef DESKTOP_VERSION
#include <qtopia/alarmserver.h> 
#endif
#ifndef _WIN32_
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#else
#include <qprocess.h>
#endif
using namespace KOrg;
extern int globalFlagBlockAgenda;
extern int globalFlagBlockStartup;

#define SYNC_MODE_NORMAL 0
#define SYNC_MODE_SHARP 1
#define SYNC_MODE_QTOPIA 2

class KOBeamPrefs : public QDialog
{
  public:
    KOBeamPrefs( QWidget *parent=0, const char *name=0 ) :
      QDialog( parent, name, true ) 
    {
        setCaption( i18n("Beam Options") );
        QVBoxLayout* lay = new QVBoxLayout( this );
        lay->setSpacing( 3 );
        lay->setMargin( 3 );
        QButtonGroup* format = new  QButtonGroup( 1, Horizontal, i18n("File format"), this );
        lay->addWidget( format ); 
        format->setExclusive ( true ) ;
        QButtonGroup* time = new  QButtonGroup(1, Horizontal, i18n("Time format"), this );
        lay->addWidget( time ); time->setExclusive ( true ) ;
        vcal = new QRadioButton(" vCalendar ", format );
        ical = new QRadioButton(" iCalendar ", format );
        vcal->setChecked( true );
        tz = new QRadioButton(i18n(" With timezone "), time );
        local = new QRadioButton(i18n(" Local time "), time );
        tz->setChecked( true );
        QPushButton * ok = new QPushButton( i18n("Beam via IR!"), this );
        lay->addWidget( ok );
        QPushButton * cancel = new QPushButton( i18n("Cancel"), this );
        lay->addWidget( cancel );
        connect ( ok,SIGNAL(clicked() ),this , SLOT ( accept() )  );
        connect (cancel, SIGNAL(clicked() ), this, SLOT ( reject())  );
        resize( 200, 200 );
    }
      
    bool beamVcal() { return vcal->isChecked(); }
    bool beamLocal() { return local->isChecked(); }
private:
    QRadioButton* vcal, *ical, *local, *tz;
};
class KOCatPrefs : public QDialog
{
  public:
    KOCatPrefs( QWidget *parent=0, const char *name=0 ) :
      QDialog( parent, name, true ) 
    {
        setCaption( i18n("Manage new Categories") );
        QVBoxLayout* lay = new QVBoxLayout( this );
        lay->setSpacing( 3 );
        lay->setMargin( 3 );
        QLabel * lab = new QLabel( i18n("After importing/loading/syncing\nthere may be new categories in\nevents or todos\nwhich are not in the category list.\nPlease choose what to do:\n "), this );
        lay->addWidget( lab ); 
        QButtonGroup* format = new  QButtonGroup( 1, Horizontal, i18n("New categories not in list:"), this );
        lay->addWidget( format ); 
        format->setExclusive ( true ) ;
        addCatBut = new QRadioButton(i18n("Add to category list"), format );
        new QRadioButton(i18n("Remove from Events/Todos"), format );
        addCatBut->setChecked( true );
        QPushButton * ok = new QPushButton( i18n("OK"), this );
        lay->addWidget( ok );
        QPushButton * cancel = new QPushButton( i18n("Cancel"), this );
        lay->addWidget( cancel );
        connect ( ok,SIGNAL(clicked() ),this , SLOT ( accept() )  );
        connect (cancel, SIGNAL(clicked() ), this, SLOT ( reject())  );
        resize( 200, 200 );
    }
      
    bool addCat() { return addCatBut->isChecked(); }
private:
    QRadioButton* addCatBut;
};



CalendarView::CalendarView( CalendarResources *calendar,
                            QWidget *parent, const char *name )
    : CalendarViewBase( parent, name ),
      mCalendar( calendar ),
      mResourceManager( calendar->resourceManager() )
{

    mEventEditor = 0;
    mTodoEditor = 0;
 
    init();
}

CalendarView::CalendarView( Calendar *calendar,
                            QWidget *parent, const char *name )
    : CalendarViewBase( parent, name ),
      mCalendar( calendar ),
      mResourceManager( 0 )
{
 
    mEventEditor = 0;
    mTodoEditor = 0;
    init();}

void CalendarView::init()
{ 
    beamDialog = new KOBeamPrefs();
    mDatePickerMode = 0;
    mCurrentSyncDevice = "";
    writeLocale();
    mViewManager = new KOViewManager( this );
    mDialogManager = new KODialogManager( this );
    mEventViewerDialog = 0;
    mModified = false;
    mReadOnly = false;
    mSelectedIncidence = 0;
    mSyncProfiles.setAutoDelete(true);
    mCalPrinter = 0;
    mFilters.setAutoDelete(true);

    mCalendar->registerObserver( this );
    // TODO: Make sure that view is updated, when calendar is changed.

    mStorage = new FileStorage( mCalendar );
    mNavigator = new DateNavigator( this, "datevav", mViewManager );

    QBoxLayout *topLayout = (QBoxLayout*)layout();
#ifndef KORG_NOSPLITTER
    // create the main layout frames.
    mPanner = new QSplitter(QSplitter::Horizontal,this,"CalendarView::Panner");
    topLayout->addWidget(mPanner);

    mLeftSplitter = new QSplitter(QSplitter::Vertical,mPanner,
                                  "CalendarView::LeftFrame");
    mPanner->setResizeMode(mLeftSplitter,QSplitter::KeepSize);

    mDateNavigator = new KDateNavigator(mLeftSplitter, mCalendar, TRUE,
                                        "CalendarView::DateNavigator", QDate::currentDate() );
    mLeftSplitter->setResizeMode(mDateNavigator,QSplitter::KeepSize);
    mTodoList = new KOTodoView(mCalendar, mLeftSplitter, "todolist");
    mFilterView = new KOFilterView(&mFilters,mLeftSplitter,"CalendarView::FilterView");

#ifdef KORG_NORESOURCEVIEW 
    mResourceView = 0;
#else
    if ( mResourceManager ) {
        mResourceView = new ResourceView( mResourceManager, mLeftSplitter );
        mResourceView->updateView();
        connect( mResourceView, SIGNAL( resourcesChanged() ),
                 SLOT( updateView() ) );
    } else {
        mResourceView = 0;
    }
#endif
    QWidget *rightBox = new QWidget( mPanner );
    QBoxLayout *rightLayout = new QVBoxLayout( rightBox );

    mNavigatorBar = new NavigatorBar( QDate::currentDate(), rightBox, "useBigPixmaps" );
    rightLayout->addWidget( mNavigatorBar );

    mRightFrame = new QWidgetStack( rightBox );
    rightLayout->addWidget( mRightFrame, 1 );

    mLeftFrame = mLeftSplitter;
#else
    QWidget *mainBox = new QWidget( this );
    QWidget *leftFrame = new QWidget( mainBox );

    QBoxLayout * mainBoxLayout;
    QBoxLayout * leftFrameLayout;
    if ( KOPrefs::instance()->mVerticalScreen ) {
        mainBoxLayout  = new QVBoxLayout(mainBox); 
        leftFrameLayout = new QHBoxLayout(leftFrame ); 
    } else {
        mainBoxLayout  = new QHBoxLayout(mainBox);
        leftFrameLayout = new  QVBoxLayout(leftFrame ); 
    }
    topLayout->addWidget( mainBox );
    mainBoxLayout->addWidget (leftFrame);
    mDateNavigator = new KDateNavigator(leftFrame, mCalendar, TRUE,
                                        "CalendarView::DateNavigator", QDate::currentDate());
    // mDateNavigator->blockSignals( true );
    leftFrameLayout->addWidget( mDateNavigator  );
    mFilterView = new KOFilterView(&mFilters,leftFrame,"CalendarView::FilterView");
    mTodoList = new KOTodoView(mCalendar, leftFrame, "todolistsmall");
    
    if (  QApplication::desktop()->width() < 480 ) {
        leftFrameLayout->addWidget(mFilterView);
        leftFrameLayout->addWidget(mTodoList, 2 );
      
    } else {
        leftFrameLayout->addWidget(mTodoList,2 );
        leftFrameLayout->addWidget(mFilterView );
    } 
    mFilterView->hide();
    QWidget *rightBox = new QWidget( mainBox );
    mainBoxLayout->addWidget ( rightBox, 10 );
    QBoxLayout *rightLayout = new QVBoxLayout( rightBox );
    mNavigatorBar = new NavigatorBar( QDate::currentDate(), rightBox, "useBigPixmaps" );
    mRightFrame = new QWidgetStack( rightBox );
    rightLayout->addWidget( mNavigatorBar );
    rightLayout->addWidget( mRightFrame, 10 );

    mLeftFrame = leftFrame;
    if ( KOPrefs::instance()->mVerticalScreen ) {
        mTodoList->setFixedHeight( mDateNavigator->sizeHint().height() );
        leftFrame->setFixedHeight( mDateNavigator->sizeHint().height() );
    } else {
        mTodoList->setFixedWidth( mDateNavigator->sizeHint().width() );
        leftFrame->setFixedWidth( mDateNavigator->sizeHint().width() );
    }
 
    //qDebug("Calendarview Size %d %d ", width(), height());
#endif
 
    connect( mNavigator, SIGNAL( datesSelected( const KCal::DateList & ) ),
             SLOT( showDates( const KCal::DateList & ) ) );
    connect( mNavigator, SIGNAL( datesSelected( const KCal::DateList & ) ),
             mDateNavigator, SLOT( selectDates( const KCal::DateList & ) ) );

    connect( mNavigatorBar, SIGNAL( goPrevYear() ),
             mNavigator, SLOT( selectPreviousYear() ) );
    connect( mNavigatorBar, SIGNAL( goNextYear() ),
             mNavigator, SLOT( selectNextYear() ) );
    connect( mNavigatorBar, SIGNAL( goPrevMonth() ),
             mNavigator, SLOT( selectPreviousMonth() ) );
    connect( mNavigatorBar, SIGNAL( goNextMonth() ),
             mNavigator, SLOT( selectNextMonth() ) );

    connect( mNavigator, SIGNAL( datesSelected( const KCal::DateList & ) ),
             mNavigatorBar, SLOT( selectDates( const KCal::DateList & ) ) );

    connect( mDateNavigator, SIGNAL( weekClicked( const QDate & ) ),
             mNavigator, SLOT( selectWeek( const QDate & ) ) );

    connect( mDateNavigator, SIGNAL( goPrevYear() ),
             mNavigator, SLOT( selectPreviousYear() ) );
    connect( mDateNavigator, SIGNAL( goNextYear() ),
             mNavigator, SLOT( selectNextYear() ) );
    connect( mDateNavigator, SIGNAL( goPrevMonth() ),
             mNavigator, SLOT( selectPreviousMonth() ) );
    connect( mDateNavigator, SIGNAL( goNextMonth() ),
             mNavigator, SLOT( selectNextMonth() ) );

    connect( mDateNavigator, SIGNAL( goPrevious() ),
             mNavigator, SLOT( selectPrevious() ) );
    connect( mDateNavigator, SIGNAL( goNext() ),
             mNavigator, SLOT( selectNext() ) );
    connect( mDateNavigator, SIGNAL( monthSelected ( int ) ),
             mNavigator, SLOT( slotMonthSelect( int ) ) );
    connect( mNavigatorBar, SIGNAL( monthSelected ( int ) ),
             mNavigator, SLOT( slotMonthSelect( int ) ) );

    connect( mDateNavigator, SIGNAL( datesSelected( const KCal::DateList & ) ),
             mNavigator, SLOT( selectDates( const KCal::DateList & ) ) );

    connect( mDateNavigator, SIGNAL( eventDropped( Event * ) ),
             SLOT( eventAdded( Event *) ) );

    connect(mDateNavigator,SIGNAL(dayPassed(QDate)),SLOT(updateView()));

    connect( this, SIGNAL( configChanged() ),
             mDateNavigator, SLOT( updateConfig() ) );

    connect( mTodoList, SIGNAL( newTodoSignal() ),
             SLOT( newTodo() ) );
    connect( mTodoList, SIGNAL( newSubTodoSignal( Todo *) ),
             SLOT( newSubTodo( Todo * ) ) );
    connect( mTodoList, SIGNAL( editTodoSignal( Todo * ) ),
             SLOT( editTodo( Todo * ) ) );
    connect( mTodoList, SIGNAL( showTodoSignal( Todo * ) ),
             SLOT( showTodo( Todo *) ) );
    connect( mTodoList, SIGNAL( deleteTodoSignal( Todo *) ),
             SLOT( deleteTodo( Todo *) ) );
    connect( this, SIGNAL( configChanged()), mTodoList, SLOT( updateConfig() ) );
    connect( mTodoList, SIGNAL( purgeCompletedSignal() ),
             SLOT( purgeCompleted() ) );
    connect( mTodoList, SIGNAL( todoModifiedSignal( Todo *, int ) ),
             SIGNAL( todoModified( Todo *, int ) ) );

    connect( mTodoList, SIGNAL( cloneTodoSignal( Incidence *  ) ),
             this, SLOT ( cloneIncidence( Incidence * ) ) );
    connect( mTodoList, SIGNAL( cancelTodoSignal( Incidence *  ) ),
             this, SLOT (cancelIncidence( Incidence * ) ) );

    connect( mTodoList, SIGNAL( moveTodoSignal( Incidence *  ) ),
             this, SLOT ( moveIncidence( Incidence * ) ) );
    connect( mTodoList, SIGNAL( beamTodoSignal( Incidence *  ) ),
            this, SLOT ( beamIncidence( Incidence * ) ) );
  
    connect( mTodoList, SIGNAL( unparentTodoSignal( Todo * ) ),
             this, SLOT ( todo_unsub( Todo * ) ) );
  
    connect( this, SIGNAL( todoModified( Todo *, int )), mTodoList,
             SLOT( updateTodo( Todo *, int ) ) );
    connect( this, SIGNAL( todoModified( Todo *, int )), this,
             SLOT( changeTodoDisplay( Todo *, int ) ) );


    connect( mFilterView, SIGNAL( filterChanged() ), SLOT( updateFilter() ) );
    connect( mFilterView, SIGNAL( editFilters() ), SLOT( editFilters() ) );
    connect( mCalendar, SIGNAL( addAlarm(const QDateTime &, const QString &  ) ), SLOT( addAlarm(const QDateTime &, const QString &  ) ) );
    connect( mCalendar, SIGNAL( removeAlarm(const QDateTime &, const QString &  ) ), SLOT( removeAlarm(const QDateTime &, const QString & ) ) );
  




    connect(QApplication::clipboard(),SIGNAL(dataChanged()),
            SLOT(checkClipboard()));
    connect( mTodoList,SIGNAL( incidenceSelected( Incidence * ) ),
             SLOT( processTodoListSelection( Incidence * ) ) );
    connect(mTodoList,SIGNAL(isModified(bool)),SLOT(setModified(bool)));

    // kdDebug() << "CalendarView::CalendarView() done" << endl; 

    mDateFrame = new QVBox(0,0,WType_Popup);
    //mDateFrame->setFrameStyle(QFrame::PopupPanel | QFrame::Raised);
    mDateFrame->setFrameStyle( QFrame::WinPanel |QFrame::Raised );
    mDateFrame->setLineWidth(3);
    mDateFrame->hide(); 
    mDateFrame->setCaption( i18n( "Pick a date to display"));
    mDatePicker = new KDatePicker ( mDateFrame , QDate::currentDate() );
 
    connect(mDatePicker,SIGNAL(dateSelected(QDate)),SLOT(slotSelectPickerDate(QDate)));
 
    mEventEditor = mDialogManager->getEventEditor();  
    mTodoEditor = mDialogManager->getTodoEditor();  
   
    mFlagEditDescription = false; 

    mSuspendTimer = new QTimer( this );
    mAlarmTimer = new QTimer( this );
    connect( mSuspendTimer, SIGNAL(  timeout ()  ), SLOT( suspendAlarm() ) );
    connect( mAlarmTimer, SIGNAL(  timeout ()  ), SLOT( timerAlarm() ) );
    mAlarmDialog = new AlarmDialog( this );
    connect( mAlarmDialog, SIGNAL( addAlarm(const QDateTime &, const QString &  ) ), SLOT( addSuspendAlarm(const QDateTime &, const QString &  ) ) );
    mAlarmDialog->setServerNotification( false );
    mAlarmDialog->setSuspendTime( KOPrefs::instance()->mAlarmSuspendTime );
}


CalendarView::~CalendarView()
{
    // kdDebug() << "~CalendarView()" << endl;
    //qDebug("CalendarView::~CalendarView() ");
    delete mDialogManager;
    delete mViewManager;
    delete mStorage;
    delete mDateFrame ;
    delete beamDialog;
    //kdDebug() << "~CalendarView() done" << endl;
}
void CalendarView::timerAlarm()
{
    //qDebug("CalendarView::timerAlarm() ");
    computeAlarm(mAlarmNotification );
}

void CalendarView::suspendAlarm()
{
    //qDebug(" CalendarView::suspendAlarm() ");
    computeAlarm(mSuspendAlarmNotification );

}

void  CalendarView::startAlarm( QString mess , QString filename)
{
    mAlarmDialog->eventNotification( mess, KOPrefs::instance()->mAlarmPlayBeeps, filename, true,KOPrefs::instance()->mAlarmBeepInterval ,KOPrefs::instance()->mAlarmSuspendCount );

}
    

void  CalendarView::computeAlarm( QString msg )
{

 QString mess = msg; 
    QString  mAlarmMessage = mess.mid( 9 ); 
    QString filename =  MainWindow::resourcePath();
    filename += "koalarm.wav";
    QString tempfilename;
    if ( mess.left( 13 )  == "suspend_alarm") {  
        bool error = false;
        int len = mess.mid( 13 ).find("+++");
        if ( len < 2 )
            error = true;
        else {
            tempfilename = mess.mid( 13, len );
            if ( !QFile::exists( tempfilename  ) )
                error = true;
        }
        if ( ! error ) {
            filename = tempfilename;
        }
        mAlarmMessage = mess.mid( 13+len+3 );
        //qDebug("suspend file %s ",tempfilename.latin1() );
        startAlarm( mAlarmMessage, filename);
        return;
    }
    if ( mess.left( 11 )  == "timer_alarm") {
        //mTimerTime = 0;
        startAlarm( mess.mid( 11 ), filename );
        return;
    } 
    if ( mess.left( 10 )  == "proc_alarm") {
        bool error = false;
        int len = mess.mid( 10 ).find("+++");
        if ( len < 2 )
            error = true;
        else {
            tempfilename = mess.mid( 10, len );
            if ( !QFile::exists( tempfilename  ) )
                error = true;
        }
        if ( error ) {
            mAlarmMessage = "Procedure Alarm\nError - File not found\n";
            mAlarmMessage += mess.mid( 10+len+3+9 );
        } else {
            //QCopEnvelope e("QPE/Application/kopi", "-writeFileSilent");
            //qDebug("-----system command %s ",tempfilename.latin1() );
#ifndef _WIN32_
            if ( vfork () == 0 ) {
                execl ( tempfilename.latin1(), 0 );
                return;
	    }
#else
		QProcess* p = new QProcess();
		p->addArgument(  tempfilename.latin1()  );
		p->start();
		return;
#endif
            
            return;
        }
       
        //qDebug("+++++++system command %s ",tempfilename.latin1() );
    }  
    if ( mess.left( 11 )  == "audio_alarm") {
        bool error = false;
        int len = mess.mid( 11 ).find("+++");
        if ( len < 2 )
            error = true;
        else {
            tempfilename = mess.mid( 11, len );
            if ( !QFile::exists( tempfilename  ) )
                error = true;
        }
        if ( ! error ) {
            filename = tempfilename;
        }
        mAlarmMessage = mess.mid( 11+len+3+9 );
        //qDebug("audio file command %s ",tempfilename.latin1() );
    } 
    if ( mess.left( 9 )  == "cal_alarm") {
        mAlarmMessage = mess.mid( 9 ) ;
    }
    
    startAlarm( mAlarmMessage, filename );


}

void CalendarView::addSuspendAlarm(const QDateTime &qdt, const QString &noti  )
{
    //qDebug("+++++addSUSPENDAlarm %s %s ", qdt.toString().latin1() , noti.latin1() );
   
    mSuspendAlarmNotification = noti;
    int ms = QDateTime::currentDateTime().secsTo( qdt )*1000;
    //qDebug("Suspend Alarm timer started with secs: %d ", ms/1000);
    mSuspendTimer->start( ms , true );

}

void CalendarView::addAlarm(const QDateTime &qdt, const QString &noti  )
{
    //qDebug("+++++addAlarm %s %s ", qdt.toString().latin1() , noti.latin1() );
    if ( ! KOPrefs::instance()->mUseInternalAlarmNotification ) {
#ifndef DESKTOP_VERSION
        AlarmServer::addAlarm ( qdt,"koalarm", noti.latin1() );
#endif     
        return;
    }
    mAlarmNotification = noti;
    int ms = QDateTime::currentDateTime().secsTo( qdt )*1000;
    //qDebug("Alarm timer started with secs: %d ", ms/1000);
    mAlarmTimer->start( ms , true );

}
void CalendarView::removeAlarm(const QDateTime &qdt, const QString &noti  )
{
    //qDebug("-----removeAlarm %s %s ", qdt.toString().latin1() , noti.latin1() );
    if ( ! KOPrefs::instance()->mUseInternalAlarmNotification ) {
#ifndef DESKTOP_VERSION
        AlarmServer::deleteAlarm (qdt ,"koalarm" ,noti.latin1() );
#endif   
        return;
    }
    mAlarmTimer->stop();
}
void CalendarView::selectWeekNum ( int num  )
{
    dateNavigator()->selectWeek( num );
    mViewManager->showWeekView();
}
KOViewManager *CalendarView::viewManager()
{
    return mViewManager;
}

KODialogManager *CalendarView::dialogManager()
{
    return mDialogManager;
}

QDate CalendarView::startDate()
{
    DateList dates = mNavigator->selectedDates();

    return dates.first();
}

QDate CalendarView::endDate()
{
    DateList dates = mNavigator->selectedDates();

    return dates.last();
}


void CalendarView::createPrinter()
{
#ifndef KORG_NOPRINTER
    if (!mCalPrinter) {
        mCalPrinter = new CalPrinter(this, mCalendar);
        connect(this, SIGNAL(configChanged()), mCalPrinter, SLOT(updateConfig()));
    }
#endif
}

void CalendarView::confSync()
{
   static KOSyncPrefsDialog* sp = 0;
    if ( ! sp ) {
        sp = new KOSyncPrefsDialog( this, "syncprefs", true );
    }
    sp->usrReadConfig();
#ifndef DESKTOP_VERSION
    sp->showMaximized();
#else
    sp->show();
#endif
    sp->exec();

}


//KOPrefs::instance()->mWriteBackFile
//KOPrefs::instance()->mWriteBackExistingOnly

//  0  syncPrefsGroup->addRadio(i18n("Take local entry on conflict"));
//  1  syncPrefsGroup->addRadio(i18n("Take remote entry on conflict"));
//  2  syncPrefsGroup->addRadio(i18n("Take newest entry on conflict"));
//  3  syncPrefsGroup->addRadio(i18n("Ask for every entry on conflict"));
//  4  syncPrefsGroup->addRadio(i18n("Force take local entry always"));
//  5  syncPrefsGroup->addRadio(i18n("Force take remote entry always"));

int  CalendarView::takeEvent( Incidence* local,  Incidence* remote, int mode , bool full )
{
 
    //void setZaurusId(int id);
    // int zaurusId() const;
    //  void setZaurusUid(int id);
   // int zaurusUid() const;
   // void setZaurusStat(int id);
   // int zaurusStat() const;
    // 0 equal
    // 1 take local
    // 2 take remote
    // 3 cancel 
    QDateTime lastSync = mLastCalendarSync;
    if ( mGlobalSyncMode == SYNC_MODE_SHARP ) {
        bool remCh, locCh;
        remCh = ( remote->zaurusUid() != local->zaurusUid() );
        locCh = ( local->zaurusStat() != local->revision() );
        //qDebug("locCh %d remCh %d locuid %d remuid %d", locCh, remCh,local->zaurusUid(), remote->zaurusUid() );
        if ( !remCh && ! locCh ) {
            //qDebug("both not changed ");
            lastSync = local->lastModified().addDays(1);
        } else {
            if ( locCh ) {
                //qDebug("loc changed %d %d", local->zaurusStat(), local->revision() );
                lastSync = local->lastModified().addDays( -1 );
                if ( !remCh )
                    remote->setLastModified( lastSync.addDays( -1 ) );
            } else {
                //qDebug(" not loc changed ");
                lastSync = local->lastModified().addDays( 1 );
                if ( remCh )
                    remote->setLastModified( lastSync.addDays( 1 ) );
                
            }
        }
        full = true;  
        if ( mode < 3 )
            mode = 3;
    } else {
          if ( local->lastModified() == remote->lastModified() )
            if ( local->revision() == remote->revision() )
                return 0;
        
    }
    // qDebug(" %d %d conflict on %s %s ", mode, full, local->summary().latin1(), remote->summary().latin1() );

    //qDebug("%s %d  %s %d", local->lastModified().toString().latin1() , local->revision(), remote->lastModified().toString().latin1(), remote->revision());  
    //qDebug("%d %d %d %d ", local->lastModified().time().second(), local->lastModified().time().msec(),  remote->lastModified().time().second(), remote->lastModified().time().msec() );
    //full = true; //debug only
    if ( full ) {
        bool equ = false;
        if ( local->type() == "Event" ) {
            equ =  (*((Event*) local) == *((Event*) remote));
        }
        else if ( local->type() =="Todo" )
            equ =  (*((Todo*) local) == (*(Todo*) remote));
        else if ( local->type() =="Journal" )
            equ =  (*((Journal*) local) == *((Journal*) remote));
        if ( equ ) {
            //qDebug("equal ");
            if ( mGlobalSyncMode == SYNC_MODE_SHARP ) {
                local->setZaurusUid( remote->zaurusUid() );
            }
            if ( mode < 4 )
                return 0;
              
        }//else //debug only
        //qDebug("not equal %s %s ", local->summary().latin1(), remote->summary().latin1());
    }
    int result;
    bool localIsNew;
    if ( full && mode < 2 )
        mode = 3;

    switch( mode ) {
    case 0:
        if ( lastSync > remote->lastModified() )
            return 1;
        if ( lastSync > local->lastModified() )
            return 2;
        return 1;
        break;
    case 1:
        if ( lastSync > remote->lastModified() )
            return 1;
        if ( lastSync > local->lastModified() )
            return 2;
        return 2;
        break;
    case 2:
        if ( local->lastModified() > remote->lastModified() )
            return 1;
        else
            return 2;
        break;
    case 3: 
        //qDebug("lsy %s  ---  lo %s  --- re %s ", lastSync.toString().latin1(), local->lastModified().toString().latin1(), remote->lastModified().toString().latin1()  );
        if ( lastSync > remote->lastModified() )
            return 1;
        if ( lastSync > local->lastModified() )
            return 2;
        //qDebug("lsy %s  ---  lo %s  --- re %s ", lastSync.toString().latin1(), local->lastModified().toString().latin1(), remote->lastModified().toString().latin1()  );
        localIsNew = local->lastModified() > remote->lastModified();
        if ( localIsNew )
            getEventViewerDialog()->setColorMode( 1 );
         else 
            getEventViewerDialog()->setColorMode( 2 );
        getEventViewerDialog()->setIncidence(local);
        if ( localIsNew )
            getEventViewerDialog()->setColorMode( 2 );
         else 
            getEventViewerDialog()->setColorMode( 1 );
        getEventViewerDialog()->addIncidence(remote);
        getEventViewerDialog()->setColorMode( 0 );
        //qDebug("local %d remote %d ",local->relatedTo(),remote->relatedTo()  );
        getEventViewerDialog()->setCaption( mCurrentSyncDevice +i18n(" : Conflict! Please choose entry!"));
        getEventViewerDialog()->showMe();
        result = getEventViewerDialog()->executeS( localIsNew );
        return result;
          
        break;
    case 4:
        return 1;
        break;
    case 5:
        return 2;
        break;
       
    default:
        break;
    }
    return 0;
}
Event* CalendarView::getLastSyncEvent()
{
    Event* lse; 
    //qDebug("CurrentSyncDevice %s ",mCurrentSyncDevice .latin1() );
    lse =   mCalendar->event( "last-syncEvent-device-"+mCurrentSyncDevice );
    if (!lse) {
        lse = new Event();
        lse->setUid( "last-syncEvent-device-"+mCurrentSyncDevice ); 
        lse->setSummary(mCurrentSyncDevice + i18n(" - sync event"));
        lse->setDtStart( mLastCalendarSync );
        lse->setDtEnd( mLastCalendarSync.addSecs( 7200 ) );
        lse->setCategories( i18n("SyncEvent") );
        lse->setReadOnly( true );
        mCalendar->addEvent( lse );
    }

    return lse;

}
void CalendarView::checkSharpEvent( Event* lastSync, Incidence* toDelete )
{
    if ( ! lastSync )
        return;
    if ( toDelete->zaurusId() < 0 )
        return;
    if ( toDelete->type() == "Journal" )
        return;
    QString des = lastSync->description();
    QString pref = "e";
    if ( toDelete->type() == "Todo" )
        pref = "t";
    des += pref+ QString::number ( toDelete->zaurusId() ) + ",";
    lastSync->setReadOnly( false );
    lastSync->setDescription( des );
    lastSync->setReadOnly( true );

}
bool  CalendarView::synchronizeCalendar( Calendar* local,  Calendar* remote, int mode )
{
    bool syncOK = true;
    int addedEvent = 0;
    int addedEventR = 0;
    int deletedEventR = 0;
    int deletedEventL = 0;
    int changedLocal = 0;
    int changedRemote = 0;
    //QPtrList<Event> el = local->rawEvents(); 
    Event* eventR;
    QString uid;
    int take;
    Event* eventL;
    Event* eventRSync;
    Event* eventLSync;
    Event* eventRSyncSharp = remote->event( "last-syncEvent-device-Sharp-DTM");
    Event* eventLSyncSharp = local->event( "last-syncEvent-device-Sharp-DTM");
    bool fullDateRange = false;
    mLastCalendarSync = QDateTime::currentDateTime();
    QDateTime modifiedCalendar = mLastCalendarSync;;
    eventR = remote->event("last-syncEvent-device-"+mCurrentSyncName );
    if ( eventR ) {
        eventRSync = (Event*) eventR->clone();
        remote->deleteEvent(eventR );

    } else {
        fullDateRange = true;
        eventRSync = new Event();
        eventRSync->setSummary(mCurrentSyncName + i18n(" - sync event"));
        eventRSync->setUid("last-syncEvent-device-"+mCurrentSyncName ); 
        eventRSync->setDtStart( mLastCalendarSync );
        eventRSync->setDtEnd( mLastCalendarSync.addSecs( 7200 ) );
        eventRSync->setCategories( i18n("SyncEvent") );
    }
    eventLSync = getLastSyncEvent();
    if ( eventLSync->dtStart() == mLastCalendarSync )
        fullDateRange = true;
   
    if ( ! fullDateRange  ) {
        if ( eventLSync->dtStart() != eventRSync->dtStart() ) {
            
            // qDebug("set fulldate to true %s %s" ,eventLSync->dtStart().toString().latin1(), eventRSync->dtStart().toString().latin1() ); 
            //qDebug("%d %d %d %d ", eventLSync->dtStart().time().second(), eventLSync->dtStart().time().msec() , eventRSync->dtStart().time().second(), eventRSync->dtStart().time().msec());
            fullDateRange = true;
        } 
    }
    if ( fullDateRange )
        mLastCalendarSync = QDateTime::currentDateTime().addDays( -100*365);
    else
        mLastCalendarSync = eventLSync->dtStart();
    // for resyncing if own file has changed
    if ( mCurrentSyncDevice == "deleteaftersync" ) {
        mLastCalendarSync = loadedFileVersion;
        qDebug("setting mLastCalendarSync ");
    }
    //qDebug("*************************** ");
    qDebug("mLastCalendarSync %s ",mLastCalendarSync.toString().latin1() );
    QPtrList<Incidence> er = remote->rawIncidences(); 
    Incidence* inR = er.first();
    Incidence* inL;
    QProgressBar bar( er.count(),0 );
    bar.setCaption (i18n("Syncing - close to abort!") );
    
    int w = 300;
    if ( QApplication::desktop()->width() < 320 )
        w = 220;
    int h = bar.sizeHint().height() ;
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
    bar.setGeometry( (dw-w)/2, (dh - h )/2 ,w,h );
    bar.show();
    int modulo = (er.count()/10)+1;
    int incCounter = 0;
    while ( inR ) {
        if ( ! bar.isVisible() )
            return false;
        if ( incCounter % modulo == 0 )
            bar.setProgress( incCounter );
        ++incCounter;
        uid = inR->uid(); 
        bool skipIncidence = false;
        if ( uid.left(21) == QString("last-syncEvent-device") )
            skipIncidence = true;
       
        qApp->processEvents();
        if ( !skipIncidence ) {
            inL = local->incidence( uid );
            if ( inL ) { // maybe conflict - same uid in both calendars
                int maxrev = inL->revision();
                if ( maxrev < inR->revision() )
                    maxrev = inR->revision();
                if ( (take = takeEvent( inL, inR, mode, fullDateRange )) > 0 ) {
                    //qDebug("take %d %s ", take, inL->summary().latin1());
                    if ( take == 3 )
                        return false;
                    if ( take == 1  ) {// take local
                        inL->setZaurusUid( inR->zaurusUid() );
                        remote->deleteIncidence( inR );
                        if ( inL->revision() < maxrev )
                            inL->setRevision( maxrev );
                        remote->addIncidence(  inL->clone() );
                        ++changedRemote;
                    } else {
                        if ( inR->revision() < maxrev )
                            inR->setRevision( maxrev );
                        local->deleteIncidence( inL );
                        local->addIncidence(  inR->clone()  );
                        ++changedLocal;
                    }
                }
            } else { //  no conflict
                if ( mGlobalSyncMode == SYNC_MODE_SHARP ) {
                    QString des = eventLSync->description(); 
                    QString pref = "e";
                    if ( inR->type() == "Todo" )
                        pref = "t";
                    if ( des.find(pref+QString::number( inR->zaurusId() ) +"," ) >= 0 && mode != 5) { // delete it
                        inR->setZaurusStat( -3 );
                        //remote->deleteIncidence( inR );
                        ++deletedEventR; 
                    } else {
                        inR->setLastModified( modifiedCalendar );
                        local->addIncidence( inR->clone() );
                        ++addedEvent;
                    }
                } else {
                    if ( inR->lastModified() > mLastCalendarSync || mode == 5 ) {
                        inR->setLastModified( modifiedCalendar );
                        local->addIncidence( inR->clone() );
                        ++addedEvent;
                    } else {
                        checkSharpEvent(eventRSyncSharp, inR);
                        remote->deleteIncidence( inR );
                        ++deletedEventR; 
                    }
                }
            }
        }
        inR =  er.next();
    }
    QPtrList<Incidence> el = local->rawIncidences(); 
    inL = el.first();
    modulo = (el.count()/10)+1;
    bar.setCaption (i18n("Add / remove events") ); 
    bar.setTotalSteps (  el.count()  ) ;
    bar.show();
    incCounter = 0;
   
    while ( inL ) {
           
        qApp->processEvents();
        if ( ! bar.isVisible() )
            return false;
        if ( incCounter % modulo == 0 )
            bar.setProgress( incCounter );
        ++incCounter;
        uid = inL->uid();
        bool skipIncidence = false;
        if ( uid.left(21) == QString("last-syncEvent-device") )
            skipIncidence = true;
        if ( mGlobalSyncMode == SYNC_MODE_SHARP && inL->type() == "Journal" )
            skipIncidence = true;
        if ( !skipIncidence )  {
            inR = remote->incidence( uid );
            if ( ! inR ) { 
                if ( mGlobalSyncMode == SYNC_MODE_SHARP ) {
                    if ( inL->zaurusId() >= 0 && mode != 4 ) {
                        local->deleteIncidence( inL );
                        ++deletedEventL; 
                    } else {
                        if ( ! KOPrefs::instance()->mWriteBackExistingOnly ) {
                            inL->setZaurusId( -1 );
                            ++addedEventR; 
                            inL->setLastModified( modifiedCalendar );
                            remote->addIncidence(  inL->clone() );
                        }
                    }
                } else {
                    if ( inL->lastModified() < mLastCalendarSync && mode != 4 ) {
                        checkSharpEvent(eventLSyncSharp, inL);
                        local->deleteIncidence( inL );
                        ++deletedEventL; 
                    } else {
                        if ( ! KOPrefs::instance()->mWriteBackExistingOnly ) {
                            ++addedEventR; 
                            inL->setLastModified( modifiedCalendar );
                            remote->addIncidence(  inL->clone() );
                        }
                    }
                }
            }
        }
        inL = el.next();
    }
     
    bar.hide();
    mLastCalendarSync = QDateTime::currentDateTime().addSecs( 1 );
    eventLSync->setReadOnly( false );
    eventLSync->setDtStart( mLastCalendarSync );
    eventRSync->setDtStart( mLastCalendarSync );
    eventLSync->setDtEnd( mLastCalendarSync.addSecs( 3600 ) );
    eventRSync->setDtEnd( mLastCalendarSync.addSecs( 3600 ) );
    eventRSync->setLocation( i18n("Remote from: ")+mCurrentSyncName ) ;
    eventLSync->setLocation(i18n("Local from: ") + mCurrentSyncName  );
    eventLSync->setReadOnly( true );
    if ( mGlobalSyncMode ==  SYNC_MODE_NORMAL)
        remote->addEvent( eventRSync );
    QString mes;
    mes .sprintf(  i18n("Synchronization summary:\n\n %d items added to local\n %d items added to remote\n %d items updated on local\n %d items updated on remote\n %d items deleted on local\n %d items deleted on remote\n"),addedEvent, addedEventR, changedLocal, changedRemote, deletedEventL, deletedEventR ); 
    if ( KOPrefs::instance()->mShowSyncSummary ) {
        KMessageBox::information(this, mes, i18n("KO/Pi Synchronization") );
    }
    qDebug( mes );
    mCalendar->checkAlarmForIncidence( 0, true );
    return syncOK;
}

void CalendarView::setSyncDevice( QString s )
{
    mCurrentSyncDevice= s;
}
void CalendarView::setSyncName( QString s )
{
    mCurrentSyncName= s;
}
bool CalendarView::syncCalendar(QString filename, int mode)
{
    mGlobalSyncMode = SYNC_MODE_NORMAL;
    CalendarLocal* calendar = new CalendarLocal();
    calendar->setTimeZoneId(KOPrefs::instance()->mTimeZoneId);
    FileStorage* storage = new FileStorage( calendar );
    bool syncOK = false;
    storage->setFileName( filename );
    // qDebug("loading ... ");
    if ( storage->load(KOPrefs::instance()->mUseQuicksave) ) {
        getEventViewerDialog()->setSyncMode( true );
        syncOK = synchronizeCalendar( mCalendar, calendar, mode );
        getEventViewerDialog()->setSyncMode( false );
        if (  syncOK ) {
           if ( KOPrefs::instance()->mWriteBackFile )
            {
                storage->setSaveFormat( new ICalFormat( KOPrefs::instance()->mUseQuicksave) );
                storage->save();
            }
        }
        setModified( true );
    }
    delete storage;
    delete calendar;
    if ( syncOK )
        updateView();
    return syncOK;
}
void CalendarView::syncSharp()
{
#ifndef DESKTOP_VERSION
    mGlobalSyncMode = SYNC_MODE_SHARP;
    //mCurrentSyncDevice = "sharp-DTM";
    if ( KOPrefs::instance()->mAskForPreferences )
        edit_sync_options(); 
    qApp->processEvents();
    CalendarLocal* calendar = new CalendarLocal();
    calendar->setTimeZoneId(KOPrefs::instance()->mTimeZoneId);
    bool syncOK = false;
    SharpFormat sharpFormat;
    if ( sharpFormat.load( calendar, mCalendar ) ) {
        getEventViewerDialog()->setSyncMode( true );
        syncOK = synchronizeCalendar( mCalendar, calendar, KOPrefs::instance()->mSyncAlgoPrefs );
        getEventViewerDialog()->setSyncMode( false );
        qApp->processEvents();
        if (  syncOK ) {
            if ( KOPrefs::instance()->mWriteBackFile )
            {
                QPtrList<Incidence> iL = mCalendar->rawIncidences(); 
                Incidence* inc = iL.first();
                while ( inc ) {
                    inc->setZaurusStat( inc->revision () );
                    inc =  iL.next();
                }
                // pending: clean last sync event description
                sharpFormat.save(calendar);
                iL = calendar->rawIncidences(); 
                inc = iL.first();
                Incidence* loc;
                while ( inc ) {
                    if ( inc->zaurusStat() == -4 ) {
                        loc = mCalendar->incidence(inc->uid() );
                        if ( loc ) {
                            loc->setZaurusId( inc->zaurusId() );
                            loc->setZaurusUid( inc->zaurusUid() );
                        }
                    }
                    inc =  iL.next();
                }
                Incidence* lse = getLastSyncEvent();
                if ( lse ) {
                    lse->setReadOnly( false );
                    lse->setDescription( "" );
                    lse->setReadOnly( true );
                }
            }
        }
        setModified( true );
    } else {
        QString question = i18n("Sorry, the database access\ncommand failed!\n\nNothing synced!\n") ;
        QMessageBox::information( 0, i18n("KO/Pi Import - ERROR"),
                                      question, i18n("Ok")) ;

    }
    delete calendar;
    updateView();
    return ;//syncOK;
#endif
}
bool CalendarView::importQtopia( const QString &categories,
                                 const QString &datebook,
                                 const QString &todolist )
{
    
   QtopiaFormat qtopiaFormat;
    qtopiaFormat.setCategoriesList (  &(KOPrefs::instance()->mCustomCategories));
    if ( !categories.isEmpty() ) qtopiaFormat.load( mCalendar, categories );
    if ( !datebook.isEmpty() ) qtopiaFormat.load( mCalendar, datebook );
    if ( !todolist.isEmpty() ) qtopiaFormat.load( mCalendar, todolist );
   
    updateView();
    return true;

#if 0
 mGlobalSyncMode = SYNC_MODE_QTOPIA;
    mCurrentSyncDevice = "qtopia-XML";
    if ( KOPrefs::instance()->mAskForPreferences )
        edit_sync_options(); 
    qApp->processEvents();
    CalendarLocal* calendar = new CalendarLocal();
    calendar->setTimeZoneId(KOPrefs::instance()->mTimeZoneId);
    bool syncOK = false;
 QtopiaFormat qtopiaFormat;
    qtopiaFormat.setCategoriesList (  &(KOPrefs::instance()->mCustomCategories));
    bool loadOk = true;
    if ( !categories.isEmpty() ) 
        loadOk = qtopiaFormat.load( calendar, categories );
    if ( loadOk && !datebook.isEmpty() ) 
        loadOk = qtopiaFormat.load( calendar, datebook );
    if ( loadOk && !todolist.isEmpty() ) 
        loadOk = qtopiaFormat.load( calendar, todolist );

    if ( loadOk  ) {
        getEventViewerDialog()->setSyncMode( true );
        syncOK = synchronizeCalendar( mCalendar, calendar, KOPrefs::instance()->mSyncAlgoPrefs );
        getEventViewerDialog()->setSyncMode( false );
        qApp->processEvents();
        if (  syncOK ) {
            if ( KOPrefs::instance()->mWriteBackFile )
            {
                // write back XML file

            }
        setModified( true );
        }
    } else {
        QString question = i18n("Sorry, the file loading\ncommand failed!\n\nNothing synced!\n") ;
        QMessageBox::information( 0, i18n("KO/Pi Sync - ERROR"),
                                      question, i18n("Ok")) ;
    }
    delete calendar;
    updateView();
    return syncOK;


#endif

}
    
void CalendarView::setSyncEventsReadOnly()
{
    Event * ev;
    QPtrList<Event> eL = mCalendar->rawEvents(); 
    ev = eL.first();
    while ( ev ) {
        if ( ev->uid().left(21) == QString("last-syncEvent-device") )
            ev->setReadOnly( true );
        ev =  eL.next();
    } 
}
bool CalendarView::openCalendar(QString filename, bool merge)
{

    if (filename.isEmpty()) {
        return false;
    }

    if (!QFile::exists(filename)) {
        KMessageBox::error(this,i18n("File does not exist:\n '%1'.").arg(filename));
        return false;
    }

    globalFlagBlockAgenda = 1;
    if (!merge) mCalendar->close();

    mStorage->setFileName( filename );

    if ( mStorage->load(KOPrefs::instance()->mUseQuicksave) ) {
        if ( merge ) ;//setModified( true );
        else {
            //setModified( true );
            mViewManager->setDocumentId( filename );
            mDialogManager->setDocumentId( filename );
            mTodoList->setDocumentId( filename );
        }
        globalFlagBlockAgenda = 2;
        // if ( getLastSyncEvent() )
        //  getLastSyncEvent()->setReadOnly( true );
        mCalendar->reInitAlarmSettings();
        setSyncEventsReadOnly();
        updateUnmanagedViews();
        updateView();
        if (  filename != MainWindow::defaultFileName() )
            saveCalendar( MainWindow::defaultFileName() );
        loadedFileVersion = QDateTime::currentDateTime();
        return true;
    } else {
        // while failing to load, the calendar object could
        // have become partially populated.  Clear it out.
        if ( !merge ) mCalendar->close();

        KMessageBox::error(this,i18n("Couldn't load calendar\n '%1'.").arg(filename));

        globalFlagBlockAgenda = 2;
        updateView();
    }
    return false;
}
void CalendarView::setLoadedFileVersion(QDateTime dt)
{
    loadedFileVersion = dt;
}
bool CalendarView::checkFileChanged(QString fn)
{
    QFileInfo finf ( fn ); 
    if ( !finf.exists() )
        return true;
    QDateTime dt = finf.lastModified ();
    if ( dt <= loadedFileVersion )
        return false;
    return true;

}
bool CalendarView::checkFileVersion(QString fn)
{
    QFileInfo finf ( fn );  
    if ( !finf.exists() )
        return true;
    QDateTime dt = finf.lastModified ();
    //qDebug("loaded file version %s",loadedFileVersion.toString().latin1());
    //qDebug("file on disk version  %s",dt.toString().latin1());
    if ( dt <= loadedFileVersion )
        return true;
   int km = KMessageBox::warningYesNoCancel(this, i18n("\nThe file on disk has changed!\nFile size: %1 bytes.\nLast modified: %2\nDo you want to:\n\n - Save and overwrite file?\n - Sync with file, then save?\n - Cancel without saving? \n").arg( QString::number( finf.size())).arg( KGlobal::locale()->formatDateTime(finf.lastModified (), true, false)) ,
                                                 i18n("KO/Pi Warning"),i18n("Overwrite"),
                                                 i18n("Sync+save"));
   
   if ( km ==  KMessageBox::Cancel )
       return false;
   if ( km ==  KMessageBox::Yes )
       return true;

    setSyncDevice("deleteaftersync" );
    KOPrefs::instance()->mAskForPreferences = true;
    KOPrefs::instance()->mSyncAlgoPrefs = 3;
    KOPrefs::instance()->mWriteBackFile = false;
    KOPrefs::instance()->mWriteBackExistingOnly = false;
    KOPrefs::instance()->mShowSyncSummary = false;
    syncCalendar( fn, 3 );
    Event * e = getLastSyncEvent();
    mCalendar->deleteEvent ( e );
    updateView();
    return true;
}

bool CalendarView::saveCalendar( QString filename )
{
 
    // Store back all unsaved data into calendar object
    // qDebug("file %s %d ",  filename.latin1()           , mViewManager->currentView()         );
    if ( mViewManager->currentView() )
        mViewManager->currentView()->flushView();

    //mStorage->setFileName( filename );

    mStorage->setSaveFormat( new ICalFormat( KOPrefs::instance()->mUseQuicksave) );
    mStorage->setFileName( filename );
    bool success;
    success = mStorage->save();
    if ( !success ) {
        return false;
    }

    return true;
}

void CalendarView::closeCalendar()
{
 
    // child windows no longer valid
    emit closingDown();

    mCalendar->close();
    setModified(false);
    updateView();
}

void CalendarView::archiveCalendar()
{
    mDialogManager->showArchiveDialog();
}


void CalendarView::readSettings()
{
 
  
    // mViewManager->showAgendaView();
    QString str;
    //qDebug("CalendarView::readSettings() ");
    // read settings from the KConfig, supplying reasonable
    // defaults where none are to be found
    KConfig *config = KOGlobals::config();
#ifndef KORG_NOSPLITTER
    config->setGroup("KOrganizer Geometry");

    QValueList<int> sizes = config->readIntListEntry("Separator1");
    if (sizes.count() != 2) {
        sizes << mDateNavigator->minimumSizeHint().width();
        sizes << 300;
    }
    mPanner->setSizes(sizes);

    sizes = config->readIntListEntry("Separator2");
    if ( ( mResourceView && sizes.count() == 4 ) ||
         ( !mResourceView && sizes.count() == 3 ) ) {
        mLeftSplitter->setSizes(sizes);
    }
#endif
    globalFlagBlockAgenda = 1;
    mViewManager->showAgendaView();
    //mViewManager->readSettings( config );
    mTodoList->restoreLayout(config,QString("Todo Layout"));
    readFilterSettings(config);
    config->setGroup( "Views" );
    int dateCount = config->readNumEntry( "ShownDatesCount", 7 );
    if ( dateCount == 5 ) mNavigator->selectWorkWeek();
    else if ( dateCount == 7 ) mNavigator->selectWeek();
    else mNavigator->selectDates( dateCount );
    // mViewManager->readSettings( config );
    updateConfig(); 
    globalFlagBlockAgenda = 2;
    mViewManager->readSettings( config );
#ifdef DESKTOP_VERSION
    config->setGroup("WidgetLayout");
    QStringList list;
    list = config->readListEntry("MainLayout");
    int x,y,w,h;
    if ( ! list.isEmpty() ) {
        x = list[0].toInt();
        y = list[1].toInt();
        w = list[2].toInt();
        h = list[3].toInt();
        topLevelWidget()->setGeometry(x,y,w,h);

    } else {
        topLevelWidget()->setGeometry( 40 ,40 , 640, 440);
    }
    list = config->readListEntry("EditEventLayout");
    if ( ! list.isEmpty() ) {
        x = list[0].toInt();
        y = list[1].toInt();
        w = list[2].toInt();
        h = list[3].toInt();
        mEventEditor->setGeometry(x,y,w,h);

    }
    list = config->readListEntry("EditTodoLayout");
    if ( ! list.isEmpty() ) {
        x = list[0].toInt();
        y = list[1].toInt();
        w = list[2].toInt();
        h = list[3].toInt();
        mTodoEditor->setGeometry(x,y,w,h);

    }
    list = config->readListEntry("ViewerLayout");
    if ( ! list.isEmpty() ) {
        x = list[0].toInt();
        y = list[1].toInt();
        w = list[2].toInt();
        h = list[3].toInt();
        getEventViewerDialog()->setGeometry(x,y,w,h);
    }
#endif


    // pending read sync settings;
    mSyncProfileNames.clear();
    mSyncProfileNames << "Profile_1";
    mSyncProfileNames << "Profile_2";
    mSyncProfileNames << "Profile_3";
    mSyncProfileNames << "Profile_4";
    mSyncProfileNames << "Profile_5";
    KSyncProfile* temp = new KSyncProfile ();
    temp->setName("Profile_1" );
    mSyncProfiles.append( temp );
    temp = new KSyncProfile ();
    temp->setName("Profile_2" );
    mSyncProfiles.append( temp );
   temp = new KSyncProfile ();
    temp->setName("Profile_3" );
    mSyncProfiles.append( temp );
    temp = new KSyncProfile ();
    temp->setName("Profile_4" );
    mSyncProfiles.append( temp );
    temp = new KSyncProfile ();
    temp->setName("Profile_5" );
    mSyncProfiles.append( temp );
}


void CalendarView::writeSettings()
{
    //  kdDebug() << "CalendarView::writeSettings" << endl;

    KConfig *config = KOGlobals::config();

#ifndef KORG_NOSPLITTER
    config->setGroup("KOrganizer Geometry");

    QValueList<int> list = mPanner->sizes();
    config->writeEntry("Separator1",list);

    list = mLeftSplitter->sizes();
    config->writeEntry("Separator2",list);
#endif

    mViewManager->writeSettings( config );
    mTodoList->saveLayout(config,QString("Todo Layout"));
    mDialogManager->writeSettings( config );
    //KOPrefs::instance()->usrWriteConfig();
    KOPrefs::instance()->writeConfig();

    writeFilterSettings(config);

    config->setGroup( "Views" );
    config->writeEntry( "ShownDatesCount", mNavigator->selectedDates().count() );

#ifdef DESKTOP_VERSION
    config->setGroup("WidgetLayout");
    QStringList list ;//= config->readListEntry("MainLayout");
    int x,y,w,h;
    QWidget* wid;
    wid = topLevelWidget();
    x = wid->geometry().x();
    y = wid->geometry().y();
    w = wid->width();
    h = wid->height();
    list.clear();
    list << QString::number( x );
    list << QString::number( y );
    list << QString::number( w );
    list << QString::number( h );
    config->writeEntry("MainLayout",list );
    
    wid = mEventEditor;
    x = wid->geometry().x();
    y = wid->geometry().y();
    w = wid->width();
    h = wid->height();
    list.clear();
    list << QString::number( x );
    list << QString::number( y );
    list << QString::number( w );
    list << QString::number( h );
    config->writeEntry("EditEventLayout",list );
 
    wid = mTodoEditor;
    x = wid->geometry().x();
    y = wid->geometry().y();
    w = wid->width();
    h = wid->height();
    list.clear();
    list << QString::number( x );
    list << QString::number( y );
    list << QString::number( w );
    list << QString::number( h );
    config->writeEntry("EditTodoLayout",list ); 
    wid = getEventViewerDialog();
    x = wid->geometry().x();
    y = wid->geometry().y();
    w = wid->width();
    h = wid->height();
    list.clear();
    list << QString::number( x );
    list << QString::number( y );
    list << QString::number( w );
    list << QString::number( h );
    config->writeEntry("ViewerLayout",list ); 
    wid = mDialogManager->getSearchDialog();
    if ( wid ) {
        x = wid->geometry().x();
        y = wid->geometry().y();
        w = wid->width();
        h = wid->height();
        list.clear();
        list << QString::number( x );
        list << QString::number( y );
        list << QString::number( w );
        list << QString::number( h );
        config->writeEntry("SearchLayout",list );
    }
#endif


    config->sync();
}

void CalendarView::readFilterSettings(KConfig *config)
{
    //  kdDebug() << "CalendarView::readFilterSettings()" << endl;

    mFilters.clear();

    config->setGroup("General");
    QStringList filterList = config->readListEntry("CalendarFilters");

    QStringList::ConstIterator it = filterList.begin();
    QStringList::ConstIterator end = filterList.end();
    while(it != end) {
        //    kdDebug() << "  filter: " << (*it) << endl;

        CalFilter *filter;
        filter = new CalFilter(*it);
        config->setGroup("Filter_" + (*it));
        //qDebug("readFilterSettings %d ",config->readNumEntry("Criteria",0) );
        filter->setCriteria(config->readNumEntry("Criteria",0));
        filter->setCategoryList(config->readListEntry("CategoryList"));
        mFilters.append(filter);

        ++it;
    }

    if (mFilters.count() == 0) {
        CalFilter *filter = new CalFilter(i18n("Default"));
        mFilters.append(filter);
    }
    mFilterView->updateFilters();
    config->setGroup("FilterView");

    mFilterView->blockSignals(true);
    mFilterView->setFiltersEnabled(config->readBoolEntry("FilterEnabled"));
    mFilterView->setSelectedFilter(config->readEntry("Current Filter"));
    mFilterView->blockSignals(false);
    // We do it manually to avoid it being done twice by the above calls
    updateFilter();
}

void CalendarView::writeFilterSettings(KConfig *config)
{
    //  kdDebug() << "CalendarView::writeFilterSettings()" << endl;

    QStringList filterList;

    CalFilter *filter = mFilters.first();
    while(filter) {
        //    kdDebug() << " fn: " << filter->name() << endl;
        filterList << filter->name();
        config->setGroup("Filter_" + filter->name());
        config->writeEntry("Criteria",filter->criteria());
        config->writeEntry("CategoryList",filter->categoryList());
        filter = mFilters.next();
    }
    config->setGroup("General");
    config->writeEntry("CalendarFilters",filterList);

    config->setGroup("FilterView");
    config->writeEntry("FilterEnabled",mFilterView->filtersEnabled());
    config->writeEntry("Current Filter",mFilterView->selectedFilter()->name());
}


void CalendarView::goToday()
{
    mNavigator->selectToday();
}

void CalendarView::goNext()
{
    mNavigator->selectNext();
}

void CalendarView::goPrevious()
{
    mNavigator->selectPrevious();
}
void CalendarView::goNextMonth()
{
    mNavigator->selectNextMonth();
}

void CalendarView::goPreviousMonth()
{
    mNavigator->selectPreviousMonth();
}
void CalendarView::writeLocale()
{
    KGlobal::locale()->setHore24Format( !KOPrefs::instance()->mPreferredTime );
    KGlobal::locale()->setWeekStartMonday( !KOPrefs::instance()->mWeekStartsOnSunday );
    KGlobal::locale()->setIntDateFormat( KOPrefs::instance()->mPreferredDate );
    KGlobal::locale()->setLanguage( KOPrefs::instance()->mPreferredLanguage );
    QString dummy = KOPrefs::instance()->mUserDateFormatLong;
    KGlobal::locale()->setDateFormat(dummy.replace( QRegExp("K"), QString(",") ));
    dummy = KOPrefs::instance()->mUserDateFormatShort;
    KGlobal::locale()->setDateFormatShort(dummy.replace( QRegExp("K"), QString(",") ));
    KGlobal::locale()->setDaylightSaving( KOPrefs::instance()->mUseDaylightsaving,
                                          KOPrefs::instance()->mDaylightsavingStart,
                                          KOPrefs::instance()->mDaylightsavingEnd );
    KGlobal::locale()->setTimezone( KOPrefs::instance()->mTimeZoneId );
}
void CalendarView::updateConfig()
{
    writeLocale();
    if ( KOPrefs::instance()->mUseAppColors )
        QApplication::setPalette( QPalette (KOPrefs::instance()->mAppColor1, KOPrefs::instance()->mAppColor2), true );
    emit configChanged();
    mTodoList->updateConfig();
    // mDateNavigator->setFont ( KOPrefs::instance()->mDateNavigatorFont);
    mCalendar->setTimeZoneId(KOPrefs::instance()->mTimeZoneId);
    // To make the "fill window" configurations work
    //mViewManager->raiseCurrentView();
}


void CalendarView::eventChanged(Event *event)
{
    changeEventDisplay(event,KOGlobals::EVENTEDITED);
    //updateUnmanagedViews();
}

void CalendarView::eventAdded(Event *event)
{
    changeEventDisplay(event,KOGlobals::EVENTADDED);
}

void CalendarView::eventToBeDeleted(Event *)
{
    kdDebug() << "CalendarView::eventToBeDeleted(): to be implemented" << endl;
}

void CalendarView::eventDeleted()
{
    changeEventDisplay(0,KOGlobals::EVENTDELETED);
}
void CalendarView::changeTodoDisplay(Todo *which, int action)
{
    changeIncidenceDisplay((Incidence *)which,  action);
}
void CalendarView::checkZaurusId( int id, bool todo )
{
    if ( id >= 0 ) { 
        Incidence* lse = mCalendar->event( "last-syncEvent-device-Sharp-DTM");
        if ( lse ) {
            QString des = lse->description();
            QString pref = "e";
            if ( todo )
                pref = "t";
            des += pref+ QString::number ( id ) + ",";
            lse->setReadOnly( false );
            lse->setDescription( des );
            lse->setReadOnly( true );
        }
    }
}
void CalendarView::changeIncidenceDisplay(Incidence *which, int action)
{
    updateUnmanagedViews();
    //qDebug(" CalendarView::changeIncidenceDisplay++++++++++++++++++++++++++  %d  %d ",which, action );
    if ( action == KOGlobals::EVENTDELETED ) { //delete
        mCalendar->checkAlarmForIncidence( 0, true );
        if ( mEventViewerDialog )
            mEventViewerDialog->hide();
    } 
    else
        mCalendar->checkAlarmForIncidence( which , false );
}

// most of the changeEventDisplays() right now just call the view's
// total update mode, but they SHOULD be recoded to be more refresh-efficient.
void CalendarView::changeEventDisplay(Event *which, int action)
{
    //  kdDebug() << "CalendarView::changeEventDisplay" << endl;
    changeIncidenceDisplay((Incidence *)which,  action);
    mDateNavigator->updateView();
    //mDialogManager->updateSearchDialog();

    if (which) {
        // If there is an event view visible update the display
        mViewManager->currentView()->changeEventDisplay(which,action);
        // TODO: check, if update needed
        //    if (which->getTodoStatus()) {
        mTodoList->updateView();
        //    }
    } else {
        mViewManager->currentView()->updateView();
    }
}


void CalendarView::updateTodoViews()
{
   
    mTodoList->updateView();
    mViewManager->currentView()->updateView();
 
}


void CalendarView::updateView(const QDate &start, const QDate &end)
{
    mTodoList->updateView();
    mViewManager->updateView(start, end);
    //mDateNavigator->updateView();
}

void CalendarView::updateView()
{
    DateList tmpList = mNavigator->selectedDates();

    // We assume that the navigator only selects consecutive days.
    updateView( tmpList.first(), tmpList.last() );
}

void CalendarView::updateUnmanagedViews()
{
    mDateNavigator->updateDayMatrix();
}

int CalendarView::msgItemDelete()
{
    return KMessageBox::warningContinueCancel(this,
                                              i18n("This item will be\npermanently deleted."),
                                              i18n("KO/Pi Confirmation"),i18n("Delete"));
}


void CalendarView::edit_cut()
{
    Event *anEvent=0;

    Incidence *incidence = mViewManager->currentView()->selectedIncidences().first();

    if (mViewManager->currentView()->isEventView()) {
        if ( incidence && incidence->type() == "Event" ) {
            anEvent = static_cast<Event *>(incidence);
        }
    }

    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }
    DndFactory factory( mCalendar );
    factory.cutEvent(anEvent);
    changeEventDisplay(anEvent, KOGlobals::EVENTDELETED);
}

void CalendarView::edit_copy()
{
    Event *anEvent=0;

    Incidence *incidence = mViewManager->currentView()->selectedIncidences().first();

    if (mViewManager->currentView()->isEventView()) {
        if ( incidence && incidence->type() == "Event" ) {
            anEvent = static_cast<Event *>(incidence);
        }
    }

    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }
    DndFactory factory( mCalendar );
    factory.copyEvent(anEvent);
}

void CalendarView::edit_paste()
{
    QDate date = mNavigator->selectedDates().first();

    DndFactory factory( mCalendar );
    Event *pastedEvent = factory.pasteEvent( date );

    changeEventDisplay( pastedEvent, KOGlobals::EVENTADDED );
}

void CalendarView::edit_options()
{
    mDialogManager->showOptionsDialog();
}
void CalendarView::edit_sync_options()
{
    //mDialogManager->showSyncOptions();
    //KOPrefs::instance()->mSyncAlgoPrefs 
    QDialog dia( this, "dia", true );
    dia.setCaption( i18n("Device: " ) +mCurrentSyncDevice );
    QButtonGroup gr ( 1,  Qt::Horizontal, i18n("Sync preferences"), &dia);
    QVBoxLayout lay ( &dia );
    lay.setSpacing( 2 );
    lay.setMargin( 3 );
    lay.addWidget(&gr);
    QRadioButton loc ( i18n("Take local entry on conflict"), &gr );
    QRadioButton rem ( i18n("Take remote entry on conflict"), &gr );
    QRadioButton newest( i18n("Take newest entry on conflict"), &gr );
    QRadioButton ask( i18n("Ask for every entry on conflict"), &gr );
    QRadioButton f_loc( i18n("Force: Take local entry always"), &gr );
    QRadioButton f_rem( i18n("Force: Take remote entry always"), &gr );
    //QRadioButton both( i18n("Take both on conflict"), &gr );
    QPushButton pb ( "OK",  &dia); 
    lay.addWidget( &pb );
    connect(&pb, SIGNAL( clicked() ), &dia, SLOT ( accept() ) ); 
    switch ( KOPrefs::instance()->mSyncAlgoPrefs ) {
    case 0:
        loc.setChecked( true);
        break;
    case 1:
        rem.setChecked( true );
        break;
    case 2:
        newest.setChecked( true);
        break;
    case 3:
        ask.setChecked( true);
        break;
    case 4:
        f_loc.setChecked( true);
        break;
    case 5:
        f_rem.setChecked( true);
        break;
    case 6:
        // both.setChecked( true);
        break; 
    default:
        break;
    }
    if ( dia.exec() ) {
        KOPrefs::instance()->mSyncAlgoPrefs = rem.isChecked()*1+newest.isChecked()*2+  ask.isChecked()*3+  f_loc.isChecked()*4+  f_rem.isChecked()*5;//+  both.isChecked()*6 ;
    }
   
}

void  CalendarView::slotSelectPickerDate( QDate  d)
{
    mDateFrame->hide();
    if ( mDatePickerMode == 1 ) {
        mNavigator->slotDaySelect( d );
    } else  if ( mDatePickerMode == 2 ) {
        if ( mMoveIncidence->type() == "Todo" ) {
            Todo * to = (Todo *) mMoveIncidence;
            QTime tim;
            if ( to->hasDueDate() ) 
                tim = to->dtDue().time();
            else {
                tim = QTime ( 0,0,0 );
                to->setFloats( true );
                to->setHasDueDate( true );
            }
            QDateTime dt ( d,tim );
            to->setDtDue( dt );
            todoChanged( to );
        } else { 
            QTime tim = mMoveIncidence->dtStart().time(); 
            int secs = mMoveIncidence->dtStart().secsTo(  mMoveIncidence->dtEnd());
            QDateTime dt ( d,tim );
            mMoveIncidence->setDtStart( dt );
            ((Event*)mMoveIncidence)->setDtEnd( dt.addSecs( secs ) );
            changeEventDisplay((Event*)mMoveIncidence, KOGlobals::EVENTEDITED);
        }
        
        mMoveIncidence->setRevision( mMoveIncidence->revision()+1  );
    }
}

void CalendarView::removeCategories()
{
    QPtrList<Incidence> incList  = mCalendar->rawIncidences(); 
    QStringList catList = KOPrefs::instance()->mCustomCategories;
    QStringList catIncList;
    QStringList newCatList;
    Incidence* inc = incList.first();
    int i;
    int count = 0;
    while ( inc ) {
        newCatList.clear();
        catIncList = QStringList::split (",", inc->categoriesStr() );
        for( i = 0; i<   catIncList.count(); ++i ) {
            if ( catList.contains (catIncList[i]))
                newCatList.append( catIncList[i] );
        }
        newCatList.sort();
        inc->setCategories( newCatList.join(",") );
        inc = incList.next();
    }
}

int CalendarView::addCategories()
{ 
    QPtrList<Incidence> incList  = mCalendar->rawIncidences(); 
    QStringList catList = KOPrefs::instance()->mCustomCategories;
    QStringList catIncList;
    Incidence* inc = incList.first();
    int i;
    int count = 0;
     while ( inc ) {
        catIncList = QStringList::split (",", inc->categoriesStr() );
        for( i = 0; i<   catIncList.count(); ++i ) {
            if ( !catList.contains (catIncList[i])) {
                catList.append( catIncList[i] );
                //qDebug("add cat %s ",  catIncList[i].latin1());
                ++count;
            }
        }
        inc = incList.next();
    }
    catList.sort();
    KOPrefs::instance()->mCustomCategories = catList;
    return count;
}

void CalendarView::manageCategories()
{
    KOCatPrefs* cp = new KOCatPrefs(); 
    cp->show();
    int w =cp->sizeHint().width() ;
    int h = cp->sizeHint().height() ;
    int dw = QApplication::desktop()->width();
    int dh = QApplication::desktop()->height();
    cp->setGeometry( (dw-w)/2, (dh - h )/2 ,w,h );
    if ( !cp->exec() ) {
        delete cp;
        return;
    }
    int count = 0;
    if ( cp->addCat() ) {
        count = addCategories();
        if ( count )
            topLevelWidget()->setCaption(QString::number( count )+ i18n(" Categories added to list! "));
    } else {
        removeCategories();
        updateView();
    }
    delete cp;
}

void CalendarView::beamIncidence(Incidence * Inc) 
{
    QPtrList<Incidence> delSel ;
    delSel.append(Inc);
    beamIncidenceList( delSel );
} 
void CalendarView::beamCalendar() 
{
    QPtrList<Incidence> delSel  = mCalendar->rawIncidences(); 
    //qDebug("beamCalendar()  ");
    beamIncidenceList( delSel );
}
void CalendarView::beamFilteredCalendar() 
{
    QPtrList<Incidence> delSel  = mCalendar->incidences(); 
    //qDebug("beamFilteredCalendar()  ");
    beamIncidenceList( delSel );
}
void  CalendarView::beamIncidenceList(QPtrList<Incidence> delSel )
{
    if ( beamDialog->exec () == QDialog::Rejected ) 
        return;
  
    QString fn = "/tmp/kopibeamfile";
    QString mes;
    bool createbup = true;
    if ( createbup ) {
        QString description = "\n";
        CalendarLocal* cal = new CalendarLocal();
        if ( beamDialog->beamLocal() )
            cal->setLocalTime();
        else
            cal->setTimeZoneId(KOPrefs::instance()->mTimeZoneId);
        Incidence *incidence = delSel.first();
        bool addText = false;
        if ( delSel.count() < 10 )
            addText = true;
        else {
            description.sprintf(i18n("  %d items?"),delSel.count() );
        }
        while ( incidence ) {
            Incidence *in = incidence->clone();
            if ( addText )
                description += in->summary() + "\n";
            cal->addIncidence( in );
            incidence = delSel.next();
        }
        if ( beamDialog->beamVcal() ) {
            fn += ".vcs";
            FileStorage storage( cal, fn, new VCalFormat );
            storage.save();
        } else {
            fn += ".ics";
            FileStorage storage( cal, fn, new ICalFormat( KOPrefs::instance()->mUseQuicksave) );
            storage.save();
        }
        delete cal;
        mes = i18n("KO/Pi: Ready for beaming");
        setCaption(mes); 
        
#ifndef DESKTOP_VERSION
        Ir *ir = new Ir( this ); 
        connect( ir, SIGNAL( done( Ir * ) ), this, SLOT( beamDone( Ir * ) ) );
        ir->send( fn, description, "text/x-vCalendar" );
#endif
    }
}
void  CalendarView::beamDone( Ir *ir )
{
#ifndef DESKTOP_VERSION
    delete ir;
#endif
}

void CalendarView::moveIncidence(Incidence * inc ) 
{
    if ( !inc ) return;
    // qDebug("showDatePickerForIncidence( ) ");
    if ( mDateFrame->isVisible() )
        mDateFrame->hide();
    else {
        int w =mDatePicker->sizeHint().width()+2*mDateFrame->lineWidth() ;
        int h = mDatePicker->sizeHint().height()+2*mDateFrame->lineWidth() ;
        int dw = QApplication::desktop()->width();
        int dh = QApplication::desktop()->height();
        mDateFrame->setGeometry( (dw-w)/2, (dh - h )/2 ,w,h );
        mDateFrame->show();
    }
    mDatePickerMode = 2;
    mMoveIncidence = inc ;
    QDate da;
    if ( mMoveIncidence->type() == "Todo" ) {
        Todo * to = (Todo *) mMoveIncidence;
            if ( to->hasDueDate() ) 
                da = to->dtDue().date();
            else
                da = QDate::currentDate();
    } else {
        da = mMoveIncidence->dtStart().date();
    }
    mDatePicker->setDate( da );
}
void CalendarView::showDatePicker( ) 
{
    //qDebug("CalendarView::showDatePicker( )  ");
    if ( mDateFrame->isVisible() )
        mDateFrame->hide();
    else {
        int w =mDatePicker->sizeHint().width() ;
        int h = mDatePicker->sizeHint().height() ;
        int dw = QApplication::desktop()->width();
        int dh = QApplication::desktop()->height();
        mDateFrame->setGeometry( (dw-w)/2, (dh - h )/2 ,w,h );
        mDateFrame->show();
    }
    mDatePickerMode = 1;
    mDatePicker->setDate( mNavigator->selectedDates().first() );
}

void CalendarView::showEventEditor()
{
#ifdef DESKTOP_VERSION
    mEventEditor->show();
#else
    mEventEditor->showMaximized();
#endif
}
void CalendarView::showTodoEditor()
{
#ifdef DESKTOP_VERSION
    mTodoEditor->show();
#else
    mTodoEditor->showMaximized();
#endif
}
void CalendarView::cancelIncidence(Incidence * inc )
{
    inc->setCancelled( ! inc->cancelled() );
    changeIncidenceDisplay( inc,KOGlobals::EVENTEDITED );
    updateView();
}
void CalendarView::cloneIncidence(Incidence * orgInc )
{ 
    Incidence * newInc = orgInc->clone();
    newInc->recreate();
 
    if ( newInc->type() == "Todo" ) {
        Todo* t = (Todo*) newInc; 
        mTodoEditor->editTodo( t );
        showTodoEditor();
        if ( mTodoEditor->exec() ) {
            mCalendar->addTodo( t );
            updateView();
        } else {
            delete t;
        }
    }
    else {
        Event*  e = (Event*)  newInc;
        mEventEditor->editEvent( e );
        showEventEditor();
        if ( mEventEditor->exec() ) {
            mCalendar->addEvent( e );
            updateView();
        } else {
            delete e;
        }
    }
}

void CalendarView::newEvent()
{
    // TODO: Replace this code by a common eventDurationHint of KOBaseView.
    KOAgendaView *aView = mViewManager->agendaView();
    if (aView) {
        if (aView->selectionStart().isValid()) {
            if (aView->selectedIsAllDay()) {
                newEvent(aView->selectionStart(),aView->selectionEnd(),true);
            } else {
                newEvent(aView->selectionStart(),aView->selectionEnd());
            }
            return;
        }
    }

    QDate date = mNavigator->selectedDates().first();
    QDateTime current =  QDateTime::currentDateTime();
    if ( date <= current.date() ) {
        int hour = current.time().hour() +1;
        newEvent( QDateTime(  current.date(), QTime( hour, 0, 0 ) ),
                  QDateTime( current.date(), QTime( hour+ KOPrefs::instance()->mDefaultDuration, 0, 0 ) ) );
    } else
        newEvent( QDateTime( date, QTime( KOPrefs::instance()->mStartTime, 0, 0 ) ),
                  QDateTime( date, QTime( KOPrefs::instance()->mStartTime +
                                          KOPrefs::instance()->mDefaultDuration, 0, 0 ) ) );
}

void CalendarView::newEvent(QDateTime fh)
{
    newEvent(fh,
             QDateTime(fh.addSecs(3600*KOPrefs::instance()->mDefaultDuration)));
}

void CalendarView::newEvent(QDate dt)
{
    newEvent(QDateTime(dt, QTime(0,0,0)),
             QDateTime(dt, QTime(0,0,0)), true);
}

void CalendarView::newEvent(QDateTime fromHint, QDateTime toHint, bool allDay)
{
  
    mEventEditor->newEvent(fromHint,toHint,allDay);
    if ( mFilterView->filtersEnabled() ) {
        CalFilter *filter = mFilterView->selectedFilter();
        if (filter && filter->showCategories()) {
            mEventEditor->setCategories(filter->categoryList().join(",") );
        }
        if ( filter )
            mEventEditor->setSecrecy( filter->getSecrecy() );
    }
    showEventEditor();
}
void  CalendarView::todoAdded(Todo * t)
{
  
    changeTodoDisplay ( t ,KOGlobals::EVENTADDED);
    updateTodoViews();
}
void  CalendarView::todoChanged(Todo * t)
{
    emit todoModified( t, 4 );
    // updateTodoViews();
}
void  CalendarView::todoToBeDeleted(Todo *)
{
    //qDebug("todoToBeDeleted(Todo *) ");
    updateTodoViews();
}
void  CalendarView::todoDeleted()
{
    //qDebug(" todoDeleted()");
    updateTodoViews();
}



void CalendarView::newTodo()
{ 
 
    mTodoEditor->newTodo(QDateTime::currentDateTime().addDays(7),0,true);
    if ( mFilterView->filtersEnabled() ) {
        CalFilter *filter = mFilterView->selectedFilter();
        if (filter && filter->showCategories()) {
            mTodoEditor->setCategories(filter->categoryList().join(",") );
        }
        if ( filter )
            mTodoEditor->setSecrecy( filter->getSecrecy() );
    }
    showTodoEditor();
}

void CalendarView::newSubTodo()
{
    Todo *todo = selectedTodo();
    if ( todo ) newSubTodo( todo );
}

void CalendarView::newSubTodo(Todo *parentEvent)
{
  
    mTodoEditor->newTodo(QDateTime::currentDateTime().addDays(7),parentEvent,true);
    showTodoEditor();
}

void CalendarView::newFloatingEvent()
{
    DateList tmpList = mNavigator->selectedDates();
    QDate date = tmpList.first();

    newEvent( QDateTime( date, QTime( 12, 0, 0 ) ),
              QDateTime( date, QTime( 12, 0, 0 ) ), true );
}


void CalendarView::editEvent( Event *event )
{

    if ( !event ) return;
    if ( event->isReadOnly() ) {
        showEvent( event );
        return;
    }
    mEventEditor->editEvent( event , mFlagEditDescription);
    showEventEditor();
}
void CalendarView::editJournal( Journal *jour )
{ 
    if ( !jour ) return;
    mDialogManager->hideSearchDialog();
    mViewManager->showJournalView();
    mNavigator->slotDaySelect( jour->dtStart().date() );
}
void CalendarView::editTodo( Todo *todo )
{
    if ( !todo ) return;
 
    if ( todo->isReadOnly() ) {
        showTodo( todo );
        return;
    }
    mTodoEditor->editTodo( todo ,mFlagEditDescription);
    showTodoEditor();

}

KOEventViewerDialog* CalendarView::getEventViewerDialog()
{
    if ( !mEventViewerDialog ) {
        mEventViewerDialog = new KOEventViewerDialog(this);
        connect( mEventViewerDialog, SIGNAL(  editIncidence( Incidence* )), this, SLOT(editIncidence( Incidence* )  ) );
        connect( this, SIGNAL(configChanged()), mEventViewerDialog, SLOT(updateConfig()));
        connect( mEventViewerDialog, SIGNAL(jumpToTime( const QDate &)), 
                 dateNavigator(), SLOT( selectWeek( const QDate & ) ) );
        connect( mEventViewerDialog, SIGNAL(showAgendaView( bool ) ), 
                 viewManager(), SLOT( showAgendaView( bool ) ) );
        mEventViewerDialog->resize( 640, 480 );
        
    }
    return mEventViewerDialog;
}
void CalendarView::showEvent(Event *event)
{
    getEventViewerDialog()->setEvent(event);
    getEventViewerDialog()->showMe();
}

void CalendarView::showTodo(Todo *event)
{
    getEventViewerDialog()->setTodo(event);
    getEventViewerDialog()->showMe();
}
void CalendarView::showJournal( Journal *jour )
{ 
    getEventViewerDialog()->setJournal(jour);
    getEventViewerDialog()->showMe();
   
}
// void CalendarView::todoModified (Todo *event, int changed)
// {
//  //  if (mDialogList.find (event) != mDialogList.end ()) {
// //     kdDebug() << "Todo modified and open" << endl;
// //     KOTodoEditor* temp = (KOTodoEditor *) mDialogList[event];
// //     temp->modified (changed);

// //   }

//   mViewManager->updateView();
// }

void CalendarView::appointment_show()
{
    Event *anEvent = 0;

    Incidence *incidence = mViewManager->currentView()->selectedIncidences().first();

    if (mViewManager->currentView()->isEventView()) {
        if ( incidence && incidence->type() == "Event" ) {
            anEvent = static_cast<Event *>(incidence);
        }
    }

    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }

    showEvent(anEvent);
}

void CalendarView::appointment_edit()
{
    Event *anEvent = 0;

    Incidence *incidence = mViewManager->currentView()->selectedIncidences().first();

    if (mViewManager->currentView()->isEventView()) {
        if ( incidence && incidence->type() == "Event" ) {
            anEvent = static_cast<Event *>(incidence);
        }
    }

    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }

    editEvent(anEvent);
}

void CalendarView::appointment_delete()
{
    Event *anEvent = 0;

    Incidence *incidence = mViewManager->currentView()->selectedIncidences().first();

    if (mViewManager->currentView()->isEventView()) {
        if ( incidence && incidence->type() == "Event" ) {
            anEvent = static_cast<Event *>(incidence);
        }
    }

    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }

    deleteEvent(anEvent);
}

void CalendarView::todo_unsub(Todo *anTodo )
{
    // Todo *anTodo = selectedTodo();
    if (!anTodo) return;
    if (!anTodo->relatedTo()) return;
    anTodo->relatedTo()->removeRelation(anTodo);
    anTodo->setRelatedTo(0);
    anTodo->updated();
    anTodo->setRelatedToUid("");
    setModified(true);
    updateView();
}

void CalendarView::deleteTodo(Todo *todo)
{
    if (!todo) {
        KNotifyClient::beep();
        return;
    }
    if (KOPrefs::instance()->mConfirm) {
        switch (msgItemDelete()) {
        case KMessageBox::Continue: // OK
            if (!todo->relations().isEmpty()) {
                KMessageBox::sorry(this,i18n("Cannot delete To-Do\nwhich has children."),
                                   i18n("Delete To-Do"));
            } else {
                checkZaurusId(  todo->zaurusId(), true );
                calendar()->deleteTodo(todo);
                changeTodoDisplay( todo,KOGlobals::EVENTDELETED );
                updateView();
            }
            break;
        } // switch
    } else {
        if (!todo->relations().isEmpty()) {
            KMessageBox::sorry(this,i18n("Cannot delete To-Do\nwhich has children."),
                               i18n("Delete To-Do"));
        } else {
            checkZaurusId(  todo->zaurusId(), true );
            mCalendar->deleteTodo(todo);
            changeTodoDisplay( todo,KOGlobals::EVENTDELETED );
            updateView();
        }
    }
    emit updateSearchDialog();
}
void CalendarView::deleteJournal(Journal *jour)
{
    if (!jour) {
        KNotifyClient::beep();
        return;
    }
    if (KOPrefs::instance()->mConfirm) {
        switch (msgItemDelete()) {
        case KMessageBox::Continue: // OK
            calendar()->deleteJournal(jour);
            updateView();
            break;
        } // switch
    } else {
        calendar()->deleteJournal(jour);;
        updateView();
    }
    emit updateSearchDialog();
}

void CalendarView::deleteEvent(Event *anEvent)
{
    if (!anEvent) {
        KNotifyClient::beep();
        return;
    }

    if (anEvent->recurrence()->doesRecur()) {
        QDate itemDate = mViewManager->currentSelectionDate();
        int km;
        if (!itemDate.isValid()) {
            //kdDebug() << "Date Not Valid" << endl;
            if (KOPrefs::instance()->mConfirm) {
                km = KMessageBox::warningContinueCancel(this,anEvent->summary() +
                                                        i18n("\nThis event recurs\nover multiple dates.\nAre you sure you want\nto delete this event\nand all its recurrences?"),
                                                        i18n("KO/Pi Confirmation"),i18n("Delete All"));
                if ( km == KMessageBox::Continue )
                    km = KMessageBox::No; // No = all below
            } else
                km = KMessageBox::No;
        } else {
            km = KMessageBox::warningYesNoCancel(this,anEvent->summary() +
                                                 i18n("\nThis event recurs\nover multiple dates.\nDo you want to delete\nall it's recurrences,\nor only the current one on:\n")+
                                                      KGlobal::locale()->formatDate(itemDate)+i18n(" ?\n\nDelete:\n"),
                                                 i18n("KO/Pi Confirmation"),i18n("Current"),
                                                 i18n("All"));
        }
        switch(km) {

        case KMessageBox::No: // Continue // all
            //qDebug("KMessageBox::No ");
            if (anEvent->organizer()==KOPrefs::instance()->email() && anEvent->attendeeCount()>0)
                schedule(Scheduler::Cancel,anEvent);

            checkZaurusId( anEvent->zaurusId());
            mCalendar->deleteEvent(anEvent);
            changeEventDisplay(anEvent,KOGlobals::EVENTDELETED);
            break;

            // Disabled because it does not work
            //#if 0
        case KMessageBox::Yes: // just this one
            //QDate qd = mNavigator->selectedDates().first();
            //if (!qd.isValid()) {
            //  kdDebug() << "no date selected, or invalid date" << endl;
            //  KNotifyClient::beep();
            //  return;
            //}
            //while (!anEvent->recursOn(qd)) qd = qd.addDays(1);
            if (itemDate!=QDate(1,1,1) || itemDate.isValid()) {
                anEvent->addExDate(itemDate);
                int duration = anEvent->recurrence()->duration();
                if ( duration > 0 ) {
                    anEvent->recurrence()->setDuration( duration - 1 );
                }
                changeEventDisplay(anEvent, KOGlobals::EVENTEDITED);
            }
            break;
            //#endif
        } // switch
    } else {
        if (KOPrefs::instance()->mConfirm) {
            switch (KMessageBox::warningContinueCancel(this,anEvent->summary() +
                                                       i18n("\nAre you sure you want\nto delete this event?"),
                                                       i18n("KO/Pi Confirmation"),i18n("Delete"))) {
            case KMessageBox::Continue: // OK
                if (anEvent->organizer()==KOPrefs::instance()->email() && anEvent->attendeeCount()>0)
                    schedule(Scheduler::Cancel,anEvent);
                checkZaurusId( anEvent->zaurusId());
                mCalendar->deleteEvent(anEvent);
                changeEventDisplay(anEvent, KOGlobals::EVENTDELETED);
                break;
            } // switch
        } else {
            if (anEvent->organizer()==KOPrefs::instance()->email() && anEvent->attendeeCount()>0)
                schedule(Scheduler::Cancel,anEvent);
            checkZaurusId( anEvent->zaurusId());
            mCalendar->deleteEvent(anEvent);
            changeEventDisplay(anEvent, KOGlobals::EVENTDELETED);
        }
    } // if-else
    emit updateSearchDialog();
}

bool CalendarView::deleteEvent(const QString &uid)
{
    Event *ev = mCalendar->event(uid);
    if (ev) {
        deleteEvent(ev);
        return true;
    } else {
        return false;
    }
}

/*****************************************************************************/

void CalendarView::action_mail()
{
#ifndef KORG_NOMAIL
    KOMailClient mailClient;

    Incidence *incidence = currentSelection();

    if (!incidence) {
        KMessageBox::sorry(this,i18n("Can't generate mail:\nNo event selected."));
        return;
    }
    if(incidence->attendeeCount() == 0 ) {
        KMessageBox::sorry(this,
                           i18n("Can't generate mail:\nNo attendees defined.\n"));
        return;
    }

    CalendarLocal cal_tmp;
    Event *event = 0;
    Event *ev = 0;
    if ( incidence && incidence->type() == "Event" ) {
        event = static_cast<Event *>(incidence);
        ev = new Event(*event);
        cal_tmp.addEvent(ev);
    }
    ICalFormat mForm( KOPrefs::instance()->mUseQuicksave);
    QString attachment = mForm.toString( &cal_tmp );
    if (ev) delete(ev);

    mailClient.mailAttendees(currentSelection(), attachment);

#endif

#if 0
    Event *anEvent = 0;
    if (mViewManager->currentView()->isEventView()) {
        anEvent = dynamic_cast<Event *>((mViewManager->currentView()->selectedIncidences()).first());
    }

    if (!anEvent) {
        KMessageBox::sorry(this,i18n("Can't generate mail:\nNo event selected."));
        return;
    }
    if(anEvent->attendeeCount() == 0 ) {
        KMessageBox::sorry(this,
                           i18n("Can't generate mail:\nNo attendees defined.\n"));
        return;
    }

    mailobject.emailEvent(anEvent);
#endif
}


void CalendarView::schedule_publish(Incidence *incidence)
{
    Event *event = 0;
    Todo *todo = 0;

    if (incidence == 0) {
        incidence = mViewManager->currentView()->selectedIncidences().first();
        if (incidence == 0) {
            incidence = mTodoList->selectedIncidences().first();
        }
    }
    if ( incidence && incidence->type() == "Event" ) {
        event = static_cast<Event *>(incidence);
    } else {
        if ( incidence && incidence->type() == "Todo" ) {
            todo = static_cast<Todo *>(incidence);
        }
    }

    if (!event && !todo) {
        KMessageBox::sorry(this,i18n("No event selected."));
        return;
    }

    PublishDialog *publishdlg = new PublishDialog();
    if (incidence->attendeeCount()>0) {
        QPtrList<Attendee> attendees = incidence->attendees();
        attendees.first();
        while ( attendees.current()!=0 ) {
            publishdlg->addAttendee(attendees.current());
            attendees.next();
        }
    }
    bool send = true;
    if ( KOPrefs::instance()->mMailClient == KOPrefs::MailClientSendmail ) {
        if ( publishdlg->exec() != QDialog::Accepted )
            send = false;
    }
    if ( send ) {
        OutgoingDialog *dlg = mDialogManager->outgoingDialog();
        if ( event ) {
            Event *ev = new Event(*event);
            ev->registerObserver(0);
            ev->clearAttendees();
            if (!dlg->addMessage(ev,Scheduler::Publish,publishdlg->addresses())) {
                delete(ev);
            }
        } else {
            if ( todo ) {
                Todo *ev = new Todo(*todo);
                ev->registerObserver(0);
                ev->clearAttendees();
                if (!dlg->addMessage(ev,Scheduler::Publish,publishdlg->addresses())) {
                    delete(ev);
                }
            }
        }
    }
    delete publishdlg;
}

void CalendarView::schedule_request(Incidence *incidence)
{
    schedule(Scheduler::Request,incidence);
}

void CalendarView::schedule_refresh(Incidence *incidence)
{
    schedule(Scheduler::Refresh,incidence);
}

void CalendarView::schedule_cancel(Incidence *incidence)
{
    schedule(Scheduler::Cancel,incidence);
}

void CalendarView::schedule_add(Incidence *incidence)
{
    schedule(Scheduler::Add,incidence);
}

void CalendarView::schedule_reply(Incidence *incidence)
{
    schedule(Scheduler::Reply,incidence);
}

void CalendarView::schedule_counter(Incidence *incidence)
{
    schedule(Scheduler::Counter,incidence);
}

void CalendarView::schedule_declinecounter(Incidence *incidence)
{
    schedule(Scheduler::Declinecounter,incidence);
}

void CalendarView::schedule_publish_freebusy(int daysToPublish)
{
    QDateTime start = QDateTime::currentDateTime();
    QDateTime end = start.addDays(daysToPublish);

    FreeBusy *freebusy = new FreeBusy(mCalendar, start, end);
    freebusy->setOrganizer(KOPrefs::instance()->email());


    PublishDialog *publishdlg = new PublishDialog();
    if ( publishdlg->exec() == QDialog::Accepted ) {
        OutgoingDialog *dlg = mDialogManager->outgoingDialog();
        if (!dlg->addMessage(freebusy,Scheduler::Publish,publishdlg->addresses())) {
            delete(freebusy);
        }
    }
    delete publishdlg;
}

void CalendarView::schedule(Scheduler::Method method, Incidence *incidence)
{
    Event *event = 0;
    Todo *todo = 0;

    if (incidence == 0) {
        incidence = mViewManager->currentView()->selectedIncidences().first();
        if (incidence == 0) {
            incidence = mTodoList->selectedIncidences().first();
        }
    }
    if ( incidence && incidence->type() == "Event" ) {
        event = static_cast<Event *>(incidence);
    }
    if ( incidence && incidence->type() == "Todo" ) {
        todo = static_cast<Todo *>(incidence);
    }

    if (!event && !todo) {
        KMessageBox::sorry(this,i18n("No event selected."));
        return;
    }

    if( incidence->attendeeCount() == 0 && method != Scheduler::Publish ) {
        KMessageBox::sorry(this,i18n("The event has no attendees."));
        return;
    }

    Event *ev = 0;
    if (event) ev = new Event(*event);
    Todo *to = 0;
    if (todo) to = new Todo(*todo);

    if (method == Scheduler::Reply || method == Scheduler::Refresh) {
        Attendee *me = incidence->attendeeByMails(KOPrefs::instance()->mAdditionalMails,KOPrefs::instance()->email());
        if (!me) {
            KMessageBox::sorry(this,i18n("Could not find your attendee entry.\nPlease check the emails."));
            return;
        }
        if (me->status()==Attendee::NeedsAction && me->RSVP() && method==Scheduler::Reply) {
            StatusDialog *statdlg = new StatusDialog(this);
            if (!statdlg->exec()==QDialog::Accepted) return;
            me->setStatus( statdlg->status() );
            delete(statdlg);
        }
        Attendee *menew = new Attendee(*me);
        if (ev) {
            ev->clearAttendees();
            ev->addAttendee(menew,false);
        } else {
            if (to) {
                todo->clearAttendees();
                todo->addAttendee(menew,false);
            }
        }
    }

    OutgoingDialog *dlg = mDialogManager->outgoingDialog();
    if (ev) {
        if ( !dlg->addMessage(ev,method) ) delete(ev);
    } else {
        if (to) {
            if ( !dlg->addMessage(to,method) ) delete(to);
        }
    }
}

void CalendarView::openAddressbook()
{
    KRun::runCommand("kaddressbook");
}

void CalendarView::setModified(bool modified)
{
    if ( modified )
        emit signalmodified();
    if (mModified != modified) {
        mModified = modified;
        emit modifiedChanged(mModified);
    }
}

bool CalendarView::isReadOnly()
{
    return mReadOnly;
}

void CalendarView::setReadOnly(bool readOnly)
{
    if (mReadOnly != readOnly) {
        mReadOnly = readOnly;
        emit readOnlyChanged(mReadOnly);
    }
}

bool CalendarView::isModified()
{
    return mModified;
}

void CalendarView::printSetup()
{
#ifndef KORG_NOPRINTER
    createPrinter();

    mCalPrinter->setupPrinter();
#endif
}

void CalendarView::print()
{
#ifndef KORG_NOPRINTER
    createPrinter();

    DateList tmpDateList = mNavigator->selectedDates();
    mCalPrinter->print(CalPrinter::Month,
                       tmpDateList.first(), tmpDateList.last());
#endif
}

void CalendarView::printPreview()
{
#ifndef KORG_NOPRINTER
    kdDebug() << "CalendarView::printPreview()" << endl;

    createPrinter();

    DateList tmpDateList = mNavigator->selectedDates();

    mViewManager->currentView()->printPreview(mCalPrinter,tmpDateList.first(),
                                              tmpDateList.last());
#endif
}

void CalendarView::exportICalendar()
{
    QString filename = KFileDialog::getSaveFileName("icalout.ics",i18n("*.ics|ICalendars"),this);

    // Force correct extension
    if (filename.right(4) != ".ics") filename += ".ics";

    FileStorage storage( mCalendar, filename, new ICalFormat( KOPrefs::instance()->mUseQuicksave) );
    storage.save();
}

bool CalendarView::exportVCalendar( QString filename )
{
    if (mCalendar->journals().count() > 0) {
        int result = KMessageBox::warningContinueCancel(this,
                                                        i18n("The journal entries can not be\nexported to a vCalendar file."),
                                                        i18n("Data Loss Warning"),i18n("Proceed"),i18n("Cancel"),
                                                        true);
        if (result != KMessageBox::Continue) return false;
    }

    //QString filename = KFileDialog::getSaveFileName("vcalout.vcs",i18n("*.vcs|VCalendars"),this);

    // Force correct extension
    if (filename.right(4) != ".vcs") filename += ".vcs";

    FileStorage storage( mCalendar, filename, new VCalFormat );
    return storage.save();
 
}

void CalendarView::eventUpdated(Incidence *)
{
    setModified();
    // Don't call updateView here. The code, which has caused the update of the
    // event is responsible for updating the view.
    //  updateView();
}

void CalendarView::adaptNavigationUnits()
{
    if (mViewManager->currentView()->isEventView()) {
        int days = mViewManager->currentView()->currentDateCount();
        if (days == 1) {
            emit changeNavStringPrev(i18n("&Previous Day"));
            emit changeNavStringNext(i18n("&Next Day"));
        } else {
            emit changeNavStringPrev(i18n("&Previous Week"));
            emit changeNavStringNext(i18n("&Next Week"));
        }
    }
}

void CalendarView::processMainViewSelection( Incidence *incidence )
{
    if ( incidence ) mTodoList->clearSelection();
    processIncidenceSelection( incidence );
}

void CalendarView::processTodoListSelection( Incidence *incidence )
{
    if ( incidence && mViewManager->currentView() ) {
        mViewManager->currentView()->clearSelection();
    }
    processIncidenceSelection( incidence );
}

void CalendarView::processIncidenceSelection( Incidence *incidence )
{
    if ( incidence == mSelectedIncidence ) return;

    mSelectedIncidence = incidence;

    emit incidenceSelected( mSelectedIncidence );

    if ( incidence && incidence->type() == "Event" ) {
        Event *event = static_cast<Event *>( incidence );
        if ( event->organizer() == KOPrefs::instance()->email() ) {
            emit organizerEventsSelected( true );
        } else {
            emit organizerEventsSelected(false);
        }
        if (event->attendeeByMails( KOPrefs::instance()->mAdditionalMails,
                                    KOPrefs::instance()->email() ) ) {
            emit groupEventsSelected( true );
        } else {
            emit groupEventsSelected(false);
        }
        return;
    } else {
        if  ( incidence && incidence->type() == "Todo" ) {
            emit todoSelected( true );
            Todo *event = static_cast<Todo *>( incidence );
            if ( event->organizer() == KOPrefs::instance()->email() ) {
                emit organizerEventsSelected( true );
            } else {
                emit organizerEventsSelected(false);
            }
            if (event->attendeeByMails( KOPrefs::instance()->mAdditionalMails,
                                        KOPrefs::instance()->email() ) ) {
                emit groupEventsSelected( true );
            } else {
                emit groupEventsSelected(false);
            }
            return;
        } else {
            emit todoSelected( false );
            emit organizerEventsSelected(false);
            emit groupEventsSelected(false);
        }
        return;
    }

    /*  if  ( incidence && incidence->type() == "Todo" ) {
        emit todoSelected( true );
        } else {
        emit todoSelected( false );
        }*/
}


void CalendarView::checkClipboard()
{
#ifndef KORG_NODND
    if (ICalDrag::canDecode(QApplication::clipboard()->data())) {
        emit pasteEnabled(true);
    } else {
        emit pasteEnabled(false);
    }
#endif
}

void CalendarView::showDates(const DateList &selectedDates)
{
    //  kdDebug() << "CalendarView::selectDates()" << endl;
  
    if ( mViewManager->currentView() ) {
        updateView( selectedDates.first(), selectedDates.last() );
    } else {
        mViewManager->showAgendaView();
    }

    QString selDates;
    selDates = KGlobal::locale()->formatDate( selectedDates.first(), true);
    if  (selectedDates.first() < selectedDates.last() )
        selDates += " - " + KGlobal::locale()->formatDate( selectedDates.last(),true);
    topLevelWidget()->setCaption( i18n("Dates: ") + selDates );
  
}

void CalendarView::editFilters()
{
    //  kdDebug() << "CalendarView::editFilters()" << endl;

    CalFilter *filter = mFilters.first();
    while(filter) {
        kdDebug() << " Filter: " << filter->name() << endl;
        filter = mFilters.next();
    }

    mDialogManager->showFilterEditDialog(&mFilters);
}
void CalendarView::toggleFilter()
{
    showFilter(! mFilterView->isVisible());
}

void CalendarView::selectFilter( int fil )
{
    mFilterView->setSelectedFilter( fil );
}
void CalendarView::showFilter(bool visible)
{
    if (visible) mFilterView->show();
    else mFilterView->hide();
}
void CalendarView::toggleFilerEnabled( )
{
    mFilterView->setFiltersEnabled ( !mFilterView->filtersEnabled()  );
    if ( !mFilterView->filtersEnabled()   ) 
        topLevelWidget()->setCaption( i18n("Filter disabled ") );
        
}
void CalendarView::updateFilter()
{
    CalFilter *filter = mFilterView->selectedFilter();
    if (filter) {
        if (mFilterView->filtersEnabled()) {
            topLevelWidget()->setCaption( i18n("Filter selected: ")+filter->name() );
            filter->setEnabled(true);
        }
        else filter->setEnabled(false);
        mCalendar->setFilter(filter);
        updateView();
    }
}

void CalendarView::filterEdited()
{
    mFilterView->updateFilters();
    updateFilter();
    writeSettings();
}


void CalendarView::takeOverEvent()
{
    Incidence *incidence = currentSelection();

    if (!incidence) return;

    incidence->setOrganizer(KOPrefs::instance()->email());
    incidence->recreate();
    incidence->setReadOnly(false);

    updateView();
}

void CalendarView::takeOverCalendar()
{
    // TODO: Create Calendar::allIncidences() function and use it here

    QPtrList<Event> events = mCalendar->events();
    for(uint i=0; i<events.count(); ++i) {
        events.at(i)->setOrganizer(KOPrefs::instance()->email());
        events.at(i)->recreate();
        events.at(i)->setReadOnly(false);
    }

    QPtrList<Todo> todos = mCalendar->todos();
    for(uint i=0; i<todos.count(); ++i) {
        todos.at(i)->setOrganizer(KOPrefs::instance()->email());
        todos.at(i)->recreate();
        todos.at(i)->setReadOnly(false);
    }

    QPtrList<Journal> journals = mCalendar->journals();
    for(uint i=0; i<journals.count(); ++i) {
        journals.at(i)->setOrganizer(KOPrefs::instance()->email());
        journals.at(i)->recreate();
        journals.at(i)->setReadOnly(false);
    }

    updateView();
}

void CalendarView::showIntro()
{
    kdDebug() << "To be implemented." << endl;
}

QWidgetStack *CalendarView::viewStack()
{
    return mRightFrame;
}

QWidget *CalendarView::leftFrame()
{
    return mLeftFrame;
}

DateNavigator *CalendarView::dateNavigator()
{
    return mNavigator;
}

KDateNavigator* CalendarView::dateNavigatorWidget()
{
    return mDateNavigator;
}
void CalendarView::toggleDateNavigatorWidget()
{
    if (mDateNavigator->isVisible())
        mDateNavigator->hide();
    else
        mDateNavigator->show();
}
void CalendarView::addView(KOrg::BaseView *view)
{
    mViewManager->addView(view);
}

void CalendarView::showView(KOrg::BaseView *view)
{
    mViewManager->showView(view, mLeftFrame->isVisible());
}

Incidence *CalendarView::currentSelection()
{
    return mViewManager->currentSelection();
}
void CalendarView::toggleAllDaySize()
{
    if ( KOPrefs::instance()->mAllDaySize > 47 )
        KOPrefs::instance()->mAllDaySize = KOPrefs::instance()->mAllDaySize /2;
    else
        KOPrefs::instance()->mAllDaySize = KOPrefs::instance()->mAllDaySize *2;
    viewManager()->agendaView()->updateConfig();
}
void CalendarView::toggleExpand()
{
    //  if ( mLeftFrame->isHidden() ) {
    //     mLeftFrame->show(); 
    //     emit calendarViewExpanded( false );
    //   } else {
    //     mLeftFrame->hide();
    //     emit calendarViewExpanded( true );
    //   }

    globalFlagBlockAgenda = 1;
    emit calendarViewExpanded( !mLeftFrame->isHidden() );
    globalFlagBlockAgenda = 5;
    mViewManager->raiseCurrentView( !mLeftFrame->isHidden() );
    //mViewManager->showView( 0, true );
}

void CalendarView::calendarModified( bool modified, Calendar * )
{
    setModified( modified );
}

Todo *CalendarView::selectedTodo()
{
    Incidence *incidence = currentSelection();
    if ( incidence && incidence->type() == "Todo" ) {
        return static_cast<Todo *>( incidence );
    }

    incidence = mTodoList->selectedIncidences().first();
    if ( incidence && incidence->type() == "Todo" ) {
        return static_cast<Todo *>( incidence );
    }

    return 0;
}

void CalendarView::dialogClosing(Incidence *in)
{
    // mDialogList.remove(in);
}

void CalendarView::showIncidence()
{
    Incidence *incidence = currentSelection();
    if ( !incidence ) incidence = mTodoList->selectedIncidences().first();
    if ( incidence ) {
        ShowIncidenceVisitor v;
        v.act( incidence, this );
    }
}
void CalendarView::editIncidenceDescription()
{ 
    mFlagEditDescription = true;
    editIncidence();
    mFlagEditDescription = false;
}
void CalendarView::editIncidence()
{
    // qDebug("editIncidence() ");
    Incidence *incidence = currentSelection();
    if ( !incidence ) incidence = mTodoList->selectedIncidences().first();
    if ( incidence ) {
        EditIncidenceVisitor v;
        v.act( incidence, this );
    }
}

void CalendarView::deleteIncidence()
{
    Incidence *incidence = currentSelection();
    if ( !incidence ) incidence = mTodoList->selectedIncidences().first();
    if ( incidence ) {
        deleteIncidence(incidence);
    }
}

void CalendarView::showIncidence(Incidence *incidence)
{
    if ( incidence ) {
        ShowIncidenceVisitor v;
        v.act( incidence, this );
    }
}

void CalendarView::editIncidence(Incidence *incidence)
{ 
    if ( incidence ) {
       
        EditIncidenceVisitor v;
        v.act( incidence, this );
        
    }
}

void CalendarView::deleteIncidence(Incidence *incidence)
{
    //qDebug(" CalendarView::deleteIncidence ");
    if ( incidence ) {
        DeleteIncidenceVisitor v;
        v.act( incidence, this );
    }
}


void CalendarView::lookForOutgoingMessages()
{
    OutgoingDialog *ogd = mDialogManager->outgoingDialog();
    ogd->loadMessages();
}

void CalendarView::lookForIncomingMessages()
{
    IncomingDialog *icd = mDialogManager->incomingDialog();
    icd->retrieve();
}

bool  CalendarView::removeCompletedSubTodos( Todo* t )
{
    bool deleteTodo = true;
    QPtrList<Incidence> subTodos;
    Incidence *aTodo;
    subTodos = t->relations();
    for (aTodo = subTodos.first(); aTodo; aTodo = subTodos.next()) {
        if (! removeCompletedSubTodos( (Todo*) aTodo ))
            deleteTodo = false;
    }
    if ( deleteTodo ) {
        if ( t->isCompleted() ) {
            checkZaurusId(  t->zaurusId(), true );
            mCalendar->deleteTodo( t );
            changeTodoDisplay( t,KOGlobals::EVENTDELETED );
        }
        else
            deleteTodo = false;
    }
    return deleteTodo;

}
void CalendarView::purgeCompleted()
{
    int result = KMessageBox::warningContinueCancel(this,
                                                    i18n("Delete all\ncompleted To-Dos?"),i18n("Purge To-Dos"),i18n("Purge"));

    if (result == KMessageBox::Continue) {

        QPtrList<Todo> todoCal;
        QPtrList<Todo> rootTodos;
        //QPtrList<Incidence> rel;
        Todo *aTodo;//, *rTodo;
        Incidence *rIncidence;
        bool childDelete = false;
        bool deletedOne = true;
        todoCal = calendar()->todos();
        for (aTodo = todoCal.first(); aTodo; aTodo = todoCal.next()) {
            if ( !aTodo->relatedTo() )
                rootTodos.append( aTodo );
        }
        for (aTodo = rootTodos.first(); aTodo; aTodo = rootTodos.next()) {
            removeCompletedSubTodos( aTodo );
        }

        updateView();
    }
}

void CalendarView::slotCalendarChanged()
{
   ; 
}

NavigatorBar *CalendarView::navigatorBar()
{
    return mNavigatorBar;
}



void CalendarView::keyPressEvent ( QKeyEvent *e) 
{
    //qDebug(" alendarView::keyPressEvent ");
    e->ignore();
}

#include "calendarview.moc"

#include "calendarviewbase.moc"
