/*
 * linux/drivers/video/pl110fb.c
 *
 * ARM PrimeCell PL110 LCD Controller Frame Buffer Driver
 *
 * Copyright (C) 2002 Lineo, Inc.
 * Based on sa1100fb.c, which is Copyright (C) Eric A. Thomas
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive for
 * more details.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/fb.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/init.h>
#include <linux/cpufreq.h>

#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/mach-types.h>
#include <asm/uaccess.h>

#include <video/fbcon.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb16.h>

#ifdef CONFIG_ARCH_LH79520
#include <asm/arch/rcpc.h>
#include <asm/arch/cpld.h>
#include <asm/arch/iocon.h>
#endif

#ifdef CONFIG_ARCH_LH7A400
#include <asm/arch/cpld.h>
#include <asm/arch/gpio.h>
#endif

/*
 * debugging?
 */
//#define DEBUG 1
/*
 * Complain if VAR is out of range.
 */
#define DEBUG_VAR 1

#include "pl110fb.h"

extern unsigned int hclkfreq_get( void);

void (*pl110fb_blank_helper)(int blank);
EXPORT_SYMBOL(pl110fb_blank_helper);


// #define _444 1
#define _555 1

#if _444
static struct pl110fb_rgb rgb_8 = {
	red:	{ offset: 0,  length: 4, },
	green:	{ offset: 0,  length: 4, },
	blue:	{ offset: 0,  length: 4, },
	transp:	{ offset: 0,  length: 0, },
};
#elif _555
static struct pl110fb_rgb rgb_8 = {
	red:	{ offset: 0,  length: 5, },
	green:	{ offset: 0,  length: 5, },
	blue:	{ offset: 0,  length: 5, },
	transp:	{ offset: 0,  length: 0, },
};
#else
#error define _444 or _555
#endif


#if 0	// 5-6-5
static struct pl110fb_rgb def_rgb_16 = {
	red:	{ offset: 11, length: 5, },
	green:	{ offset: 5,  length: 6, },
	blue:	{ offset: 0,  length: 5, },
	transp:	{ offset: 0,  length: 0, },
};
#else // 5-5-5
static struct pl110fb_rgb def_rgb_16 = {
	red:	{ offset: 10, length: 5, },
	green:	{ offset: 5,  length: 5, },
	blue:	{ offset: 0,  length: 5, },
	transp:	{ offset: 0,  length: 0, },
};
#endif


#define VERTICAL_REFRESH	60	/* optimum refresh rate, in Hz. */


#if defined(CONFIG_ARCH_LH79520) || defined(CONFIG_ARCH_LH7A400)
static struct pl110fb_mach_info lh_info __initdata = {
#if defined(CONFIG_PL110_LQ35)
	/*
	 * LQ035Q7DB02
	 */
	bpp:		16,
	xres:		240,			yres:		320,
	hsync_len:	61, /* hsw */		vsync_len:	1,  /* vsw */
	left_margin:	22, /* hbp */		upper_margin:	7,  /* vbp */
	right_margin:	21, /* hfp */		lower_margin:	5,  /* vfp */

	sync:		FB_SYNC_VERT_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,

	LCDtiming2:	LCD_TIMING2_IPC,

	LCDtiming3:	0,

	LCDcontrol:     LCD_CTRL_TFT | LCD_CTRL_BW_COLOR | LCD_CTRL_WATERMARK,

	LCDICPcontrol:	LCDICP_CONTROL_CLSEN | LCDICP_CONTROL_SPSEN,

	LCDICPsetup:	LCDICP_SETUP_MODE_HRTFT | LCDICP_SETUP_VERT_NORMAL | LCDICP_SETUP_HORIZ_NORMAL,

	LCDICPtiming1:	LCDICP_TIMING1_PSDEL(9) | LCDICP_TIMING1_REVDEL(200) | LCDICP_TIMING1_LPDEL(14),

	LCDICPtiming2:	LCDICP_TIMING2_PSDEL2(209) | LCDICP_TIMING2_SPLVALUE(84),

#elif defined(CONFIG_PL110_LQ39)
	/*
	 * The Sharp LQ039Q2DS53 panel takes an RGB666 signal,
	 * but we provide it with an RGB555 signal instead (def_rgb_16).
	 */

	/* bpp set based on DIP switches on LCD board */
	xres:		320,			yres:		240,
	hsync_len:	13, /* hsw */		vsync_len:	2,  /* vsw */
	left_margin:	21, /* hbp */		upper_margin:	5,  /* vbp */
	right_margin:	11, /* hfp */		lower_margin:	5,  /* vfp */

	sync:		FB_SYNC_VERT_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT,

	LCDtiming2:	LCD_TIMING2_IHS | LCD_TIMING2_IPC,

	LCDtiming3:	0,

	LCDcontrol:     LCD_CTRL_TFT | LCD_CTRL_BW_COLOR | LCD_CTRL_WATERMARK,

	LCDICPcontrol:	LCDICP_CONTROL_CLSEN | LCDICP_CONTROL_SPSEN,

	LCDICPsetup:	LCDICP_SETUP_MODE_HRTFT | LCDICP_SETUP_VERT_NORMAL | LCDICP_SETUP_HORIZ_NORMAL,

	LCDICPtiming1:	LCDICP_TIMING1_PSDEL(9) | LCDICP_TIMING1_REVDEL(3) | LCDICP_TIMING1_LPDEL(14),

	LCDICPtiming2:	LCDICP_TIMING2_PSDEL2(209) | LCDICP_TIMING2_SPLVALUE(34),

#elif defined(CONFIG_PL110_LM57)
	bpp:            8,
	xres:		320,		yres:		240,
#elif defined(CONFIG_PL110_LQ57)
	bpp:            8,
	xres:		240,		yres:		320,
#elif defined(CONFIG_PL110_LQ64)
	/* LQ64D343 */
	bpp:            16,
	xres:		640,		yres:		480,

	hsync_len:	15, /* hsw */		vsync_len:	4,  /* vsw */
	left_margin:	97, /* hbp */		upper_margin:	27,  /* vbp */
	right_margin:	51, /* hfp */		lower_margin:	1,  /* vfp */

	sync:		0, /* helps build LDCtiming2 */

	LCDtiming2:	0,

	LCDtiming3:	0,

	LCDcontrol:     LCD_CTRL_TFT | LCD_CTRL_BW_COLOR | LCD_CTRL_WATERMARK,

	LCDICPcontrol:	0,

	LCDICPsetup:	LCDICP_SETUP_MODE_BYPASS,

	LCDICPtiming1:	0,

	LCDICPtiming2:	0,
#elif defined(CONFIG_PL110_LQ121)
	bpp:		4,
	xres:		320,		yres:		240,
#elif defined(CONFIG_PL110_LQ101)
#else
#error "You must have an LCD panel configured"
#endif
};
#endif // CONFIG_ARCH_LH79520 || CONFIG_ARCH_LH7A400



static struct pl110fb_mach_info * __init
pl110fb_get_machine_info(struct pl110fb_info *fbi)
{
	struct pl110fb_mach_info *inf = &lh_info;

#if defined(CONFIG_LPD_79520_10)
	inf->bpp = 16;
#endif
#if defined (CONFIG_LH79520_EVB) && defined(CONFIG_ARCH_LH79520)

	if( machine_is_lh79520evb()) {
		cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
		u8 dipSw = (u8)~(cpld->display_dip_sw & 0xff);

	    /* set bpp based on LCD board dip switch 0 */
	    inf->bpp = (dipSw & 1 ? 8 : 16);
	}
#endif
#if defined(CONFIG_ARCH_LH7A400)
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
	u8 dipSw = (u8)~(cpld->u3.dispDipSw & 0xff);

	if( machine_is_lh7a400evb()) {
	    /* set bpp based on LCD board dip switch 0 */
	    inf->bpp = (dipSw & 1 ? 8 : 16);
	}
#endif
	return inf;
}


static int pl110fb_activate_var(struct fb_var_screeninfo *var, struct pl110fb_info *);
static void set_ctrlr_state(struct pl110fb_info *fbi, u_int state);


static inline void
pl110fb_schedule_task(struct pl110fb_info *fbi, u_int state)
{
	unsigned long flags;

	local_irq_save(flags);
	/*
	 * We need to handle two requests being made at the same time.
	 * There are two important cases:
	 *  1. When we are changing VT (C_REENABLE) while unblanking (C_ENABLE)
	 *     We must perform the unblanking, which will do our REENABLE for us.
	 *  2. When we are blanking, but immediately unblank before we have
	 *     blanked.  We do the "REENABLE" thing here as well, just to be sure.
	 */
	if (fbi->task_state == C_ENABLE && state == C_REENABLE)
		state = (u_int) -1;
	if (fbi->task_state == C_DISABLE && state == C_ENABLE)
		state = C_REENABLE;

	if (state != (u_int)-1) {
		fbi->task_state = state;
		schedule_task(&fbi->task);
	}
	local_irq_restore(flags);
}

/*
 * Get the VAR structure pointer for the specified console
 */
static inline struct fb_var_screeninfo *
get_con_var(struct fb_info *info, int con)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	return (con == fbi->currcon || con == -1) ? &fbi->fb.var : &fb_display[con].var;
}

/*
 * Get the DISPLAY structure pointer for the specified console
 */
static inline struct display *
get_con_display(struct fb_info *info, int con)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	return (con < 0) ? fbi->fb.disp : &fb_display[con];
}

/*
 * Get the CMAP pointer for the specified console
 */
static inline struct fb_cmap *
get_con_cmap(struct fb_info *info, int con)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;

	return (con == fbi->currcon || con == -1) ? &fbi->fb.cmap : &fb_display[con].cmap;
}

static inline u_int
chan_to_field(u_int chan, struct fb_bitfield *bf)
{
	chan &= 0xffff;
	chan >>= 16 - bf->length;
	return chan << bf->offset;
}


static int
pl110fb_setpalettereg(u_int regno, u_int red, u_int green, u_int blue,
		       u_int trans, struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	u_int val, ret = 1;
	lcdRegs_t    *LCD = (lcdRegs_t *)IO_ADDRESS(LCD_PHYS);

	if (regno < fbi->palette_size) {

#if _444 // 4:4:4 
		val  = ((red   >> 4)  & 0x0F00) >> 7;	// bits 4:0
		val |= ((green >> 8)  & 0x00F0) << 2;	// bits 9:5
		val |= ((blue  >> 12) & 0x000F) << 11;	// bits 14:10
#else // 5:5:5
		val  = (red   & 0xF800) >> 11;		// bits 4:0
		val |= (green & 0xF800) >>  6;		// bits 9:5
		val |= (blue  & 0xF800) >>  1;		// bits 14:10
		val |= 0x8000;						// always set intensity bit
#endif

		if( regno & 1)			/* setting higher number entry */
		    LCD->palette[regno >> 1] = (LCD->palette[regno >> 1] & 0x0000ffff) | (val << 16);
		else				/* setting lower number entry	*/
		    LCD->palette[regno >> 1] = (LCD->palette[regno >> 1] & 0xffff0000) | val;

		/* */
		DPRINTK("reg=%x  r=%x  g=%x  b=%x  t=%x    addr=%p  val=%x  entry=%x\n",
			regno, red, green, blue, trans, &LCD->palette[regno >> 1], val,
			LCD->palette[regno >> 1]);
		/* */

		ret = 0;
	}
	return ret;
}

static int
pl110fb_setcolreg(u_int regno, u_int red, u_int green, u_int blue,
		   u_int trans, struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	struct display *disp = get_con_display(info, fbi->currcon);
	u_int val;
	int ret = 1;

	/*
	 * If inverse mode was selected, invert all the colours
	 * rather than the register number.  The register number
	 * is what you poke into the framebuffer to produce the
	 * colour you requested.
	 */
	if (disp->inverse) {
		red   = 0xffff - red;
		green = 0xffff - green;
		blue  = 0xffff - blue;
	}

	/*
	 * If greyscale is true, then we convert the RGB value
	 * to greyscale no mater what visual we are using.
	 */
	if (fbi->fb.var.grayscale)
		red = green = blue = (19595 * red + 38470 * green +
					7471 * blue) >> 16;

	switch (fbi->fb.disp->visual) {
	case FB_VISUAL_TRUECOLOR:
		/*
		 * 12 or 16-bit True Colour.  We encode the RGB value
		 * according to the RGB bitfield information.
		 */
		if (regno < 16) {
			u16 *pal = fbi->fb.pseudo_palette;

			val  = chan_to_field(red, &fbi->fb.var.red);
			val |= chan_to_field(green, &fbi->fb.var.green);
			val |= chan_to_field(blue, &fbi->fb.var.blue);

			pal[regno] = val;
			ret = 0;
		}
		break;

	case FB_VISUAL_STATIC_PSEUDOCOLOR:
	case FB_VISUAL_PSEUDOCOLOR:
		ret = pl110fb_setpalettereg(regno, red, green, blue, trans, info);
		break;
	}

	return ret;
}

/*
 *  pl110fb_display_dma_period()
 *    Calculate the minimum period (in picoseconds) between two DMA
 *    requests for the LCD controller.
 */
static unsigned int
pl110fb_display_dma_period(struct fb_var_screeninfo *var)
{
	unsigned int mem_bits_per_pixel;

	mem_bits_per_pixel = var->bits_per_pixel;
	if (mem_bits_per_pixel == 12)
		mem_bits_per_pixel = 16;

	/*
	 * Period = pixclock * bits_per_byte * bytes_per_transfer
	 *		/ memory_bits_per_pixel;
	 */
	return var->pixclock * 8 * 16 / mem_bits_per_pixel;
}

/*
 *  pl110fb_decode_var():
 *    Get the video params out of 'var'. If a value doesn't fit, round it up,
 *    if it's too big, return -EINVAL.
 *
 *    Suggestion: Round up in the following order: bits_per_pixel, xres,
 *    yres, xres_virtual, yres_virtual, xoffset, yoffset, grayscale,
 *    bitfields, horizontal timing, vertical timing.
 */
static int
pl110fb_validate_var(struct fb_var_screeninfo *var,
		      struct pl110fb_info *fbi)
{
	int ret = -EINVAL;

	if (var->xres < MIN_XRES)
		var->xres = MIN_XRES;
	if (var->yres < MIN_YRES)
		var->yres = MIN_YRES;
	if (var->xres > fbi->max_xres)
		var->xres = fbi->max_xres;
	if (var->yres > fbi->max_yres)
		var->yres = fbi->max_yres;
	var->xres_virtual =
	    var->xres_virtual < var->xres ? var->xres : var->xres_virtual;
	var->yres_virtual =
	    var->yres_virtual < var->yres ? var->yres : var->yres_virtual;

	DPRINTK("var->bits_per_pixel=%d\n", var->bits_per_pixel);
	switch (var->bits_per_pixel) {
#ifdef FBCON_HAS_CFB4
	case 4:		ret = 0; break;
#endif
#ifdef FBCON_HAS_CFB8
	case 8:		ret = 0; break;
#endif
#ifdef FBCON_HAS_CFB16
	case 16:	ret = 0; break;
#endif
	default:
		break;
	}

#ifdef CONFIG_CPU_FREQ
	printk(KERN_DEBUG "dma period = %d ps, clock = %d kHz\n",
		pl110fb_display_dma_period(var),
		cpufreq_get(smp_processor_id()));
#endif

	return ret;
}


static void
pl110fb_hw_set_var(struct fb_var_screeninfo *var, struct pl110fb_info *fbi)
{
	u_long palette_mem_size;

	fbi->palette_size = var->bits_per_pixel == 8 ? 256 : 16;

	palette_mem_size = fbi->palette_size * sizeof(u16);

	DPRINTK("palette_mem_size = 0x%08lx\n", (u_long) palette_mem_size);

	fb_set_cmap(&fbi->fb.cmap, 1, pl110fb_setcolreg, &fbi->fb);

	pl110fb_activate_var(var, fbi);
}

/*
 * pl110fb_set_var():
 *	Set the user defined part of the display for the specified console
 */
static int
pl110fb_set_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	struct fb_var_screeninfo *dvar = get_con_var(&fbi->fb, con);
	struct display *display = get_con_display(&fbi->fb, con);
	int err, chgvar = 0, rgbidx;

	DPRINTK("called\n");

	/*
	 * Decode var contents into a par structure, adjusting any
	 * out of range values.
	 */
	err = pl110fb_validate_var(var, fbi);
	if (err) {
	    DPRINTK( "pl110fb_validate_var returned err=%d\n", err);
	    return err;
	}

	if (var->activate & FB_ACTIVATE_TEST)
		return 0;

	if ((var->activate & FB_ACTIVATE_MASK) != FB_ACTIVATE_NOW)
		return -EINVAL;

	if (dvar->xres != var->xres)
		chgvar = 1;
	if (dvar->yres != var->yres)
		chgvar = 1;
	if (dvar->xres_virtual != var->xres_virtual)
		chgvar = 1;
	if (dvar->yres_virtual != var->yres_virtual)
		chgvar = 1;
	if (dvar->bits_per_pixel != var->bits_per_pixel)
		chgvar = 1;
	if (con < 0)
		chgvar = 0;

	switch (var->bits_per_pixel) {
#ifdef FBCON_HAS_CFB4
	case 4:
		if (fbi->cmap_static)
			display->visual	= FB_VISUAL_STATIC_PSEUDOCOLOR;
		else
			display->visual	= FB_VISUAL_PSEUDOCOLOR;
		display->line_length	= var->xres / 2;
		display->dispsw		= &fbcon_cfb4;
		rgbidx			= RGB_8;
		break;
#endif
#ifdef FBCON_HAS_CFB8
	case 8:
		if (fbi->cmap_static)
			display->visual	= FB_VISUAL_STATIC_PSEUDOCOLOR;
		else
			display->visual	= FB_VISUAL_PSEUDOCOLOR;
		display->line_length	= var->xres;
		display->dispsw		= &fbcon_cfb8;
		rgbidx			= RGB_8;
		break;
#endif
#ifdef FBCON_HAS_CFB16
	case 16:
		display->visual		= FB_VISUAL_TRUECOLOR;
		display->line_length	= var->xres * 2;
		display->dispsw		= &fbcon_cfb16;
		display->dispsw_data	= fbi->fb.pseudo_palette;
		rgbidx			= RGB_16;
		break;
#endif
	default:
		rgbidx = 0;
		display->dispsw = &fbcon_dummy;
		break;
	}

	display->screen_base	= fbi->screen_cpu;
	display->next_line	= display->line_length;
	display->type		= fbi->fb.fix.type;
	display->type_aux	= fbi->fb.fix.type_aux;
	display->ypanstep	= fbi->fb.fix.ypanstep;
	display->ywrapstep	= fbi->fb.fix.ywrapstep;
	display->can_soft_blank	= 1;
	display->inverse	= fbi->cmap_inverse;

	*dvar			= *var;
	dvar->activate		&= ~FB_ACTIVATE_ALL;

	/*
	 * Copy the RGB parameters for this display
	 * from the machine specific parameters.
	 */
	dvar->red		= fbi->rgb[rgbidx]->red;
	dvar->green		= fbi->rgb[rgbidx]->green;
	dvar->blue		= fbi->rgb[rgbidx]->blue;
	dvar->transp		= fbi->rgb[rgbidx]->transp;

	DPRINTK("RGBT length = %d:%d:%d:%d\n",
		dvar->red.length, dvar->green.length, dvar->blue.length,
		dvar->transp.length);

	DPRINTK("RGBT offset = %d:%d:%d:%d\n",
		dvar->red.offset, dvar->green.offset, dvar->blue.offset,
		dvar->transp.offset);

	/*
	 * Update the old var.  The fbcon drivers still use this.
	 * Once they are using fbi->fb.var, this can be dropped.
	 */
	display->var = *dvar;

	/*
	 * If we are setting all the virtual consoles, also set the
	 * defaults used to create new consoles.
	 */
	if (var->activate & FB_ACTIVATE_ALL)
		fbi->fb.disp->var = *dvar;

	/*
	 * If the console has changed and the console has defined
	 * a changevar function, call that function.
	 */
	if (chgvar && info && fbi->fb.changevar)
		fbi->fb.changevar(con);

	/* If the current console is selected, activate the new var. */
	if (con != fbi->currcon)
		return 0;

	pl110fb_hw_set_var(dvar, fbi);

	return 0;
}

static int
__do_set_cmap(struct fb_cmap *cmap, int kspc, int con,
	      struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	struct fb_cmap *dcmap = get_con_cmap(info, con);
	int err = 0;

	if (con == -1)
		con = fbi->currcon;

	/* no colormap allocated? (we always have "this" colour map allocated) */
	if (con >= 0)
		err = fb_alloc_cmap(&fb_display[con].cmap, fbi->palette_size, 0);

	if (!err && con == fbi->currcon)
		err = fb_set_cmap(cmap, kspc, pl110fb_setcolreg, info);

	if (!err)
		fb_copy_cmap(cmap, dcmap, kspc ? 0 : 1);

	return err;
}

static int
pl110fb_set_cmap(struct fb_cmap *cmap, int kspc, int con,
		  struct fb_info *info)
{
	struct display *disp = get_con_display(info, con);

	if (disp->visual == FB_VISUAL_TRUECOLOR ||
	    disp->visual == FB_VISUAL_STATIC_PSEUDOCOLOR)
		return -EINVAL;

	return __do_set_cmap(cmap, kspc, con, info);
}

static int
pl110fb_get_fix(struct fb_fix_screeninfo *fix, int con, struct fb_info *info)
{
	struct display *display = get_con_display(info, con);

	*fix = info->fix;

	fix->line_length = display->line_length;
	fix->visual	 = display->visual;
	return 0;
}

static int
pl110fb_get_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
	*var = *get_con_var(info, con);
	return 0;
}

static int
pl110fb_get_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
	struct fb_cmap *dcmap = get_con_cmap(info, con);
	fb_copy_cmap(dcmap, cmap, kspc ? 0 : 2);
	return 0;
}

static struct fb_ops pl110fb_ops = {
	owner:		THIS_MODULE,
	fb_get_fix:	pl110fb_get_fix,
	fb_get_var:	pl110fb_get_var,
	fb_set_var:	pl110fb_set_var,
	fb_get_cmap:	pl110fb_get_cmap,
	fb_set_cmap:	pl110fb_set_cmap,
};

/*
 *  pl110fb_switch():       
 *	Change to the specified console.  Palette and video mode
 *      are changed to the console's stored parameters.
 *
 *	Uh oh, this can be called from a tasklet (IRQ)
 */
static int
pl110fb_switch(int con, struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	struct display *disp;
	struct fb_cmap *cmap;

	DPRINTK("con=%d info->modename=%s\n", con, fbi->fb.modename);

	if (con == fbi->currcon)
		return 0;

	if (fbi->currcon >= 0) {
		disp = fb_display + fbi->currcon;

		/*
		 * Save the old colormap and video mode.
		 */
		disp->var = fbi->fb.var;

		if (disp->cmap.len)
			fb_copy_cmap(&fbi->fb.cmap, &disp->cmap, 0);
	}

	fbi->currcon = con;
	disp = fb_display + con;

	/*
	 * Make sure that our colourmap contains 256 entries.
	 */
	fb_alloc_cmap(&fbi->fb.cmap, 256, 0);

	if (disp->cmap.len)
		cmap = &disp->cmap;
	else
		cmap = fb_default_cmap(1 << disp->var.bits_per_pixel);

	fb_copy_cmap(cmap, &fbi->fb.cmap, 0);

	fbi->fb.var = disp->var;
	fbi->fb.var.activate = FB_ACTIVATE_NOW;

	pl110fb_set_var(&fbi->fb.var, con, info);
	return 0;
}

/*
 * Formal definition of the VESA spec:
 *  On
 *  	This refers to the state of the display when it is in full operation
 *  Stand-By
 *  	This defines an optional operating state of minimal power reduction with
 *  	the shortest recovery time
 *  Suspend
 *  	This refers to a level of power management in which substantial power
 *  	reduction is achieved by the display.  The display can have a longer 
 *  	recovery time from this state than from the Stand-by state
 *  Off
 *  	This indicates that the display is consuming the lowest level of power
 *  	and is non-operational. Recovery from this state may optionally require
 *  	the user to manually power on the monitor
 *
 *  Now, the fbdev driver adds an additional state, (blank), where they
 *  turn off the video (maybe by colormap tricks), but don't mess with the
 *  video itself: think of it semantically between on and Stand-By.
 *
 *  So here's what we should do in our fbdev blank routine:
 *
 *  	VESA_NO_BLANKING (mode 0)	Video on,  front/back light on
 *  	VESA_VSYNC_SUSPEND (mode 1)  	Video on,  front/back light off
 *  	VESA_HSYNC_SUSPEND (mode 2)  	Video on,  front/back light off
 *  	VESA_POWERDOWN (mode 3)		Video off, front/back light off
 *
 *  This will match the matrox implementation.
 */
/*
 * pl110fb_blank():
 *	Blank the display by setting all palette values to zero.  Note, the 
 * 	12 and 16 bpp modes don't really use the palette, so this will not
 *      blank the display in all modes.  
 */
static void
pl110fb_blank(int blank, struct fb_info *info)
{
	struct pl110fb_info *fbi = (struct pl110fb_info *)info;
	int i;

	DPRINTK("pl110fb_blank: blank=%d info->modename=%s\n", blank,
		fbi->fb.modename);

	switch (blank) {
	case VESA_POWERDOWN:
	case VESA_VSYNC_SUSPEND:
	case VESA_HSYNC_SUSPEND:
		if (fbi->fb.disp->visual == FB_VISUAL_PSEUDOCOLOR ||
		    fbi->fb.disp->visual == FB_VISUAL_STATIC_PSEUDOCOLOR)
			for (i = 0; i < fbi->palette_size; i++)
				pl110fb_setpalettereg(i, 0, 0, 0, 0, info);
		pl110fb_schedule_task(fbi, C_DISABLE);
		if (pl110fb_blank_helper)
			pl110fb_blank_helper(blank);
		break;

	case VESA_NO_BLANKING:
		if (pl110fb_blank_helper)
			pl110fb_blank_helper(blank);
		if (fbi->fb.disp->visual == FB_VISUAL_PSEUDOCOLOR ||
		    fbi->fb.disp->visual == FB_VISUAL_STATIC_PSEUDOCOLOR)
			fb_set_cmap(&fbi->fb.cmap, 1, pl110fb_setcolreg, info);
		pl110fb_schedule_task(fbi, C_ENABLE);
	}
}

static int
pl110fb_updatevar(int con, struct fb_info *info)
{
	DPRINTK("entered\n");
	return 0;
}

/*
 * Calculate the PCD value from the clock rate (in picoseconds).
 * We take account of the PPCR clock setting.
 */
static inline int
get_pcd(unsigned int pixclock)
{
	unsigned int pcd;

	if (pixclock) {
		pcd = hclkfreq_get() / 100000;
		pcd *= pixclock;
		pcd /= 10000000;
		pcd += 1;	/* make up for integer math truncations */
	} else {
		/*
		 * People seem to be missing this message.  Make it big.
		 * Make it stand out.  Make sure people see it.
		 */
		printk(KERN_WARNING "******************************************************\n");
		printk(KERN_WARNING "**            ZERO PIXEL CLOCK DETECTED             **\n");
		printk(KERN_WARNING "** You are using a zero pixclock.  This means that  **\n");
		printk(KERN_WARNING "** clock scaling will not be able to adjust your    **\n");
		printk(KERN_WARNING "** your timing parameters appropriately, and the    **\n");
		printk(KERN_WARNING "** bandwidth calculations will fail to work.  This  **\n");
		printk(KERN_WARNING "** will shortly become an error condition, which    **\n");
		printk(KERN_WARNING "** will prevent your LCD display working.  Please   **\n");
		printk(KERN_WARNING "** send your patches in as soon as possible to shut **\n");
		printk(KERN_WARNING "** this message up.                                 **\n");
		printk(KERN_WARNING "******************************************************\n");
		pcd = 0;
	}

	DPRINTK( "pcd=%d\n", pcd);

	return pcd;
}

/*
 * pl110fb_activate_var():
 *	Configures LCD Controller based on entries in var parameter.  Settings are      
 *	only written to the controller if changes were made.  
 */
static int
pl110fb_activate_var(struct fb_var_screeninfo *var, struct pl110fb_info *fbi)
{
	struct pl110fb_lcd_reg new_regs;
//	u_int half_screen_size;
	u_int yres, pcd = get_pcd( var->pixclock);
	u_long flags;
	lcdRegs_t    *LCD = (lcdRegs_t *)IO_ADDRESS(LCD_PHYS);
	lcdicpRegs_t *LCDICP = (lcdicpRegs_t *)IO_ADDRESS(LCDICP_PHYS);

	DPRINTK("Configuring pl110 LCD\n");
	DPRINTK( "LCD=%p   LCDICP=%p\n", LCD, LCDICP);

	DPRINTK("var: xres=%d hslen=%d lm=%d rm=%d\n",
		var->xres, var->hsync_len,
		var->left_margin, var->right_margin);
	DPRINTK("var: yres=%d vslen=%d um=%d bm=%d\n",
		var->yres, var->vsync_len,
		var->upper_margin, var->lower_margin);

#if DEBUG_VAR
	if (var->xres < 16        || var->xres > 1024)
		printk(KERN_ERR "%s: invalid xres %d\n",
			fbi->fb.fix.id, var->xres);
	if (var->hsync_len < 1    || var->hsync_len > 64)
		printk(KERN_ERR "%s: invalid hsync_len %d\n",
			fbi->fb.fix.id, var->hsync_len);
	if (var->left_margin < 1  || var->left_margin > 255)
		printk(KERN_ERR "%s: invalid left_margin %d\n",
			fbi->fb.fix.id, var->left_margin);
	if (var->right_margin < 1 || var->right_margin > 255)
		printk(KERN_ERR "%s: invalid right_margin %d\n",
			fbi->fb.fix.id, var->right_margin);
	if (var->yres < 1         || var->yres > 1024)
		printk(KERN_ERR "%s: invalid yres %d\n",
			fbi->fb.fix.id, var->yres);
	if (var->vsync_len < 1    || var->vsync_len > 64)
		printk(KERN_ERR "%s: invalid vsync_len %d\n",
			fbi->fb.fix.id, var->vsync_len);
	if (var->upper_margin < 0 || var->upper_margin > 255)
		printk(KERN_ERR "%s: invalid upper_margin %d\n",
			fbi->fb.fix.id, var->upper_margin);
	if (var->lower_margin < 0 || var->lower_margin > 255)
		printk(KERN_ERR "%s: invalid lower_margin %d\n",
			fbi->fb.fix.id, var->lower_margin);
#endif
	if( var->bits_per_pixel == 8) {
	    new_regs.LCDcontrol = LCD_CTRL_BPP8;
	 } else if( var->bits_per_pixel == 16) {
	    new_regs.LCDcontrol = LCD_CTRL_BPP16 | LCD_CTRL_BGR;
	}

	new_regs.LCDcontrol |= fbi->LCDcontrol | LCD_CTRL_ENABLE;
	    
	new_regs.LCDtiming0 =
	    LCD_TIMING0_HFP(var->right_margin) +
	    LCD_TIMING0_HBP(var->left_margin) +
	    LCD_TIMING0_HSW(var->hsync_len) +
	    LCD_TIMING0_PPL(var->xres);

	/*
	 * If we have a dual scan LCD, then we need to halve
	 * the YRES parameter.
	 */
	yres = var->yres;
	if( fbi->LCDcontrol & LCD_CTRL_DUAL)
		yres /= 2;

	new_regs.LCDtiming1 =
		LCD_TIMING1_VBP(var->upper_margin) +
		LCD_TIMING1_VFP(var->lower_margin) +
		LCD_TIMING1_VSW(var->vsync_len) +
		LCD_TIMING1_LPP(yres);

	new_regs.LCDtiming2 = ~0x1 & (fbi->LCDtiming2 | 
		LCD_TIMING2_CPL(var->xres) |
		LCD_TIMING2_PCD(pcd) |
		(var->sync & FB_SYNC_HOR_HIGH_ACT  ? 0 : LCD_TIMING2_IHS) |
		(var->sync & FB_SYNC_VERT_HIGH_ACT ? 0 : LCD_TIMING2_IVS));

	new_regs.LCDtiming3    = fbi->LCDtiming3;
	new_regs.LCDICPsetup   = fbi->LCDICPsetup | LCDICP_SETUP_PPL(var->xres);
#if defined(CONFIG_ARCH_LH7A400)
	new_regs.LCDICPsetup   |= LCDICP_SETUP_POWER;
#endif

	new_regs.LCDICPcontrol = fbi->LCDICPcontrol;
	new_regs.LCDICPtiming1 = fbi->LCDICPtiming1;
	new_regs.LCDICPtiming2 = fbi->LCDICPtiming2;

	DPRINTK("new LCDtiming0    = 0x%08x\n", (u32)new_regs.LCDtiming0);
	DPRINTK("new LCDtiming1    = 0x%08x\n", (u32)new_regs.LCDtiming1);
	DPRINTK("new LCDtiming2    = 0x%08x\n", (u32)new_regs.LCDtiming2);
	DPRINTK("new LCDtiming3    = 0x%08x\n", (u32)new_regs.LCDtiming3);
	DPRINTK("new LCDcontrol    = 0x%08x\n", (u32)new_regs.LCDcontrol);
	DPRINTK("new LCDICPsetup   = 0x%08x\n", (u32)new_regs.LCDICPsetup);
	DPRINTK("new LCDICPcontrol = 0x%08x\n", (u32)new_regs.LCDICPcontrol);
	DPRINTK("new LCDICPtiming1 = 0x%08x\n", (u32)new_regs.LCDICPtiming1);
	DPRINTK("new LCDICPtiming2 = 0x%08x\n", (u32)new_regs.LCDICPtiming2);


	/* Update shadow copy atomically */
	local_irq_save(flags);
	fbi->upbase = fbi->screen_dma;

	fbi->reg_LCDtiming0    = new_regs.LCDtiming0;
	fbi->reg_LCDtiming1    = new_regs.LCDtiming1;
	fbi->reg_LCDtiming2    = new_regs.LCDtiming2;
	fbi->reg_LCDtiming3    = new_regs.LCDtiming3;
	fbi->reg_LCDcontrol    = new_regs.LCDcontrol;
	fbi->reg_LCDICPsetup   = new_regs.LCDICPsetup;
	fbi->reg_LCDICPcontrol = new_regs.LCDICPcontrol;
	fbi->reg_LCDICPtiming1 = new_regs.LCDICPtiming1;
	fbi->reg_LCDICPtiming2 = new_regs.LCDICPtiming2;
	local_irq_restore(flags);

	/*
	 * Only update the registers if the controller is enabled
	 * and something has changed.
	 */
	if ((LCD->timing0 != fbi->reg_LCDtiming0)       || (LCD->timing1 != fbi->reg_LCDtiming1) ||
	    (LCD->timing2 != fbi->reg_LCDtiming2)       || (LCD->timing3 != fbi->reg_LCDtiming2) ||
	    (LCD->control != fbi->reg_LCDcontrol)       || (LCDICP->setup != fbi->reg_LCDICPsetup) ||
	    (LCDICP->timing1 != fbi->reg_LCDICPtiming1) || (LCDICP->timing2 != fbi->reg_LCDICPtiming2) ||
	    (LCD->upbase != fbi->upbase)                || (LCD->lpbase != fbi->lpbase))
		pl110fb_schedule_task(fbi, C_REENABLE);

	return 0;
}

/*
 * NOTE!  The following functions are purely helpers for set_ctrlr_state.
 * Do not call them directly; set_ctrlr_state does the correct serialisation
 * to ensure that things happen in the right way 100% of time time.
 *	-- rmk
 */

static void
pl110fb_backlight_on(struct pl110fb_info *fbi)
{
    DPRINTK("backlight on\n");

#if defined (CONFIG_LH79520_EVB) && defined(CONFIG_ARCH_LH79520)
    if( machine_is_lh79520evb()) {
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	cpld->lcd_pwr_cntl |= CPLD_BACKLIGHT_ON;
    }
#endif

#ifdef CONFIG_ARCH_LH7A400
    if( machine_is_lh7a400evb()) {
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	cpld->lcd_pwr_cntl |= CPLD_BACKLIGHT_ON;
    }
#endif
}

static void
pl110fb_backlight_off(struct pl110fb_info *fbi)
{
    DPRINTK("backlight off\n");

#if defined (CONFIG_LH79520_EVB) && defined(CONFIG_ARCH_LH79520)
    if( machine_is_lh79520evb()) {
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	cpld->lcd_pwr_cntl &= ~CPLD_BACKLIGHT_ON;
    }
#endif
#ifdef CONFIG_ARCH_LH7A400
    if( machine_is_lh7a400evb()) {
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	cpld->lcd_pwr_cntl &= ~CPLD_BACKLIGHT_ON;
    }
#endif
}

static void
pl110fb_power_up_lcd(struct pl110fb_info *fbi)
{
	DPRINTK("LCD power on\n");

#if defined (CONFIG_LH79520_EVB) && defined(CONFIG_ARCH_LH79520)
	if( machine_is_lh79520evb()) {
	    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	    cpld->lcd_pwr_cntl |= CPLD_LCDP_EN;
	}
#endif
#ifdef CONFIG_ARCH_LH7A400
	if( machine_is_lh7a400evb()) {
	    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	    cpld->lcd_pwr_cntl |= CPLD_LCDP_EN;
	}
#endif
}

static void
pl110fb_power_down_lcd(struct pl110fb_info *fbi)
{
	DPRINTK("LCD power off\n");
#if defined (CONFIG_LH79520_EVB) && defined(CONFIG_ARCH_LH79520)
	if( machine_is_lh79520evb()) {
	    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	    cpld->lcd_pwr_cntl &= ~CPLD_LCDP_EN;
	}
#endif
#ifdef CONFIG_ARCH_LH7A400
	if( machine_is_lh7a400evb()) {
	    cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;

	    cpld->lcd_pwr_cntl &= ~CPLD_LCDP_EN;
	}
#endif
}


static void
pl110fb_setup_hw(struct pl110fb_info *fbi)
{
#ifdef CONFIG_ARCH_LH79520
    rcpcRegs_t  *rcpc  = (rcpcRegs_t *)IO_ADDRESS( RCPC_PHYS);
    ioconRegs_t *iocon = (ioconRegs_t *)IO_ADDRESS( IOCON_PHYS);

    rcpc->control       |= RCPC_CTRL_WRTLOCK_ENABLED;		/* unlock RCPC registers */
    barrier();

    /*
     * use HCLK for the LCD clock.
     */
    rcpc->spareClkCtrl   &= ~RCPC_SPARE_CLKCTRL_LCDCLK_DISABLE; /* enable LCDCLK	*/
    rcpc->spareClkSel    &= ~RCPC_SCLKSEL_LCDCLK;		/* LCDCLK from HCLK	*/
    rcpc->spare0Prescale &= 0;
    rcpc->AHBClkCtrl     &= ~RCPC_CLKCTRL_DMAC_DISABLE;		/* ensure DMA gets a clock */
    rcpc->control        &= ~RCPC_CTRL_WRTLOCK_ENABLED;		/* lock RCPC registers	*/
    
    /* set the pin mux to enable all required LCD signals and disable the rest */
	if (LCD_CTRL_TFT == (fbi->reg_LCDcontrol & LCD_CTRL_TFT)) {
		if (0 == (LCDICP_SETUP_MODE_HRTFT & fbi->reg_LCDICPsetup)) {
			/* tft */
			iocon->LCDMux = LCDMUX_CLD12   | /* Table 11-5: not used by TFT */
				LCDMUX_CLD13   | /* blue[0] */
				LCDMUX_CLD14   | /* blue[1] */
				LCDMUX_CLD15   | /* blue[2] */
				LCDMUX_CLDEN   | /* LCD Enable */
				LCDMUX_CLVDDEN | /* CLVDDEN */
				LCDMUX_PIOC2   | /* don't need external LCD clock */
				LCDMUX_CLCP    | /* DCLK */
				LCDMUX_CLD16   | /* blue[3] */
				LCDMUX_CLLP    | /* LLP (Table 11-5 bruce) */
				LCDMUX_CLD17   | /* blue[4] */
				LCDMUX_CLFP    | /* FP */
				LCDMUX_CLD2    | /* red[1] */
				LCDMUX_CLD3    | /* red[2] */
				LCDMUX_CLD4    | /* red[3] */
				LCDMUX_CLD5    | /* red[4] */
				LCDMUX_CLD6    | /* Table 11-5: not used by TFT */
				LCDMUX_CLD7    | /* green[0] */
				LCDMUX_CLD8    | /* green[1] */
				LCDMUX_CLD9    | /* green[2] */
				LCDMUX_CLD10   | /* green[3] */
				LCDMUX_CLD11;    /* green[4] */
		} else {
			/* hrtft */
			iocon->LCDMux = LCDMUX_CLREV 
				| LCDMUX_CLXCLK	/* what!!!??? */
				| LCDMUX_PIOC2		/* don't need external LCD clock */
				| LCDMUX_CLD13		/* blue[0] */ 
				| LCDMUX_CLD14		/* blue[1] */
				| LCDMUX_CLD15		/* blue[2] */
				| LCDMUX_CLD16		/* blue[3] */
				| LCDMUX_CLD17		/* blue[4] */
				| LCDMUX_CLSPL		/* SPL */
				| LCDMUX_CLS		/* CLS */
				| LCDMUX_CLCP		/* DCLK */
				| LCDMUX_CLP		/* LP */
				| LCDMUX_CLSPS		/* SPS */
				| LCDMUX_CLD2		/* red[1] */
				| LCDMUX_CLD3		/* red[2] */
				| LCDMUX_CLD4		/* red[3] */
				| LCDMUX_CLD5		/* red[4] */
				| LCDMUX_CPS		/* PS */
				| LCDMUX_CLD7		/* green[0] */
				| LCDMUX_CLD8		/* green[1] */
				| LCDMUX_CLD9		/* green[2] */
				| LCDMUX_CLD10		/* green[3] */
				| LCDMUX_CLD11;		/* green[4] */
		}
	} else {
		/* stn */
		printk("STN Pinout not yet implemented\n");
	}

    DPRINTK( "IOCON->LCDMux=%x\n", iocon->LCDMux);
#endif
#ifdef CONFIG_ARCH_LH7A400
    gpioRegs_t *gpio = (gpioRegs_t *)IO_ADDRESS(GPIO_PHYS);

    gpio->pinmux |= (GPIO_PINMUX_PEOCON | GPIO_PINMUX_PDOCON);	/* route LCD data bits	*/

    DPRINTK( "gpio: pinmux=%x  pdddr=0x%x  peddr=0x%x\n",
			 gpio->pinmux, gpio->pdddr, gpio->peddr);	// DDD
#endif
}

static void
pl110fb_enable_controller(struct pl110fb_info *fbi)
{
	lcdRegs_t *LCD = (lcdRegs_t *)IO_ADDRESS(LCD_PHYS);
	lcdicpRegs_t *LCDICP = (lcdicpRegs_t *)IO_ADDRESS(LCDICP_PHYS);

#if defined(CONFIG_LH79520_EVB) || defined(CONFIG_ARCH_LH7A400)
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
#endif
	DPRINTK("Enabling LCD controller\n");

	LCDICP->control = fbi->reg_LCDICPcontrol;
	LCDICP->setup   = fbi->reg_LCDICPsetup;
	LCDICP->timing1 = fbi->reg_LCDICPtiming1;
	LCDICP->timing2 = fbi->reg_LCDICPtiming2;
	LCD->timing0    = fbi->reg_LCDtiming0;
	LCD->timing1    = fbi->reg_LCDtiming1;
	LCD->timing2    = fbi->reg_LCDtiming2;
	LCD->timing3    = fbi->reg_LCDtiming3;
	LCD->upbase     = fbi->upbase;
#if defined(CONFIG_ARCH_LH7A400)
	LCD->lpoverflow = fbi->upbase;
#endif
	LCD->intrEnable = 0;
	LCD->control    = fbi->reg_LCDcontrol;

	/* 
	 * enable lcd output
	 */
#if defined(CONFIG_LH79520_EVB) || defined(CONFIG_ARCH_LH7A400)
	cpld->lcd_pwr_cntl |= CPLD_LCD_OE;
#endif

	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(20 * HZ / 1000);

	LCD->control |= LCD_CTRL_PWR;


#if defined(CONFIG_ARCH_LH7A400)
	DPRINTK("real LCDoverflow   = %p\n",     (void *)LCD->lpoverflow);
#endif
	DPRINTK("real LCDupbase     = %p\n",     (void *)LCD->upbase);
	DPRINTK("real LCDlpbase     = %p\n",     (void *)LCD->lpbase);
	DPRINTK("real LCDtiming0    = 0x%08x\n", LCD->timing0);
	DPRINTK("real LCDtiming1    = 0x%08x\n", LCD->timing1);
	DPRINTK("real LCDtiming2    = 0x%08x\n", LCD->timing2);
	DPRINTK("real LCDtiming3    = 0x%08x\n", LCD->timing3);
	DPRINTK("real LCDcontrol    = 0x%08x\n", LCD->control);
	DPRINTK("real LCDICPsetup   = 0x%08x\n", LCDICP->setup);
	DPRINTK("real LCDICPcontrol = 0x%08x\n", LCDICP->control);
	DPRINTK("real LCDICPtiming1 = 0x%08x\n", LCDICP->timing1);
	DPRINTK("real LCDICPtiming2 = 0x%08x\n", LCDICP->timing2);
}


static void
pl110fb_disable_controller(struct pl110fb_info *fbi)
{
	lcdRegs_t *LCD = (lcdRegs_t *)IO_ADDRESS(LCD_PHYS);
	lcdicpRegs_t *LCDICP = (lcdicpRegs_t *)IO_ADDRESS(LCDICP_PHYS);

#if defined(CONFIG_LH79520_EVB) || defined(CONFIG_ARCH_LH7A400)
	cpldRegs_t *cpld = (cpldRegs_t *)CPLD_BASE;
#endif

	DECLARE_WAITQUEUE(wait, current);

	DPRINTK("Disabling LCD controller\n");

	add_wait_queue(&fbi->ctrlr_wait, &wait);
	set_current_state(TASK_UNINTERRUPTIBLE);

	LCD->intrEnable |= LCD_STATUS_VCOMP;	/* allow VCOMP interrupts */

	/* turn off LCD power, wait a bit, then disable the controller */
	LCD->control    &= ~LCD_CTRL_PWR;

	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(20 * HZ / 1000);

	LCD->control = 0;
	LCDICP->control = 0;

#if defined(CONFIG_LH79520_EVB) || defined(CONFIG_ARCH_LH7A400)
	cpld->lcd_pwr_cntl &= ~CPLD_LCD_OE;
#endif

	schedule_timeout(20 * HZ / 1000);
	current->state = TASK_RUNNING;
	remove_wait_queue(&fbi->ctrlr_wait, &wait);
}


/*
 *  pl110fb_handle_irq: Handle 'LCD DONE' interrupts.
 */
static void
pl110fb_handle_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct pl110fb_info *fbi = dev_id;
	lcdRegs_t *LCD = (lcdRegs_t *)IO_ADDRESS(LCD_PHYS);

	unsigned int intr = LCD->maskedIntrStatus;

	DPRINTK("IRQ: status=0x%x\n", intr);

	if( intr & LCD_STATUS_VCOMP) {		/* vertical compare interrupt	*/
		LCD->intrEnable = 0;		/* only want one interrupt	*/
		wake_up(&fbi->ctrlr_wait);
	}

	LCD->rawIntrStatus = 0;			/* clear interrupt		*/
}

/*
 * This function must be called from task context only, since it will
 * sleep when disabling the LCD controller, or if we get two contending
 * processes trying to alter state.
 */
static void
set_ctrlr_state(struct pl110fb_info *fbi, u_int state)
{
	u_int old_state;

	down(&fbi->ctrlr_sem);

	old_state = fbi->state;

	switch (state) {
	case C_DISABLE_CLKCHANGE:
		/*
		 * Disable controller for clock change.  If the
		 * controller is already disabled, then do nothing.
		 */
		if (old_state != C_DISABLE && old_state != C_DISABLE_PM) {
			fbi->state = state;
			pl110fb_disable_controller(fbi);
		}
		break;

	case C_DISABLE_PM:
	case C_DISABLE:
		/*
		 * Disable controller
		 */
		if (old_state != C_DISABLE) {
			fbi->state = state;

			pl110fb_backlight_off(fbi);
			if (old_state != C_DISABLE_CLKCHANGE)
				pl110fb_disable_controller(fbi);
			pl110fb_power_down_lcd(fbi);
		}
		break;

	case C_ENABLE_CLKCHANGE:
		/*
		 * Enable the controller after clock change.  Only
		 * do this if we were disabled for the clock change.
		 */
		if (old_state == C_DISABLE_CLKCHANGE) {
			fbi->state = C_ENABLE;
			pl110fb_enable_controller(fbi);
		}
		break;

	case C_REENABLE:
		/*
		 * Re-enable the controller only if it was already
		 * enabled.  This is so we reprogram the control
		 * registers.
		 */
		if (old_state == C_ENABLE) {
			pl110fb_disable_controller(fbi);
			pl110fb_setup_hw(fbi);
			pl110fb_enable_controller(fbi);
		}
		break;

	case C_ENABLE_PM:
		/*
		 * Re-enable the controller after PM.  This is not
		 * perfect - think about the case where we were doing
		 * a clock change, and we suspended half-way through.
		 */
		if (old_state != C_DISABLE_PM)
			break;
		/* fall through */

	case C_ENABLE:
		/*
		 * Power up the LCD screen, enable controller, and
		 * turn on the backlight.
		 */
		if (old_state != C_ENABLE) {
			fbi->state = C_ENABLE;
			pl110fb_setup_hw(fbi);
			pl110fb_power_up_lcd(fbi);
			pl110fb_enable_controller(fbi);
			pl110fb_backlight_on(fbi);
		}
		break;
	}
	up(&fbi->ctrlr_sem);
}


/*
 * Our LCD controller task (which is called when we blank or unblank)
 * via keventd.
 */
static void
pl110fb_task(void *dummy)
{
	struct pl110fb_info *fbi = dummy;
	u_int state = xchg(&fbi->task_state, -1);

	set_ctrlr_state(fbi, state);
}

#ifdef CONFIG_CPU_FREQ
/*
 * Calculate the minimum DMA period over all displays that we own.
 * This, together with the SDRAM bandwidth defines the slowest CPU
 * frequency that can be selected.
 */
static unsigned int
pl110fb_min_dma_period(struct pl110fb_info *fbi)
{
	unsigned int min_period = (unsigned int)-1;
	int i;

	for (i = 0; i < MAX_NR_CONSOLES; i++) {
		unsigned int period;

		/*
		 * Do we own this display?
		 */
		if (fb_display[i].fb_info != &fbi->fb)
			continue;

		/*
		 * Ok, calculate its DMA period
		 */
		period = pl110fb_display_dma_period(get_con_var(&fbi->fb, i));
		if (period < min_period)
			min_period = period;
	}

	return min_period;
}

/*
 * CPU clock speed change handler.  We need to adjust the LCD timing
 * parameters when the CPU clock is adjusted by the power management
 * subsystem.
 */
static int
pl110fb_clkchg_notifier(struct notifier_block *nb, unsigned long val,
			 void *data)
{
	struct pl110fb_info *fbi = TO_INF(nb, clockchg);
	struct cpufreq_minmax *mm = data;
	u_int pcd;

	switch (val) {
	case CPUFREQ_MINMAX:
		printk(KERN_DEBUG "min dma period: %d ps, old clock %d kHz, "
			"new clock %d kHz\n", pl110fb_min_dma_period(fbi),
			mm->cur_freq, mm->new_freq);
		/* todo: fill in min/max values */
		break;

	case CPUFREQ_PRECHANGE:
		set_ctrlr_state(fbi, C_DISABLE_CLKCHANGE);
		break;

	case CPUFREQ_POSTCHANGE:
		pcd = get_pcd(fbi->fb.var.pixclock);
		fbi->reg_lccr3 = (fbi->reg_lccr3 & ~0xff) | LCCR3_PixClkDiv(pcd);	// DDD
		set_ctrlr_state(fbi, C_ENABLE_CLKCHANGE);
		break;
	}
	return 0;
}
#endif

#ifdef CONFIG_PM
/*
 * Power management hook.  Note that we won't be called from IRQ context,
 * unlike the blank functions above, so we may sleep.
 */
static int
pl110fb_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
	struct pl110fb_info *fbi = pm_dev->data;

	DPRINTK("pm_callback: %d\n", req);

	if (req == PM_SUSPEND || req == PM_RESUME) {
		int state = (int)data;

		if (state == 0) {
			/* Enter D0. */
			set_ctrlr_state(fbi, C_ENABLE_PM);
		} else {
			/* Enter D1-D3.  Disable the LCD controller.  */
			set_ctrlr_state(fbi, C_DISABLE_PM);
		}
	}
	DPRINTK("done\n");
	return 0;
}
#endif

/*
 * pl110fb_map_video_memory():
 *      Allocates the DRAM memory for the frame buffer.  This buffer is  
 *	remapped into a non-cached, non-buffered, memory region to  
 *      allow palette and pixel writes to occur without flushing the 
 *      cache.  Once this area is remapped, all virtual memory
 *      access to the video memory should occur at the new region.
 */
static int __init
pl110fb_map_video_memory(struct pl110fb_info *fbi)
{
	/*
	 * We reserve one page for the palette, plus the size
	 * of the framebuffer.
	 */
	fbi->map_size = PAGE_ALIGN(fbi->fb.fix.smem_len);
	fbi->map_cpu = consistent_alloc(GFP_KERNEL, fbi->map_size,
					&fbi->map_dma);

	if (fbi->map_cpu) {
		fbi->screen_cpu = fbi->map_cpu;
		fbi->screen_dma = fbi->map_dma;
		fbi->fb.fix.smem_start = fbi->screen_dma;
	}

	DPRINTK( "fix.smem_len = %d  map_cpu = 0x%x    screen_cpu = 0x%x     screen_dma = 0x%x\n",
		fbi->fb.fix.smem_len, fbi->map_cpu, fbi->screen_cpu, fbi->screen_dma);

	return fbi->map_cpu ? 0 : -ENOMEM;
}

/* Fake monspecs to fill in fbinfo structure */
static struct fb_monspecs monspecs __initdata = {
	30000, 70000, 50, 65, 0	/* Generic */
};


static struct pl110fb_info * __init
pl110fb_init_fbinfo(void)
{
	struct pl110fb_mach_info *inf;
	struct pl110fb_info *fbi;
	int pixelsPerSecond;

	fbi = kmalloc(sizeof(struct pl110fb_info) + sizeof(struct display) +
		      sizeof(u16) * 16, GFP_KERNEL);
	if (!fbi)
		return NULL;

	memset(fbi, 0, sizeof(struct pl110fb_info) + sizeof(struct display));

	fbi->currcon		= -1;

	strcpy(fbi->fb.fix.id, PL110_NAME);

	fbi->fb.fix.type	= FB_TYPE_PACKED_PIXELS;
	fbi->fb.fix.type_aux	= 0;
	fbi->fb.fix.xpanstep	= 0;
	fbi->fb.fix.ypanstep	= 0;
	fbi->fb.fix.ywrapstep	= 0;
	fbi->fb.fix.accel	= FB_ACCEL_NONE;

	fbi->fb.var.nonstd	= 0;
	fbi->fb.var.activate	= FB_ACTIVATE_NOW;
	fbi->fb.var.height	= -1;
	fbi->fb.var.width	= -1;
	fbi->fb.var.accel_flags	= 0;
	fbi->fb.var.vmode	= FB_VMODE_NONINTERLACED;

	strcpy(fbi->fb.modename, PL110_NAME);
	strcpy(fbi->fb.fontname, "Acorn8x8");

	fbi->fb.fbops		= &pl110fb_ops;
	fbi->fb.changevar	= NULL;
	fbi->fb.switch_con	= pl110fb_switch;
	fbi->fb.updatevar	= pl110fb_updatevar;
	fbi->fb.blank		= pl110fb_blank;
	fbi->fb.flags		= FBINFO_FLAG_DEFAULT;
	fbi->fb.node		= -1;
	fbi->fb.monspecs	= monspecs;
	fbi->fb.disp		= (struct display *)(fbi + 1);
	fbi->fb.pseudo_palette	= (void *)(fbi->fb.disp + 1);

	fbi->rgb[RGB_8]		= &rgb_8;
	fbi->rgb[RGB_16]	= &def_rgb_16;

	inf = pl110fb_get_machine_info(fbi);

	/*
	 * Calculate pixclock.  pixclock is the time in picoseconds spent
	 * drawing a pixel.  The time (in seconds) to draw a pixel is
	 * the inverse of how many pixels we draw in a second (pixelsPerSecond).
	 *
	 * pixelsPerSecond is xres*yres*refresh, plus all the overhead time
	 * (horizontal and vertical front and back porches, plus horizontal
	 * and vertical sync lengths).
	 *
	 * 
	 */
	pixelsPerSecond =
	    (inf->xres + inf->hsync_len + inf->left_margin  + inf->right_margin) *
	    (inf->yres + inf->vsync_len + inf->upper_margin + inf->lower_margin) *
	    VERTICAL_REFRESH;

	inf->pixclock = 1000000000 / (pixelsPerSecond / 1000);

	DPRINTK( "pixelsPerSecond=%d    pixclock=%d\n", pixelsPerSecond,  inf->pixclock);


	fbi->max_xres			= inf->xres;
	fbi->fb.var.xres		= inf->xres;
	fbi->fb.var.xres_virtual	= inf->xres;
	fbi->max_yres			= inf->yres;
	fbi->fb.var.yres		= inf->yres;
	fbi->fb.var.yres_virtual	= inf->yres;
	fbi->max_bpp			= inf->bpp;
	fbi->fb.var.bits_per_pixel	= inf->bpp;
	fbi->fb.var.pixclock		= inf->pixclock;
	fbi->fb.var.hsync_len		= inf->hsync_len;
	fbi->fb.var.left_margin		= inf->left_margin;
	fbi->fb.var.right_margin	= inf->right_margin;
	fbi->fb.var.vsync_len		= inf->vsync_len;
	fbi->fb.var.upper_margin	= inf->upper_margin;
	fbi->fb.var.lower_margin	= inf->lower_margin;
	fbi->fb.var.sync		= inf->sync;
	fbi->fb.var.grayscale		= inf->cmap_greyscale;
	fbi->cmap_inverse		= inf->cmap_inverse;
	fbi->cmap_static		= inf->cmap_static;
	fbi->LCDtiming2			= inf->LCDtiming2;
	fbi->LCDtiming3			= inf->LCDtiming3;
	fbi->LCDcontrol			= inf->LCDcontrol;
	fbi->LCDICPsetup		= inf->LCDICPsetup;
	fbi->LCDICPcontrol		= inf->LCDICPcontrol;
	fbi->LCDICPtiming1		= inf->LCDICPtiming1;
	fbi->LCDICPtiming2		= inf->LCDICPtiming2;
	fbi->state			= C_DISABLE;
	fbi->task_state			= (u_char)-1;
	fbi->fb.fix.smem_len		= fbi->max_xres * fbi->max_yres *
					  fbi->max_bpp / 8;

	init_waitqueue_head(&fbi->ctrlr_wait);
	INIT_TQUEUE(&fbi->task, pl110fb_task, fbi);
	init_MUTEX(&fbi->ctrlr_sem);

	return fbi;
}

int __init
pl110fb_init(void)
{
	struct pl110fb_info *fbi;
	int ret;

	DPRINTK( "\n\npl110fb_init\n");

	DPRINTK( "cpu clock = %d     HCLK = %d\n", cpufreq_get(0), hclkfreq_get());

	fbi = pl110fb_init_fbinfo();
	ret = -ENOMEM;
	if (!fbi)
	    goto failed;

	/* Initialize video memory */
	ret = pl110fb_map_video_memory(fbi);
	if (ret)
	    goto failed;

	ret = request_irq(IRQ_LCD, pl110fb_handle_irq, SA_INTERRUPT,
			  fbi->fb.fix.id, fbi);
	if (ret) {
	    printk(KERN_ERR "pl110fb: request_irq failed: ret=%d\n", ret);
	    goto failed;
	}

	pl110fb_set_var(&fbi->fb.var, -1, &fbi->fb);

	ret = register_framebuffer(&fbi->fb);
	if (ret < 0)
		goto failed;

#ifdef CONFIG_PM
	/*
	 * Note that the console registers this as well, but we want to
	 * power down the display prior to sleeping.
	 */
	fbi->pm = pm_register(PM_SYS_DEV, PM_SYS_VGA, pl110fb_pm_callback);
	if (fbi->pm)
		fbi->pm->data = fbi;
#endif
#ifdef CONFIG_CPU_FREQ
	fbi->clockchg.notifier_call = pl110fb_clkchg_notifier;
	cpufreq_register_notifier(&fbi->clockchg);
#endif

	/*
	 * Ok, now enable the LCD controller
	 */
	set_ctrlr_state(fbi, C_ENABLE);

	/* This driver cannot be unloaded at the moment */
	MOD_INC_USE_COUNT;

	return 0;

failed:
	if (fbi)
		kfree(fbi);
	return ret;
}

int __init
pl110fb_setup(char *options)
{
	return 0;
}

MODULE_DESCRIPTION("ARM PL110 framebuffer driver");
MODULE_LICENSE("GPL");

