/*
 * Hardware definitions for HP iPAQ Handheld Computers
 *
 * Copyright 2000-2003 Hewlett-Packard Company.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Jamey Hicks.
 *
 * History:
 *
 * 2002-08-23   Jamey Hicks        GPIO and IRQ support for iPAQ H5400
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/tty.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/pm.h>
#include <linux/bootmem.h>
#include <linux/delay.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/setup.h>
#include <asm/io.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/arch/h5400-gpio.h>
#include <asm/arch/h5400-asic.h>
#include <asm/arch/h5400-irqs.h>
#include <asm/arch/h5400-init.h>

#include <asm/arch/irq.h>
#include <asm/types.h>

#include "generic.h"

unsigned long h5400_magic = 0x5400beef;
int h5400_pointer = 0;
#define BUFSIZE 2048
unsigned long h5400_buffer[BUFSIZE+2] = { 0xdead5400, };


void
h5400_asic_write_register (unsigned long address, unsigned long value)
{
	h5400_buffer[h5400_pointer++] = address | 1;
	h5400_buffer[h5400_pointer++] = value;
	if (h5400_pointer >= BUFSIZE) h5400_pointer = 0;
		
	if (0) printk("H5400: W %p %p\n", (void *)address, (void *)value);
	__raw_writel (value, address);
}

unsigned long
h5400_asic_read_register (unsigned long address)
{
	unsigned long result;

	h5400_buffer[h5400_pointer++] = address | 0;
	result =  __raw_readl (address);
	h5400_buffer[h5400_pointer++] = result;
	if (h5400_pointer >= BUFSIZE) h5400_pointer = 0;

	return result;
}

/***********************************************************************************/
/*      ASIC IRQ demux                                                             */
/***********************************************************************************/

#define MAX_ASIC_ISR_LOOPS    1000
static void h5400_irq_demux( int irq, void *dev_id, struct pt_regs *regs )
{
	int i;
	
	if (0) printk("%s: interrupt received\n", __FUNCTION__);

	for ( i = 0 ; (i < MAX_ASIC_ISR_LOOPS); i++ ) {
		u32 pending;
		int asic_irq;
		
		pending = h5400_asic_read_register (H5400_ASIC_IC_INTPND);
		if (!pending)
			break;

		asic_irq = H5400_ASIC_IC_IRQ_START + (31 - __builtin_clz (pending));
		do_IRQ (asic_irq, regs);
	}

	if ( i >= MAX_ASIC_ISR_LOOPS && H5400_ASIC_IC_INTPND ) {
		u32 pending = h5400_asic_read_register (H5400_ASIC_IC_INTPND);
		printk("%s: interrupt processing overrun pending=0x%08x\n", __FUNCTION__, pending);
	}
}

static struct irqaction h5400_irq = {
	name:     "h5400_asic",
	handler:  h5400_irq_demux,
	flags:    SA_INTERRUPT
};

/***********************************************************************************/
/*      ASIC EPS IRQ demux                                                         */
/***********************************************************************************/

static u32 eps_irq_mask[] = {
	H5400_ASIC_PCMCIA_EPS_CD0_N, /* IRQ_GPIO_H5400_EPS_CD0 */
	H5400_ASIC_PCMCIA_EPS_CD1_N, /* IRQ_GPIO_H5400_EPS_CD1 */
	H5400_ASIC_PCMCIA_EPS_IRQ0_N, /* IRQ_GPIO_H5400_EPS_IRQ0 */
	H5400_ASIC_PCMCIA_EPS_IRQ1_N, /* IRQ_GPIO_H5400_EPS_IRQ1 */
	(H5400_ASIC_PCMCIA_EPS_ODET0_N|H5400_ASIC_PCMCIA_EPS_ODET1_N),/* IRQ_GPIO_H5400_EPS_ODET */
	H5400_ASIC_PCMCIA_EPS_BATT_FLT/* IRQ_GPIO_H5400_EPS_BATT_FAULT */
};

static void h5400_asic_eps_irq_demux( int irq, void *dev_id, struct pt_regs *regs )
{
	int i;

	if (0) printk("%s: interrupt received irq=%d\n", __FUNCTION__, irq);

	for ( i = 0 ; (i < MAX_ASIC_ISR_LOOPS) && h5400_asic_read_register(H5400_ASIC_PCMCIA_IC) ; i++ ) {
		int j;
		int eps_pending = h5400_asic_read_register (H5400_ASIC_PCMCIA_IP);
		if (0) printk("%s: eps_pending=0x%08x\n", __FUNCTION__, eps_pending);
		for ( j = 0 ; j < H5400_ASIC_EPS_IRQ_COUNT ; j++ )
			if ( eps_pending & eps_irq_mask[j] )
				do_IRQ(j + H5400_ASIC_EPS_IRQ_START, regs);
	}

	if ( i >= MAX_ASIC_ISR_LOOPS && h5400_asic_read_register(H5400_ASIC_PCMCIA_IP) )
		printk("%s: interrupt processing overrun pending=0x%08x\n", __FUNCTION__, h5400_asic_read_register(H5400_ASIC_PCMCIA_IP));
}

static struct irqaction h5400_asic_eps_irq = {
	name:     "h5400_asic_eps",
	handler:  h5400_asic_eps_irq_demux,
	flags:    SA_INTERRUPT
};

/***********************************************************************************/
/*      ASIC GPIO IRQ demux                                                        */
/***********************************************************************************/

static void h5400_asic_gpio_irq_demux( int irq, void *dev_id, struct pt_regs *regs )
{
	unsigned long pending;
	int loop;

	for (loop = 0; loop < MAX_ASIC_ISR_LOOPS; loop++)	
	{
		int i;
		pending = h5400_asic_read_register (H5400_ASIC_GPIO_INTPND);

		if (!pending)
			break;
		
		for (i = 0; i < H5400_ASIC_GPIO_IRQ_COUNT; i++) {
			if (pending & (1 << i))
				do_IRQ (H5400_ASIC_GPIO_IRQ_START + i, regs);
		}
	}

	if (pending)
		printk ("%s: demux overrun, pending=%08lx\n", __FUNCTION__, pending);
}

static struct irqaction h5400_asic_gpio_irq = {
	name:     "h5400 gpio",
	handler:  h5400_asic_gpio_irq_demux,
	flags:    SA_INTERRUPT
};

/***********************************************************************************/
/*      IRQ handling                                                               */
/***********************************************************************************/

/* mask_ack <- IRQ is first serviced.
       mask <- IRQ is disabled.  
     unmask <- IRQ is enabled */

static void h5400_asic_mask_ack_ic_irq( unsigned int irq )
{
	int mask = 1 << (irq - H5400_ASIC_IC_IRQ_START);
	unsigned long value;

	h5400_asic_write_register (H5400_ASIC_IC_SRCPND, mask);
	h5400_asic_write_register (H5400_ASIC_IC_INTPND, mask);

 	if (0) printk("%s: irq=%d mask=0x%x pending=0x%08x\n", __FUNCTION__, irq, mask, H5400_ASIC_IC_INTPND);
}

static void h5400_asic_mask_ic_irq( unsigned int irq )
{
	int mask = 1 << (irq - H5400_ASIC_IC_IRQ_START);

	H5400_ASIC_SET_BIT (H5400_ASIC_IC_INTMSK, mask);
}

static void h5400_asic_unmask_ic_irq( unsigned int irq )
{
	int mask = 1 << (irq - H5400_ASIC_IC_IRQ_START);

	H5400_ASIC_CLEAR_BIT (H5400_ASIC_IC_INTMSK, mask);
}

static void h5400_asic_mask_ack_eps_irq( unsigned int irq )
{
	int mask = eps_irq_mask[irq - H5400_ASIC_EPS_IRQ_START];

	H5400_ASIC_SET_BIT (H5400_ASIC_IC_INTMSK, mask);
	h5400_asic_write_register (H5400_ASIC_PCMCIA_IP, mask);
}

static void h5400_asic_mask_eps_irq( unsigned int irq )
{
	int mask = eps_irq_mask[irq - H5400_ASIC_EPS_IRQ_START];

	H5400_ASIC_CLEAR_BIT (H5400_ASIC_PCMCIA_IC, mask);
}

static void h5400_asic_unmask_eps_irq( unsigned int irq )
{
	int mask = eps_irq_mask[irq - H5400_ASIC_EPS_IRQ_START];

	H5400_ASIC_SET_BIT (H5400_ASIC_PCMCIA_IC, mask);
}

static void h5400_asic_mask_ack_gpio_irq (unsigned int irq)
{
	H5400_ASIC_CLEAR_BIT (H5400_ASIC_GPIO_ENINT2, 1 << (irq - H5400_ASIC_GPIO_IRQ_START));
	h5400_asic_write_register (H5400_ASIC_GPIO_INTPND, 1 << (irq - H5400_ASIC_GPIO_IRQ_START));
}

static void h5400_asic_mask_gpio_irq (unsigned int irq)
{
	H5400_ASIC_CLEAR_BIT (H5400_ASIC_GPIO_ENINT2, 1 << (irq - H5400_ASIC_GPIO_IRQ_START));
}

static void h5400_asic_unmask_gpio_irq (unsigned int irq)
{
	H5400_ASIC_SET_BIT (H5400_ASIC_GPIO_ENINT2, 1 << (irq - H5400_ASIC_GPIO_IRQ_START));
}

void __init h5400_asic_init_irq(void)
{
	int i;
	for ( i = 0 ; i < H5400_ASIC_IC_IRQ_COUNT ; i++ ) {
		int irq = i + H5400_ASIC_IC_IRQ_START;
		irq_desc[irq].valid    = 1;
		irq_desc[irq].probe_ok = 1;
		irq_desc[irq].no_unmask = 1;
		irq_desc[irq].mask_ack = h5400_asic_mask_ack_ic_irq;
		irq_desc[irq].mask     = h5400_asic_mask_ic_irq;
		irq_desc[irq].unmask   = h5400_asic_unmask_ic_irq;
	}

	h5400_asic_write_register (H5400_ASIC_PCMCIA_IC, 0);    /* nothing enabled */
	h5400_asic_write_register (H5400_ASIC_PCMCIA_IP, 0xFF); /* clear anything here now */
	h5400_asic_write_register (H5400_ASIC_PCMCIA_IM, 0x2555);
	for ( i = 0 ; i < H5400_ASIC_EPS_IRQ_COUNT ; i++ ) {
		int irq = i + H5400_ASIC_EPS_IRQ_START;
		irq_desc[irq].valid    = 1;
		irq_desc[irq].probe_ok = 1;
		irq_desc[irq].no_unmask = 1;
		irq_desc[irq].mask_ack = h5400_asic_mask_ack_eps_irq;
		irq_desc[irq].mask     = h5400_asic_mask_eps_irq;
		irq_desc[irq].unmask   = h5400_asic_unmask_eps_irq;
	}

	for (i = 0; i < H5400_ASIC_GPIO_IRQ_COUNT; i++) {
		int irq = i + H5400_ASIC_GPIO_IRQ_START;
		irq_desc[irq].valid    = 1;
		irq_desc[irq].probe_ok = 1;
		irq_desc[irq].no_unmask = 1;
		irq_desc[irq].mask_ack = h5400_asic_mask_ack_gpio_irq;
		irq_desc[irq].mask     = h5400_asic_mask_gpio_irq;
		irq_desc[irq].unmask   = h5400_asic_unmask_gpio_irq;
	}

#if 0
	printk("%s: gpio10=%x gedr[10]=%d asic_ic_srcpnd=0x%08x asic_ic_intpnd=0x%08x asic_pcmcia_ip=0x%08x\n",
	       __FUNCTION__,
	       GPLR(GPIO_NR_H5400_ASIC_INT_N)&GPIO_bit(GPIO_NR_H5400_ASIC_INT_N),
	       GEDR(GPIO_NR_H5400_ASIC_INT_N)&GPIO_bit(GPIO_NR_H5400_ASIC_INT_N),
	       H5400_ASIC_IC_INTPND,
	       H5400_ASIC_IC_SRCPND,
	       H5400_ASIC_PCMCIA_IP);
#endif

	/* all ints off */
	h5400_asic_write_register (H5400_ASIC_IC_INTMSK, 0xffffffff);

	/* clear out any pending irqs */
	for (i = 0; i < 32; i++) {
		if (h5400_asic_read_register (H5400_ASIC_IC_INTPND) == 0)
			break;
		h5400_asic_write_register (H5400_ASIC_IC_SRCPND, 0xffffffff);
		h5400_asic_write_register (H5400_ASIC_IC_INTPND, 0xffffffff);
	}

	irq_desc[IRQ_GPIO(GPIO_NR_H5400_ASIC_INT_N)].reentrant = 1;
	irq_desc[IRQ_H5400_PCMCIA].reentrant = 1;
	irq_desc[IRQ_H5400_GPIO].reentrant = 1;
 
        /* note: set_GPIO_IRQ_edge takes a gpio number not a mask on pxa!! */
	set_GPIO_IRQ_edge( GPIO_NR_H5400_ASIC_INT_N, GPIO_FALLING_EDGE );
	setup_arm_irq( IRQ_GPIO(GPIO_NR_H5400_ASIC_INT_N), &h5400_irq );
	setup_arm_irq( IRQ_H5400_PCMCIA, &h5400_asic_eps_irq );
	setup_arm_irq( IRQ_H5400_GPIO, &h5400_asic_gpio_irq );
}

static void __init h5400_init_irq( void )
{
	/* Initialize standard IRQs */
	pxa_init_irq();

	h5400_asic_init_irq();  
}

/***********************************************************************************/
/*      EGPIO, etc                                                                 */
/***********************************************************************************/

static void h5400_video_power_on (int setp)
{
	SET_H5400_ASIC_GPIO (GPB, MQ_POWER_ON, setp);
}

static void h5400_video_lcd_enable (int setp)
{
	SET_H5400_ASIC_GPIO (GPB, LCD_EN, setp);
}

static void h5400_control_egpio( enum ipaq_egpio_type x, int setp )
{
	switch (x) {
	case IPAQ_EGPIO_LCD_POWER:
		h5400_video_power_on( setp );
		break;
	case IPAQ_EGPIO_LCD_ENABLE:
		h5400_video_lcd_enable( setp );
		break;
	case IPAQ_EGPIO_OPT_NVRAM_ON:
		printk("opt_nvram_n: before gplr=%x setp=%d\n", GPLR(GPIO_NR_H5400_OPT_NVRAM), setp);
		SET_H5400_GPIO( OPT_NVRAM, setp );
		printk("opt_nvram_n: after gplr=%x setp=%d\n", GPLR(GPIO_NR_H5400_OPT_NVRAM), setp);
		break;
	case IPAQ_EGPIO_OPT_ON:
		SET_H5400_ASIC_GPIO( GPA, OPT_ON_N, !setp );
		printk("opt_on: gpa=%x setp=%d\n", H5400_ASIC_GPIO_GPA_DAT, setp);
		break;
	case IPAQ_EGPIO_CARD_RESET:
		printk("%s: card_reset: setp=%d\n", __FUNCTION__, setp);
		if (setp)
			h5400_asic_write_register (H5400_ASIC_PCMCIA_CC, H5400_ASIC_PCMCIA_CC_RESET);
		else
			h5400_asic_write_register (H5400_ASIC_PCMCIA_CC, 0);
		break;
	case IPAQ_EGPIO_OPT_RESET:
		SET_H5400_ASIC_GPIO( GPA, OPT_RESET, setp );
		break;
	case IPAQ_EGPIO_RS232_ON:
		/* automatic */
		break;
	case IPAQ_EGPIO_BLUETOOTH_ON:
		/* apply reset */
		SET_H5400_GPIO(BT_M_RESET, 0);

		/* select normal operation */
		SET_H5400_GPIO(BT_ENV_0, 1);
		SET_H5400_GPIO(BT_ENV_1, 1);

		/* apply power */
		SET_H5400_ASIC_GPIO(GPB, BLUETOOTH_3V0_ON, setp);
		SET_H5400_GPIO(BT_2V8_N, !setp);

		/* release reset */
		if (setp) {
			udelay(100);
			SET_H5400_GPIO(BT_M_RESET, 1);
		}
		break;
	case IPAQ_EGPIO_VPP_ON:
		/* always on */
		break;

	case IPAQ_EGPIO_CODEC_NRESET:
	case IPAQ_EGPIO_AUDIO_ON:
	case IPAQ_EGPIO_QMUTE:
	case IPAQ_EGPIO_IR_ON:
	case IPAQ_EGPIO_IR_FSEL:
	default:
		printk("%s: unhandled ipaq gpio=%d\n", __FUNCTION__, x);
	}
}

static unsigned long h5400_read_egpio( enum ipaq_egpio_type x)
{
	switch (x) {
	case IPAQ_EGPIO_PCMCIA_CD0_N:
		return h5400_asic_read_register(H5400_ASIC_PCMCIA_EPS) & H5400_ASIC_PCMCIA_EPS_CD0_N;
	case IPAQ_EGPIO_PCMCIA_CD1_N:
		return h5400_asic_read_register(H5400_ASIC_PCMCIA_EPS) & H5400_ASIC_PCMCIA_EPS_CD1_N;
	case IPAQ_EGPIO_PCMCIA_IRQ0:
		return h5400_asic_read_register(H5400_ASIC_PCMCIA_EPS) & H5400_ASIC_PCMCIA_EPS_IRQ0_N;
	case IPAQ_EGPIO_PCMCIA_IRQ1:
		return h5400_asic_read_register(H5400_ASIC_PCMCIA_EPS) & H5400_ASIC_PCMCIA_EPS_IRQ1_N;
        default:
		printk("%s:%d: unknown ipaq_egpio_type=%d\n", __FUNCTION__, __LINE__, x);
		return 0;
	}
}

struct egpio_irq_info h5400_egpio_irq_info[] = {
	{ IPAQ_EGPIO_PCMCIA_CD0_N, 0, IRQ_GPIO_H5400_EPS_CD0 }, 
	{ IPAQ_EGPIO_PCMCIA_CD1_N, 0, IRQ_GPIO_H5400_EPS_CD1 },
	{ IPAQ_EGPIO_PCMCIA_IRQ0,  0, IRQ_GPIO_H5400_EPS_IRQ0 },
	{ IPAQ_EGPIO_PCMCIA_IRQ1,  0, IRQ_GPIO_H5400_EPS_IRQ1 },
	{ 0, 0 }
}; 

static int h5400_egpio_irq_number(enum ipaq_egpio_type egpio_nr)
{
	struct egpio_irq_info *info = h5400_egpio_irq_info;
	while (info->irq != 0) {
		if (info->egpio_nr == egpio_nr) {
			if (1) printk("%s: egpio_nr=%d irq=%d\n", __FUNCTION__, egpio_nr, info->irq);
			return info->irq;
		}
		info++;
	}

	printk("%s: unhandled egpio_nr=%d\n", __FUNCTION__, egpio_nr); 
	return -EINVAL;
}

static __inline__ void 
fix_msc (void)
{
	/* fix CS0 for h5400 flash. */
	/* fix CS1 for MediaQ chip.  select 16-bit bus and vlio.  */
	/* fix CS5 for SAMCOP.  */
	MSC0 = 0x129c24f2;
	(void)MSC0;
	MSC1 = 0x7ff424fa;
	(void)MSC1;
	MSC2 = 0x7ff47ff4;
	(void)MSC2;

	MDREFR |= 0x02080000;
}

static int h5400_pm_callback( int req )
{
	int result = 0;
	static int gpio_a, gpio_b;		/* ASIC GPIOs */
	static int gpio_0, gpio_1, gpio_2;	/* PXA GPIOs */
	
	printk("%s: %d\n", __FUNCTION__, req);

	switch (req) {
	case PM_RESUME:
		/* bootldr will have screwed up MSCs, so set them right again */
		fix_msc ();

		h5400_asic_write_register (H5400_ASIC_GPIO_GPA_DAT, gpio_a);
		h5400_asic_write_register (H5400_ASIC_GPIO_GPB_DAT, gpio_b);

		GPSR0 = gpio_0;
		GPCR0 = ~gpio_0;
		GPSR1 = gpio_1;
		GPCR1 = ~gpio_1;
		GPSR2 = gpio_2;
		GPCR2 = ~gpio_2;

		if ( ipaq_model_ops.pm_callback_aux )
			result = ipaq_model_ops.pm_callback_aux(req);
		break;

	case PM_SUSPEND:
		if ( ipaq_model_ops.pm_callback_aux &&
		     ((result = ipaq_model_ops.pm_callback_aux(req)) != 0))
			return result;
		
		gpio_a = h5400_asic_read_register (H5400_ASIC_GPIO_GPA_DAT);
		gpio_b = h5400_asic_read_register (H5400_ASIC_GPIO_GPB_DAT);

		gpio_0 = GPLR0;
		gpio_1 = GPLR1;
		gpio_2 = GPLR2;
		break;

	default:
		printk("%s: unrecognized PM callback\n", __FUNCTION__);
		break;
	}
	return result;
}

/***********************************************************************************/
/*      Miscellaneous                                                              */
/***********************************************************************************/

static void h5400_set_led (enum led_color color, int duty_time, int cycle_time)
{
	static const u8 h5400_lednum[5] = { 4, 0, 1, 2, 3 };
	static unsigned long led_state;
	int lednum, ledvalue;
	unsigned long flags;

	lednum = h5400_lednum[color];
	ledvalue = duty_time ? 0x3 | (duty_time << 4) | (cycle_time << 12) : 0;

	local_irq_save (flags);

	/* Clock needs to be left enabled if any LEDs are flashing */
	if (duty_time && (duty_time < cycle_time))
		led_state |= 1 << lednum;
	else
		led_state &= ~1 << lednum;

	H5400_ASIC_SET_BIT (H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_LED_CLKEN);

	h5400_asic_write_register (H5400_ASIC_LED_CONTROL(lednum), ledvalue);
	
	if (led_state == 0)
		H5400_ASIC_CLEAR_BIT (H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_LED_CLKEN);

	local_irq_restore (flags);
}

void h5400_asic_clock_enable (u32 bit, int on)
{
	unsigned long flags;

	local_irq_save (flags);

	if (on)
		H5400_ASIC_SET_BIT (H5400_ASIC_CPM_ClockControl, bit);
	else
		H5400_ASIC_CLEAR_BIT (H5400_ASIC_CPM_ClockControl, bit);

	local_irq_restore (flags);
}



/***********************************************************************************/
/*      Initialisation                                                             */
/***********************************************************************************/

static struct ipaq_model_ops h5400_model_ops __initdata = {
	.generic_name = "5400",
	.control      = h5400_control_egpio,
	.read         = h5400_read_egpio,
	.pm_callback  = h5400_pm_callback,
	.irq_number   = h5400_egpio_irq_number,
	.set_led      = h5400_set_led
};

static short h5400_gpio_modes[] __initdata = {
	GPIO_NR_H5400_OPT_INT | GPIO_IN ,
	GPIO_NR_H5400_BACKUP_POWER | GPIO_IN ,
	GPIO_NR_H5400_ACTION_BUTTON | GPIO_IN ,
#if 0
	GPIO_NR_H5400_COM_DCD_SOMETHING | GPIO_OUT ,
	GPIO_NR_H5400_RESET_BUTTON_AGAIN_N | GPIO_ ,
	GPIO_NR_H5400_RSO_N | GPIO_ ,
#endif
	GPIO_NR_H5400_ASIC_INT_N | GPIO_IN ,
	GPIO_NR_H5400_BT_ENV_0 | GPIO_OUT ,
	GPIO_NR_H5400_BT_ENV_1 | GPIO_OUT ,
	GPIO_NR_H5400_BT_WU | GPIO_IN ,

#if 0
	GPIO_NR_H5400_COM_RXD | GPIO_ ,
	GPIO_NR_H5400_COM_CTS | GPIO_ ,
	GPIO_NR_H5400_COM_DCD | GPIO_ ,
	GPIO_NR_H5400_COM_DSR | GPIO_ ,
	GPIO_NR_H5400_COM_RI | GPIO_ ,
	GPIO_NR_H5400_COM_TXD | GPIO_ ,
	GPIO_NR_H5400_COM_DTR | GPIO_ ,
	GPIO_NR_H5400_COM_RTS | GPIO_ ,
#endif

#if 0
	GPIO_NR_H5400_BT_RXD | GPIO_ ,
	GPIO_NR_H5400_BT_TXD | GPIO_ ,
	GPIO_NR_H5400_BT_CTS | GPIO_ ,
	GPIO_NR_H5400_BT_RTS | GPIO_ ,
#endif
#if 0
	GPIO_NR_H5400_IRDA_RXD | GPIO_ ,
	GPIO_NR_H5400_IRDA_TXD | GPIO_ ,
#endif

#if 0
	GPIO_NR_H5400_IRDA_SD | GPIO_ ,
#endif

	GPIO_NR_H5400_POWER_SD_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_POWER_FP_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_POWER_ACCEL_N | GPIO_OUT | GPIO_ACTIVE_LOW,

	GPIO_NR_H5400_OPT_NVRAM | GPIO_OUT ,
	GPIO_NR_H5400_CHG_EN | GPIO_OUT ,
	GPIO_NR_H5400_USB_PULLUP | GPIO_OUT ,
	GPIO_NR_H5400_BT_2V8_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_EXT_CHG_RATE | GPIO_OUT ,

	GPIO_NR_H5400_CIR_RESET | GPIO_OUT ,
	GPIO_NR_H5400_POWER_LIGHT_SENSOR_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_BT_M_RESET | GPIO_OUT ,
	GPIO_NR_H5400_STD_CHG_RATE | GPIO_OUT ,
	GPIO_NR_H5400_SD_WP_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_MOTOR_ON_N | GPIO_OUT | GPIO_ACTIVE_LOW,
	GPIO_NR_H5400_HEADPHONE_DETECT | GPIO_IN ,
	GPIO_NR_H5400_USB_CHG_RATE | GPIO_OUT ,
};

static struct map_desc h5400_io_desc[] __initdata = {
 /* virtual            physical           length      domain     r  w  c  b */
  { H3600_BANK_2_VIRT, H3600_BANK_2_PHYS, 0x02800000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 2  CS#2 */
  { H3600_BANK_4_VIRT, H3600_BANK_4_PHYS, 0x00800000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 4  CS#4 */
  { H3600_BANK_5_VIRT, H3600_BANK_5_PHYS, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 5  CS#5 */
  LAST_DESC
};

static void __init h5400_map_io(void)
{
	int i;

	pxa_map_io();
	iotable_init(h5400_io_desc);

	/* Configure power management stuff. */
	PWER = PWER_GPIO0 | PWER_RTC;
	PFER = PWER_GPIO0 | PWER_RTC;
	PRER = 0;
	PCFR = PCFR_OPDE;
	CKEN = CKEN6_FFUART;

	h5400_asic_write_register (H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_LED_CLKEN);
	h5400_asic_write_register (H5400_ASIC_LED_LEDPS, 0xf42400);		/* 4Hz */

	H5400_ASIC_SET_BIT (H5400_ASIC_CPM_ClockControl, H5400_ASIC_CPM_CLKCON_GPIO_CLKEN);
	H5400_ASIC_SET_BIT (H5400_ASIC_GPIO_GPA_CON2, 0x16);

	for (i = 0; i < ARRAY_SIZE(h5400_gpio_modes); i++) {
		int mode = h5400_gpio_modes[i];
		set_GPIO_mode(mode);
	}

#if FixMeLater
	/* Add wakeup on AC plug/unplug */
	PWER  |= PWER_GPIO12;
#endif

	fix_msc ();

#if 0
	/* touchscreen works, asic locks up */
	h5400_asic_write_register (H5400_ASIC_CPM_ClockSleep, H5400_ASIC_CPM_CLKSLEEP_UCLK_ON);
	/* asic does not lock up but touchscreen malfunctions */
	h5400_asic_write_register (H5400_ASIC_CPM_ClockSleep, H5400_ASIC_CPM_CLKSLEEP_HALF_CLK);
#else
	/* ryan says this works: */
	h5400_asic_write_register (H5400_ASIC_CPM_ClockSleep, 0);
#endif

	/* Set sleep states for PXA GPIOs */
	PGSR0 = GPSRx_SleepValue;
	PGSR1 = GPSRy_SleepValue;
	PGSR2 = GPSRz_SleepValue;

	/* Turn off all LEDs */
	ipaq_set_led (GREEN_LED, 0, 0);
	ipaq_set_led (BLUE_LED, 0, 0);
	ipaq_set_led (YELLOW_LED, 0, 0);
	ipaq_set_led (RED_LED, 0, 0);

	ipaq_model_ops = h5400_model_ops;
}

MACHINE_START(H5400, "HP iPAQ H5400")
	MAINTAINER("HP Labs, Cambridge Research Labs")
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	BOOT_PARAMS(0xa0000100)
	MAPIO(h5400_map_io)
	INITIRQ(h5400_init_irq)
MACHINE_END

EXPORT_SYMBOL(h5400_asic_clock_enable);
EXPORT_SYMBOL(h5400_asic_write_register);
EXPORT_SYMBOL(h5400_asic_read_register);
