/*
 *  linux/drivers/char/axim_key.c
 *
 *  Copyright (C) 2003 Martin Demin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/compiler.h>
#include <linux/interrupt.h>

#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/irq.h>
#include <asm/arch/hardware.h>
#include <linux/devfs_fs_kernel.h>
//#include <linux/axim_key.h>
#include <linux/h3600_keyboard.h>

/*
 * TSBUF_SIZE must be a power of two
 */
#define AXIM_KEY_MINOR	2

#define TSBUF_SIZE	256
#define NEXT(index)	(((index) + 1) & (TSBUF_SIZE - 1))
#define IRQ_TOUCHSCREEN (32+22)

#define AXIM_KEY_POWER		0
#define AXIM_KEY_RECORD		10
#define AXIM_KEY_CALENDAR	11
#define AXIM_KEY_CONTACTS	12
#define AXIM_KEY_MAIL		13
#define AXIM_KEY_START 		14
#define AXIM_KEY_CENTER		27

#define AXIM_KEY_POWER_IRQ		(0+22)
#define AXIM_KEY_RECORD_IRQ		(10+22)
#define AXIM_KEY_CALENDAR_IRQ		(11+22)
#define AXIM_KEY_CONTACTS_IRQ		(12+22)
#define AXIM_KEY_MAIL_IRQ		(13+22)
#define AXIM_KEY_START_IRQ 		(14+22)
#define AXIM_KEY_CENTER_IRQ		(27+22)





int buttons [16];



static DECLARE_WAIT_QUEUE_HEAD(queue);
static DECLARE_MUTEX(open_sem);
static spinlock_t tailptr_lock = SPIN_LOCK_UNLOCKED;
static struct fasync_struct *fasync;
static int gMajor,raw_read;		
static devfs_handle_t devfs_key, devfs_key_dir, devfs_keyraw;
static unsigned long last_x,last_y,ljiffies,pressed;


static unsigned char button_to_scancode[] = {
        0, /* unused */
        H3600_SCANCODE_RECORD,   /* 1 -> record button */
        H3600_SCANCODE_CALENDAR, /* 2 -> calendar */
        H3600_SCANCODE_CONTACTS, /* 3 -> contact */
        H3600_SCANCODE_Q,        /* 4 -> Q button */
        H3600_SCANCODE_START,    /* 5 -> start menu */
        H3600_SCANCODE_UP,       /* 6 -> up */
        H3600_SCANCODE_RIGHT,    /* 7 -> right */
        H3600_SCANCODE_LEFT,     /* 8 -> left */
        H3600_SCANCODE_DOWN,     /* 9 -> down */
	H3600_SCANCODE_ACTION,   /* 10 -> action button (synthesized, not from Atmel) */
	H3600_SCANCODE_SUSPEND,  /* 11 -> power button (synthesized, not from Atmel)  */
	0, 0, 0, 0               /* pad out to 16 total bytes */
};

static int axim_read_buttons()
{
	int x;
	unsigned long gplr,gplr2;
	
	x=0;
	gplr=GPLR0;
	if(!(gplr&1)) x=11;  // Power
	if((gplr>>10)&1) x=1; // Record
	if((gplr>>11)&1) x=2; // Calendar
	if((gplr>>12)&1) x=3; // contacts
	if((gplr>>13)&1) x=4; // mail
	if((gplr>>14)&1) x=5; // start
	if(!((gplr>>27)&1)) // push down
	{
	    gplr2=GPLR2 & 0xf;
	    switch(gplr2)
	    {
#ifndef CONFIG_AXIM_KEY_FIX
		case 0xC: x=7; break; // right
		case 0x3: x=8; break; // left
		case 0x9: x=9; break; // down
		case 0x6: x=6; break; // up
#else
		case 0xC: x=8; break; // left
		case 0x3: x=7; break; // right
		case 0x9: x=6; break; // up
		case 0x6: x=9; break; // down
#endif
		default:
		x=10;		
	    }
	}
//	printk("Button pressed: %d\n",x);
	return x;
}

static void
axim_key_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long x,i;

	x=axim_read_buttons();
	
// release all but one pressed
	for(i=1;i<16;i++)
	    if(buttons[i]&&i!=x) // if button changed to 0
	    {
		buttons[i]=0;
		handle_scancode( button_to_scancode[i], 0 );   
	    }
// send "pressed" scancode	    
	if(!buttons[x])
	{
	    handle_scancode( button_to_scancode[x], 1 );   
	    buttons[x]=1;
	}
	
	
	wake_up_interruptible(&queue);
	kill_fasync(&fasync, SIGIO, POLL_IN);

}


static ssize_t
axim_key_read(struct file *filp, char *buf, size_t count, loff_t *l)
{
//	unsigned long x,y,q,nx,ny,dx,dy;
	unsigned short data[1];
	ssize_t written = 0;

	while (/*(volatile int)head != (volatile int)tail &&*/ count >= sizeof data) {

	
	data[0]=button_to_scancode[pressed];

		if (copy_to_user(buf, data, 1))
			return -EFAULT;
		count -= sizeof data;
		buf += sizeof data;
		written += sizeof data;
	}
	return written ? written : -EINVAL;
}

//int poll;

static unsigned int
axim_key_poll(struct file *filp, poll_table *wait)
{
	poll_wait(filp, &queue, wait);
	pressed=axim_read_buttons();
/*	if(pressed) return POLLIN | POLLRDNORM;
	return 0;*/
	return pressed ? POLLIN | POLLRDNORM : 0;
}

static int
axim_key_ioctl(struct inode *inode, struct file *filp,
		unsigned int cmd, unsigned long arg)
{
	int retval = 0;

        return retval;
}

static int
axim_key_open(struct inode *inode, struct file *filp)
{
	return 0;
}

static int
axim_key_fasync(int fd, struct file *filp, int on)
{
	return fasync_helper(fd, filp, on, &fasync);
}

static int
axim_key_release(struct inode *inode, struct file *filp)
{
	axim_key_fasync(-1, filp, 0);
	return 0;
}

static struct file_operations axim_key_fops = {
	owner:		THIS_MODULE,
	read:		axim_key_read,
	poll:		axim_key_poll,
	ioctl:		axim_key_ioctl,
	open:		axim_key_open,
	release:	axim_key_release,
	fasync:		axim_key_fasync,
};


static int h3600_key_open_generic(struct inode * inode, struct file * filp)
{
        unsigned int minor = MINOR( inode->i_rdev );   /* Extract the minor number */
	int result;

	if ( minor > 2 ) {
		printk("%s: bad minor = %d\n", __FUNCTION__, minor );
		return -ENODEV;
	}

        if (0) printk("%s: minor=%d\n", __FUNCTION__,minor);

	if ( !filp->private_data ) {
		switch (minor) {
		case AXIM_KEY_MINOR:
//			filp->private_data = &g_touchscreen.filtered;
			filp->f_op = &axim_key_fops;
			break;
		}
	}

	result = filp->f_op->open( inode, filp );
	if ( !result ) 
		return result;

	return 0;
}


struct file_operations generic_fops = {
	open:     h3600_key_open_generic
};

static struct miscdevice axim_key_miscdev = {
        AXIM_KEY_MINOR,
        "Axim_key",
        &axim_key_fops
};


/*
 * Initialization and exit routines
 */
int __init
axim_key_init(void)
{
	int retval,i;

        printk("%s: registering char device\n", __FUNCTION__);

        gMajor = devfs_register_chrdev(0, "axim_key", &generic_fops);
        if (gMajor < 0) {
                printk("%s: can't get major number\n", __FUNCTION__);
                return gMajor;
        }

        devfs_key_dir = devfs_mk_dir(NULL, "touchscreen", NULL);
	if ( !devfs_key_dir ) return -EBUSY;

        devfs_key     = devfs_register( devfs_key_dir, "key", DEVFS_FL_DEFAULT,
				       gMajor, AXIM_KEY_MINOR, 
				       S_IFCHR | S_IRUSR | S_IWUSR, 
				       &axim_key_fops, NULL );
#if 0
	set_GPIO_IRQ_edge(AXIM_KEY_POWER,
			  GPIO_BOTH_EDGES);
	if ((retval = request_irq(AXIM_KEY_POWER_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_power", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}
#endif
	set_GPIO_IRQ_edge(AXIM_KEY_CALENDAR,
			  GPIO_BOTH_EDGES);
	if ((retval = request_irq(AXIM_KEY_CALENDAR_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_calendar", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}
	set_GPIO_IRQ_edge(AXIM_KEY_CONTACTS,
			  GPIO_BOTH_EDGES);
	if ((retval = request_irq(AXIM_KEY_CONTACTS_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_contacts", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}
	set_GPIO_IRQ_edge(AXIM_KEY_MAIL,
			  GPIO_BOTH_EDGES);
	if ((retval = request_irq(AXIM_KEY_MAIL_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_mail", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}
	set_GPIO_IRQ_edge(AXIM_KEY_START,
			  GPIO_BOTH_EDGES);

	if ((retval = request_irq(AXIM_KEY_START_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_home", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}
	set_GPIO_IRQ_edge(AXIM_KEY_CENTER,
			  GPIO_BOTH_EDGES);
	if ((retval = request_irq(AXIM_KEY_CENTER_IRQ, axim_key_handler,
			SA_INTERRUPT, "axim_key_center", 0))) {
		printk(KERN_WARNING "axim_key: failed to get IRQ\n");
		return retval;
	}

	for(i=1;i<16;i++)
    	    buttons[i]=0;

	printk(KERN_NOTICE "Axim buttons driver initialised\n");

	return 0;
}

void __exit
axim_key_exit(void)
{
        devfs_unregister(devfs_key);
	devfs_unregister(devfs_key_dir);
	free_irq(AXIM_KEY_CENTER_IRQ,0);
//	free_irq(AXIM_KEY_POWER_IRQ,0);
	free_irq(AXIM_KEY_START_IRQ,0);
	free_irq(AXIM_KEY_CALENDAR_IRQ,0);
	free_irq(AXIM_KEY_CONTACTS_IRQ,0);
	free_irq(AXIM_KEY_MAIL_IRQ,0);
	
	devfs_unregister_chrdev(gMajor, "axim_key");
}

module_init(axim_key_init);
module_exit(axim_key_exit);

MODULE_AUTHOR("Tak-Shing Chan <chan@aleph1.co.uk>");
MODULE_DESCRIPTION("axim buttons driver");
MODULE_SUPPORTED_DEVICE("buttons/axim");
