/*
* Driver interface to the touchscreen on the iPAQ H1900
*
* Copyright (C) 2003 Joshua Wise
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* HAL code based on h5400_asic_io.c, which is
*  Copyright (C) 2003 Compaq Computer Corporation.
*
* Author:  Joshua Wise <joshua at joshuawise.com>
*          June 2003
*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/config.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>

#include <asm/arch/hardware.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/irq.h>

#include <asm/mach/irq.h>
#include <asm/arch-pxa/h1900-gpio.h>
#include <asm/hardware/ipaq-asic3.h>

#define SAMPLE_TIMEOUT 10	/* sample every 10ms */

MODULE_AUTHOR("Joshua Wise");
MODULE_DESCRIPTION("Touchscreen support for the iPAQ H1900");

static struct timer_list timer;

static unsigned int spi_ctrl(unsigned char ctrlbyte)
{
	/* prevent a race condition in the while loops */
	int antirace;
	
	SSDR = ctrlbyte;
	
	/* wait until we have completed sending. */
	antirace = 100000;
	while ((SSSR & SSSR_BSY) && antirace)
	{
		udelay(10);
		antirace--;
	};
	
	if (antirace == 0)
	{
		printk("%s: warning: timeout while waiting for sssr_bsy to clear\n", __FUNCTION__);
	};
	
	antirace = 100000;
	/* wait for the receive fifo to become not empty. */
	while ((~SSSR & SSSR_RNE) && antirace)
	{
		udelay(10);
		antirace--;
	};
	
	if (antirace == 0)
	{
		printk("%s: warning: timeout while waiting for sssr_rne to set\n", __FUNCTION__);
	};
	
	return SSDR & 0xFFFF;
};

#define CTRL_START 0x80
#define CTRL_YPOS  0x10
#define CTRL_Z1POS 0x30
#define CTRL_Z2POS 0x40
#define CTRL_XPOS  0x50

static void h1900_pen(int irq, void* data, struct pt_regs *regs)
{
	int pressed;
	unsigned char ctrlbyte;
	unsigned int bytes[4];
	
	pressed = GPLR(GPIO_NR_H1900_PEN_IRQ_N) & GPIO_bit(GPIO_NR_H1900_PEN_IRQ_N);
	pressed = !pressed;
#ifdef UBERDEBUGGY
	printk("%s: state %d\n", __FUNCTION__, pressed);
#endif
	if (!pressed)
	{
		/* tell the HAL that we have been released. */
		h3600_hal_touchpanel(0, 0, 0);
		return;
	};
	
	/* read positions. */
	bytes[0] = spi_ctrl(CTRL_START | CTRL_YPOS);
	bytes[1] = spi_ctrl(CTRL_START | CTRL_Z1POS);
	bytes[2] = spi_ctrl(CTRL_START | CTRL_Z2POS);
	bytes[3] = spi_ctrl(CTRL_START | CTRL_XPOS);
	
	/* output positions. */
	/* <Hymie> You're an uberdebuggy, NonToxic
	 * <Hymie> YOU"RE AN UBERDEBUGGY */
#ifdef UBERDEBUGGY
	printk("%s: %03X-%03x-%03x-%03x\n", __FUNCTION__, bytes[0], bytes[1], bytes[2], bytes[3]);
#endif

	/* tell the HAL about the position */
	h3600_hal_touchpanel(bytes[3], bytes[0], 1);
	
	/* and, timerize it. */
	mod_timer (&timer, jiffies + (SAMPLE_TIMEOUT * HZ) / 1000);
};

static void h1900_ts_timer(unsigned long nr)
{
	h1900_pen(0,NULL,NULL);
};

int h1900_ts_init( void )
{
	if ( !machine_is_h1900() ) {
		printk("%s: unknown iPAQ model %s\n", __FUNCTION__, h3600_generic_name() );
		return -ENODEV;
	}
	
	printk("%s: initializing the touchscreen.\n", __FUNCTION__);
	
	/* disable before mucking about */
	SSCR0 &= ~SSCR0_SSE_ENABLED;
	
	/* set to Microwire mode */
	SSCR0 &= ~0x00000030;
	SSCR0 |= SSCR0_FRF_MICROWIRE;
	
	/* don't use the external clock */
	SSCR0 &= ~SSCR0_ECS_USE_SSPEXTCLK;
	
	/* set to 12-bit mode */
	SSCR0 &= ~0xF;
	SSCR0 |= 0xB;
	
	/* set to 100mhz. */
	SSCR0 &= ~0xFF00;
	SSCR0 |=  0x1100; /* 17 decimal */
	
	/* set to 8-bit command packets */
	SSCR1 &= ~SSCR1_MWDS_16BIT;
	
	/* set polarity and phase to 0 */
	SSCR1 &= ~SSCR1_SPO_FALLING;
	SSCR1 &= ~SSCR1_SPH;

	/* no interrupts, and no loopback mode. */
	SSCR1 &= ~(SSCR1_RIE|SSCR1_TIE|SSCR1_LBM);
	
	/* now to set up GPIOs... */
	GPDR(GPIO23_SCLK) |=  GPIO_bit(GPIO23_SCLK);
	GPDR(GPIO24_SFRM) |=  GPIO_bit(GPIO24_SFRM);
	GPDR(GPIO25_STXD) |=  GPIO_bit(GPIO25_STXD);
	GPDR(GPIO26_SRXD) &= ~GPIO_bit(GPIO26_SRXD);
	set_GPIO_mode(GPIO23_SCLK_MD);
	set_GPIO_mode(GPIO24_SFRM_MD);
	set_GPIO_mode(GPIO25_STXD_MD);
	set_GPIO_mode(GPIO26_SRXD_MD);
	
	/* and enable it! */
	SSCR0 |= SSCR0_SSE_ENABLED;	

	/* now set up the pen action GPIO */
	set_GPIO_IRQ_edge(GPIO_NR_H1900_PEN_IRQ_N, GPIO_BOTH_EDGES);
	request_irq (IRQ_GPIO(GPIO_NR_H1900_PEN_IRQ_N), h1900_pen, SA_SAMPLE_RANDOM, "h1900 pen", NULL);

	/* we may have to do a read here to make sure that penint comes on, in case it's in a weird state. */
	spi_ctrl(CTRL_START);
	
	/* set up the timer. */
	init_timer(&timer);
	timer.function = h1900_ts_timer;
	timer.data = (unsigned long)NULL;
	
	return 0;
}

void h1900_ts_exit( void )
{
	SSCR0 &= ~SSCR0_SSE_ENABLED;
	
	/* make sure to free the IRQ... */
	free_irq(IRQ_GPIO(GPIO_NR_H1900_PEN_IRQ_N), NULL);
	
	/* now free the timer... */
	del_timer_sync (&timer);
}

module_init(h1900_ts_init)
module_exit(h1900_ts_exit)
