#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <linux/wait.h>
#include <linux/timer.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/tqueue.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>

static int ignore_cis_vcc = 1;

static dev_info_t dev_info = "hc_sl811_cs";
static dev_link_t *dev_list = NULL;

static dev_link_t *sl811hs_attach (void);
static void sl811hs_detach (dev_link_t *link);
static void sl811hs_release (u_long arg);

extern void sl811_hc_interrupt (int irq, void *__hci, struct pt_regs *r);

static void cs_error(client_handle_t handle, int func, int ret)
{
	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
}

#define CS_CHECK(fn, args...) \
while ((last_ret = CardServices(last_fn = (fn), args)) != 0) goto cs_failed

#define CFG_CHECK2(fn, args...) \
do { int ret = CardServices(fn, args); \
if (ret != 0) { \
	cs_error(link->handle, fn, ret); \
	goto next_entry; \
} \
} while (0)

static dev_node_t node;

static int 
sl811hs_config (dev_link_t *link)
{
	int ret;
	tuple_t tuple;
	cisparse_t parse;
	int last_fn, last_ret;
	u_char buf[64];
	config_info_t conf;
	cistpl_cftable_entry_t dflt = { 0 };
	void *hci;

	tuple.DesiredTuple = CISTPL_CONFIG;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);
	tuple.TupleOffset = 0;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	CS_CHECK(GetTupleData, link->handle, &tuple);
	CS_CHECK(ParseTuple, link->handle, &tuple, &parse);
	link->conf.ConfigBase = parse.config.base;
	link->conf.Present = parse.config.rmask[0];

	CS_CHECK(GetConfigurationInfo, link->handle, &conf);
	link->conf.Vcc = conf.Vcc;

	/* Look for an appropriate configuration table entry in the CIS */
	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	for (;;) {
		cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
		CFG_CHECK2(GetTupleData, link->handle, &tuple);
		CFG_CHECK2(ParseTuple, link->handle, &tuple, &parse);

		if (cfg->flags & CISTPL_CFTABLE_DEFAULT)
			dflt = *cfg;
		if (cfg->index == 0) {
			printk("index is 0\n");
			goto next_entry;
		}
		link->conf.ConfigIndex = cfg->index;
	
		/* Use power settings for Vcc and Vpp if present */
		/*  Note that the CIS values need to be rescaled */
		if (cfg->vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != cfg->vcc.param[CISTPL_POWER_VNOM] /
			    10000 && !ignore_cis_vcc) {
				goto next_entry;
			}
		} else if (dflt.vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != dflt.vcc.param[CISTPL_POWER_VNOM] /
			    10000 && !ignore_cis_vcc) {
				goto next_entry;
			}
		}

		if (cfg->vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				cfg->vpp1.param[CISTPL_POWER_VNOM] / 10000;
		else if (dflt.vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				dflt.vpp1.param[CISTPL_POWER_VNOM] / 10000;

		/* Do we need to allocate an interrupt? */
		if (cfg->irq.IRQInfo1 || dflt.irq.IRQInfo1)
			link->conf.Attributes |= CONF_ENABLE_IRQ;

		/* IO window settings */
		link->io.NumPorts1 = link->io.NumPorts2 = 0;
		if ((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
			cistpl_io_t *io = (cfg->io.nwin) ? &cfg->io : &dflt.io;
			link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
			if (!(io->flags & CISTPL_IO_8BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
			if (!(io->flags & CISTPL_IO_16BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
			link->io.IOAddrLines = io->flags &
				CISTPL_IO_LINES_MASK;
			link->io.BasePort1 = io->win[0].base;
			link->io.NumPorts1 = io->win[0].len;
			if (io->nwin > 1) {
				link->io.Attributes2 = link->io.Attributes1;
				link->io.BasePort2 = io->win[1].base;
				link->io.NumPorts2 = io->win[1].len;
			}
		}

		/* This reserves IO space but doesn't actually enable it */
		CFG_CHECK2(RequestIO, link->handle, &link->io);

		/* This configuration table entry is OK */
		break;

	next_entry:
		CS_CHECK(GetNextTuple, link->handle, &tuple);
	}

	/*
	 * Allocate an interrupt line.  Note that this does not assign a
	 * handler to the interrupt, unless the 'Handler' member of the
	 * irq structure is initialized.
	 */
	if (link->conf.Attributes & CONF_ENABLE_IRQ) {
		int i;
		link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
		link->irq.IRQInfo1 = IRQ_LEVEL_ID;
		link->irq.IRQInfo2 = 0;
		link->irq.Handler = NULL;
		link->irq.Instance = NULL;
		CS_CHECK(RequestIRQ, link->handle, &link->irq);
	}

	/*
	 * This actually configures the PCMCIA socket -- setting up
	 * the I/O windows and the interrupt mapping, and putting the
	 * card and host interface into "Memory and IO" mode.
	 */
	CS_CHECK(RequestConfiguration, link->handle, &link->conf);

	ret = sl811_hc_add_hci (link->io.BasePort1, link->io.BasePort1 + 1,
				link->irq.AssignedIRQ, &link->priv);
	if (ret) {
		printk ("register hci %d\n", ret);
		return ret;
	}

	/* Finally, report what we've done */
	printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d",
	       dev_info, link->conf.ConfigIndex,
	       link->conf.Vcc / 10, link->conf.Vcc % 10);
	if (link->conf.Vpp1)
		printk(", Vpp %d.%d", link->conf.Vpp1 / 10,
		       link->conf.Vpp1 % 10);
	if (link->conf.Attributes & CONF_ENABLE_IRQ)
		printk(", irq %d", link->irq.AssignedIRQ);
	if (link->io.NumPorts1)
		printk(", io 0x%04x-0x%04x", link->io.BasePort1,
		       link->io.BasePort1+link->io.NumPorts1-1);
	if (link->io.NumPorts2)
		printk(" & 0x%04x-0x%04x", link->io.BasePort2,
		       link->io.BasePort2+link->io.NumPorts2-1);
	printk("\n");

	link->dev = &node;

	link->state |= DEV_CONFIG;
	link->state &= ~DEV_CONFIG_PENDING;

	return 0;

 cs_failed:
	cs_error(link->handle, last_fn, last_ret);
	sl811hs_release((u_long)link);
	return 1;
}


static void 
sl811hs_release (u_long arg)
{
	dev_link_t *link = (dev_link_t *)arg;

	if (link->open) {
		printk("%s: release postponed, '%s' still open\n",
		      dev_info, link->dev->dev_name);
		link->state |= DEV_STALE_CONFIG;
		return;
	}

	sl811_hc_release_hci (link->priv);

	CardServices(ReleaseConfiguration, link->handle);
	if (link->io.NumPorts1)
		CardServices(ReleaseIO, link->handle, &link->io);
	if (link->irq.AssignedIRQ)
		CardServices(ReleaseIRQ, link->handle, &link->irq);

	link->state &= ~DEV_CONFIG;
}

static int
sl811hs_event (event_t event, int priority, event_callback_args_t *args)
{
	dev_link_t *link = args->client_data;

	switch (event) {
	case CS_EVENT_CARD_INSERTION:
		link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
		sl811hs_config (link);
		break;

	case CS_EVENT_CARD_REMOVAL:
		link->state &= ~DEV_PRESENT;
		if (link->state & DEV_CONFIG) {
			sl811hs_release((u_long)link);
		}
		break;

	case CS_EVENT_PM_SUSPEND:
		link->state |= DEV_SUSPEND;
		sl811_hc_suspend (link->priv);
		break;

	case CS_EVENT_RESET_PHYSICAL:
		if (link->state & DEV_CONFIG) {
			if (link->open) {
				/* ... */
			}
			CardServices(ReleaseConfiguration, link->handle);
		}
		break;

	case CS_EVENT_PM_RESUME:
		link->state &= ~DEV_SUSPEND;
		sl811_hc_resume (link->priv);
		break;

	case CS_EVENT_CARD_RESET:
		if (link->state & DEV_CONFIG) {
			CardServices(RequestConfiguration, link->handle,
				     &link->conf);
			if (link->open) {
				/* ... */
			}
		}
		break;

	default:
		printk("sl811hs: unknown event %d\n", event);
		break;
	}
	return 0;
}

/* allocate local data and register with CardServices
 * initialize dev_link structure, but do not configure the card yet */
static dev_link_t *
sl811hs_attach (void)
{
	dev_link_t *link;
	client_reg_t client_reg;
	int ret;
#if 0
	local_info_t *local;
#endif

	for (link = dev_list; link; link = link->next) {
		if (link->state & DEV_STALE_LINK) {
			printk("%s: flushing stale link\n", dev_info);
			sl811hs_detach(link);
		}
	}

	link = kmalloc(sizeof(dev_link_t), GFP_KERNEL);
	if (link == NULL)
		return NULL;

	memset(link, 0, sizeof(dev_link_t));

	link->release.function = &sl811hs_release;
	link->release.data = (u_long)link;

	link->conf.Vcc = 33;
	link->conf.IntType = INT_MEMORY_AND_IO;

	/* register with CardServices */
	link->next = dev_list;
	dev_list = link;
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_IO_CLIENT;
	client_reg.EventMask = CS_EVENT_CARD_INSERTION |
		CS_EVENT_CARD_REMOVAL |
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &sl811hs_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;
	ret = CardServices (RegisterClient, &link->handle, &client_reg);
	if (ret != CS_SUCCESS) {
		cs_error(link->handle, RegisterClient, ret);
		sl811hs_detach (link);
		return NULL;
	}
	return link;
}

static void 
sl811hs_detach (dev_link_t *link)
{
	dev_link_t **linkp;

	for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
		if (*linkp == link)
			break;
	if (*linkp == NULL) {
		printk(KERN_WARNING "%s: Attempt to detach non-existing "
		       "PCMCIA client\n", dev_info);
		return;
	}

	del_timer(&link->release);
	if (link->state & DEV_CONFIG) {
		printk("%s: detach postponed, '%s' still locked\n",
		       dev_info, link->dev->dev_name);
		sl811hs_release((u_long)link);
		if (link->state & DEV_STALE_CONFIG) {
			link->state |= DEV_STALE_LINK;
			return;
		}
	}

	if (link->handle) {
		int res = CardServices(DeregisterClient, link->handle);
		if (res) {
			printk("CardService(DeregisterClient) => %d\n", res);
			cs_error(link->handle, DeregisterClient, res);
		}
	}

	*linkp = link->next;

	kfree (link);
}

static int __init 
sl811hs_cs_init (void)
{
	servinfo_t serv;

	CardServices(GetCardServicesInfo, &serv);
	if (serv.Revision != CS_RELEASE_CODE) {
		printk(KERN_NOTICE
		       "%s: CardServices release does not match!\n", dev_info);
		return -1;
	}

	register_pccard_driver(&dev_info, &sl811hs_attach, &sl811hs_detach);  
	return 0;
}

static void __exit
sl811hs_cs_exit (void)
{
	unregister_pccard_driver(&dev_info);
}

module_init(sl811hs_cs_init);
module_exit(sl811hs_cs_exit);

MODULE_AUTHOR("Nexus Electronics Ltd");
MODULE_DESCRIPTION("Support for SL811HS USB host adapter");
MODULE_SUPPORTED_DEVICE("SL811HS USB host");
MODULE_LICENSE("GPL");
