/* minivol - A tiny volume control tray app

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/


#include <libmb/mb.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef ENABLE_NLS
# include <libintl.h>
# define _(text) gettext(text)
#else
# define _(text) (text)
#endif

#ifdef USE_PNG
#define IMG_EXT "png"
#else
#define IMG_EXT "xpm"
#endif

#define POPUP_WIDTH  33
#define POPUP_HEIGHT 130
#define POPUP_PTR_SZ 10

#define VOLUME_ZERO 0
#define VOLUME_LOW  1
#define VOLUME_MID  2
#define VOLUME_HIGH 3

#define VOLUME_ZERO_IMG "minivol-zero."  IMG_EXT
#define VOLUME_LOW_IMG  "minivol-low."   IMG_EXT
#define VOLUME_MID_IMG  "minivol-mid."   IMG_EXT
#define VOLUME_HIGH_IMG "minivol-high." IMG_EXT

static char *ImgLookup[64] = {
  VOLUME_ZERO_IMG,
  VOLUME_LOW_IMG,
  VOLUME_MID_IMG,
  VOLUME_HIGH_IMG
};

static char *ThemeName = NULL;
static MBPixbuf *pb;
static MBPixbufImage *Imgs[4] = { 0,0,0,0 }, *ImgsScaled[4] = { 0,0,0,0 };

static Window WinPopup;
static int    mixerfd;
static Bool   PopupIsMapped = False;
static int    PopupYOffset  = 5;
static Bool   SliderActive  = False;
static int vol, orig_vol;
static unsigned char *err_str = NULL;

static void 
popup_shape(MBTrayApp *app, int fx, int width, 
	    int height, Bool flip, Bool is_vert)
{
  Display *dpy    = mb_tray_app_xdisplay(app);
  Window win_root = mb_tray_app_xrootwin(app);
  int screen      = mb_tray_app_xscreen(app);

  XPoint tri[3];
  Pixmap mask;
  
  GC ShapeGC;
  int hoff = 10 ;
  
  mask = XCreatePixmap( dpy, win_root, width, height, 1 );
  ShapeGC = XCreateGC( dpy, mask, 0, 0 );
  
  XSetForeground( dpy, ShapeGC, BlackPixel( dpy, screen ));
  XFillRectangle( dpy, mask, ShapeGC, 0, 0, width, height );
  XSetForeground( dpy, ShapeGC, WhitePixel( dpy, screen ));
  XSetBackground( dpy, ShapeGC, BlackPixel( dpy, screen ));
  
  if (flip)
    {
      XFillRectangle(dpy, mask, ShapeGC, 10, 0, width-20, height-hoff);
      XFillRectangle(dpy, mask, ShapeGC, 0, 10, width, height-hoff-20);
      
      XFillArc( dpy, mask, ShapeGC, 0, 0, 21, 21, 90*64, 90*64 );
      XFillArc( dpy, mask, ShapeGC, width-21, 0, 21, 21, 0, 90*64 );
      
      if ( (fx+hoff) > width)
	XFillRectangle( dpy, mask, ShapeGC, width-21, height-21-hoff,
			21, 21);
      else
	XFillArc( dpy, mask, ShapeGC, width-21, height-21-hoff,
		  21, 21, 270*64, 90*64 );

      if ( (fx-hoff) < 0 )
	XFillRectangle( dpy, mask, ShapeGC, 0, height-21-hoff,
			21, 21);
      else
	XFillArc( dpy, mask, ShapeGC, 0, height-21-hoff,
		  21, 21, 180*64, 90*64 );

      tri[0].x = fx;	        tri[0].y = height;
      tri[1].x = fx+hoff;	tri[1].y = height-hoff;
      tri[2].x = fx-hoff;	tri[2].y = height-hoff;;

   } else {
      XFillRectangle(dpy, mask, ShapeGC, 10, hoff, width-20, height-hoff);
      XFillRectangle(dpy, mask, ShapeGC, 0, hoff+10, width, height-hoff-20);
      XFillRectangle(dpy, mask, ShapeGC, 0, hoff, width-20, height-hoff-20);
      
      //XFillArc( dpy, mask, ShapeGC, 0, hoff, 21, 21, 90*64, 90*64 );
      XFillArc( dpy, mask, ShapeGC, width-21, hoff, 21, 21, 0, 90*64 );
      XFillArc( dpy, mask, ShapeGC, width-21, height-21,
		21, 21, 270*64, 90*64 );
      XFillArc( dpy, mask, ShapeGC, 0, height-21, 21, 21, 180*64, 90*64 );
      
      tri[0].x = fx;	        tri[0].y = 0;
      tri[1].x = fx+hoff;       tri[1].y = hoff;
      tri[2].x = fx-hoff;	tri[2].y = hoff;
   }

  if (!is_vert)
    XFillPolygon( dpy, mask, ShapeGC, tri, 3, Nonconvex, CoordModeOrigin );
   
   XShapeCombineMask( dpy, WinPopup, ShapeBounding, 0, 0, mask, ShapeSet);
   
   XFreePixmap( dpy, mask );
   XFreeGC( dpy, ShapeGC );
}


static void
popup_create(MBTrayApp *app)
{
  Display *dpy    = mb_tray_app_xdisplay(app);
  Window win_root = mb_tray_app_xrootwin(app);
  int screen      = mb_tray_app_xscreen(app);
  GC  gc;
  int i;
  XSetWindowAttributes attr;
  XColor xcol_bg, exact;
  Pixmap popup_backing;

  gc = XCreateGC(dpy, win_root, 0, NULL);

  XAllocNamedColor(dpy, DefaultColormap(dpy, screen), "gold", 
		   &xcol_bg, &exact);

  XSetBackground(dpy, gc, xcol_bg.pixel);
  XSetForeground(dpy, gc, xcol_bg.pixel);

  popup_backing = XCreatePixmap(dpy, win_root, POPUP_WIDTH, POPUP_HEIGHT, 
				DefaultDepth(dpy, screen));

  XFillRectangle(dpy, popup_backing, gc, 0, 0, POPUP_WIDTH, POPUP_HEIGHT);

  XSetForeground(dpy, gc, BlackPixel(dpy, screen));
  
  for(i=0;i<=50;i++)
    XDrawLine(dpy, popup_backing, gc, 
	      4+((i/2)), 4+(i*2)+PopupYOffset, 
	      4+25, 4+(i*2)+PopupYOffset );


  attr.override_redirect = True;
  attr.event_mask = ButtonPressMask|ExposureMask
    |ButtonMotionMask|PointerMotionHintMask;
  /* attr.background_pixel = xcol_bg.pixel; */
  attr.background_pixmap = popup_backing;

  /*   attr.background_pixel = bg_col.pixel;
       attr.border_pixel = fg_col.pixel; */
  
  WinPopup = XCreateWindow(dpy, win_root,
			    400, 100, POPUP_WIDTH, POPUP_HEIGHT, 0,
			    DefaultDepth(dpy, DefaultScreen(dpy)),
			    CopyFromParent,
			    DefaultVisual(dpy, DefaultScreen(dpy)),
			    CWOverrideRedirect|CWEventMask|
			    /*CWBackPixel|*/ CWBackPixmap,
			    &attr);

  XFreePixmap(dpy, popup_backing);

  popup_shape(app, 8, POPUP_WIDTH, POPUP_HEIGHT, (PopupYOffset == 5),
	      mb_tray_app_tray_is_vertical(app));

  XFreeGC( dpy, gc );

  XSelectInput(dpy, WinPopup, VisibilityChangeMask|ExposureMask
	       |ButtonPressMask|ButtonReleaseMask|
	       ButtonMotionMask|PointerMotionMask); 
}

void popup_destroy(MBTrayApp *app)
{
  Display *dpy    = mb_tray_app_xdisplay(app);
  XDestroyWindow(dpy, WinPopup);
}

void popup_draw_slider(MBTrayApp *app, int volume)
{
  Display *dpy    = mb_tray_app_xdisplay(app);
  Window win_root = mb_tray_app_xrootwin(app);
  GC gc;

  gc = XCreateGC(dpy, win_root, 0, NULL);
  XClearWindow(dpy, WinPopup);
  XDrawRectangle(dpy, WinPopup, gc, 2, (100-volume) + PopupYOffset, 29, 8); 
  XFreeGC( dpy, gc );
}

int set_vol(MBTrayApp *app, int volume)
{
  int new_vol;

  vol = volume;

  if (vol>100) vol = 100;
  if (vol<0)   vol = 0;
  
  new_vol = vol | (vol << 8);
  if (ioctl(mixerfd, SOUND_MIXER_WRITE_VOLUME, &new_vol)
      == -1)
    fprintf(stderr, "minivol: unable to set volume\n");
    
  popup_draw_slider(app, vol);

  mb_tray_app_repaint(app);

  return vol;
}

/* -- tray app callbacks -- */

void
paint_callback (MBTrayApp *app, Drawable drw )
{
  int img_index = 0;
  MBPixbufImage *img_backing = NULL;
  
  img_backing = mb_tray_app_get_background (app, pb);

  if (!err_str)
    {
      img_index = vol/25;

      if (img_index > 3) img_index = 3;
      if (img_index < 0) img_index = 0;
    }

  /* CurrentVolLevel */
  mb_pixbuf_img_composite(pb, img_backing, 
			  ImgsScaled[img_index], 
			  0, 0);

  mb_pixbuf_img_render_to_drawable(pb, img_backing, drw, 0, 0);

  mb_pixbuf_img_free( pb, img_backing );
}

void
resize_callback (MBTrayApp *app, int w, int h )
{
  int i;

  for (i=0; i<4; i++)
    {
      if (ImgsScaled[i] != NULL) mb_pixbuf_img_free(pb, ImgsScaled[i]);
      ImgsScaled[i] = mb_pixbuf_img_scale(pb, Imgs[i], w, h);
    }
}

void
xevent_callback (MBTrayApp *app, XEvent *xevent)
{
  Display *dpy    = mb_tray_app_xdisplay(app);
  Window win_root = mb_tray_app_xrootwin(app);

  switch (xevent->type) 
    {
    case ButtonPress:
      if (err_str)
	{
	  mb_tray_app_tray_send_message(app, err_str, 5000);
	  break;
	}

      if (xevent->xmotion.window == WinPopup)
	{
	  orig_vol = vol = set_vol(app, 100-(xevent->xmotion.y-2-PopupYOffset));
	  SliderActive = True;
	} else {
	  Window win_mouse_root, win_mouse;
	  int popup_win_x, popup_win_y, x, y;
	  unsigned int mouse_mask;
	  
	  if (PopupIsMapped)
	    {
	      XUngrabPointer(dpy, CurrentTime);
	      popup_destroy(app);
	      PopupIsMapped = False;
	    } else {
	      XWindowAttributes root_attr;
	      
	      XGetWindowAttributes(dpy, win_root, &root_attr);

	      XGrabPointer(dpy, win_root, True,
			   (ButtonPressMask|ButtonReleaseMask),
			   GrabModeAsync,GrabModeAsync, 
			   None, None, CurrentTime);
	      
	      
	      XQueryPointer(dpy, win_root,
			    &win_mouse_root, &win_mouse,
			    &x, &y, &popup_win_x, &popup_win_y, 
			    &mouse_mask);
	      
	
	      if (mb_tray_app_tray_is_vertical (app))
		{
		  if (x > (DisplayWidth(mb_tray_app_xdisplay(app), 
					mb_tray_app_xscreen(app)) /2) )
		    x -= ( mb_tray_app_width(app) + POPUP_WIDTH );
		  else
		    x += mb_tray_app_width(app);

		  if ( (y - POPUP_HEIGHT) < 0 ) 
		    y = 0;
		  else if ( (y + POPUP_HEIGHT) > root_attr.height )
		    y = root_attr.height - POPUP_HEIGHT;
 		}
	      else
		{
		  if (y < POPUP_HEIGHT)
		    { PopupYOffset = 15; y = mb_tray_app_height(app); }
		  else
		    { PopupYOffset = 5; y -= (POPUP_HEIGHT + mb_tray_app_height(app)); }
		  x -= (mb_tray_app_width(app)/2);
		}

	      popup_create(app);

	      XMoveWindow(dpy, WinPopup, x, y);   
	      /*  (PopupYOffset == 5) ? y-136 : 20); */
	      XMapRaised(dpy, WinPopup);
	      popup_draw_slider(app, vol);
	      PopupIsMapped = True;
	    }
	}
      break;
    case ButtonRelease:
      SliderActive = False;
      break;
    case MotionNotify:
      if (SliderActive)
	{
	  int vol = 100-(xevent->xmotion.y+2-PopupYOffset);
	  if ((vol > 0) && abs(vol-orig_vol) > PopupYOffset)
	    { 
	      orig_vol = set_vol(app, vol);
	      popup_draw_slider(app, vol);
	    } 
	  
	}
      break;
    }
}

void
load_icons(void)
{
 int   i;
 char *icon_path;

  for (i=0; i<4; i++)
    {
      if (Imgs[i] != NULL) mb_pixbuf_img_free(pb, Imgs[i]);
      icon_path = mb_dot_desktop_icon_get_full_path (ThemeName, 
						     32, 
						     ImgLookup[i]);
      
      if (icon_path == NULL 
	  || !(Imgs[i] = mb_pixbuf_img_new_from_file(pb, icon_path)))
	{
	  fprintf(stderr, "minivol: failed to load icon\n" );
	  exit(1);
	}

      free(icon_path);
    }
}

void 
theme_callback (MBTrayApp *app, char *theme_name)
{
  printf("%s() THEME CB with %s\n", __func__, theme_name);
  if (!theme_name) return;
  if (ThemeName) free(ThemeName);
  ThemeName = strdup(theme_name);
  load_icons(); 	
  resize_callback (app, mb_tray_app_width(app), mb_tray_app_width(app) );
}


int
main( int argc, char *argv[])
{
  char *mixer_dev = "/dev/mixer";
  int devmask = 0;

  MBTrayApp *app = NULL;

#if ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, DATADIR "/locale");
  bind_textdomain_codeset (PACKAGE, "UTF-8"); 
  textdomain (PACKAGE);
#endif

   if ((mixerfd = open(mixer_dev, O_RDWR)) < 0) {
     err_str = _("Cant open mixer device");
   }

   if (!err_str && ioctl(mixerfd, SOUND_MIXER_READ_DEVMASK, &devmask) == -1) {
     err_str = _("Cant open mixer device for reading");
   }

   /* SOUND_MIXER_WRITE to set volume ... */
   if (!err_str && ioctl(mixerfd, SOUND_MIXER_READ_VOLUME, &orig_vol)==-1)
   {
     err_str = _("Cant open mixer device for writing");
   }
   if (!err_str)
     vol = ( (orig_vol & 0xff) + ((orig_vol >> 8) & 0xff) ) / 2;

   app = mb_tray_app_new ( _("Volume Control"),
			   resize_callback,
			   paint_callback,
			   &argc,
			   &argv );  
   
   if (!app) exit(0); 

   pb = mb_pixbuf_new(mb_tray_app_xdisplay(app), 
		      mb_tray_app_xscreen(app));
   
   mb_tray_app_set_theme_change_callback (app, theme_callback );

   mb_tray_app_set_xevent_callback (app, xevent_callback );
   
   load_icons();

   mb_tray_app_set_icon(app, pb, Imgs[3]);
   
   mb_tray_app_main (app);
   
   return 1;
}
