/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Functions for manipulating file descriptor sets.
 *
 * Author: Mike Butler,    mgb@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *
 * $Id: UtFd.h,v 1.1.1.1 2003/10/24 10:34:57 br1 Exp $
 */

#ifndef _UtFd_h
#define _UtFd_h

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>

 // Useful utilities on file descriptors...
/* Compute the union of two FD sets... */
/* #### This is CPU expensive...  Is there a better portable way?? */
inline fd_set 
FD_UNION(const fd_set &s1, const fd_set &s2)
{ 
  static int fdlimit = 0;
  fd_set fds;
  FD_ZERO(&fds);

  if(!fdlimit) {
    struct rlimit limit;
    assert(!getrlimit(RLIMIT_NOFILE, &limit));
    fdlimit = limit.rlim_cur;
  }
  for(int fd = 0; fd < fdlimit; fd++) 
    if(FD_ISSET(fd, &s1) || FD_ISSET(fd, &s2)) FD_SET(fd, &fds);
      
  return(fds);
}

/* Count the number of set file descriptors... */
/* #### This is CPU expensive...  Is there a better portable way?? */
inline int 
FD_COUNT(const fd_set &s)
{ 
  static int fdlimit = 0;
  int count = 0;

  if(!fdlimit) {
    struct rlimit limit;
    assert(!getrlimit(RLIMIT_NOFILE, &limit));
    fdlimit = limit.rlim_cur;
  }

  for(int fd = 0; fd < fdlimit; fd++) 
    if(FD_ISSET(fd, &s)) count++;
      
  return(count);
}

/* Compute the intersection of two FD sets... */
inline fd_set 
FD_INTERSECT(const fd_set &s1, const fd_set &s2)
{ 
  static int fdlimit = 0;
  fd_set fds;
  FD_ZERO(&fds);

  if(!fdlimit) {
    struct rlimit limit;
    assert(!getrlimit(RLIMIT_NOFILE, &limit));
    fdlimit = limit.rlim_cur;
  }

  for(int fd = 0; fd < fdlimit; fd++) 
    if(FD_ISSET(fd, &s1) && FD_ISSET(fd, &s2)) FD_SET(fd, &fds);
      
  return(fds);
}

#endif
