/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef GenericChannel_H
#define GenericChannel_H

#include "Channel.h"

#include "Statistics.h"

#include "GenericReadBuffer.h"

//
// Set the verbosity level.
//

#define PANIC
#undef  TEST
#undef  DEBUG

//
// Define this to log a line when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// This class implements the client
// side compression of X protocol.
//

class GenericChannel : public Channel
{
  public:

  GenericChannel(Transport *transport, Compressor *compressor,
                     Decompressor *decompressor);

  virtual ~GenericChannel();

  virtual int handleRead(EncodeBuffer &encodeBuffer);

  virtual int handleWrite(const unsigned char *message, unsigned int length);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                              const unsigned char *buffer, const unsigned int size);

  virtual int handleSplit(EncodeBuffer &encodeBuffer, int packetLimit);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                unsigned char *buffer, const unsigned int size);

  virtual int handleUnsplit(DecodeBuffer &decodeBuffer);

  virtual int handleAbortSplit(EncodeBuffer &encodeBuffer);

  virtual int handleAbortSplit(DecodeBuffer &decodeBuffer);

  virtual int handleMotion(EncodeBuffer &encodeBuffer, int forceMotion = 0)
  {
    return 0;
  }

  virtual int handleWakeup(EncodeBuffer &encodeBuffer)
  {
    return 0;
  }

  virtual int handleConfiguration();

  virtual int handleFinish();

  virtual int handleNotify(T_notification_type type)
  {
    return 0;
  }
    
  virtual int handleReset();

  virtual int needSplit() const
  {
    return 0;
  }

  virtual int needMotion() const
  {
    return 0;
  }

  virtual int needWakeup() const
  {
    return 0;
  }

  virtual int updateWakeup()
  {
    return 0;
  }

  virtual int needLimit()
  {
    return (transport_ -> length() >=
                control -> TransportGenericBufferLimit);
  }

  virtual int getLimit()
  {
    return (control -> TransportGenericBufferLimit);
  }

  virtual T_channel_type getType() = 0;

  protected:

  //
  // Model generic channels' encoding and decoding
  // policy.
  //

  virtual int isCompressed() = 0;

  //
  // Set flush policy.
  //

  virtual int isPrioritized() = 0;

  virtual int isFlushable() = 0;

  //
  // Generic channel have a lower congestion threshold
  // than X channels.
  //

  virtual int isCongested()
  {
    return (transport_ -> blocked() == 1);
  }

  //
  // Generic channels are considered not reliable.
  //

  virtual int isReliable()
  {
    return 0;
  }

  //
  // Register protocol bits for
  // specific service.
  //

  virtual void addProtocolBits(unsigned int bitsIn, unsigned int bitsOut) = 0;

  //
  // Channel's own read buffer.
  //

  GenericReadBuffer readBuffer_;

  private:

  //
  // Keep track of object's creation
  // and deletion.
  //

  #ifdef REFERENCES

  static int references_;

  #endif
};

class SyncChannel : public GenericChannel
{
  public:

  SyncChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~SyncChannel()
  {
  }

  protected:

  virtual T_channel_type getType()
  {
    return CHANNEL_SYNC;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isFlushable()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addSyncBits(bitsIn, bitsOut);
    }
  }
};

class KeybdChannel : public GenericChannel
{
  public:

  KeybdChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~KeybdChannel()
  {
  }

  protected:

  virtual T_channel_type getType()
  {
    return CHANNEL_KEYBD;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  //
  // Reduce latency on embedded keyboard channel.
  //

  virtual int isPrioritized()
  {
    return 1;
  }

  virtual int isFlushable()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addKeybdBits(bitsIn, bitsOut);
    }
  }
};

class SambaChannel : public GenericChannel
{
  public:

  SambaChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~SambaChannel()
  {
  }

  protected:

  virtual T_channel_type getType()
  {
    return CHANNEL_SAMBA;
  }

  virtual int isCompressed()
  {
    return 1;
  }

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isFlushable()
  {
    return 0;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addSambaBits(bitsIn, bitsOut);
    }
  }
};

class MediaChannel : public GenericChannel
{
  public:

  MediaChannel(Transport *transport, Compressor *compressor,
                   Decompressor *decompressor)

    : GenericChannel(transport, compressor, decompressor)
  {
  }

  virtual ~MediaChannel()
  {
  }

  protected:

  virtual T_channel_type getType()
  {
    return CHANNEL_MEDIA;
  }

  virtual int isCompressed()
  {
    return 0;
  }

  //
  // Reduce latency on media channel.
  //

  virtual int isPrioritized()
  {
    return 0;
  }

  virtual int isFlushable()
  {
    return 1;
  }

  virtual void addProtocolBits(unsigned int bitsIn,
                                   unsigned int bitsOut)
  {
    if (control -> CollectStatistics)
    {
      statistics -> addMediaBits(bitsIn, bitsOut);
    }
  }
};

#endif /* GenericChannel_H */
