/* Input pipe
 * Ben Lynn
 */
/*
Copyright (C) 2002 Benjamin Lynn (blynn@cs.stanford.edu)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define INPIPELINK "inpipe"

#include <stdio.h>
#include <string.h>

//for atoi
#include <stdlib.h>

//for tolower
#include <ctype.h>

#include <pthread.h>

//xmms includes
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "common.h"
#include "config.h"
#include "in.h"

//need following declarations so internal XMMS functions can be called
//(ripped from XMMS source)
//from xmms.c
gboolean playlist_load(gchar * inpipefile);
//modified by Lutz Mndle, lmaendle@csi.com
//to be compatible with xmms >= 1.2.8
//gboolean playlist_save(gchar * inpipefile);
gboolean playlist_save(gchar * inpipefile, gboolean is_pls);

//from widget.h
void lock_widget_list(GList * wlist);
void unlock_widget_list(GList * wlist);

//from main.c
void mainwin_general_menu_callback(gpointer cb_data, guint action, GtkWidget * w);
enum
{
	MAINWIN_GENERAL_ABOUT, MAINWIN_GENERAL_PLAYFILE, MAINWIN_GENERAL_PLAYDIRECTORY,
	MAINWIN_GENERAL_PLAYLOCATION, MAINWIN_GENERAL_FILEINFO,
	MAINWIN_GENERAL_SHOWMWIN, MAINWIN_GENERAL_SHOWPLWIN,
	MAINWIN_GENERAL_SHOWEQWIN, MAINWIN_GENERAL_PREV, MAINWIN_GENERAL_PLAY,
	MAINWIN_GENERAL_PAUSE, MAINWIN_GENERAL_STOP, MAINWIN_GENERAL_NEXT,
	MAINWIN_GENERAL_STOPFADE, MAINWIN_GENERAL_BACK5SEC,
	MAINWIN_GENERAL_FWD5SEC, MAINWIN_GENERAL_START, MAINWIN_GENERAL_BACK10,
	MAINWIN_GENERAL_FWD10, MAINWIN_GENERAL_JTT, MAINWIN_GENERAL_JTF,
	MAINWIN_GENERAL_EXIT
};

enum {
    default_skip = 5,
};

pipe_t inpipe;

static jumptable_t playlist_jt;

static GString *infile;

static gchar *inpipelink;

extern GList *mainwin_wlist;

static void add_playlist_command(gchar *s, void (* function)())
{
    jumptable_add(playlist_jt, s, function);
}

void in_open()
{
    pipe_assign(inpipe, infile->str);

    pipe_open(inpipe);
    if (!inpipe->is_open) {
	xmms_show_message("XMMSPipe Error", "Error opening input pipe", "Ok",
		FALSE, NULL, NULL);
	return;
    }

    if (config->idauto) {
	FILE *fp;
	gchar *idprefix;

	idprefix = g_strconcat(config->idprefix->str, username, NULL);
	fp = fopen(idprefix, "w");
	if (fp) {
	    fprintf(fp, "%s\n", inpipe->name);
	    fclose(fp);
	}
	g_free(idprefix);
    }

    if (!session) common_symlink(inpipe->name, inpipelink);
}

void in_close()
{
    pipe_close(inpipe);
}

static void set_playlist_pos()
{
    gint playlist_length = xmms_remote_get_playlist_length(session);
    gint i;

    if (arg) {
	i = atoi(arg);
	if (i >= 1 && i <= playlist_length) {
	    xmms_remote_set_playlist_pos(session, i - 1);
	}
    }
}

static void playlist_delete()
{
    gint playlist_length = xmms_remote_get_playlist_length(session);
    gint i;

    if (arg) {
	i = atoi(arg);
	if (i >= 1 && i <= playlist_length) {
	    xmms_remote_playlist_delete(session, i - 1);
	}
    }
}

static void playlist_delete_current()
{
    gint playlist_length = xmms_remote_get_playlist_length(session);
    gint i;
    if (playlist_length > 0) {
	i = xmms_remote_get_playlist_pos(session);
	xmms_remote_playlist_delete(session, i);
    }
}

/* skipf and skipb thanks to Andy Gimblett */
static void skip_forward()
{
    gint time = xmms_remote_get_output_time(session);
    double d, skip = default_skip;

    if (arg) {
	d = atof(arg);
	if (d >= 0.0) {
	    skip = d;
	}
    }
    xmms_remote_jump_to_time(session, time + 1000 * skip);
}

static void skip_backward()
{
    gint time = xmms_remote_get_output_time(session);
    double d, skip = default_skip;

    if (arg) {
	d = atof(arg);
	if (d > 0.0) {
	    skip = d;
	}
    }
    xmms_remote_jump_to_time(session, time - 1000 * skip);
}

static void playlist_prev()
{
    xmms_remote_playlist_prev(session);
}

static void playlist_next()
{
    xmms_remote_playlist_next(session);
}

static void playlist_clear()
{
    xmms_remote_playlist_clear(session);
}

static void playlist_add()
{
    GList *songlist = NULL;

    if (arg) {
	songlist = g_list_append(songlist, arg);
	xmms_remote_playlist_add(session, songlist);
	g_list_free(songlist);
    }
}

static void play()
{
    xmms_remote_play(session);
}

static void local_pause()
{
    xmms_remote_pause(session);
}

static void play_pause()
{
    if (xmms_remote_is_playing(session))
	xmms_remote_pause(session);
    else
	xmms_remote_play(session);
}

static void stop()
{
    xmms_remote_stop(session);
}

static void local_playlist_load()
{
    if (arg) playlist_load(arg);
}

static void local_playlist_save()
{
//modified by Lutz Mndle, lmaendle@csi.com
//to be compatible with xmms >= 1.2.8
//    if (arg) playlist_save(arg,);
    if (arg) playlist_save(arg,FALSE);
}

static gint last_vl;
static gint last_vr;
static gint mute_flag = 0;

static void mute()
{
    if (mute_flag) {
	mute_flag = 0;
	xmms_remote_set_volume(session, last_vl, last_vr);
    } else {
	mute_flag++;
	xmms_remote_get_volume(session, &last_vl, &last_vr);
	xmms_remote_set_volume(session, 0, 0);
    }
}

static void jump_to(gchar *searchstr)
{
    gint oldpos = xmms_remote_get_playlist_pos(session);
    gint i;
    gint playlist_length = xmms_remote_get_playlist_length(session);
    gchar *s, *s1, *s2;
    GList *foundlist = NULL, *here = NULL;

    s2 = g_strdup(searchstr);
    for (s = s2; *s; s++) {
	*s = tolower(*s);
    }

    for (i=0; i<playlist_length; i++) {
	s1 = g_strdup(xmms_remote_get_playlist_file(session, i));
	for (s = s1; *s; s++) {
	    *s = tolower(*s);
	}
	if (strstr(s1, s2)) {
	    foundlist = g_list_append(foundlist, GINT_TO_POINTER(i));
	    if (oldpos == i) here = g_list_last(foundlist);
	}
	g_free(s1);
    }
    g_free(s2);

    if (here) {
	if (here->next) {
	    xmms_remote_set_playlist_pos(session, GPOINTER_TO_INT(here->next->data));
	} else {
	    xmms_remote_set_playlist_pos(session, GPOINTER_TO_INT(foundlist->data));
	}
    } else if (foundlist) {
	xmms_remote_set_playlist_pos(session, GPOINTER_TO_INT(foundlist->data));
    }
    g_list_free(foundlist);
}

static void jump()
{
    if (arg) jump_to(arg);
}

static void open_jump_window()
{
    lock_widget_list(mainwin_wlist);
    mainwin_general_menu_callback(0, MAINWIN_GENERAL_JTF, 0);
    unlock_widget_list(mainwin_wlist);
}

static void jump_to_time()
{
    if (arg) xmms_remote_jump_to_time(session, atoi(arg));
}

static void quit()
{
    xmms_remote_quit(session);
}

static void toggle_repeat()
{
    xmms_remote_toggle_repeat(session);
}

static void toggle_shuffle()
{
    xmms_remote_toggle_shuffle(session);
}

static void repeat()
{
    if (arg) {
	if (!strcmp(arg, "on")) {
	    if (xmms_remote_is_repeat(session) == FALSE) xmms_remote_toggle_repeat(session);
	}
	if (!strcmp(arg, "off")) {
	    if (xmms_remote_is_repeat(session) == TRUE) xmms_remote_toggle_repeat(session);
	}
	if (!strcmp(arg, "toggle")) {
	    xmms_remote_toggle_repeat(session);
	}
    }
}

static void shuffle()
{
    if (arg) {
	if (!strcmp(arg, "on")) {
	    if (xmms_remote_is_shuffle(session) == FALSE) xmms_remote_toggle_shuffle(session);
	}
	if (!strcmp(arg, "off")) {
	    if (xmms_remote_is_shuffle(session) == TRUE) xmms_remote_toggle_shuffle(session);
	}
	if (!strcmp(arg, "toggle")) {
	    xmms_remote_toggle_shuffle(session);
	}
    }
}

static void add_volume()
{
    if (arg) {
	gint i;

	if (mute_flag) {
	    mute_flag = 0;
	    xmms_remote_set_volume(session, last_vl, last_vr);
	} else {
	    xmms_remote_get_volume(session, &last_vl, &last_vr);
	}
	i = atoi(arg);
	last_vl = volume_clip(last_vl + i);
	last_vr = volume_clip(last_vr + i);
	xmms_remote_set_volume(session, last_vl, last_vr);
    }
}

static void set_volume()
{
    if (arg) {
	gint l, r;
	gchar *s = arg;

	argsplit(s);

	l = atoi(s);

	if (arg) r = atoi(arg);
	else r = l;

	if (mute_flag) mute_flag = 0;

	l = volume_clip(l);
	r = volume_clip(r);
	xmms_remote_set_volume(session, l, r);
    }
}

static void set_balance()
{
    if (arg) xmms_remote_set_balance(session, balance_clip(atoi(arg)));
}

static void window_change(gboolean (* predicate)(gint),
	void (* change)(gint, gboolean))
{
    if (!strcmp(arg, "show") || !strcmp(arg, "on")) {
	change(session, TRUE);
    }
    if (!strcmp(arg, "hide") || !strcmp(arg, "off")) {
	change(session, FALSE);
    }
    if (!strcmp(arg, "toggle")) {
	change(session, !predicate(session));
    }
}

static void window()
{
    char *s;
    int allflag = 0;
    if (!arg) return;
    s = arg;
    argsplit(s);
    if (!arg) return;
    if (!strcmp(s, "all")) allflag = 1;
    if (allflag || !strcmp(s, "main")) {
	window_change(xmms_remote_is_main_win, xmms_remote_main_win_toggle);
    }
    if (allflag || !strcmp(s, "eq")) {
	window_change(xmms_remote_is_eq_win, xmms_remote_eq_win_toggle);
    }
    if (allflag || !strcmp(s, "pl")) {
	window_change(xmms_remote_is_pl_win, xmms_remote_pl_win_toggle);
    }
}

static void playlist_command()
{
    subcommand(playlist_jt);
}

static void id_setauto()
{
    setauto(&config->idauto);
}

static void id_setprefix()
{
    if (arg) {
	g_string_assign(config->idprefix, arg);
    }
}

static void id_command()
{
    gchar *s;
    if (!arg) return;
    s = arg;
    argsplit(s);
    if (!strcmp(s, "auto")) {
	id_setauto();
    }
    if (!strcmp(s, "prefix")) {
	id_setprefix();
    }
}

static void set_preamp()
{
    if (!arg) return;
    xmms_remote_set_eq_preamp(session, atof(arg));
}

static void set_band()
{
    if (arg) {
	gint band;
	gfloat value;
	gchar *s = arg;

	argsplit(s);

	band = atoi(s);

	if (arg) value = atof(arg);
	else value = 0.0;

	xmms_remote_set_eq_band(session, band, value);
    }
}

void in_init()
{
    pipe_init(inpipe);
    jumptable_init(playlist_jt);
    infile = g_string_new(config->inprefix->str);
    g_string_append(infile, filesuffix);
    inpipelink = g_strconcat(homedir, "/.xmms/", INPIPELINK, NULL);

    in_open();

    add_command("set_playlist_pos", set_playlist_pos);
    add_command("playlist_delete", playlist_delete);
    add_command("delete", playlist_delete);
    add_command("playlist_delete_current", playlist_delete_current);
    add_command("delete_current", playlist_delete_current);
    add_command("skip_forward", skip_forward);
    add_command("skipf", skip_forward);
    add_command("skip_backward", skip_backward);
    add_command("skipb", skip_backward);
    add_command("playlist_prev", playlist_prev);
    add_command("prev", playlist_prev);
    add_command("playlist_next", playlist_next);
    add_command("next", playlist_next);
    add_command("playlist_clear", playlist_clear);
    add_command("clear", playlist_clear);
    add_command("playlist_add", playlist_add);
    add_command("add", playlist_add);
    add_command("jump", jump);
    add_command("jump_window", open_jump_window);
    add_command("toggle_repeat", toggle_repeat);
    add_command("toggle_shuffle", toggle_shuffle);
    add_command("playlist_save", local_playlist_save);
    add_command("save", local_playlist_save);
    add_command("playlist_load", local_playlist_load);
    add_command("load", local_playlist_load);

    add_command("mute", mute);
    add_command("stop", stop);
    add_command("jump_to_time", jump_to_time);
    add_command("quit", quit);
    add_command("add_volume", add_volume);
    add_command("play", play);
    add_command("pause", local_pause);
    add_command("play_pause", play_pause);

    add_command("volume", set_volume);
    add_command("balance", set_balance);
    add_command("playlist", playlist_command);
    add_playlist_command("load", local_playlist_load);
    add_playlist_command("save", local_playlist_save);
    add_playlist_command("add", playlist_add);
    add_playlist_command("jump_window", open_jump_window);
    add_playlist_command("set_position", set_playlist_pos);
    add_playlist_command("set_pos", set_playlist_pos);
    add_playlist_command("goto", set_playlist_pos);
    add_command("goto", set_playlist_pos);
    add_playlist_command("delete_current", playlist_delete_current);
    add_playlist_command("delete", playlist_delete);
    add_playlist_command("prev", playlist_prev);
    add_playlist_command("next", playlist_next);
    add_playlist_command("clear", playlist_clear);
    add_playlist_command("jump", jump);

    add_command("shuffle", shuffle);
    add_command("repeat", repeat);
    add_command("window", window);
    add_command("saveconfig", config_save);

    add_command("id", id_command);

    add_command("preamp", set_preamp);
    add_command("band", set_band);
}

void in_free()
{
    pipe_free(inpipe);
    jumptable_free(playlist_jt);
}

void in_reopen()
{
    if (inpipe->is_open) in_close();
    in_open();
}
