/*
   SHARP CE-AG06 SDL frontend
   Copyright 2005 Alexander Chukov <sash@pdaXrom.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <zlib.h>
#include <jpeglib.h>
#include <errno.h>
#include <time.h>

#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>

#include "zcamera.h"
#include "font.h"

#define WIDTH	320
#define HEIGHT	240

static int cam_width, cam_height;
static SDL_Surface *screen;
static int exitRequested;
static char *buf;
static int rotate;
static int zoom;
static int mode;
static fbcon_font_desc *gfx_font;

#define OSD_SHOW_TIME	2

static clock_t OSD_timer;
static char *OSD_text;

int write_jpeg (char *filepath, char *filename, unsigned short *data, int width, int height, int quality)
{
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr jerr;
    FILE *fp;
    int i, j;
    unsigned char *line;
    int line_length;
    char filename_full[256];

    unsigned char *rgb;
  
    sprintf(filename_full,"%s/%s",filepath, filename);
    if (NULL == (fp = fopen (filename_full, "w"))) {
        fprintf (stderr, "write_jpeg: can't open %s: %s\n", filename_full,
                 strerror (errno));
        return -1;
    }

    cinfo.err = jpeg_std_error (&jerr);
    jpeg_create_compress (&cinfo);
    jpeg_stdio_dest (&cinfo, fp);
    cinfo.image_width = width;
    cinfo.image_height = height;
    cinfo.input_components = 3;
    cinfo.in_color_space = JCS_RGB;
    jpeg_set_defaults (&cinfo);
    jpeg_set_quality (&cinfo, quality, TRUE);
    jpeg_start_compress (&cinfo, TRUE);
    line_length = width * 3;

    rgb = (unsigned char *) malloc(line_length);


    for (i = 0, line = data; i < height; i++, line += line_length) {
	for (j = 0; j < width; j++) {
	    rgb[j * 3 + 0] = ((data[i * width + j] >> 11) << 3) & 0xff;
	    rgb[j * 3 + 1] = ((data[i * width + j] >> 5 ) << 2) & 0xff;
	    rgb[j * 3 + 2] = ((data[i * width + j]      ) << 3) & 0xff;
	}

	jpeg_write_scanlines (&cinfo, &rgb, 1);
    }

    free(rgb);

    jpeg_finish_compress (&(cinfo));
    jpeg_destroy_compress (&(cinfo));

    fclose (fp);

    return 0;
}

char *snap_filename (char *base, char *ext)
{
  static time_t last = 0;
  static int count = 0;
  static char *filename = NULL;
  
  time_t now;
  struct tm *tm;
  char timestamp[32];
  char *pt;

  time (&now);
  tm = localtime (&now);

  if (last != now)
    count = 0;
  last = now;
  count++;
   

  if (filename != NULL)
    free (filename);
  filename  = malloc (strlen (base) + strlen (ext) + 32);

  strftime (timestamp, 31, "%Y%m%d-%H%M%S", tm);
  sprintf (filename, "%s-%s-%d.%s", base, timestamp, count, ext);

  pt = filename;
  while (*pt != 0)
    {
      if (*pt == '/')
        *pt = '|';
      pt++;
    }

  return filename;
}

void write_image(char *data, int w, int h)
{
    char *filename;
    char dir[128];
    filename = snap_filename("zcam", "jpg");
    printf("%s\n", filename);
    sprintf(dir, "%s/Documents", getenv("HOME"));
    write_jpeg(dir, filename, (unsigned short *) data, w, h, 75);
    free(filename);
}

static inline void set_pixel_col(unsigned short *scr, int x, int y, unsigned short c) {
    scr[y * screen->w + x] = c;
}

void put_char(unsigned short *scr, int x, int y, unsigned int c, unsigned short fgcol, unsigned short bgcol)
{
    int i,j,bits;

    for (i = 0; i < gfx_font->height; i++) {
	bits = gfx_font->data [gfx_font->height * c + i];
	for (j = 0; j < gfx_font->width; j++, bits <<= 1) {
	    if (bits & 0x80) {
		set_pixel_col (scr, x + j, y + i, fgcol);
	    } else {
		set_pixel_col (scr, x + j, y + i, bgcol);
	    }
	}
    }
}

void put_string(unsigned short *scr, int x, int y, char *s, unsigned short fgcol, unsigned short bgcol)
{
    int i;
    int tx = x;
    
    for (i = 0; *s; i++, x += gfx_font->width, s++) {
	if (*s == '\n') {
	    x = tx - gfx_font->width;
	    y += gfx_font->height;
	} else put_char (scr, x, y, (unsigned char) *s, fgcol, bgcol);
	if (x + gfx_font->width * 2 - 1 > screen->w) {
	    x = tx - gfx_font->width;
	    y += gfx_font->height;
	}
    }
}

#define getR(v)	((((v) >> 11) << 3) & 0xff)
#define getG(v)	((((v) >> 5 ) << 2) & 0xff)
#define getB(v) ((((v)      ) << 3) & 0xff)

void downscale_image(unsigned short *src, unsigned short *dst, int w, int h)
{
    int i, j;
    h >>= 1;
    w >>= 1;
    for (j = 0; j < h; j++) {
	for (i = 0; i < w; i++) {
	    unsigned int r,g,b;
	    unsigned int pu = *(unsigned int *) src;
	    unsigned int pd = *(unsigned int *) ((unsigned short *)src + (w << 1));
	    r = (getR(pu & 0xffff) + getR(pu >> 16) + getR(pd & 0xffff) + getR(pd >> 16)) >> 2;
	    g = (getG(pu & 0xffff) + getG(pu >> 16) + getG(pd & 0xffff) + getG(pd >> 16)) >> 2;
	    b = (getB(pu & 0xffff) + getB(pu >> 16) + getB(pd & 0xffff) + getB(pd >> 16)) >> 2;
	    *dst = ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);


	    src+=2;
	    dst++;

////	    dst[j * w + i] = src[j * w * 4 + i * 2];
	}
	src += (w << 1);
    }
}

int CamThread(void *data)
{
    int fShutter = 0;
    int _mode = -1;
    int _rotate;
    int _zoom;
    int _OSD_timer = 0;
    
    cam_open();

    buf = (char *) malloc(640*480*2);

    do {
	SDL_Flip( screen );

	if (_mode != mode) {
	    _mode = mode;
	    _rotate = 0;
	    _zoom = 1;
	    rotate = _rotate;
	    zoom = _zoom;
	    fShutter = 0;
	    
	    /* restore default state */
	    cam_setCaptureFrame(cam_width, cam_height, _zoom * 256, _rotate);
	    cam_clearShutterLatch();

	    if (_mode) {
		cam_width = 640;
		cam_height = 480;
	    } else {
		cam_width = 320;
		cam_height = 240;
	    }
	} else {
	    _rotate = rotate;
	    _zoom = zoom;
	}

	if (OSD_timer) {
	    _OSD_timer = clock();
	    OSD_timer = 0;
	}

	cam_setCaptureFrame(cam_width, cam_height, _zoom * 256, _rotate);
    
	//sleep(1);
	usleep(100 * (_mode?3:1));
    
	cam_captureFrame(cam_width, cam_height, _zoom * 256, _mode?buf:screen->pixels);
	
	if (_mode)
	    downscale_image((unsigned short *)buf, (unsigned short *)screen->pixels, cam_width, cam_height);

	if (_OSD_timer) {
	    if ((_OSD_timer + CLOCKS_PER_SEC * OSD_SHOW_TIME) > clock()) {
		put_string(screen->pixels, 2, 2, OSD_text, 0xffff, 0x0000);
	    } else {
		_OSD_timer = 0;
	    }
	}
	
	if (cam_keyOn()) {
	    if (fShutter == 0) {
		system("play /usr/share/sounds/camera.wav");
		write_image(_mode?buf:screen->pixels, cam_width, cam_height);
		fShutter++;
	    } else {
		fShutter = 0;
	    }
	} else {
	}

    } while( exitRequested == 0);	//

    free(buf);
    
    cam_close();

    return 0;
}

int main(int argc, char *argv[])
{
    SDL_Thread *thread;
    SDL_Event event;
    int done;
    
    SDL_Init(SDL_INIT_VIDEO);
    
    cam_width = WIDTH;
    cam_height= HEIGHT;
    
    screen = SDL_SetVideoMode(cam_width, cam_height, 16, SDL_HWSURFACE|SDL_DOUBLEBUF);

    SDL_WM_SetCaption("zcamera", "zcamera");
    
    gfx_font = &font_vga_8x8;
    rotate = 0;
    zoom = 1;
    mode = 1;
    exitRequested = 0;
    
    thread = SDL_CreateThread(CamThread, NULL);
    if (thread == NULL) {
	fprintf(stderr, "Couldn't create thread: %s\n", SDL_GetError());
	exit(1);
    }

    done = 0;

    while ( !done && SDL_WaitEvent(&event) ) {
	switch(event.type) {
	    case SDL_QUIT:
		done=1;
		break;
	    case SDL_KEYDOWN: {
		SDLKey sdlkey=event.key.keysym.sym;
		int k=0;
		switch(sdlkey){
		    case SDLK_UP:	
			if (mode || rotate)
			    break;
			zoom += (zoom < 2)?1:0; 
			OSD_text = "x2";
			OSD_timer = 1;
			break;
		    case SDLK_DOWN:
			if (mode || rotate)
			    break;
			zoom -= (zoom > 1)?1:0; 
			OSD_text = "x1";
			OSD_timer = 1;
			break;
		    case SDLK_LEFT:	
			if (mode || (zoom == 2))
			    break;
			rotate = 0;
			OSD_text = "Portrait";
			OSD_timer = 1;
			break;
		    case SDLK_RIGHT:	
			if (mode || (zoom == 2))
			    break;
			rotate = 1;
			OSD_text = "Landscape";
			OSD_timer = 1;
			break;
		    case SDLK_ESCAPE:	
			done = 1; 
			break;
		    case SDLK_RETURN:
			mode = !mode;
			if (mode)
			    OSD_text = "VGA";
			else
			    OSD_text = "QVGA";
			OSD_timer = 1;
			break;
		    default: k = (int)sdlkey;
		    }
		}; break;
	    default:
		break;
	}
    }

    exitRequested=1;
    SDL_WaitThread(thread, NULL);

    SDL_Quit();
    return 0;
}
