/*
 *  Copyright (C) 2005 David J. Kessler <dkessler@kopsisengineering.com>
 *
 *  Derived from kbdd by Nils Faerber:
 *  Copyright (C) 2004,2005 Nils Faerber <nils.faerber@kernelconcepts.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA
 */

#define MAIN 1

#include "zkbdd.h"

#include <string.h>
#include <errno.h>
#include <limits.h>
#include <signal.h>
#include <dirent.h>

#include "uinput.h"
#include "dev_uinput.h"
#include "ztokens.h"

#define ARRAY_SIZE(A)	(sizeof(A)/sizeof(A[0]))

/* This is where we remember the keyboard type specified
 * on the command line. */
char keyboard_type[64];

char debug=0;
int uindev=0;

/* The following globals correspond to fields that should
 * be defined in the driver script file. Any that aren't
 * defined will end up set to 0 later so don't try to
 * establish defaults here. */
unsigned char scancode_fn = 0;
unsigned char scancode_punc = 0;
unsigned char scancode_num = 0;
unsigned char scancode_shiftl = 0;
unsigned char scancode_shiftr = 0;
unsigned char scancode_caps = 0;
int has_num_lock = 0;
int has_punc_lock = 0;
int has_fn_lock = 0;

/* The is the global state object for the Lua VM */
lua_State* L;

/* The following is correct for C7XX, C8XX, 6000, 1000, and
 * 3X00 series Zaurii. If used with older models (5000D, 5500)
 * specify "-p /dev/ttyS2" on the command line. */

#define DEFAULT_TTS "/dev/ttyS1"
char TTY_PORT[PATH_MAX] = DEFAULT_TTS;

unsigned int keys_down[8] = {0, 0, 0, 0, 0, 0, 0, 0};

void set_keydown(unsigned char key)
{
	int i;
	int j;

	i = key / 32;
	j = key % 32;

	keys_down[i] = keys_down[i] | (1 << j);

	if (debug) {
		for (i = 0; i < 8; i++) {
			printf("0x%08x\n", keys_down[i]);
		}
	}
}

void clear_keydown(unsigned char key)
{
	int i;
	int j;

	i = key / 32;
	j = key % 32;

	keys_down[i] = keys_down[i] & (0xffffffff ^ (1 << j));

	if (debug) {
		for (i = 0; i < 8; i++) {
			printf("0x%08x\n", keys_down[i]);
		}
	}
}

int open_serial(char *port, speed_t  baud)
{
	int fd, res;
	struct termios ssetup;

	fd=open(port, O_RDWR);
	if (fd <= 0) {
		perror("open serial");
		printf("opening %s failed\n", port);
		return (-1);
	}

	res = tcgetattr(fd, &ssetup);
	if (res < 0) {
		perror("tcgetattr");
		return (-1);
	}

	ssetup.c_iflag = 0L;
	ssetup.c_oflag = 0L;
	ssetup.c_cflag &= ~(CSTOPB|PARENB|CRTSCTS);
	ssetup.c_cflag |= (CS8|CLOCAL);
	ssetup.c_lflag = 0L;
	ssetup.c_cc[VTIME] = 0;
	ssetup.c_cc[VMIN] = 1;
	cfsetispeed(&ssetup, baud);
	cfsetospeed(&ssetup, baud);

	res=tcsetattr(fd, TCSANOW, &ssetup);
	if (res < 0) {
		perror("tcsetattr");
		return (-1);
	}

	return (fd);
}


void send_z_key(unsigned short keycode, unsigned int key_down)
{
	if (!keycode) return;

	if ( key_down ) {
		if (keycode & 0x80) {
			if (debug) {
				printf("press %02x\n", ZK_MODESWITCH);
			}
			else {
				dev_uinput_key(uindev, ZK_MODESWITCH, KEY_PRESSED);
			}
		}
		if (debug) {
			printf("press %02x\n", keycode & 0x7f);
		}
		else {
			dev_uinput_key(uindev, keycode & 0x7f, KEY_PRESSED);
		}
		set_keydown(keycode);
	} 
	else {
		if (debug) {
			printf("release %02x\n", keycode & 0x7f);
		}
		else {
			dev_uinput_key(uindev, keycode & 0x7f, KEY_RELEASED);
		}
		if (keycode & 0x80) {
			if (debug) {
				printf("release %02x\n", ZK_MODESWITCH);
			}
			else {
				dev_uinput_key(uindev, ZK_MODESWITCH, KEY_RELEASED);
			}
		}
		clear_keydown(keycode);
	}
}

unsigned char lookup_scancode(char *map, unsigned char key)
{
	unsigned char keycode;

	if (debug) {
		printf("lookup_scancode(%s, 0x%02x)\n", map, key);
	}

	lua_pushstring(L, map);
	lua_gettable(L, LUA_GLOBALSINDEX);
	lua_rawgeti(L, -1, key);
	keycode = lua_tonumber(L, -1);
	if (debug) printf("keycode: 0x%02x\n", keycode);
	lua_pop(L, 2);

	return keycode;
}

int run_driver(void)
{
	int fd;
	int l;
	unsigned char buf[16];
	unsigned char key;
	unsigned int  key_down;
	unsigned char keycode;
	int shift_on = 0;
	int caps_on = 0;
	int num_on = 0;
	int num_lock = 0;
	int fn_on = 0;
	int punc_on = 0;

	if (debug) {
	       	printf("opening %s\n", TTY_PORT);
	}
	fd = open_serial(TTY_PORT, B9600);
	if (fd <= 0) {
		return (-1);
	}

	while (fd > 0) {

		l = read (fd, buf, 1);
		if (l == 1)
		{
			lua_getglobal(L, "do_scancode");
			lua_pushnumber(L, buf[0]);
			lua_call(L, 1, 2);
			key = (unsigned char)lua_tonumber(L, -2);
			key_down = (unsigned char)lua_tonumber(L, -1);
			lua_pop(L, 2);

			/* if (debug) printf("** key: 0x%02x, down: %d\n", key, key_down); */

			if (key == 0) {
				continue;
			}

			/* detect fn */
			if (key == scancode_fn) {
				if (key_down) {
					fn_on++;
					if (debug) printf("fn = %d\n", fn_on);
				}
				else {
					fn_on--;
					if (debug) printf("fn = %d\n", fn_on);
				}
			}

			/* detect punct */
			if (key == scancode_punc) {
				if (key_down) {
					punc_on++;
					if (debug) printf("punc = %d\n", punc_on);
				}
				else {
					punc_on--;
					if (debug) printf("punc = %d\n", punc_on);
				}
			}

			/* detect num */
			if (key == scancode_num) {
				if (key_down && has_num_lock) {
					if (shift_on) {
						/* shifted num is num lock */
						num_lock = !num_lock;
						if (debug) printf("numlock = %d\n", num_lock);
					}
					else {
						num_on++;
						if (debug) printf("num = %d\n", num_on);
						if (debug) printf("numlock = %d\n", num_lock);
					}
				}
				else {
					if (shift_on) {
						/* ignore */
						continue;
					}
					else {
						num_on--;
						if (debug) printf("num = %d\n", num_on);
						if (debug) printf("numlock = %d\n", num_lock);
					}
				}
			}

			/* detect shift */
			if (key == scancode_shiftl || key == scancode_shiftr) {
				if (key_down) {
					shift_on++;
					if (debug) printf("shift = %d\n", shift_on);
					if (debug) printf("caps = %d\n", caps_on);
				}
				else {
					shift_on--;
					if (debug) printf("shift = %d\n", shift_on);
					if (debug) printf("caps = %d\n", caps_on);
				}
			}

			/* detect caps lock */
			if (key == scancode_caps) {
				if (!key_down) {
					caps_on = !caps_on;
					if (debug) printf("caps = %d\n", caps_on);
				}
			}

			/* now we know all the modifiers, we can handle the key */
			
			if (fn_on) {
				keycode = lookup_scancode("map_fn", key);
				send_z_key(keycode, key_down);
				continue;
			}

			if (punc_on) {
				keycode = lookup_scancode("map_punc", key);
				send_z_key(keycode, key_down);
				continue;
			}

			if (num_on || num_lock) {
				keycode = lookup_scancode("map_num", key);
				send_z_key(keycode, key_down);
				continue;
			}

			if ((shift_on && !caps_on) || (!shift_on && caps_on)) {
				if (debug) printf("doing shift case\n");
				keycode = lookup_scancode("map_shift", key);
				if (!keycode) {
					keycode = lookup_scancode("map_normal", key);
					if (key_down) {
						/* send shift down */
						send_z_key(ZK_SHIFT, key_down);
						/* send key down */
						send_z_key(keycode, key_down);
					}
					else {
						/* send key up */
						send_z_key(keycode, key_down);
						/* send shift up */
						send_z_key(ZK_SHIFT, key_down);
					}
				}
				else {
					send_z_key(keycode, key_down);
				}
			}
			else {
				keycode = lookup_scancode("map_normal", key);
				send_z_key(keycode, key_down);
			}
		}
	}

	return 0;
}

void list_drivers(char *fmt)
{
	DIR* dir;
	struct dirent* entry;
	char* ext;

	dir = opendir("/usr/share/zkbdd/drivers");
	if (dir) {
		while ((entry = readdir(dir)) != NULL) {
			ext = strstr(entry->d_name, ".lua");
			if (ext != NULL && ext != entry->d_name) {
				*ext = 0;
				if (strncmp(entry->d_name, "ztokens", 7)) {
					printf(fmt, entry->d_name);
				}
			}
		}
		closedir(dir);
	}
}

void print_usage(char *arg0)
{
	printf ("zkbdd %s\n", VERSION);
	printf ("Usage:\n");
	printf ("%s [-d] [-h] [-l] [-c <config file>] -p <serial-port> -t <kbd type>\n", arg0);
	printf ("-d\tenable debugging output\n");
	printf ("-h\tprint this help\n");
	printf ("-l\tlist drivers\n");
	printf ("-c <config file>\n");
	printf ("\tRead port and type from config file\n");
	printf ("-p <serial-port>\n");
	printf ("\tspecify serial port device, default %s\n", DEFAULT_TTS);
	printf ("-t <kbd type>\n");
	printf ("\tspecify the serial keyboard type, supported are:\n");
	list_drivers("\t\t%s\n");
	printf ("Example:\n\t%s -t %s\n", arg0, "pocketop");
}

int find_kbd_type(const char *ktype)
{
	/* ultimately this guy needs to look through a search
	 * path and find the specified "ktype.lua" file
	 */

	strncpy(keyboard_type, ktype, 64);
	return 1;
}

void parse_config(const char *path, int *kbdtype, char port[])
{
	char *needle;
	FILE *fd;
	char buf[PATH_MAX];
        
        fd = fopen(path, "r");
	if (!fd) {
		printf("could not open config file %s\n", path);
		return;
	}
        
        while (!feof(fd)) {
		fgets(buf, PATH_MAX, fd);
		
		if (*buf == '#' || *buf == '\0') {
			/* It's a comment or a blank line */
			continue;
		}
		
		if ((needle = strstr(buf, "port:")) != 0) {
			needle += 5; 
			/* Trim whitespaces */
			while (isspace(*needle)) {
				needle++;
			}
			
			while (isspace(needle[strlen(needle)-1])) {
				needle[strlen(needle)-1] = '\0';
			}
			strncpy(port, needle, PATH_MAX);
		} 
		else if ((needle = strstr(buf, "type:")) != 0) {
			needle += 5; 
			/* Trim whitespaces */
			while (isspace(*needle)) {
				needle++;
			}
			
			while (isspace(needle[strlen(needle)-1])) {
				needle[strlen(needle)-1] = '\0';
			}
					
			*kbdtype = find_kbd_type(needle);
		}
	}
}
          
int driver_setting_num(char *key)
{
	int value;

	if (key == NULL) {
		printf("driver_setting_num: null key\n");
	}

	lua_pushstring(L, key);
	lua_gettable(L, LUA_GLOBALSINDEX);
	value = lua_tonumber(L, -1);
	if (debug) printf("driver_setting_num(%s) = %d\n", key, value);
	lua_pop(L, 1);

	return value;
}

void driver_setting_string(char *key, char *dest, int len)
{
	const char *value;

	lua_pushstring(L, key);
	lua_gettable(L, LUA_GLOBALSINDEX);
	value = lua_tostring(L, -1);
	if (value) {
		if (debug) printf("driver_setting_num(%s) = %s\n", key, value);
		strncpy(dest, value, len);
	}

	lua_pop(L, 1);
}

void handle_sigterm(int signal)
{
	int i = 0;
	int j = 0;

	printf("zkbdd shutting down\n");

	/* clear out any down keys */
	for (i = 0; i < 8; i++) {
		for (j = 0; j < 8; j++) {
			if (keys_down[i] & (1 << j)) {
				send_z_key(i * 8 + j, 0);
			}
		}
	}
	
	dev_uinput_close(uindev);
	lua_close(L);
	exit(EXIT_SUCCESS);
}

#define DESCRIPTION_LEN 80
#define VERSION_LEN 32

int main(int argc, char **argv)
{
	int optc;
	int kbdtype = -1;

	char driver[PATH_MAX] = "/usr/share/zkbdd/drivers/";
	
	char description[DESCRIPTION_LEN] = "Unknown";
	char version[VERSION_LEN] = "?.?";

	/* parse the options */
	while ((optc = getopt(argc, argv, "c:t:p:dhl")) != -1) {
		switch ((char)optc) {
			case 'h':
				print_usage(argv[0]);
				exit(1);
				break;
			case 'l':
				list_drivers("%s\n");
				exit(1);
				break;
			case 'd':
				debug = 1;
				break;
			case 't':
				kbdtype = find_kbd_type(optarg);
				break;
			case 'p':
				strncpy(TTY_PORT, optarg, PATH_MAX);
				break;
			case 'c':
				parse_config(optarg, &kbdtype, TTY_PORT);
				break;
		}
	}

	if (kbdtype == -1) {
		/* check the default config file */
		parse_config("/etc/zkbdd.conf", &kbdtype, TTY_PORT);
		if (kbdtype == -1) {
			print_usage(argv[0]);
			exit(1);
		}
	}
        
	/* initialize Lua */
	L = lua_open();
	lua_baselibopen(L);
	lua_dofile(L, "/usr/share/zkbdd/drivers/ztokens.lua");
	strncat(driver, keyboard_type, PATH_MAX);
	strncat(driver, ".lua", PATH_MAX);
	lua_dofile(L, driver);

	/* open the uinput device */
	uindev = dev_uinput_init();
	if (uindev <= 0) {
		printf("init uinput failed\n");
		exit (1);
	}

	/* hook up signal handler(s) */
	signal(SIGTERM, handle_sigterm);
	signal(SIGKILL, handle_sigterm);
	signal(SIGQUIT, handle_sigterm);
	signal(SIGSTOP, handle_sigterm);
	signal(SIGINT, handle_sigterm);

	driver_setting_string("description", description, DESCRIPTION_LEN);
	driver_setting_string("version", version, VERSION_LEN);
	printf("Zkbdd using driver: %s, version %s\n", description, version);

	scancode_fn = driver_setting_num("scancode_fn");
	scancode_punc = driver_setting_num("scancode_punc");
	scancode_num = driver_setting_num("scancode_num");
	scancode_shiftl = driver_setting_num("scancode_shiftl");
	scancode_shiftr = driver_setting_num("scancode_shiftr");
	scancode_caps = driver_setting_num("scancode_caps");
	has_num_lock = driver_setting_num("has_num_lock");
	has_punc_lock = driver_setting_num("has_punc_lock");
	has_fn_lock = driver_setting_num("has_fn_lock");

	run_driver();

	lua_close(L);
	return 0;
}

