/**********************************************************************
*
*  main.c
*
*  Descripion  - This is a common ioctl code which can be used to
*                communicate with the nvaudio hardware
*
*  Copyright (c) 2002-2003 NVIDIA Corporation
*
***********************************************************************
*/
#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <nvioctl.h>
#include "commlib.h"

static int audio_fd = 0;
static struct nv_ioctl * nvioctl = NULL;

static const char* codec_name[] =
{   
    [0] = " NONE ",
    [1] = " REALTEK ",
    [2] = " ADI ",
    [3] = " SIGMATEL ",
};

/* Function to check mixer is valid */
int is_valid()
{
    int i = 0;

    if((nvioctl == 0) && (audio_fd == 0)) {
        return -1 ;
    }


    /* clear the ioctl struct now */
    nvioctl->cmd  = 0;
    for(i=0; i < 4; i++)
         nvioctl->val[i] = 0;
    memset(nvioctl->name,0,strlen(nvioctl->name));

    return 0;
}

/*
 Function to open the mixer for nvaudio.
*/
int open_mixer()
{
    audio_fd   = open("/dev/mixer",O_WRONLY,0);
    if(audio_fd == -1) {
        return -1;
    }

    nvioctl = (struct nv_ioctl *) malloc(sizeof(struct nv_ioctl));
    if(nvioctl == NULL) {
        close(audio_fd);
        audio_fd = 0;
        return -1;
    }
    return audio_fd;
}

/*
  Free the mixer pointer
*/
void free_mixer()
{
    if(nvioctl)
        free(nvioctl);
    close(audio_fd);
    nvioctl  = 0;
    audio_fd = 0;
}

/*
  Function to set the Speaker selection
*/
int set_speaker_select(int option)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd =  NV_SET_SPKRSELECT;
    switch (option)
    {
        case 2:     /* 2 speakers*/
            nvioctl->val[1] = NV_SPKR_STEREO;
            break;
        case 0:    /* head phones*/
            nvioctl->val[1] = NV_SPKR_STEREO;
            break;
        case 4:   /* 4 speakers*/
            nvioctl->val[1] = NV_SPKR_QUAD;
            break;
        case 6:  /* 6 speakers*/
            nvioctl->val[1] = NV_SPKR_51;
            break;
    }

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;

    }

    return 0;
}

/*
    Function to get the speaker selection
*/
int get_speaker_select(void)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    =  NV_GET_SPKRSELECT;
    
    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return nvioctl->val[1];
}

/* Function to get swap linein for s. left & right */
int get_swaplinein( void )
{
	if( -1 == is_valid()) {
        return -1 ;
    }
	/* support only for realtek codec 
	check the codec type and return */

	nvioctl->cmd    = NV_GET_ONOFF;
	nvioctl->val[0] = SWAP_LINEIN;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return nvioctl->val[1];
}

/* Function to set swap linein for s. left & right */
int set_swaplinein( int enablebit)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_ONOFF;
	nvioctl->val[0] = SWAP_LINEIN;
    nvioctl->val[1] = enablebit;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;

    }

    return 0;
}

/* Function to get swap micin for center & lfe */
int get_swapmicin( void )
{
	if( -1 == is_valid()) {
        return -1 ;
    }
	/* support only for realtek codec 
	check the codec type and return */

	nvioctl->cmd    = NV_GET_ONOFF;
	nvioctl->val[0] = SWAP_MICIN;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return nvioctl->val[1];
}

/* Function to set swap micwin for center & lfe */
int set_swapmicin( int enablebit)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_ONOFF;
	nvioctl->val[0] = SWAP_MICIN;
    nvioctl->val[1] = enablebit;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;

    }

    return 0;
}

/* Function to get the Analog Output Status */
int get_analogout( void)
{
    if( -1 == is_valid()) {
        return -1 ;
    }
    nvioctl->cmd    = NV_GET_ANALOGOUT;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return nvioctl->val[1];
}

/* Function to set the Analog Output Status */
int set_analogout( int enablebit)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_ANALOGOUT;
    nvioctl->val[1] = enablebit;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;

    }

    return 0;
}

/* Function to get the Digital Output Status */
int get_digitalout ( void )
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_GET_DIGITALOUT;
    
    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return nvioctl->val[1];
}

/* Function to set the Digital Output Status */
int set_digitalout ( int enablebit)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_DIGITALOUT;
    nvioctl->val[1] = enablebit;

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return 0;
}

/* Function to set the Mic  Boost */
int set_mic_boost ( int enablebit)
{
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_MICBOOST;
    nvioctl->val[1] = enablebit;
    
    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
        return -1;
    }

    return 0;
}

/* Function to set the volumes for common registers */
int set_common_volregister( int mixernum, int mixvalue)
{
    unsigned int cmd = -1;
    int value        = 0;
    int bCommon      = 0;

    if( -1 == is_valid()) {
        return -1 ;
    }

    value = mixvalue | (mixvalue << 8);

    switch(mixernum) {
        case SOUND_MIXER_VOLUME:
        case SOUND_MIXER_PCM:
        case SOUND_MIXER_LINE:
        case SOUND_MIXER_MIC:
        case SOUND_MIXER_CD:
        case SOUND_MIXER_LINE1:
        case SOUND_MIXER_IGAIN:
        case SOUND_MIXER_PHONEOUT:
        case SOUND_MIXER_VIDEO:
        case SOUND_MIXER_PHONEIN:
            cmd     = MIXER_WRITE(mixernum);
            bCommon = 1;

            if(ioctl(audio_fd,cmd ,&value) == -1) {
                return -1;
            }

            break;
    }

    if(!bCommon) {
        cmd             = SOUND_MIXER_PRIVATE1;
        nvioctl->cmd    = NV_SET_PREMIXVOL;
        nvioctl->val[1] = mixvalue;
        switch(mixernum) {
            case NV_MIXER_PRELEFT:
                 nvioctl->val[0] = PREMIX_LEFT;
                 break;
            case NV_MIXER_PRERIGHT:
                nvioctl->val[0] = PREMIX_RIGHT;
                break;
            case NV_MIXER_PRERLEFT:
                nvioctl->val[0] = PREMIX_RLEFT;
                break;
            case NV_MIXER_PRERRIGHT:
                nvioctl->val[0] = PREMIX_RRIGHT;
                break;
            case NV_MIXER_PRECENTER:
                nvioctl->val[0] = PREMIX_CENTER;
                break;
            case NV_MIXER_PRESUB:
                nvioctl->val[0] = PREMIX_SUB;
                break;
            
      }

      if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
            return -1;
      }
    }

    return 0;
}

/* Function to get the volumes for common registers */
int get_common_volregister( int mixernum, int *mute)
{
    int regval  = -1;
    int bfound  = 0;

    if( -1 == is_valid()) {
        return -1 ;
    }
    
    *mute = 0;
    nvioctl->cmd  = NV_GET_PREMIXVOL;
    
    switch(mixernum) {
        case NV_MIXER_MICBOOST:
            nvioctl->cmd  = NV_GET_MICBOOST;
            bfound = 1;
            break;
        case NV_MIXER_PRELEFT:
            nvioctl->val[0] = PREMIX_LEFT;
            bfound = 1;
            break;
        case NV_MIXER_PRERIGHT:
            nvioctl->val[0] = PREMIX_RIGHT;
            bfound = 1;
            break;
        case NV_MIXER_PRERLEFT:
            nvioctl->val[0] = PREMIX_RLEFT;
            bfound = 1;
            break;
        case NV_MIXER_PRERRIGHT:
            nvioctl->val[0] = PREMIX_RRIGHT;
            bfound = 1;
            break;
        case NV_MIXER_PRECENTER:
            nvioctl->val[0] = PREMIX_CENTER;
            bfound = 1;
            break;
        case NV_MIXER_PRESUB:
            nvioctl->val[0] = PREMIX_SUB;
            bfound = 1;
            break;
    }

    if(bfound) {
        if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
            return -1;
        }
        regval = nvioctl->val[1];
        if((mixernum >= NV_MIXER_PRELEFT) || (mixernum <= NV_MIXER_PRESUB)) {
            *mute = nvioctl->val[0];
        }

    } else  {

        if(ioctl(audio_fd,MIXER_READ(mixernum), &regval) == -1 ) {
            return -1;
        }
    }
    return regval;
}

/* function to set mute state */
int set_mute( int mixernum, int value)
{
    
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_SET_MIXERMUTE;
    
    switch(mixernum) {
        case SOUND_MIXER_VOLUME :
            nvioctl->val[0] = NV_MASTER_MUTE;
            nvioctl->val[1] = value;
        break;
    }

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
            return -1;
    }

    return 0;
}

/* function to get mute state */
int get_mute( int mixernum)
{
    int regvalue    = 0;

    if( -1 == is_valid()) {
        return -1 ;
    }
    nvioctl->cmd    = NV_GET_MIXERMUTE;
    
    switch(mixernum) {
        case SOUND_MIXER_VOLUME :
            nvioctl->val[0] = NV_MASTER_MUTE;
            break;
    }

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1) {
            return -1;
    }

    regvalue = nvioctl->val[1];
    return regvalue;
}

/* Function to set the record input */
int set_record_channel(int inport)
{
    int recport = 0;

    if( -1 == is_valid()) {
        return -1 ;
    }

    /* read the existing selection */
    if(ioctl(audio_fd,MIXER_READ(SOUND_MIXER_RECSRC), &recport) == -1 ){
       return -1;
    }

    recport |= inport;
    if((ioctl(audio_fd, MIXER_WRITE(SOUND_MIXER_RECSRC), &recport) ==-1)) {
        return -1;
    }

    return 0;
}

/* Function to get the record input */
int get_record_channel()
{
    int recport = -1;

    if( -1 == is_valid()) {
        return -1 ;
    }

    /* read the existing selection */
    if(ioctl(audio_fd,MIXER_READ(SOUND_MIXER_RECSRC), &recport) == -1 ){
        return -1;
    }

    return recport;
}

/*Function to find the codec type*/
int get_nvinfo(int index, char* retvalue, int* retType)
{
    int value   = 0;
    char buf[4] = {0};
    
    if( -1 == is_valid()) {
        return -1 ;
    }

    nvioctl->cmd    = NV_GET_INFO;
    strcpy(nvioctl->name,"");

    switch(index) {

        case NV_MIXER_LINUXVER:
            nvioctl->val[0] = LINUX_INFO;
            break;
        case NV_MIXER_DRIVERVER:
            nvioctl->val[0] = DRIVER_INFO;
            break;
        case NV_MIXER_CODECVER:
            nvioctl->val[0] = CODEC_INFO;
            break;
    }

    if(ioctl(audio_fd,SOUND_MIXER_PRIVATE1,nvioctl) == -1 ){
        return -1;
    }

    value = nvioctl->val[1];

    switch(index) {
        case NV_MIXER_LINUXVER:
            sprintf(buf,"%d",(value >> 16));
            strcpy(retvalue,buf);
            strcat(retvalue,".");
            sprintf(buf,"%d",((value >> 8) & 0x00ff));
            strcat(retvalue,buf);
            strcat(retvalue,".");
            sprintf(buf,"%d",(value & 0x0000ff));
            strcat(retvalue,buf);
            break;
        case NV_MIXER_DRIVERVER:
            strcpy(retvalue,nvioctl->name);
            break;
        case NV_MIXER_CODECVER:
            strncpy(retvalue,codec_name[value],strlen(codec_name[value])); 
            *retType = value;
            break;

    }
    

    return 0;
}

