/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <math.h>
#include <stdint.h>

#include <glib-object.h>
#include <glib.h>
#include <gdk/gdkkeysyms.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <goffice/utils/go-color.h>
#include <gtkgesture.h>

#include "application.h"
#include "debug.h"
#include "display.h"
#include "renderer.h"
#include "slide-show.h"

struct _CriaSlideShowPrivate {
	CriaDisplay		* display;
	GnomeCanvasGroup	* group;
	GtkGestureHandler	* gesture_handler;
};

enum {
	PROP_0,
	PROP_PRESENTATION,
	PROP_SLIDE
};

enum {
	SIGNAL,
	N_SIGNALS
};

static	void	cria_slide_show_init	       (CriaSlideShow	* slide_show);
static	void	cria_slide_show_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void	cria_slide_show_next_page_event(GtkGestureHandler *gh,
						gpointer user_data_a,
						gpointer user_data_b);
static void	cria_slide_show_prev_page_event(GtkGestureHandler *gh,
						gpointer user_data_a,
						gpointer user_data_b);
static	void	cria_slide_show_render	       (CriaSlideShow	* slide_show);
static	void	cria_slide_show_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
static	guint	cria_slide_show_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_show_signal	       (CriaSlideShow	* template,
						const	gchar	* string);
#endif
typedef enum {
	KEYS_NONE,
	KEYS_FORWARD,
	KEYS_BACK,
	KEYS_CLOSE
} CriaSlideEditorAction;

static CriaSlideEditorAction
cria_key_press_event_cb_next_slide(GtkWidget* widget, GdkEventKey* event, gpointer data) {
	guint	  action = KEYS_NONE;
	guint	  slide = 0;
	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW (widget), FALSE);
	
	slide = cria_slide_show_get_slide (CRIA_SLIDE_SHOW (widget));

	/* TODO If we want to make it configurable we have to use a
	 * method instead of a switch statement */
	switch (event->keyval) {
	case GDK_space:
	case GDK_Right:
	case GDK_KP_Enter:
	case GDK_Return:
	case GDK_Page_Down:
		action = KEYS_FORWARD;
		break;
	case GDK_BackSpace:
	case GDK_Left:
	case GDK_Page_Up:
		action = KEYS_BACK;
		break;
	case GDK_Escape:
		action = KEYS_CLOSE;
		break;
	default:
		g_debug ("cria_key_press_event(): this key is not handled");
		break;
	}

	return action;
}

static gboolean
cria_key_press_event_cb(GtkWidget* widget, GdkEventKey* event, gpointer data) {
	guint			  slide  = 0;
	CriaSlideEditorAction	  action = KEYS_NONE;

	g_assert(CRIA_IS_SLIDE_SHOW(widget));

	action = cria_key_press_event_cb_next_slide(widget, event, data);
	slide = cria_slide_show_get_slide(CRIA_SLIDE_SHOW(widget));
	
	if(slide >= cria_presentation_n_slides(cria_slide_show_get_presentation(CRIA_SLIDE_SHOW(widget))) &&
	   action != KEYS_BACK) {
		g_debug("SlideShow::cbKeyPressEvent(): after show, we're quitting now");
		gtk_object_destroy(GTK_OBJECT(widget));
		return TRUE;
	}
	
	switch(action) {
	case KEYS_FORWARD:
		g_debug("SlideShow::cbKeyPressEvent(): next slide");
		if(slide < cria_presentation_n_slides(cria_slide_show_get_presentation(CRIA_SLIDE_SHOW(widget)))) {
			cria_slide_show_set_slide(CRIA_SLIDE_SHOW(widget), slide + 1);
		}
		break;
	case KEYS_BACK:
		g_debug("SlideShow::cbKeyPressEvent(): previous slide");
		if(slide > 0) {
			cria_slide_show_set_slide(CRIA_SLIDE_SHOW(widget), slide - 1);
		}
		break;
	case KEYS_CLOSE:
		g_debug("SlideShow::cbKeyPressEvent(): close slide show");
		gtk_object_destroy(GTK_OBJECT(widget));
		break;
	case KEYS_NONE:
		g_debug("SlideShow::cbKeyPressEvent(): nothing to do");
		return FALSE;
	}

	return TRUE;
}

static gboolean
cria_slide_show_button_press_event(GtkWidget* self, GdkEventButton* ev) {
	if(cria_slide_show_get_slide(CRIA_SLIDE_SHOW(self)) >= cria_presentation_n_slides(cria_slide_show_get_presentation(CRIA_SLIDE_SHOW(self)))) {
		gtk_object_destroy(GTK_OBJECT(self));
		return TRUE;
	}
	
	return FALSE;
}

static void
cria_slide_show_class_init (CriaSlideShowClass	* cria_slide_show_class) {
	GObjectClass	* g_object_class;
	GtkWidgetClass	* widget_class;
	CriaBlock	* block;

	g_debug ("cria_slide_show_class_init(%i): start", __LINE__);

	/* create the after show slide */
	cria_slide_show_class->after_show = cria_slide_new(NULL);
	cria_slide_set_title(cria_slide_show_class->after_show, _("End of Presentation"));
#warning "static SlideShow::init(): FIXME: add a background color to the after show slide"
	block = cria_block_new(cria_slide_show_class->after_show, "message");
	/* TRANSLATORS: there is a UTF-8 Character for the ellipsis (...), please use it and not three dots */
	cria_block_set_text(block, _("Press a Key or click to close this Window..."));
	cria_block_set_color(block, RGBA_TO_UINT(0xaa, 0xaa, 0xaa, 0xff));
	cria_block_set_alignment(block, CRIA_ALIGNMENT_CENTER);
	cria_block_set_valignment(block, CRIA_ALIGNMENT_MIDDLE);
	{
		GOColor  color = RGBA_BLACK;
		GoPoint* size = cria_slide_get_size(cria_slide_show_class->after_show);
		g_debug("static SlideShow::init(): Slide size %llix%lli", size->x, size->y);
#warning "static SlideShow::init(): FIXME: get rid of g_object_set here"
		cria_slide_set_background_color(cria_slide_show_class->after_show, &color);
		g_object_set(G_OBJECT(block),
			     "left", (go_unit_t)0,
			     "right", size->x,
			     "top", (go_unit_t)0,
			     "bottom", size->y,
			     NULL);
	}
	cria_slide_add_block(cria_slide_show_class->after_show, block);

	g_object_class = G_OBJECT_CLASS(cria_slide_show_class);

	/* setting up signal system */
#if 0
	cria_slide_show_class->signal = cria_slide_show_signal;

	cria_slide_show_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE_SHOW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideShowClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_show_set_property;
	g_object_class->get_property = cria_slide_show_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_PRESENTATION,
			g_param_spec_object (
				"presentation",
				"Presentation",
				"The presentation currently shown by this CriaSlideShow",
				CRIA_TYPE_PRESENTATION,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_SLIDE,
			g_param_spec_uint (
				"slide",
				"Slide",
				"The slide that's presented in this moment or that we're "
				"currently switching to",
				0,
				G_MAXUINT,
				0,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);

	/* setting up the widget class */
	widget_class = GTK_WIDGET_CLASS(cria_slide_show_class);
	widget_class->button_press_event = cria_slide_show_button_press_event;

	g_debug ("cria_slide_show_class_init(%i): end", __LINE__);
}

guint
cria_slide_show_get_slide (CriaSlideShow*self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW (self), 0);

	return self->slide;
}

CriaPresentation *
cria_slide_show_get_presentation (
		CriaSlideShow	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW(self), NULL);
	
	return self->presentation;
}

static void
cria_slide_show_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaSlideShow	* self;

	self = CRIA_SLIDE_SHOW (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->presentation);
		break;
	case PROP_SLIDE:
		g_value_set_uint (
				value,
				self->slide);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_show_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaSlideShowClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_show_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideShow),
			0,
			(GInstanceInitFunc)cria_slide_show_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_WINDOW,
				"CriaSlideShow",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_show_init(CriaSlideShow *self) {
	GdkColor	* color = g_new0 (GdkColor, 1);

	self->slide = -1;
	gdk_color_parse (
			"black",
			color);
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));
	
	g_debug ("cria_slide_show_init(%i): start", __LINE__);

	/* setting up our private data */
	self->priv = g_new0 (CriaSlideShowPrivate, 1);

	/* setting up the canvas */
	self->priv->display = cria_display_new();
	cria_display_set_zoom(self->priv->display, 1.0);

	gdk_colormap_alloc_color(gtk_widget_get_colormap(GTK_WIDGET(self->priv->display)),
				 color,
				 FALSE,
				 TRUE);
	gtk_widget_modify_bg(GTK_WIDGET(self->priv->display), GTK_STATE_NORMAL, color);

#warning "SlideShow::init(): FIXME: this one will cause a bug later"
	gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(self->priv->display), 1.0);
#warning "SlideShow::init(): FIXME: This one cannot work here as we set the presentation later"

	/* setting up gestures */
	self->priv->gesture_handler = gtk_gesture_handler_new (GTK_WIDGET (self->priv->display));
	/* adding gestures */
	gtk_gesture_add_callback (self->priv->gesture_handler, "012",
			GTK_GESTURE_FUNC (cria_slide_show_next_page_event),
			self, NULL);
	gtk_gesture_add_callback (self->priv->gesture_handler, "210",
			GTK_GESTURE_FUNC (cria_slide_show_prev_page_event),
			self, NULL);
	
	gtk_window_fullscreen (GTK_WINDOW (self));
	gtk_container_add(GTK_CONTAINER(self),
			  GTK_WIDGET(self->priv->display));
	
	g_signal_connect_swapped(G_OBJECT(self->priv->display), "key-release-event",
				 G_CALLBACK(cria_key_press_event_cb), self);

	gtk_widget_show_all(GTK_WIDGET(self));
	g_debug("cria_slide_show_init(%i): end", __LINE__);
}

static void
cria_slide_show_next_page_event(GtkGestureHandler* gh, gpointer user_data_a, gpointer user_data_b) {
	CriaSlideShow	* self;
	
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (user_data_a));
	g_return_if_fail (CRIA_SLIDE_SHOW(user_data_a)->slide <= cria_presentation_n_slides (cria_slide_show_get_presentation (CRIA_SLIDE_SHOW(user_data_a))));

	self = CRIA_SLIDE_SHOW (user_data_a);

	if (self->slide < cria_presentation_n_slides (cria_slide_show_get_presentation (self)))
	{
		cria_slide_show_set_slide (
				self,
				self->slide + 1);
	}
	else
	{
		gtk_object_destroy (GTK_OBJECT (self));
	}
}

GtkWidget*
cria_slide_show_new(CriaPresentation* p) {
	g_assert(p != NULL);
	g_assert(CRIA_IS_PRESENTATION(p));

	return g_object_new(CRIA_TYPE_SLIDE_SHOW, "presentation", p, NULL);
}

static void
cria_slide_show_prev_page_event (
		GtkGestureHandler *gh,
		gpointer user_data_a,
		gpointer user_data_b)
{
	CriaSlideShow   * self;

	g_return_if_fail (CRIA_IS_SLIDE_SHOW (user_data_a));

	self = CRIA_SLIDE_SHOW (user_data_a);

	if (self->slide > 0)
	{
		cria_slide_show_set_slide (
				self,
				self->slide - 1);
	}
		
}

static void
cria_slide_show_render(CriaSlideShow* self) {
	CriaSlide	* slide;
	GoPoint		* slide_s,
			* display_s;
	gchar		* title;

	g_assert(self != NULL && CRIA_IS_SLIDE_SHOW(self));
	g_assert(self->presentation != NULL && CRIA_IS_PRESENTATION(self->presentation));
	
	g_debug("SlideShow::render(): start");
	
	if(self->slide < cria_presentation_n_slides(cria_slide_show_get_presentation(self))) {
		slide = cria_presentation_get_slide(cria_slide_show_get_presentation(self), self->slide);
	} else {
		slide = CRIA_SLIDE_SHOW_GET_CLASS(self)->after_show;
	}

	slide_s = cria_slide_get_size(slide);
	display_s = cria_slide_get_display_size(slide, gdk_screen_get_default());
	cria_display_set_sizes(self->priv->display, slide_s, display_s);

	cria_renderer_render_slide(self->priv->display, slide);
	
	/* update window's title */
	title = g_strdup_printf(_("%s"), cria_slide_get_title(slide));
	g_debug("SlideShow::render(): title: %s", title);
	gtk_window_set_title(GTK_WINDOW(self), title);

	g_free(title);
	g_free(slide_s);
	g_free(display_s);

	g_debug("SlideShow::render(): end");
}

void
cria_slide_show_set_presentation(CriaSlideShow* self, CriaPresentation* presentation) {
	g_return_if_fail (CRIA_IS_SLIDE_SHOW(self));

	g_debug ("cria_slide_show_set_presentation(%i): start", __LINE__);

	if (self->presentation != NULL)
		g_object_unref (self->presentation);

	self->presentation = presentation;
	if (self->presentation)
		g_object_ref (presentation);

	/* ensure we don't pass invalid slide numbers */
	cria_slide_show_set_slide(self, self->slide);

	cria_slide_show_render (self);

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug ("cria_slide_show_set_presentation(%i): set presentation 0x%x", __LINE__, (uintptr_t)self->presentation);
	g_debug ("cria_slide_show_set_presentation(%i): end", __LINE__);
}

static void
cria_slide_show_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideShow	* self;
	
	self = CRIA_SLIDE_SHOW (object);
	
	switch (prop_id)
	{
	case PROP_PRESENTATION:
		cria_slide_show_set_presentation (
				self,
				g_value_get_object (value));
		break;
	case PROP_SLIDE:
		cria_slide_show_set_slide (
				self,
				g_value_get_uint (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_slide_show_set_slide(CriaSlideShow* self, guint slide) {
	g_assert(self != NULL && CRIA_IS_SLIDE_SHOW(self));
	g_assert(self->presentation != NULL && CRIA_IS_PRESENTATION(self->presentation));
	g_assert(slide == -1 || slide <= cria_presentation_n_slides(self->presentation));

	if(slide == -1) {
		slide = 0;
	}

	if(slide != self->slide) {
		self->slide = slide;
	
		if(slide == cria_presentation_n_slides(self->presentation)) {
			g_debug("SlideShow::setSlide(): setting to the after show slide");
		} else {
			g_debug("SlideShow::setSlide(): setting to %d/%d", slide + 1, cria_presentation_n_slides(self->presentation));
		}
		
		cria_slide_show_render(self);

		g_object_notify(G_OBJECT(self), "slide");
	} else {
		g_debug("SlideShow::setSlide(): slide hasn't changed");
	}
}

