import rox
from rox import g

DELETE = 1
INSERT = 2

# As operations are performed, they are added to 'in_progress'.
# When complete, the whole group is added to undo_buffer.

class Buffer(g.TextBuffer):
	"A buffer that deals with undo/redo."

	def __init__(self):
		g.TextBuffer.__init__(self, None)
		self.in_progress = None
		self.undo_buffer = []
		self.redo_buffer = []	# Will be None during undo or redo
	
	def start_undo_history(self):
		def begin(buffer): self.in_progress = []
		
		self.connect('begin-user-action', begin)
		def end_action(buffer):
			assert self.in_progress is not None
			if self.in_progress:
				self.undo_buffer.append(self.in_progress)
				self.in_progress = []
			self.in_progress = None
		self.connect('end-user-action', end_action)
		self.connect('delete-range', self.delete_range)
		self.connect('insert-text', self.insert_text)
	
	def delete_range(self, buffer, start, end):
		self.in_progress.append((INSERT,
				 start.get_offset(),
				 buffer.get_slice(start, end, g.TRUE)))
		if self.redo_buffer:
			self.redo_buffer = []
	
	def insert_text(self, buffer, where, text, len):
		text = text[:len]	# PyGtk bug?
		start = where.get_offset()
		self.in_progress.append((DELETE,
					start, start + len, text))
		if self.redo_buffer:
			self.redo_buffer = []
	
	try:
		g.TextBuffer(None).insert_at_cursor('hello')
	except:
		# Old version of pygtk
		def insert(self, iter, text):
			g.TextBuffer.insert(self, iter, text, -1)
		def insert_at_cursor(self, text):
			g.TextBuffer.insert_at_cursor(self, text, -1)

	def do_action(self, op):
		if op[0] == DELETE:
			start = self.get_iter_at_offset(op[1])
			end = self.get_iter_at_offset(op[2])
			self.delete(start, end)
		elif op[0] == INSERT:
			start = self.get_iter_at_offset(op[1])
			self.insert(start, op[2])
		else:
			rox.alert('Unknown entry in undo buffer!\n' + `op`)

	def undo(self, hist = None):
		if hist is None: hist = self.undo_buffer

		if not hist:
			g.gdk.beep()
			return	# Nothing to undo/redo

		assert not self.in_progress

		# List of actions to perform
		group = hist.pop()
		
		old_redo, self.redo_buffer = self.redo_buffer, None

		try:
			self.begin_user_action()
			group.reverse()
			for action in group:
				self.do_action(action)
		finally:
			self.redo_buffer = old_redo
			if hist is self.undo_buffer:
				self.redo_buffer.append(self.in_progress)
			else:
				self.undo_buffer.append(self.in_progress)
			self.in_progress = []

			self.end_user_action()
	
	def redo(self): self.undo(hist = self.redo_buffer)
