/******************************************************************************
//             INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2001 Intel Corporation. All Rights Reserved.
//
//  VSS: 
//		$Workfile: anc.c $
//		$Revision: 1 $
//		$Date: 03-09-10 13:18 $
//
//  Description:
//		Adaptive Noise Cancellation using FIR LMS adaptive filter 
//
******************************************************************************/

/* Standard includes */
#include <string.h>

/* Standard IPP definitions */
#include <ippdefs.h>
#include <ippSP.h>

/* External interface */
#include "anc.h"

#ifdef PERFORMANCE_TIMER
#include "oscr.h"
	
/* OSCR timer controls */
#define		START_TIMER		{ixs_get_OSCR_val(&t1);}
#define		STOP_TIMER(x)	{ixs_get_OSCR_val(&t2); g_TimerOSCR[x]+=(t2-t1); g_NumBlocks[x]++;}
#define		RESET_TIMER(x)	{g_TimerOSCR[x]=0; g_NumBlocks[x]=0;}

/* OSCR timer constants */
#define     FC				3250000.0			/* OSCR clock frequency */
#define		NUM_TIMERS		4					/* Number of timers */
#define		T_LMS			0					/* LMS adaptive filter */
#define		T_ACOUSTIC_PATH	1					/* Acoustic path filter */
#define		T_TONE_GEN		2					/* Tone generator */
#define		T_NOISE_GEN		3					/* Noise generator */

/* OSCR timer tick and block count vectors */
int g_TimerOSCR[NUM_TIMERS]={0,0,0,0};
int g_NumBlocks[NUM_TIMERS]={0,0,0,0};

/* OSCR timer display labels */
char *g_TimerDesc[NUM_TIMERS]=
{
	"LMS adaptive filter   ",
	"Acoustic path filter  ",
	"Tone generator        ",
	"Noise generator       "
};
#else

#define		START_TIMER   
#define		STOP_TIMER(x) 

#endif

/* LMS Adaptation and acoustic path parameters */
#define		LMS_LEN			15
#define		AP_TAPS			15
#define		LMS_STEPSIZE	1

/* Fixed sinusoidal parameters */
#define		AQ15			32767				/* Q15 */
#define		PHASE			0					/* 0, Q15 */
#define		F				1000				/* Desired tone frequency, in Hz */				
#define		FQ15			((F<<15)/8000)		/* F Hz for Fs=8000 */

/* SNR computation parameters */
#define		LOG10			4716				/* for sf=-11 */
#define		LSF				4096.0				/* Scalefactor for ln() */
#define		LOGLSF			17035				/* ln(LSF)*2^sf (sf=11)

/* Globals */
IppToneState_16s		g_ToneState;
IppRandGaussState_16s	g_GaussState;
Ipp16s					g_GaussStdDev;
int						g_SNR;
Ipp32s					g_TapsLMS[LMS_LEN];
int						g_TapsLen=LMS_LEN;
Ipp16s					g_LMS_DelayLine[2*LMS_LEN];
Ipp32s					g_StepSize=LMS_STEPSIZE;
int						g_LMS_DelayLineIndex=0;

/* Filter coefficients */
#include "Acoustic_Path.h"

/* Acoustic path filter memories */
Ipp16s DelayLineAP[2*AP_TAPS];
int DelayLineIndexAP=0;

/* Adjust random signal parameters */
void SetNoiseVariance(int sd)
{
	g_GaussStdDev=sd;
	ippsRandGaussInit_16s(&g_GaussState, 0, g_GaussStdDev, 0);
}

/* Apply ANC and generate requested output 

   OutputANC - noise-cancelled output
   Signal    - clean input, lengthx1 vector
   Noise     - noise "reference microphone" signal, lengthx1 vector
   SigNoise  - signal+noise processed through simulated acoustic path, lengthx1 vector
   length    - block size
   mode      - output mode: 0 s+n(processed by acoustic path), 1-clean input signal, 2-ANC output
*/

void BlockANC(Ipp16s *OutputANC, Ipp16s *Signal, Ipp16s *Noise, Ipp16s *SigNoise, int length, int mode )
{
	static int initialized=0;
	Ipp32s NoiseEnergy, SignalEnergy;
	Ipp16s tmp;
	int i;
	unsigned long t1, t2;
	float snr;
	
	/* Setup timer and signal generators */
	if (!initialized)
	{
		/* Initialize signal generators */
		ippsRandGaussInit_16s(&g_GaussState, 0, g_GaussStdDev, 0);
		ippsToneInitQ15_16s(&g_ToneState, AQ15, FQ15, PHASE);

		/* Clear filter memories */
		ippsZero_16s(DelayLineAP,2*AP_TAPS);
		for(i=0;i<LMS_LEN;i++)
			g_TapsLMS[i]=0;
		for(i=0;i<2*LMS_LEN;i++)
			g_LMS_DelayLine[i]=0;
		initialized=1;
	}

	/* Clear signal buffers */
	//ippsZero_16s(OutputANC,length);
	ippsZero_16s(Noise,length);
	ippsZero_16s(SigNoise,length);
	//ippsZero_16s(Signal,length);
	ippsCopy_16s(OutputANC,Signal,length);

	/* Generate noise */
	START_TIMER;
	ippsRandGauss_16s(Noise,length,&g_GaussState);
	STOP_TIMER(T_NOISE_GEN);

	/* Apply simulated acoustic path */
	START_TIMER;
	ippsFIR_Direct_16s(Noise,SigNoise,length,AcousticPath,AP_TAPS,DelayLineAP,&DelayLineIndexAP);
	STOP_TIMER(T_ACOUSTIC_PATH);

	/* Generate tone */
	START_TIMER;
	//ippsToneQ15_16s(Signal,length,&g_ToneState);
	STOP_TIMER(T_TONE_GEN);

	/* Update SNR */
	ippsNorm_L2_16s32s(Signal,length,&SignalEnergy);
	ippsNorm_L2_16s32s(Noise,length,&NoiseEnergy);

	/* Compute LMS target signal = signal + noise, modified by acoustic path */
	ippsAdd_16s_I(Signal,SigNoise,length);

	/* Apply LMS to the current block */
	START_TIMER;
	for(i=0;i<length;i++)
	{
		ippsFIRLMSOne_Direct_16s(Noise[i],SigNoise[i],&(OutputANC[i]),
			                     g_TapsLMS, g_TapsLen, g_StepSize, 
								 g_LMS_DelayLine, &g_LMS_DelayLineIndex);
	}
	STOP_TIMER(T_LMS);

	/* Update ANC output - remove estimated noise from s+n */
	ippsSub_16s_I(SigNoise,OutputANC,length);
		
	/* Update SNR */
	if (NoiseEnergy!=0)
	{
		if (SignalEnergy>=NoiseEnergy)
		{
			g_SNR=SignalEnergy/NoiseEnergy;
			tmp=g_SNR&0xffff;
			ippsLn_16s_ISfs(&tmp,1,-11);
			g_SNR=((20*tmp)/LOG10);
		}
		/* Handle SigEng/NoiEng < 1 using log(ab)=log(a)+log(b)=>log(a)=log(ab)-log(b) */
		else
		{
			snr=(float)SignalEnergy/(float)NoiseEnergy*LSF;
			g_SNR=snr+0.5;
			tmp=g_SNR&0xffff;
			ippsLn_16s_ISfs(&tmp,1,-11);
			tmp-=LOGLSF;
			g_SNR=((20*tmp)/LOG10);
		}
	}
	else
		g_SNR=100000;

	/* Apply filters as specified by mode */
	switch(mode)
	{
		case 0:
			/* Return signal+noise */
			ippsCopy_16s(SigNoise,OutputANC,length);
			break;
		case 1:
			/* Return signal  */
			ippsCopy_16s(Signal,OutputANC,length);
			break;
		case 2:
			/* Noise cancelled output already in the output buffer */
			break;
	}
}

/* Return SNR */
void GetSNR(int *snr)
{
	*snr=g_SNR;
}

#ifdef PERFORMANCE_TIMER
/* 
    Compute CPU cost associated with a particular timer   

	n				- timer index (n<0 gives number of timers available)
	N				- block size
	SampleRate		- sample rate
	Desc			- timer label (w_str)
*/
int GetPerformanceTimer(int n, int N, int SampleRate, char *Desc)
{
	Ipp64s PercentageCPU;
	int val;

	/* Return number of timers, otherwise %CPU associated with a particular timer */
	if (n<0)
		return(NUM_TIMERS);
	else
	{
		wcscpy(Desc,g_TimerDesc[n]);
		PercentageCPU=(Ipp64s)g_TimerOSCR[n]*(Ipp64s)SampleRate*100/(Ipp64s)(g_NumBlocks[n]*N)/FC;
		val=(int)PercentageCPU;
		return(val);
	}
}

/* Clear performance timers */
void ResetPerformanceTimers(void)
{
	int i;
	for(i=0;i<NUM_TIMERS;i++)
		RESET_TIMER(i);
}

/* Open OSCR timer device */
void InitTimer()
{
	ixs_open_OSCR();
}

/* Close OSCR timer device */
void ShutdownTimer()
{
	ixs_close_OSCR();
}

#endif

