/******************************************************************************
//               INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2000 Intel Corporation. All Rights Reserved.
//
//  VSS:
//		$Workfile: reverb.c $
//      $Revision: 1 $
//      $Date: 03-09-10 17:09 $
//      $Archive: /ipp_se/primitive/ipp4/ipp40/beta/cotulla/install/arm_linux/sample/Reverb/reverb.c $
//
//  Description:
//      Reverb example
//
******************************************************************************/

/************
Header files
*************/

/* Linux audio output device interface */
#include	<fcntl.h>					
#include	<unistd.h>
#include	<sys/ioctl.h>
#include	<linux/soundcard.h>
#include	<sched.h>
#include	<signal.h>

/* Console, file, and string I/O */
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>

/* Standard IPP definitions */
#include	"ippdefs.h"
#include	"ippSP.h"

/* Interface to IPP-based examples */
#include	"schroeder.h"

/* Local constants */
#define		FS				48000		/* Sample rate */
#define		N				16384		/* Audio buffer size */

/* User interface constants */
#define		UI_STACK_SIZE	2500		/* User interface stack size, in 32-bit words */					
#define		MSGBUF_LEN      1000		/* User interface keyboard buffer size, in bytes */
#define		PBUF_LEN		100			/* User interface parsing buffer */
#define		MAX_STR			256			/* Maximum string length, in characters */
#define		MAX_FN			256			/* Maximum filename length, in characters */

/*************
 Prototypes 
**************/
void			DisplayConsoleStartupBanner		(void);
void			ProcessCommandLineArguments		(int, char **, FILE **, FILE **, float *, int *);
void			RenderSound						(int *, Ipp16s *, int);
void			GetAudioInput					(int *dev, Ipp16s *sig, int len);
int				OpenAudioDevice					(int *, int, int);
void			CloseFile						(FILE *);
void			CloseAudioDevice				(int);
int				ReadInputFile					(FILE *fpi, Ipp16s *pcm, int len);
void			WriteOutputFile					(FILE *, Ipp16s *, int);
int				UserInterface					(void);
void			Usage							(void);

/*************
 Global data 
**************/
int g_Quit=0;					/* Quit flag, set asynchronously by the user interface */
int g_WaitForUI=0;				/* Prevents asynchronous RT60 change by the UI */

/*************************************
 Console-mode application entry point 
**************************************/
/******************************************************************************************
// Name:			 main
//
// Description:		 Modified Schroeder reverberator sample code
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//
*******************************************************************************************/
int main(int argc, char **argv)
{
	FILE *fpi;					/* Input file */							
	FILE *fpo;					/* Output file */							
	Ipp16s s[N];				/* Input PCM speech buffer */
	Ipp16s shat[N];				/* Output PCM speech buffer */
	int AudioDevice;			/* Pointer to the (Linux) audio output device */
	int AudioDevEnable;			/* Audio device enable flag */
	int StackUI[UI_STACK_SIZE];	/* User interface stack (child thread) */
	int status=-1;				/* Audio device status */
	int UIpid;					/* User interface PID */
	int frame=1;				/* Frame counter for file I/O */
	int FirstTime=1;			/* Set reverb time after Schroeder has started */
	float Tr;					/* Reverb time */

	/****************************************************
	Startup banner, parse command line, open I/O devices  
	*****************************************************/
	DisplayConsoleStartupBanner();	
	ProcessCommandLineArguments(argc,argv,&fpi,&fpo,&Tr,&AudioDevEnable);

	#ifdef PERFORMANCE_TIMER
		/* Initialize performance timer */
		InitTimer();
	#endif

	/* Spawn user interface thread */
	UIpid=__clone(UserInterface,&(StackUI[UI_STACK_SIZE]),CLONE_VM|CLONE_FS);

	/* Open audio device */
	/* Current driver opens at 48 kHz regardless of requested rate */
	if (AudioDevEnable)
	{
		status=OpenAudioDevice(&AudioDevice,2,48000);
		if (status<0)
		{
			printf("Unable to open audio device\n");
			exit(0);
		}
	}
	printf("\n");

	/* Apply reverberation to the audio buffer until user aborts */
	while(!g_Quit)
	{
		/* Fill audio input buffer */
		if (status>=0)
			GetAudioInput(&AudioDevice,s,N);
		else
		{
			ippsZero_16s(s,N);
			if (ReadInputFile(fpi,s,N)<N)
				g_Quit=1;
		}

		/* Block while the user interface thread is chagning the reverb time */
		while(g_WaitForUI);

		/* Apply reverb to current audio input buffer */
		ApplyReverb(s,shat,N,FS);

		/* Reverb time must be set after first application */
		if (FirstTime)
		{
			SetReverbTime(Tr);
			FirstTime=0;
		}

		/* Render current audio buffer */
		if (status>=0)
			RenderSound(&AudioDevice,shat,N);
		else 
		{
			WriteOutputFile(fpo,shat,N);
			printf("Frame %d\r",frame++);
			fflush(stdout);
		}
	}

	/* Close I/O devices */	
	CloseFile(fpi);
	CloseFile(fpo);
	if (status>=0)
		CloseAudioDevice(AudioDevice);

	/* Deallocate Schroeder storage */
	ShutdownReverb();

	#ifdef PERFORMANCE_TIMER
		/* Close performance timer device driver */
		ShutdownTimer();
	#endif

	/* Shutdown message */
	printf("\n\nDone.\n\n");

	/* End UI process */
	kill(UIpid,SIGKILL);

} /* End of main() */

/********************************
 Console management and file I/O 
*********************************/

/*********************************************************************************************
// Name:			 DisplayConsoleStartupBanner
//
// Description:		 Display startup message on the text console
//
// Input Arguments:  None.
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void DisplayConsoleStartupBanner()
{
	int i;
	for(i=0;i<50;i++) printf("\n");
	printf("\n\nIntel(R) Integrated Performance Primitives (IPP)\nfor Intel(R) XScale(TM) Microarchitecture.\n\n");
	printf("Modified Schroeder Artificial Reverberation Example, Version 1.00\n\n");
	printf("Copyright (C) 2000-01, Intel Corporation.  All Rights Reserved.\n\n");
}

/*********************************************************************************************
// Name:			 ProcessCommandLineArguments
//
// Description:		 Extracts run-time parameters from the command line
//					 Opens I/O files
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//					 fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None
**********************************************************************************************/
void ProcessCommandLineArguments(int argc, char **argv, FILE **fpi, FILE **fpo, float *Tr, int *AudioDev)
{
	char ifile[MAX_FN],ofile[MAX_FN];
	char *arg;
	int i;

	/* Establish defaults */
	strcpy(ifile,"sound.in");
	strcpy(ofile,"sound.out");
	*Tr=1.5;
	*AudioDev=0;
	
	/* Usage */
	if ((argc>1)&&(strcmp(argv[1],"--help")==0))
		Usage();

    /* Parse arguments */
	for(i=1;i<argc;i++)
	{
		arg=argv[i];
		switch(arg[1])
		{
			case 'a':
				*AudioDev=1;
				break;
			case 'i':
				strcpy(ifile,&(arg[2]));
				break;
			case 'o':
				strcpy(ofile,&(arg[2]));
				break;
			case 't':
				*Tr=atof(&(arg[2]));
				break;
			default:
				Usage();
				break;
		}; 
	}

	if (!(*AudioDev))
	{
		/* Open input file */
		if ((*fpi=fopen(ifile,"rb"))==NULL)
		{
			printf("Unable to open %s\n",ifile);
			exit(1);
		}
		else
			printf("Input file:\t%s\n",ifile);	
	
		/* Open output file */
		if ((*fpo=fopen(ofile,"wb"))==NULL)
		{
			printf("Unable to create %s\n",ofile);
			exit(1);
		}
		else
			printf("Output file:\t%s\n",ofile);
	}
	else
	{
		*fpi=*fpo=NULL;
		printf("Sound I/O:\tfull-duplex audio device\n");
	}

	/* Show configuration */
	printf("Reverb time:\t%2.1f seconds\n\n",*Tr);
}

/*********************************************************************************************
// Name:			 Usage
//
// Description:		 Display usage banner
//
// Returns:			 None
**********************************************************************************************/
void Usage(void)
{
	printf("Usage: reverb -i<input file> -o<output file> -t<reverb time> -a [--help], \n\nwhere\n\n");
	printf("<input file> is the desired binary pcm input file\n");
	printf("<output file> is the desired binary pcm output file\n");
	printf("<reverb time> is the desired reverb time, in seconds (""RT60"")\n");
	printf("-a forces the use of the full-duplex audio device instead of the specified files\n\n");
	printf("No input file forces use of the audio device for input/output.\n\n");
	printf("--help displays usage\n\n");
	printf("\n");
	exit(0);
}

/*********************************************************************************************
// Name:			 CloseOutputFile
//
// Description:		 Close PCM binary output file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseOutputFile(FILE *fpo)
{
	if (fpo!=NULL) fclose(fpo);
}

/*****************************
 Linux Audio Device Interface
******************************/

/***************************************************************************************
// Name:			 RenderSound
//
// Description:		 Write pcm audio samples to the audio output device. 
//					 Assumes 44.1 kHz sample rate
//
// Input Arguments:  fp					- Pointer to the audio output device. 
//					 pcm				- Pointer to a buffer of interleaved pcm audio.
//					 channels			- Number of channels contained in the buffer.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void RenderSound(int *fp, Ipp16s *pcm, int len)
{
	int i,j;
	static Ipp16s obuf[2*N];		/* audio output buffer */

	/* Replicate signal on both left and right channels */
	for(i=j=0;i<2*len;i+=2,j++)
		obuf[i]=obuf[i+1]=pcm[j];

	/* Pass audio output buffer to the audio output device */
	write(*fp,obuf,sizeof(Ipp16s)*len*2);
}

/***************************************************************************************
// Name:			 OpenAudioDevice
//
// Description:		 Initialize Linux audio output device. 
//
// Input Arguments:  dev				- Pointer to the audio output device. 
//					 channels			- Number of audio channels.
//
// Returns:			 Status				 0	 = success, device ready
//										-100 = unable to open device
//										-101 = unable to select 16-bit stereo PCM
//										-102 = unable to select 44100 Hz sample rate
//										-103 = unable to select specified # of channels 
//
****************************************************************************************/
int OpenAudioDevice(int *dev, int channels, int SampleRate)
{
	long param;
	int status=0;

	/* Open audio device (UDA1341 on the Linux SA-1110 Microprocessor Evaluation Module) */
	if ((*dev=open("/dev/dsp",O_RDWR))==-1) status=-100;

	/* Request 16-bit, little-endian, PCM data format */
	param = AFMT_S16_LE;
	if ((ioctl(*dev,SNDCTL_DSP_SETFMT,&param))!=0) { close (*dev); status=-101; }
  
	/* Request sample rate */
	param=SampleRate;
	if ((ioctl (*dev,SNDCTL_DSP_SPEED,&param))!=0) { close (*dev); status=-102; }

	/* Request number of channels */
	param = channels;
	if ((ioctl(*dev,SNDCTL_DSP_CHANNELS,&param))!=0) { close (*dev); status=-103; }

	/* Trap unsuccessful attempts to open audio device */
	if (status<0) { fprintf(stderr,"\nUnable to open audio device.\n"); fflush(stderr); return(status); }
	else return 0;
}

/***************************************************************************************
// Name:			 GetAudioInput
//
// Description:		 Read pcm audio from the audio input device. 
//					 Assumes 44.1 kHz input stream
//
// Input Arguments:  dev				- Pointer to the audio input device. 
//					 sig				- Pointer to a buffer of interleaved pcm audio.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void GetAudioInput(int *dev, Ipp16s *sig, int len)
{
	int i,j;					/* Buffer indices */
	static Ipp16s pcm[2*N];

	/* Fill pcm audio input buffer */
	read(*dev,pcm,2*len*sizeof(Ipp16s));

	/* Extract right channel from the L/R interleaved data since
	   resampler requires contiguous (non-interleaved) samples */
	for(i=j=0;j<(2*len);i++,j+=2)
		//sig[i]=pcm[j];
		sig[i]=(((int)pcm[j]+(int)pcm[j+1])>>1)&0xffff;
}


/*********************************************************************************************
// Name:			 CloseAudioDevice
//
// Description:		 Close audio output device
//
// Input Arguments:  AudioDevice		- Pointer to the audio output device
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseAudioDevice(int AudioDevice)
{
	close(AudioDevice);
}

/*********************************************************************************************
// Name:			 ReadInputFile
//
// Description:		 Read binary file contents to a pcm buffer
//
// Input Arguments:  fpi				- Pointer to the input file stream
//					 pcm				- PCM buffer pointer
//					 N					- Number of samples to read
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
int ReadInputFile(FILE *fpi, Ipp16s *pcm, int len)
{
	if (fpi!=NULL) 
		return(fread(pcm,sizeof(Ipp16s),len,fpi));
	else
		return(0);
}

/*********************************************************************************************
// Name:			 WriteOutputFile
//
// Description:		 Write PCM buffer contents to a binary file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//					 pcm				- PCM output buffer pointer
//					 N					- Number of samples to write
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void WriteOutputFile(FILE *fpo, Ipp16s *pcm, int len)
{
	if (fpo!=NULL) 
		fwrite(pcm,sizeof(Ipp16s),len,fpo);
}

/*********************************************************************************************
// Name:			 CloseFile
//
// Description:		 Close PCM binary output file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseFile(FILE *fp)
{
	if (fp!=NULL) fclose(fp);
}

/*********************************************************************************************
// Name:			 User interface thread
//
// Description:		 Keyboard character-based user interface
//					 Thread terminates with the parent
//
// Returns:			 None				
**********************************************************************************************/
int UserInterface()
{
	char msg[MSGBUF_LEN], tstr[PBUF_LEN];	/* Keyboard message and parsing buffer */
	Ipp16s ebuf[N]; 
	int i,p,n;
	char name[256];
	float Tr;

	/* UI main loop */
	while(1)
	{
		/* Clear keyboard message buffer */
		for(i=0;i<MSGBUF_LEN;i++)
			msg[i]='\0';
		/* Clear parse buffer */
		for(i=0;i<PBUF_LEN;i++)
			tstr[i]='\0';

		/* Get user input from the keyboard */
		gets(msg);

		/* Parse and act on user input */
		switch(msg[0])
		{
			/* t<time> = set reverb time */
			case 't':
				Tr=atof(&(msg[1]));
				printf("Reverb time = %2.1f seconds\n",Tr);

				/* Force the real-time reverberator to wait for Tr change to complete */
				g_WaitForUI=1;
				SetReverbTime(Tr);
				g_WaitForUI=0;
				break;

			/* h = help */
			case 'h':
			case 'H':
				printf("\n\nCommands:\n");
				printf("t<time>  - set reverb time (RT60), in seconds\n");
				printf("h                   - display command list\n");
				#ifdef PERFORMANCE_TIMER
				printf("p                   - display performance measurements\n");
				#endif
				printf("q                   - QUIT\n");
				printf("\n\n");
				break;

			#ifdef PERFORMANCE_TIMER
			/* p = display performance data */
			case 'p':
			case 'P':
				n=GetPerformanceTimer(-1,0,1,NULL);
				printf("\n\nOSCR Performance Measurements\n\n");
				printf("\nOperation\tCPU Utilization\n\n");
				for(i=0;i<n;i++)
				{
					p=GetPerformanceTimer(i,N,FS,name);
					printf("%s\t\t%d %%\n",name,p);
				}
				printf("\n\n");
				break;
			#endif

			/* q = Quit */
			case 'q':
			case 'Q':
				g_Quit=1;
				break;
			default:
				break;
		}
	}
}
