/*
 * linux/drivers/pcmcia/av500_cf.c
 *
 * Compact Flash driver for the AV500's module slot
 *
 * Copyright (C) 2003 LSE Linux & Security Experts GmbH
 *
 *  The code is basically a copy of sa1100_generic.c, minced with
 *  some details about the AV500's special hardware setup.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include <linux/config.h>

#ifndef CONFIG_AV500_REV10

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/spinlock.h>
#include <linux/completion.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/mm.h>
#include <linux/notifier.h>
#include <linux/proc_fs.h>
#include <linux/version.h>
#include <linux/cpufreq.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/ss.h>
#include <pcmcia/bus_ops.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/cistpl.h>

#include <asm/semaphore.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/arch/gpio.h>
#include <asm/arch/mux.h>
#include <asm/irq.h>

//#define PCMCIA_DEBUG 4
//#define DEBUG_CF_CPLD 1

#include "cs_internal.h"
#include "av500_cf.h"

/*
 * Address Space Definition:
 *
 * CF slot is connected to: EMIFS_CS2
 * CF Card Physical Base:   0x08000000
 *      Attribute Memory:   0x08000000
 *                   CIS:   - .....1FF
 *         Configuration:   - .....3FF
 *              Card I/O:   0x08001000
 *         Common Memory:   0x08000400
 *                          - ....0FFF
 *
 * CF CPLD Physical Base:   0x08006000
 *
 */

#define CF_DATA_PHYS_BASE   0x08000000
#define CF_CPLD_PHYS_BASE   0x08006000

#ifdef CONFIG_AV500_REV13
#define CF_IRQ          INT_MPUIO5
#define CF_PIN_READY    5
#define CF_VSEL_3V3     4
#define CF_VSEL_5V      3
extern void vpx3226_request_power(void);
extern void vpx3226_release_power(void);
#else
#define CF_IRQ		INT_GPIO3
#define CF_PIN_READY	3
#define CF_VSEL_3V3     4
static inline void vpx3226_request_power(void) {}
static inline void vpx3226_release_power(void) {}
#endif

extern void av500_report_cfpower(int);

#define CF_PIN_UARTGATE 11

#define CF_PHYS_ATTR_MEM    (CF_DATA_PHYS_BASE)
#define CF_PHYS_COMM_MEM    (CF_DATA_PHYS_BASE+0x0400)
#define CF_PHYS_IO_SPACE    (CF_DATA_PHYS_BASE+0x1000)

#define CF_CPLD_BASE    cf_cpld_base
#define CF_RESET_REG    (CF_CPLD_BASE+8)

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
#endif


/* This structure maintains housekeeping state for each socket, such
 * as the last known values of the card detect pins, or the Card Services
 * callback value associated with the socket:
 */
static const int av500_pcmcia_socket_count = AV500_PCMCIA_MAX_SOCK;
static struct av500_pcmcia_socket av500_pcmcia_socket[AV500_PCMCIA_MAX_SOCK];

#define PCMCIA_SOCKET(x)	(av500_pcmcia_socket + (x))

/* Returned by the low-level PCMCIA interface: */
static struct pcmcia_low_level *pcmcia_low_level;

static struct timer_list poll_timer;
static struct tq_struct av500_pcmcia_task;

/* iomapped base of cf registers */
static unsigned long cf_cpld_base;

static int av500_pcmciall_card_ready;

/* module initialisation and cleanup */
int init_driver(void);
void cleanup_driver(void);

/*****************************************************************************
 *
 * AV500 COMPACT FLASH DRIVER LOWLEVEL FUNCTIONS
 *
 ****************************************************************************/

/*
 * cf_card_reset()
 *
 * set the RESET pin of the compact flash slot
 *
 * input:
 *      1: RESET pin to HIGH
 *      0: RESET pin to LOW
 *
 * output:
 *      void
 */
static inline void cf_card_reset(int state)
{
        if (state)
                writew(1, CF_RESET_REG);
        else
                writew(0, CF_RESET_REG);
}

/*
 * cf_core_reset()
 *
 * reset the compact flash CPLD
 *
 * input:
 *      1: RESET HIGH (active)
 *      0: RESET LOW
 *
 * output:
 *      void
 */
static inline void cf_core_reset(int state)
{
        if (state)
                writew(2, CF_RESET_REG);
        else
                writew(0, CF_RESET_REG);
}

/*
 * cf_card_detect()
 *
 * return state of the CF CD pins
 *
 * input:
 *      void
 *
 * output:
 *      bitwise OR of CF_CD2(2), CF_CD1(1)
 */
static inline int cf_card_detect(void)
{
#define CF_CD1  0x01
#define CF_CD2  0x02
        return readw(CF_RESET_REG) & 0x0003;
}

/*
 * cf_voltage_detect()
 *
 * detect the voltage based on the VS1 pin
 *
 * input:
 *      void
 *
 * output:
 *	boolean value reflecting state of CF_VS1
 */
static inline int cf_vs1_get(void)
{
	return (inb(OMAP1510_UART2_BASE + 0x18) & (1 << 4)) == (1 << 4);
}

static inline int cf_card_ready(void)
{
#ifdef CONFIG_AV500_REV13
	return omap_mpuio_get(CF_PIN_READY);
#else	
	return omap_gpio_get(CF_PIN_READY);
#endif
}

static void memdump(const unsigned char* mem, int count)
{
        const unsigned char* p = (const unsigned char*)mem;

        while (count > 0) {
                int i;
                printk(KERN_DEBUG "%08x: ", (unsigned int)p-(unsigned int)mem);

                for (i=0; i < 8; i++)
                        printk("%02x ", p[i]);

                printk(" | ");
                for (i=0; i < 8; i++)
                        printk("%c", isalnum(p[i])?p[i]:'.');

                printk("\n");

                count -= 8;
                p += 8;
        }
}



static int av500_pcmciall_init(struct pcmcia_init *init){
        int irq, res;
        unsigned long flags;

        DEBUG(2, "%s()\n", __FUNCTION__);

        cf_cpld_base = (  unsigned long )ioremap(  CF_CPLD_PHYS_BASE, SZ_4K );
        if (  cf_cpld_base == 0 ) {
                printk(KERN_ERR "unable to map CF CPLD region\n" );
                return -1;
        }

        /* initialize external bus with slow, failsafe values */
        REGL(EMIFS_CS2_CONFIG)=  (0  << 21) |		// increment address for second half of 32bit access
                                 (0  << 20) |		// 16 bit access width
                                 (0  << 16) |		// asynchronous reads, no page mode
                                 (3  << 12) |		// WE pulse length 3 clock cycles (WELEN)
                                 (1  << 8 ) |		// 1 clocks wait for write (WRWST)
                                 (4  << 4 ) |		// 6 clocks wait for read  (RDWST)
                                 (0  << 2 ) |		// don't retime data from external bus
                                 (2  << 0 ) ;		// EMIFS reference clock = TC / 6

#if defined(CONFIG_OMAP_INNOVATOR)
        omap_cfg_reg(gpio6);
        omap_cfg_reg(gpio3);
        omap_cfg_reg(gpio4);
        omap_cfg_reg(w5_mpuio5);
#endif

#if defined (CONFIG_AV500_REV13)
        omap_gpio_set(CF_VSEL_3V3);
        omap_gpio_set(CF_VSEL_5V);
        omap_gpio_dir(CF_VSEL_3V3, GPIO_DIR_OUT);
        omap_gpio_dir(CF_VSEL_5V, GPIO_DIR_OUT);
	
	omap_mpuio_dir(CF_PIN_READY, GPIO_DIR_IN);
	omap_mpuio_irq(CF_PIN_READY, GPIO_IRQTRIGGER_FALLING);
#else
	omap_gpio_set(CF_VSEL_3V3);
	omap_gpio_dir(CF_VSEL_3V3, GPIO_DIR_OUT);
	omap_gpio_dir(CF_PIN_READY, GPIO_DIR_IN);
	/* cannot configure edge sensivity on GPIOs,
	 * always triggers on falling edge */
#endif
	
	omap_gpio_dir(CF_PIN_UARTGATE, GPIO_DIR_IN);
	
        return 1;
}

static int av500_pcmciall_shutdown(void)
{
        DEBUG(2, "av500_pcmciall_shutdown()\n");

        omap_gpio_set(CF_VSEL_3V3);
#ifdef CONFIG_AV500_REV13
	omap_gpio_set(CF_VSEL_5V);
#endif	
        iounmap( ( void* )cf_cpld_base );

        return 0;
}

static int av500_pcmciall_socket_state(struct pcmcia_state_array
				       *state_array)
{
	int card_detect;
	
	int i;
	
        DEBUG(4, "av500_pcmciall_socket_state()\n");

        if (state_array->size<1)
                return -1;

        memset(state_array->state, 0,
                (state_array->size)*sizeof(struct pcmcia_state));

	card_detect = cf_card_detect();


        state_array->state[0].detect = (card_detect == 0)? 1:0;
        if (av500_pcmciall_card_ready == 1)
		state_array->state[0].ready  = 1;
	else {
		av500_pcmciall_card_ready = cf_card_ready();
		state_array->state[0].ready  = cf_card_ready();
	}
        state_array->state[0].bvd1   = 1;
        state_array->state[0].bvd2   = 1;
        state_array->state[0].wrprot = 0;
        state_array->state[0].vs_3v  = cf_vs1_get();
        state_array->state[0].vs_Xv  = !cf_vs1_get();

        return 1;
}

static int av500_pcmciall_get_irq_info(struct pcmcia_irq_info *info)
{
        DEBUG(2, "av500_pcmciall_get_irq_info() for socket %u\n", info->sock);

        if(info->sock>0)
                return -1;

        info->irq=CF_IRQ;

        return 0;
}

static int av500_pcmciall_configure_socket(const struct pcmcia_configure
					   *configure)
{
        unsigned long flags;
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(configure->sock);

        DEBUG(2, "av500_pcmciall_configure_socket() for socket %u\n", configure->sock);

        if(configure->sock > 0)
                return -1;

        save_flags_cli(flags);

        switch(configure->vcc){
        case 0:
                /* switch off voltage converters */
                omap_gpio_set(CF_VSEL_3V3);
#ifdef CONFIG_AV500_REV13
		omap_gpio_set(CF_VSEL_5V);
#endif	
		/* VPX video decoder no longer needed, save energy */
		vpx3226_release_power();
		av500_report_cfpower(0);
                break;

        case 33:
#ifdef CONFIG_AV500_REV13
                /* switch off 5V */
                omap_gpio_set(CF_VSEL_5V);
#endif	
                /* switch on 3.3V */
                omap_gpio_clr(CF_VSEL_3V3);
		/* 
		 * VPX video decoder needs to be powered because it shares data and address lines with
		 * the CF slot. So we power it up but don't initialize it. It will stay in sleep mode
		 * with outputs in high-z 
		 */ 
		vpx3226_request_power();
		av500_report_cfpower(1);
                break;

        case 50:
        default:
                printk(KERN_ERR "%s(): unrecognized Vcc %u\n", __FUNCTION__,
                        configure->vcc);

                /* switch off voltage converters */
                omap_gpio_set(CF_VSEL_3V3);
#ifdef CONFIG_AV500_REV13
		omap_gpio_set(CF_VSEL_5V);
#endif	
                restore_flags(flags);
                return -1;
        }


	if (configure->reset)
		av500_pcmciall_card_ready = 0;

        /* reset the card if asked */
        cf_card_reset(configure->reset);

        if (configure->irq) {
                if (irq_disabled(CF_IRQ))
                        enable_irq(CF_IRQ);
        } else {
                if (!irq_disabled(CF_IRQ))
                        disable_irq(CF_IRQ);
        }

        /* Silently ignore Vpp, output enable, speaker enable. */
        restore_flags(flags);
        return 0;
}

static int av500_pcmciall_socket_init(int sock)
{
        DEBUG(2, "av500_pcmciall_socket_init() for socket %u\n", sock);
	cf_core_reset(1);
	cf_core_reset(0);
        return 0;
}

static int av500_pcmciall_socket_suspend(int sock)
{
        DEBUG(2, "av500_pcmciall_socket_suspend() for socket %u\n", sock);
        return 0;
}

struct pcmcia_low_level av500_pcmcia_ops = {
        init:                   av500_pcmciall_init,
        shutdown:               av500_pcmciall_shutdown,
        socket_state:           av500_pcmciall_socket_state,
        get_irq_info:           av500_pcmciall_get_irq_info,
        configure_socket:       av500_pcmciall_configure_socket,

        socket_init:            av500_pcmciall_socket_init,
        socket_suspend:         av500_pcmciall_socket_suspend,
};



/******************************************************************************
 *
 * AV500 COMPACT FLASH GENERIC FUNCTIONS
 *
 *****************************************************************************/

/*
 * av500_pcmcia_state_to_config
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 *
 * Convert PCMCIA socket state to our socket configure structure.
 */
static struct pcmcia_configure
av500_pcmcia_state_to_config(unsigned int sock, socket_state_t *state)
{
        struct pcmcia_configure conf;

        conf.sock    = sock;
        conf.vcc     = state->Vcc;
        conf.vpp     = state->Vpp;
        conf.output  = state->flags & SS_OUTPUT_ENA ? 1 : 0;
        conf.speaker = state->flags & SS_SPKR_ENA ? 1 : 0;
        conf.reset   = state->flags & SS_RESET ? 1 : 0;
        conf.irq     = state->io_irq != 0;

        return conf;
}

/* av500_pcmcia_init()
 * ^^^^^^^^^^^^^^^^^^^^
 *
 * (Re-)Initialise the socket, turning on status interrupts
 * and PCMCIA bus.  This must wait for power to stabilise
 * so that the card status signals report correctly.
 *
 * Returns: 0
 */
static int av500_pcmcia_init(unsigned int sock)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        struct pcmcia_configure conf;

        DEBUG(2, "%s(): initializing socket %u\n", __FUNCTION__, sock);

        skt->cs_state = dead_socket;

        conf = av500_pcmcia_state_to_config(sock, &dead_socket);

        pcmcia_low_level->configure_socket(&conf);

        return pcmcia_low_level->socket_init(sock);
}

/*
 * av500_pcmcia_suspend()
 * ^^^^^^^^^^^^^^^^^^^^^^^
 *
 * Remove power on the socket, disable IRQs from the card.
 * Turn off status interrupts, and disable the PCMCIA bus.
 *
 * Returns: 0
 */
static int av500_pcmcia_suspend(unsigned int sock)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        struct pcmcia_configure conf;
        int ret;

        DEBUG(2, "%s(): suspending socket %u\n", __FUNCTION__, sock);

        conf = av500_pcmcia_state_to_config(sock, &dead_socket);

        ret = pcmcia_low_level->configure_socket(&conf);

        if (ret == 0) {
                skt->cs_state = dead_socket;
                ret = pcmcia_low_level->socket_suspend(sock);
        }

        return ret;
}


/* av500_pcmcia_events()
 * ^^^^^^^^^^^^^^^^^^^^^^
 * Helper routine to generate a Card Services event mask based on
 * state information obtained from the kernel low-level PCMCIA layer
 * in a recent (and previous) sampling. Updates `prev_state'.
 *
 * Returns: an event mask for the given socket state.
 */
static inline unsigned int
av500_pcmcia_events(struct pcmcia_state *state,
		     struct pcmcia_state *prev_state,
		     unsigned int mask, unsigned int flags)
{
        unsigned int events = 0;

        if (state->detect != prev_state->detect) {
                DEBUG(3, "%s(): card detect value %u\n", __FUNCTION__, state->detect);

                events |= SS_DETECT;
        }

        if (state->ready != prev_state->ready) {
                DEBUG(3, "%s(): card ready value %u\n", __FUNCTION__, state->ready);

                events |= flags & SS_IOCARD ? 0 : SS_READY;
        }

        if (state->bvd1 != prev_state->bvd1) {
                DEBUG(3, "%s(): card BVD1 value %u\n", __FUNCTION__, state->bvd1);

                events |= flags & SS_IOCARD ? SS_STSCHG : SS_BATDEAD;
        }

        if (state->bvd2 != prev_state->bvd2) {
                DEBUG(3, "%s(): card BVD2 value %u\n", __FUNCTION__, state->bvd2);

                events |= flags & SS_IOCARD ? 0 : SS_BATWARN;
        }

        *prev_state = *state;

        events &= mask;

        DEBUG(4, "events: %s%s%s%s%s%s\n",
                events == 0         ? "<NONE>"   : "",
                events & SS_DETECT  ? "DETECT "  : "",
                events & SS_READY   ? "READY "   : "",
                events & SS_BATDEAD ? "BATDEAD " : "",
                events & SS_BATWARN ? "BATWARN " : "",
                events & SS_STSCHG  ? "STSCHG "  : "");

        return events;
}  /* av500_pcmcia_events() */


/* av500_pcmcia_task_handler()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Processes serviceable socket events using the "eventd" thread context.
 *
 * Event processing (specifically, the invocation of the Card Services event
 * callback) occurs in this thread rather than in the actual interrupt
 * handler due to the use of scheduling operations in the PCMCIA core.
 */
static void av500_pcmcia_task_handler(void *data)
{
        struct pcmcia_state state[AV500_PCMCIA_MAX_SOCK];
        struct pcmcia_state_array state_array;
        unsigned int all_events;

        DEBUG(4, "%s(): entering PCMCIA monitoring thread\n", __FUNCTION__);

        state_array.size = av500_pcmcia_socket_count;
        state_array.state = state;

        do {
                unsigned int events;
                int ret, i;

                memset(state, 0, sizeof(state));

                DEBUG(4, "%s(): interrogating low-level PCMCIA service\n", __FUNCTION__);

                ret = pcmcia_low_level->socket_state(&state_array);
                if (ret < 0) {
                        printk(KERN_ERR "av500_pcmcia: unable to read socket status\n");
                        break;
                }

                all_events = 0;

                for (i = 0; i < state_array.size; i++, all_events |= events) {
                        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(i);

                        events = av500_pcmcia_events(&state[i], &skt->k_state,
                                                        skt->cs_state.csc_mask,
                                                        skt->cs_state.flags);

                        if (events && av500_pcmcia_socket[i].handler != NULL)
                                skt->handler(skt->handler_info, events);
                }
        } while(all_events);

}  /* av500_pcmcia_task_handler() */

static struct tq_struct av500_pcmcia_task = {
	routine: av500_pcmcia_task_handler
};


/* av500_pcmcia_poll_event()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Let's poll for events in addition to IRQs since IRQ only is unreliable...
 */
static void av500_pcmcia_poll_event(unsigned long dummy)
{
        DEBUG(4, "%s(): polling for events\n", __FUNCTION__);
        poll_timer.function = av500_pcmcia_poll_event;
        poll_timer.expires = jiffies + AV500_PCMCIA_POLL_PERIOD;
        add_timer(&poll_timer);
        schedule_task(&av500_pcmcia_task);
}


/* av500_pcmcia_interrupt()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^
 * Service routine for socket driver interrupts (requested by the
 * low-level PCMCIA init() operation via av500_pcmcia_thread()).
 * The actual interrupt-servicing work is performed by
 * av500_pcmcia_thread(), largely because the Card Services event-
 * handling code performs scheduling operations which cannot be
 * executed from within an interrupt context.
 */
static void av500_pcmcia_interrupt(int irq, void *dev, struct pt_regs *regs)
{
        DEBUG(4, "%s(): servicing IRQ %d\n", __FUNCTION__, irq);
        schedule_task(&av500_pcmcia_task);
}


/* av500_pcmcia_register_callback()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the register_callback() operation for the in-kernel
 * PCMCIA service (formerly SS_RegisterCallback in Card Services). If
 * the function pointer `handler' is not NULL, remember the callback
 * location in the state for `sock', and increment the usage counter
 * for the driver module. (The callback is invoked from the interrupt
 * service routine, av500_pcmcia_interrupt(), to notify Card Services
 * of interesting events.) Otherwise, clear the callback pointer in the
 * socket state and decrement the module usage count.
 *
 * Returns: 0
 */
static int
av500_pcmcia_register_callback(unsigned int sock,
				void (*handler)(void *, unsigned int),
				void *info)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);

        if (handler == NULL) {
                skt->handler = NULL;
                MOD_DEC_USE_COUNT;
        } else {
                MOD_INC_USE_COUNT;
                skt->handler_info = info;
                skt->handler = handler;
        }

        return 0;
}


/* av500_pcmcia_inquire_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the inquire_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_InquireSocket in Card Services). Of note is
 * the setting of the SS_CAP_PAGE_REGS bit in the `features' field of
 * `cap' to "trick" Card Services into tolerating large "I/O memory"
 * addresses. Also set is SS_CAP_STATIC_MAP, which disables the memory
 * resource database check. (Mapped memory is set up within the socket
 * driver itself.)
 *
 * In conjunction with the STATIC_MAP capability is a new field,
 * `io_offset', recommended by David Hinds. Rather than go through
 * the SetIOMap interface (which is not quite suited for communicating
 * window locations up from the socket driver), we just pass up
 * an offset which is applied to client-requested base I/O addresses
 * in alloc_io_space().
 *
 * SS_CAP_PAGE_REGS: used by setup_cis_mem() in cistpl.c to set the
 *   force_low argument to validate_mem() in rsrc_mgr.c -- since in
 *   general, the mapped * addresses of the PCMCIA memory regions
 *   will not be within 0xffff, setting force_low would be
 *   undesirable.
 *
 * SS_CAP_STATIC_MAP: don't bother with the (user-configured) memory
 *   resource database; we instead pass up physical address ranges
 *   and allow other parts of Card Services to deal with remapping.
 *
 * SS_CAP_PCCARD: we can deal with 16-bit PCMCIA & CF cards, but
 *   not 32-bit CardBus devices.
 *
 * Return value is irrelevant; the pcmcia subsystem ignores it.
 */
static int
av500_pcmcia_inquire_socket(unsigned int sock, socket_cap_t *cap)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        int ret = -1;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        if (sock < av500_pcmcia_socket_count) {
                cap->features  = SS_CAP_PAGE_REGS | SS_CAP_STATIC_MAP | SS_CAP_PCCARD;
                cap->irq_mask  = 0;
                cap->map_size  = PAGE_SIZE;
                cap->pci_irq   = skt->irq;
                cap->io_offset = (unsigned long)skt->virt_io;

                ret = 0;
        }

        return ret;
}


/* av500_pcmcia_get_status()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_status() operation for the in-kernel PCMCIA
 * service (formerly SS_GetStatus in Card Services). Essentially just
 * fills in bits in `status' according to internal driver state or
 * the value of the voltage detect chipselect register.
 *
 * As a debugging note, during card startup, the PCMCIA core issues
 * three set_socket() commands in a row the first with RESET deasserted,
 * the second with RESET asserted, and the last with RESET deasserted
 * again. Following the third set_socket(), a get_status() command will
 * be issued. The kernel is looking for the SS_READY flag (see
 * setup_socket(), reset_socket(), and unreset_socket() in cs.c).
 *
 * Returns: 0
 */
static int
av500_pcmcia_get_status(unsigned int sock, unsigned int *status)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        struct pcmcia_state state[AV500_PCMCIA_MAX_SOCK];
        struct pcmcia_state_array state_array;
        unsigned int stat;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        state_array.size = av500_pcmcia_socket_count;
        state_array.state = state;

        memset(state, 0, sizeof(state));

        if ((pcmcia_low_level->socket_state(&state_array)) < 0) {
                printk(KERN_ERR "av500_pcmcia: unable to get socket status\n");
                return -1;
        }

        skt->k_state = state[sock];

        stat = state[sock].detect ? SS_DETECT : 0;
        stat |= state[sock].ready ? SS_READY  : 0;
        stat |= state[sock].vs_3v ? SS_3VCARD : 0;
        stat |= state[sock].vs_Xv ? SS_XVCARD : 0;

        /* The power status of individual sockets is not available
        * explicitly from the hardware, so we just remember the state
        * and regurgitate it upon request:
        */
        stat |= skt->cs_state.Vcc ? SS_POWERON : 0;

        if (skt->cs_state.flags & SS_IOCARD)
                stat |= state[sock].bvd1 ? SS_STSCHG : 0;
        else {
                if (state[sock].bvd1 == 0)
                        stat |= SS_BATDEAD;
                else if (state[sock].bvd2 == 0)
                        stat |= SS_BATWARN;
        }

        DEBUG(3, "\tstatus: %s%s%s%s%s%s%s%s\n",
                stat & SS_DETECT  ? "DETECT "  : "",
                stat & SS_READY   ? "READY "   : "",
                stat & SS_BATDEAD ? "BATDEAD " : "",
                stat & SS_BATWARN ? "BATWARN " : "",
                stat & SS_POWERON ? "POWERON " : "",
                stat & SS_STSCHG  ? "STSCHG "  : "",
                stat & SS_3VCARD  ? "3VCARD "  : "",
                stat & SS_XVCARD  ? "XVCARD "  : "");

        *status = stat;

        return 0;
}  /* av500_pcmcia_get_status() */


/* av500_pcmcia_get_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_GetSocket in Card Services). Not a very
 * exciting routine.
 *
 * Returns: 0
 */
static int
av500_pcmcia_get_socket(unsigned int sock, socket_state_t *state)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        *state = skt->cs_state;

        return 0;
}

/* av500_pcmcia_set_socket()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_socket() operation for the in-kernel PCMCIA
 * service (formerly SS_SetSocket in Card Services). We more or
 * less punt all of this work and let the kernel handle the details
 * of power configuration, reset, &c. We also record the value of
 * `state' in order to regurgitate it to the PCMCIA core later.
 *
 * Returns: 0
 */
static int
av500_pcmcia_set_socket(unsigned int sock, socket_state_t *state)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        struct pcmcia_configure conf;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        DEBUG(3, "\tmask:  %s%s%s%s%s%s\n\tflags: %s%s%s%s%s%s\n",
                (state->csc_mask==0)?"<NONE>":"",
                (state->csc_mask&SS_DETECT)?"DETECT ":"",
                (state->csc_mask&SS_READY)?"READY ":"",
                (state->csc_mask&SS_BATDEAD)?"BATDEAD ":"",
                (state->csc_mask&SS_BATWARN)?"BATWARN ":"",
                (state->csc_mask&SS_STSCHG)?"STSCHG ":"",
                (state->flags==0)?"<NONE>":"",
                (state->flags&SS_PWR_AUTO)?"PWR_AUTO ":"",
                (state->flags&SS_IOCARD)?"IOCARD ":"",
                (state->flags&SS_RESET)?"RESET ":"",
                (state->flags&SS_SPKR_ENA)?"SPKR_ENA ":"",
                (state->flags&SS_OUTPUT_ENA)?"OUTPUT_ENA ":"");
        DEBUG(3, "\tVcc %d  Vpp %d  irq %d\n",
                state->Vcc, state->Vpp, state->io_irq);

        conf = av500_pcmcia_state_to_config(sock, state);

        if (pcmcia_low_level->configure_socket(&conf) < 0) {
                printk(KERN_ERR "av500_pcmcia: unable to configure socket %d\n", sock);
                return -1;
        }

        skt->cs_state = *state;

        return 0;
}  /* av500_pcmcia_set_socket() */


/* av500_pcmcia_get_io_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_io_map() operation for the in-kernel PCMCIA
 * service (formerly SS_GetIOMap in Card Services). Just returns an
 * I/O map descriptor which was assigned earlier by a set_io_map().
 *
 * Returns: 0 on success, -1 if the map index was out of range
 */
static int
av500_pcmcia_get_io_map(unsigned int sock, struct pccard_io_map *map)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        int ret = -1;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        if (map->map < MAX_IO_WIN) {
                *map = skt->io_map[map->map];
                ret = 0;
        }

        return ret;
}


/* av500_pcmcia_set_io_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_io_map() operation for the in-kernel PCMCIA
 * service (formerly SS_SetIOMap in Card Services). We configure
 * the map speed as requested, but override the address ranges
 * supplied by Card Services.
 *
 * Returns: 0 on success, -1 on error
 */
static int
av500_pcmcia_set_io_map(unsigned int sock, struct pccard_io_map *map)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        DEBUG(3, "\tmap %u  speed %u\n\tstart 0x%08x  stop 0x%08x\n",
                map->map, map->speed, map->start, map->stop);
        DEBUG(3, "\tflags: %s%s%s%s%s%s%s%s\n",
                (map->flags==0)?"<NONE>":"",
                (map->flags&MAP_ACTIVE)?"ACTIVE ":"",
                (map->flags&MAP_16BIT)?"16BIT ":"",
                (map->flags&MAP_AUTOSZ)?"AUTOSZ ":"",
                (map->flags&MAP_0WS)?"0WS ":"",
                (map->flags&MAP_WRPROT)?"WRPROT ":"",
                (map->flags&MAP_USE_WAIT)?"USE_WAIT ":"",
                (map->flags&MAP_PREFETCH)?"PREFETCH ":"");

        if (map->map >= MAX_IO_WIN) {
                printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
                        map->map);
                return -1;
        }

        if (map->flags & MAP_ACTIVE) {
                skt->speed_io = map->speed;
                /* TODO: set the appropriate timing parameters here
                 *       currently, our hardware does not allow this.
                 */
        }

        if (map->stop == 1)
                map->stop = PAGE_SIZE-1;

        map->stop -= map->start;
        map->stop += (unsigned long)skt->virt_io;
        map->start = (unsigned long)skt->virt_io;

        skt->io_map[map->map] = *map;

        return 0;
}  /* av500_pcmcia_set_io_map() */


/* av500_pcmcia_get_mem_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the get_mem_map() operation for the in-kernel PCMCIA
 * service (formerly SS_GetMemMap in Card Services). Just returns a
 *  memory map descriptor which was assigned earlier by a
 *  set_mem_map() request.
 *
 * Returns: 0 on success, -1 if the map index was out of range
 */
static int
av500_pcmcia_get_mem_map(unsigned int sock, struct pccard_mem_map *map)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        int ret = -1;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        if (map->map < MAX_WIN) {
                *map = skt->mem_map[map->map];
                ret = 0;
        }

        return ret;
}


/* av500_pcmcia_set_mem_map()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the set_mem_map() operation for the in-kernel PCMCIA
 * service (formerly SS_SetMemMap in Card Services). We configure
 * the map speed as requested, but override the address ranges
 * supplied by Card Services.
 *
 * Returns: 0 on success, -1 on error
 */
static int
av500_pcmcia_set_mem_map(unsigned int sock, struct pccard_mem_map *map)
{
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        unsigned long start;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        DEBUG(3, "\tmap %u speed %u sys_start %08lx sys_stop %08lx card_start %08x\n",
                map->map, map->speed, map->sys_start, map->sys_stop, map->card_start);
        DEBUG(3, "\tflags: %s%s%s%s%s%s%s%s\n",
                (map->flags==0)?"<NONE>":"",
                (map->flags&MAP_ACTIVE)?"ACTIVE ":"",
                (map->flags&MAP_16BIT)?"16BIT ":"",
                (map->flags&MAP_AUTOSZ)?"AUTOSZ ":"",
                (map->flags&MAP_0WS)?"0WS ":"",
                (map->flags&MAP_WRPROT)?"WRPROT ":"",
                (map->flags&MAP_ATTRIB)?"ATTRIB ":"",
                (map->flags&MAP_USE_WAIT)?"USE_WAIT ":"");

        if (map->map >= MAX_WIN){
                printk(KERN_ERR "%s(): map (%d) out of range\n", __FUNCTION__,
                        map->map);
                return -1;
        }

        if (map->flags & MAP_ACTIVE) {
                unsigned int speed = map->speed;

                if (map->flags & MAP_ATTRIB) {
                        skt->speed_attr = speed;
                } else {
                        skt->speed_mem = speed;
                }

        }

        start = (map->flags & MAP_ATTRIB) ? skt->phys_attr : skt->phys_mem;

        if (map->sys_stop == 0)
                map->sys_stop = PAGE_SIZE-1;

        map->sys_stop -= map->sys_start;
        map->sys_stop += start;
        map->sys_start = start;

        skt->mem_map[map->map] = *map;

        return 0;
}  /* av500_pcmcia_set_mem_map() */


#if defined(CONFIG_PROC_FS)

/* av500_pcmcia_proc_status()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the /proc/bus/pccard/??/status file.
 *
 * Returns: the number of characters added to the buffer
 */
static int
av500_pcmcia_proc_status(char *buf, char **start, off_t pos,
			  int count, int *eof, void *data)
{
        unsigned int sock = (unsigned int)data;
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(sock);
        char *p = buf;

        p+=sprintf(p, "k_state  : %s%s%s%s%s%s%s\n",
                skt->k_state.detect ? "detect " : "",
                skt->k_state.ready  ? "ready "  : "",
                skt->k_state.bvd1   ? "bvd1 "   : "",
                skt->k_state.bvd2   ? "bvd2 "   : "",
                skt->k_state.wrprot ? "wrprot " : "",
                skt->k_state.vs_3v  ? "vs_3v "  : "",
                skt->k_state.vs_Xv  ? "vs_Xv "  : "");

        p+=sprintf(p, "status   : %s%s%s%s%s%s%s%s%s\n",
                skt->k_state.detect ? "SS_DETECT " : "",
                skt->k_state.ready  ? "SS_READY " : "",
                skt->cs_state.Vcc   ? "SS_POWERON " : "",
                skt->cs_state.flags & SS_IOCARD ? "SS_IOCARD " : "",
                (skt->cs_state.flags & SS_IOCARD &&
                skt->k_state.bvd1) ? "SS_STSCHG " : "",
                ((skt->cs_state.flags & SS_IOCARD)==0 &&
                (skt->k_state.bvd1==0)) ? "SS_BATDEAD " : "",
                ((skt->cs_state.flags & SS_IOCARD)==0 &&
                (skt->k_state.bvd2==0)) ? "SS_BATWARN " : "",
                skt->k_state.vs_3v  ? "SS_3VCARD " : "",
                skt->k_state.vs_Xv  ? "SS_XVCARD " : "");

        p+=sprintf(p, "mask     : %s%s%s%s%s\n",
                skt->cs_state.csc_mask & SS_DETECT  ? "SS_DETECT "  : "",
                skt->cs_state.csc_mask & SS_READY   ? "SS_READY "   : "",
                skt->cs_state.csc_mask & SS_BATDEAD ? "SS_BATDEAD " : "",
                skt->cs_state.csc_mask & SS_BATWARN ? "SS_BATWARN " : "",
                skt->cs_state.csc_mask & SS_STSCHG  ? "SS_STSCHG "  : "");

        p+=sprintf(p, "cs_flags : %s%s%s%s%s\n",
                skt->cs_state.flags & SS_PWR_AUTO   ? "SS_PWR_AUTO "   : "",
                skt->cs_state.flags & SS_IOCARD     ? "SS_IOCARD "     : "",
                skt->cs_state.flags & SS_RESET      ? "SS_RESET "      : "",
                skt->cs_state.flags & SS_SPKR_ENA   ? "SS_SPKR_ENA "   : "",
                skt->cs_state.flags & SS_OUTPUT_ENA ? "SS_OUTPUT_ENA " : "");

        p+=sprintf(p, "Vcc      : %d\n", skt->cs_state.Vcc);
        p+=sprintf(p, "Vpp      : %d\n", skt->cs_state.Vpp);
        p+=sprintf(p, "IRQ      : %d\n", skt->cs_state.io_irq);

        p+=sprintf(p, "I/O      : %u\n", skt->speed_io);
        p+=sprintf(p, "attribute: %u\n", skt->speed_attr);
        p+=sprintf(p, "common   : %u\n", skt->speed_mem);

        return p-buf;
}

/* av500_pcmcia_proc_setup()
 * ^^^^^^^^^^^^^^^^^^^^^^^^^^
 * Implements the proc_setup() operation for the in-kernel PCMCIA
 * service (formerly SS_ProcSetup in Card Services).
 *
 * Returns: 0 on success, -1 on error
 */
static void
av500_pcmcia_proc_setup(unsigned int sock, struct proc_dir_entry *base)
{
        struct proc_dir_entry *entry;

        DEBUG(2, "%s() for sock %u\n", __FUNCTION__, sock);

        if ((entry = create_proc_entry("status", 0, base)) == NULL){
                printk(KERN_ERR "unable to install \"status\" procfs entry\n");
                return;
        }

        entry->read_proc = av500_pcmcia_proc_status;
        entry->data = (void *)sock;
}

#endif  /* defined(CONFIG_PROC_FS) */

static struct pccard_operations av500_pcmcia_operations = {
        init:			av500_pcmcia_init,
        suspend:		av500_pcmcia_suspend,
        register_callback:	av500_pcmcia_register_callback,
        inquire_socket:	        av500_pcmcia_inquire_socket,
        get_status:		av500_pcmcia_get_status,
        get_socket:		av500_pcmcia_get_socket,
        set_socket:		av500_pcmcia_set_socket,
        get_io_map:		av500_pcmcia_get_io_map,
        set_io_map:		av500_pcmcia_set_io_map,
        get_mem_map:		av500_pcmcia_get_mem_map,
        set_mem_map:		av500_pcmcia_set_mem_map,
#ifdef CONFIG_PROC_FS
        proc_setup:		av500_pcmcia_proc_setup
#endif
};


static int __init av500_pcmcia_machine_init(void)
{
        pcmcia_low_level = &av500_pcmcia_ops;
        return 0;
}


/* driver entry point for initialization */
int init_driver(void)
{
        servinfo_t info;
        struct pcmcia_init pcmcia_init;
        struct pcmcia_state state[AV500_PCMCIA_MAX_SOCK];
        struct pcmcia_state_array state_array;
        struct av500_pcmcia_socket *skt = PCMCIA_SOCKET(0);
        struct pcmcia_irq_info irq_info;
        int ret;

        printk(KERN_INFO "PMA400 PCMCIA (CS release %s)\n", CS_RELEASE);

        CardServices(GetCardServicesInfo, &info);

        if (info.Revision != CS_RELEASE_CODE) {
                printk(KERN_ERR "Card Services release codes do not match\n");
                return -EINVAL;
        }

        ret = av500_pcmcia_machine_init();
        if (ret)
                return ret;

        pcmcia_init.handler = av500_pcmcia_interrupt;

        ret = pcmcia_low_level->init(&pcmcia_init);
        if (ret < 0) {
                printk(KERN_ERR "Unable to initialize kernel PCMCIA service (%d).\n", ret);
                return ret == -1 ? -EIO : ret;
        }

        state_array.size  = av500_pcmcia_socket_count;
        state_array.state = state;

        memset(state, 0, sizeof(state));

        if (pcmcia_low_level->socket_state(&state_array) < 0) {
                pcmcia_low_level->shutdown();
                printk(KERN_ERR "Unable to get PCMCIA status from kernel.\n");
                return -EIO;
        }

        irq_info.sock = 0;
        irq_info.irq  = -1;
        ret = pcmcia_low_level->get_irq_info(&irq_info);
        if (ret < 0)
                printk(KERN_ERR "Unable to get IRQ for socket 0 (%d)\n", ret);

        skt->irq        = irq_info.irq;
        skt->k_state    = state[0];
        skt->speed_io   = 0;
        skt->speed_attr = 0;
        skt->speed_mem  = 0;
        skt->phys_attr  = CF_PHYS_ATTR_MEM;
        skt->phys_mem   = CF_PHYS_COMM_MEM;
        skt->virt_io    = ioremap(CF_PHYS_IO_SPACE, SZ_4K);

        if (skt->virt_io == NULL) {
                ret = -ENOMEM;
                goto out_err;
        }

        /* Only advertise as many sockets as we can detect */
        ret = register_ss_entry(av500_pcmcia_socket_count, &av500_pcmcia_operations);
        if (ret < 0) {
                printk(KERN_ERR "Unable to register sockets\n");
                goto out_unmap;
        }

        /*
         * Start the event poll timer.  It will reschedule by itself afterwards.
         */
        av500_pcmcia_poll_event(0);

        return 0;

out_unmap:
	iounmap(skt->virt_io);
out_err:
        pcmcia_low_level->shutdown();

        return ret;
}

void cleanup_driver(void)
{
        del_timer_sync(&poll_timer);

        unregister_ss_entry(&av500_pcmcia_operations);

        iounmap(av500_pcmcia_socket[0].virt_io);

        pcmcia_low_level->shutdown();

        flush_scheduled_tasks();
}

module_init(init_driver);
module_exit(cleanup_driver);

MODULE_AUTHOR("Matthias Welwarsky <welwarsky@archos.com>");
MODULE_DESCRIPTION("Compact Flash driver for the PMA400");
MODULE_LICENSE("Dual MPL/GPL");

EXPORT_NO_SYMBOLS;

#endif
