/*
 * Hardware definitions for HP iPAQ Handheld Computers
 *
 * Copyright 2000-2003 Hewlett-Packard Company.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Authors: Jamey Hicks <jamey.hicks at hp.com>, Joshua Wise <joshua at joshuawise.com>.
 *
 * History:
 *
 * 2003-05-14	Joshua Wise        Adapted for the HP iPAQ H1900
 * 2002-08-23   Jamey Hicks        Adapted for use with PXA250-based iPAQs
 * 2001-10-??   Andrew Christian   Added support for iPAQ H3800
 *                                 and abstracted EGPIO interface.
 *
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/tty.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/bootmem.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/setup.h>
#include <asm/keyboard.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/arch/h1900-gpio.h>
#include <linux/h3600_keyboard.h>

#include <asm/hardware/ipaq-asic3.h>

#include <asm/arch/irq.h>
#include <asm/types.h>

#include <linux/serial_core.h>

#include "generic.h"

/* whether it's safe for charger stuff to write to LEDs. */
int h1900_leds;

extern int h3600_kbd_translate(unsigned char scancode, unsigned char *keycode, char raw_mode);
extern char h3600_kbd_unexpected_up(unsigned char keycode);

static void msleep(unsigned int msec)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout( (msec * HZ + 999) / 1000);
}

/*
  On screen enable, we get 
  
     h3800_video_power_on(1)
     LCD controller starts
     h3800_video_lcd_enable(1)

  On screen disable, we get
  
     h3800_video_lcd_enable(0)
     LCD controller stops
     h3800_video_power_on(0)
*/

static void h1900_video_power_on( int setp )
{
	if (machine_is_h1900())
	{
		if (setp) {
			GPSR(48) = GPIO_bit(48);	/* -7V to LCD -  VGL_EN*/
			msleep(50);
			GPCR(50) = GPIO_bit(50);	/* 2.5V to LCD - DVDD_EN */
			msleep(5);
		} else {
			msleep(5);
			GPCR(50) = GPIO_bit(50);
			msleep(50);
			GPCR(48) = GPIO_bit(48);
		};
	};
}

static void h1900_video_lcd_enable( int setp )
{
	/* NOP */
}


static void h1900_control_egpio( enum ipaq_egpio_type x, int setp )
{
	switch (x) {
	case IPAQ_EGPIO_LCD_POWER:
		h1900_video_power_on( setp );
		break;
	case IPAQ_EGPIO_LCD_ENABLE:
		break;
	case IPAQ_EGPIO_CODEC_NRESET:
	case IPAQ_EGPIO_AUDIO_ON:
	case IPAQ_EGPIO_QMUTE:
	case IPAQ_EGPIO_CARD_RESET:
	case IPAQ_EGPIO_OPT_ON:
	case IPAQ_EGPIO_OPT_NVRAM_ON:
	case IPAQ_EGPIO_OPT_RESET:
	case IPAQ_EGPIO_RS232_ON:
	case IPAQ_EGPIO_BLUETOOTH_ON:
		printk("%s: error - should not be called\n", __FUNCTION__);
		break;
	case IPAQ_EGPIO_IR_ON:
//		CLEAR_ASIC3( GPIO3_IR_ON_N );
		printk("%s: FIXME: what is the GPIO for IR?\n", __FUNCTION__);
		break;
	case IPAQ_EGPIO_IR_FSEL:
		break;
	case IPAQ_EGPIO_VPP_ON:
		if (setp)
			GPSR(37) = GPIO_bit(37);
		else
			GPCR(37) = GPIO_bit(37);
		break;
	default:
		printk("%s: unhandled egpio=%d\n", __FUNCTION__, x);
	}
}

static unsigned long h1900_read_egpio( enum ipaq_egpio_type x)
{
	printk("%s:%d: No readable GPIOs on H1900...\n", __FUNCTION__, __LINE__);
	return 0;
}

/* We need to fix ASIC2 GPIO over suspend/resume.  At the moment,
   it doesn't appear that ASIC3 GPIO has the same problem */

static int h1900_pm_callback( int req )
{
	return 0;
}

static int h1900_egpio_irq_number(enum ipaq_egpio_type egpio_nr)
{
	printk("%s: unhandled egpio_nr=%d\n", __FUNCTION__, egpio_nr); 
	return -EINVAL;
}

static void h1900_set_led (int color, int duty_time, int cycle_time)
{
	h1900_leds = 0;
	if (duty_time == 0)
		h1900_leds = 1;
		/* safe for charger to set LEDs without messing with application settings */
			
	if (color == RED_LED)
	{		
		H3900_ASIC3_LED_0_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_0_PeriodTime = cycle_time;
		H3900_ASIC3_LED_0_DutyTime = 0;
		udelay(1);
		H3900_ASIC3_LED_0_DutyTime = duty_time;
		
		H3900_ASIC3_LED_1_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_1_PeriodTime = cycle_time;
		H3900_ASIC3_LED_1_DutyTime = 0;
	};
	
	if (color == GREEN_LED)
	{
		H3900_ASIC3_LED_1_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_1_PeriodTime = cycle_time;
		H3900_ASIC3_LED_1_DutyTime = 0;
		udelay(1);
		H3900_ASIC3_LED_1_DutyTime = duty_time;
		
		H3900_ASIC3_LED_0_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_0_PeriodTime = cycle_time;
		H3900_ASIC3_LED_0_DutyTime = 0;	
	};
	
	if (color == YELLOW_LED)
	{
		H3900_ASIC3_LED_1_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_1_PeriodTime = cycle_time;
		H3900_ASIC3_LED_1_DutyTime = 0;
				
		H3900_ASIC3_LED_0_TimeBase = 0x6 | LEDTBS_BLINK;
		H3900_ASIC3_LED_0_PeriodTime = cycle_time;
		H3900_ASIC3_LED_0_DutyTime = 0;
		
		udelay(1);
		H3900_ASIC3_LED_1_DutyTime = duty_time;
		H3900_ASIC3_LED_0_DutyTime = duty_time;
	};
	
	if (color == BLUE_LED)
		printk(KERN_DEBUG "h1900.c: aiee! we don't have a blue led, you doofus!\n");
}

/* 
 * GPIO16: LCD_PWM: control brightness of frontlight via pulse width modulation 
 * GPIO36: Frontlight power enable (I suggest not enabling without running pwm
 */
static void h1900_backlight_power (int on)
{
	if (on)
		GPSR(36) = GPIO_bit(36);
	else
		GPCR(36) = GPIO_bit(36);
}

static struct ipaq_model_ops h1900_model_ops __initdata = {
	.generic_name = "1900",
	.control      = h1900_control_egpio,
	.read         = h1900_read_egpio,
	.pm_callback  = h1900_pm_callback,
	.irq_number   = h1900_egpio_irq_number,
	.set_led      = h1900_set_led,
	.backlight_power = h1900_backlight_power
};


static void h1900_charge(int irq, void *dev_id, struct pt_regs *regs)
{
	int charge;
	
	if (	!(GPLR(GPIO_NR_H1900_AC_IN_N) & GPIO_bit(GPIO_NR_H1900_AC_IN_N)) &&
		(GPLR(GPIO_NR_H1900_MBAT_IN) & GPIO_bit(GPIO_NR_H1900_MBAT_IN))		)
		charge=1;
	else
		charge=0;
		
	if (charge)
	{
		int dtime;
		
		if (h1900_leds)
		{
			/* if we're charging... */
			if (GPLR(GPIO_NR_H1900_CHARGING) & GPIO_bit(GPIO_NR_H1900_CHARGING))
				dtime = 0x80;	/* set duty time so we blink. */
			else			/* otherwise... */
				dtime = 0x101;	/* keep it solid. */
			
			H3900_ASIC3_LED_1_TimeBase = 0x6 | LEDTBS_BLINK;
			H3900_ASIC3_LED_1_PeriodTime = 0x100;
			
			H3900_ASIC3_LED_0_TimeBase = 0x6 | LEDTBS_BLINK;
			H3900_ASIC3_LED_0_PeriodTime = 0x100;
		
			H3900_ASIC3_LED_0_DutyTime = 0;
			H3900_ASIC3_LED_1_DutyTime = 0;
			
			udelay(20);	// asic voodoo - possibly need a whole duty cycle?
			
			H3900_ASIC3_LED_1_DutyTime = dtime;
			H3900_ASIC3_LED_0_DutyTime = dtime;
		};
		
		GPSR(GPIO_NR_H1900_CHARGER_EN) = GPIO_bit(GPIO_NR_H1900_CHARGER_EN);
	} else {
		if (h1900_leds)
		{
			H3900_ASIC3_LED_1_TimeBase = 0x6 | LEDTBS_BLINK;
			H3900_ASIC3_LED_1_PeriodTime = 0x100;
			H3900_ASIC3_LED_1_DutyTime = 0;
		
			H3900_ASIC3_LED_0_TimeBase = 0x6 | LEDTBS_BLINK;
			H3900_ASIC3_LED_0_PeriodTime = 0x100;
			H3900_ASIC3_LED_0_DutyTime = 0;
			
			udelay(20);
			
			/* blink red if there's no battery. */
			if (!(GPLR(GPIO_NR_H1900_MBAT_IN) & GPIO_bit(GPIO_NR_H1900_MBAT_IN)))
				H3900_ASIC3_LED_0_DutyTime = 0x40;
		}
		
		GPCR(GPIO_NR_H1900_CHARGER_EN) = GPIO_bit(GPIO_NR_H1900_CHARGER_EN);
	};
};

static struct irqaction h1900_charge_irq = {
	name:     "h1900_charge",
	handler:  h1900_charge,
	flags:    SA_INTERRUPT
};

static void __init h1900_init_irq( void )
{
	int i;
	
	/* Initialize standard IRQs */
	pxa_init_irq();
	
	set_GPIO_IRQ_edge(GPIO_NR_H1900_AC_IN_N, GPIO_BOTH_EDGES);	/* call us if we insert AC.. */
	set_GPIO_IRQ_edge(GPIO_NR_H1900_MBAT_IN, GPIO_BOTH_EDGES);	/* plug in the battery.. */
	set_GPIO_IRQ_edge(GPIO_NR_H1900_CHARGING, GPIO_BOTH_EDGES);	/* .. or start/stop charging. */
	setup_arm_irq(IRQ_GPIO(GPIO_NR_H1900_AC_IN_N), &h1900_charge_irq);
	setup_arm_irq(IRQ_GPIO(GPIO_NR_H1900_MBAT_IN), &h1900_charge_irq);
	setup_arm_irq(IRQ_GPIO(GPIO_NR_H1900_CHARGING), &h1900_charge_irq);	
}

static struct map_desc h1900_io_desc[] __initdata = {
 /* virtual            physical           length      domain     r  w  c  b */
  { H1900_ASIC3_VIRT,    H1900_ASIC3_PHYS,    0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 3  CS#3 */
  { H1900_ASIC3_SD_VIRT, H1900_ASIC3_SD_PHYS, 0x00800000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 4  CS#4 */
  LAST_DESC
}; 

/*
 * Common map_io initialization
 */
static void __init h1900_map_io(void)
{
	pxa_map_io();
	
	iotable_init(h1900_io_desc);
	
#if 0
	PGSR0 = GPSRx_SleepValue;
	PGSR1 = GPSRy_SleepValue;
	PGSR2 = GPSRz_SleepValue;
	
	/* Set up GPIO direction and alternate function registers */
	GAFR0_L = GAFR0x_InitValue;
	GAFR0_U = GAFR1x_InitValue;
	GAFR1_L = GAFR0y_InitValue;
	GAFR1_U = GAFR1y_InitValue;
	GAFR2_L = GAFR0z_InitValue;
	GAFR2_U = GAFR1z_InitValue;
	GPDR0 = GPDRx_InitValue;
	GPDR1 = GPDRy_InitValue;
	GPDR2 = GPDRz_InitValue;
	GPCR0 = 0x0fffffff;       /* All outputs are set low by default */

#endif
	/* Add wakeup on AC plug/unplug (and resume button) */
	PWER = PWER_RTC | PWER_GPIO4 | PWER_GPIO0;
	PFER = PWER_RTC | PWER_GPIO4 | PWER_GPIO0;
	PRER =            PWER_GPIO4 | PWER_GPIO0;
	PCFR = PCFR_OPDE;
	
	h1900_leds = 1;		/* it's by default safe to set LEDs for charging. */
				/* (usermode LED settings will overwrite.) */
				
	/* fake a charge interrupt so that we start charging if need be. */
	h1900_charge(0,NULL,NULL);
	
	ipaq_model_ops = h1900_model_ops;
}
 
/************************* H1900 *************************/

/* <jamey> NonToxic: try reading 0x48000008, bit 3 turned on (value 8) means 16bit */
int detect_h1900() 
{
	static int h1900 = 0;
	if (!h1900)
	{
		if (io_p2v(0x48000008) & 0x8)
		{
			printk("ipaq.c: Detected iPAQ H1900.\n");
			h1900 = 2;
		} else
			h1900 = 1;
	};
	
	return h1900-1;
};

MACHINE_START(H1900, "HP iPAQ H1900")
	MAINTAINER("HP Labs, Cambridge Research Labs")
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	BOOT_PARAMS(0xa0000100)
	MAPIO(h1900_map_io)
	INITIRQ(h1900_init_irq)
MACHINE_END

