/*
 * Low-level MMC/SD functions for the ASIC3 MMC/SD Controller
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * Some parts of code based on iPAQ 3800 MMC driver code
 *
 * Author: Pawel Kolodziejski (aquadran)
 *
 * comments:
 * - initilise card procedure fail sometimes for my MMC card,
 * - add code for handle interrupt card detection (removal, insertion).
 * - add code for support PM.
 * - multi block mode not tested (MMC core driver doesn't support it)
 * - 3900 must disable bus host clock, and use asic2 MMC clock
 */

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>

#include <asm/unaligned.h>

#include <asm/arch/hardware.h>
#include <asm/arch/h3900_asic.h>

#include <linux/mmc/mmc_ll.h>


struct response_info {
	int length;
	u16 cdc_flags;
};

static struct response_info rinfo[] = {
        { 0,  SD_CTRL_COMMAND_RESPONSE_TYPE_NORMAL  }, // R0
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R1  }, // R1
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R1B }, // R1b
        { 16, SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R2  }, // R2
        { 16, SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R2  }, // R2
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R3  }, // R3
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R3  }, // R4
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R3  }, // R5
        { 5,  SD_CTRL_COMMAND_RESPONSE_TYPE_EXT_R3  }, // R6
};

struct mmc_data {
	struct mmc_request *request;
	int                flag;
};

static struct mmc_data g_data;

#ifdef CONFIG_MMC_DEBUG

struct cmd_to_name {
	int   id;
	char *name;
};

static struct cmd_to_name cmd_names[] = {
	{ MMC_CIM_RESET, "CIM_RESET" },
	{ MMC_GO_IDLE_STATE, "GO_IDLE_STATE" },
	{ MMC_SEND_OP_COND, "SEND_OP_COND" },
	{ MMC_ALL_SEND_CID, "ALL_SEND_CID" },
	{ MMC_SET_RELATIVE_ADDR, "SET_RELATIVE_ADDR" },
	{ MMC_SET_DSR, "SET_DSR" },
	{ MMC_SELECT_CARD, "SELECT_CARD" },
	{ MMC_SEND_CSD, "SEND_CSD" },
	{ MMC_SEND_CID, "SEND_CID" },
	{ MMC_READ_DAT_UNTIL_STOP, "READ_DAT_UNTIL_STOP" },
	{ MMC_STOP_TRANSMISSION, "STOP_TRANSMISSION" },
	{ MMC_SEND_STATUS	, "SEND_STATUS	" },
	{ MMC_GO_INACTIVE_STATE, "GO_INACTIVE_STATE" },
	{ MMC_SET_BLOCKLEN, "SET_BLOCKLEN" },
	{ MMC_READ_SINGLE_BLOCK, "READ_SINGLE_BLOCK" },
	{ MMC_READ_MULTIPLE_BLOCK, "READ_MULTIPLE_BLOCK" },
	{ MMC_WRITE_DAT_UNTIL_STOP, "WRITE_DAT_UNTIL_STOP" },
	{ MMC_SET_BLOCK_COUNT, "SET_BLOCK_COUNT" },
	{ MMC_WRITE_BLOCK, "WRITE_BLOCK" },
	{ MMC_WRITE_MULTIPLE_BLOCK, "WRITE_MULTIPLE_BLOCK" },
	{ MMC_PROGRAM_CID, "PROGRAM_CID" },
	{ MMC_PROGRAM_CSD, "PROGRAM_CSD" },
	{ MMC_SET_WRITE_PROT, "SET_WRITE_PROT" },
	{ MMC_CLR_WRITE_PROT, "CLR_WRITE_PROT" },
	{ MMC_SEND_WRITE_PROT, "SEND_WRITE_PROT" },
	{ MMC_ERASE_GROUP_START, "ERASE_GROUP_START" },
	{ MMC_ERASE_GROUP_END, "ERASE_GROUP_END" },
	{ MMC_ERASE, "ERASE" },
	{ MMC_FAST_IO, "FAST_IO" },
	{ MMC_GO_IRQ_STATE, "GO_IRQ_STATE" },
	{ MMC_LOCK_UNLOCK, "LOCK_UNLOCK" },
	{ MMC_APP_CMD, "APP_CMD" },
	{ MMC_GEN_CMD, "GEN_CMD" },
	{ SD_APP_OP_COND, "SD_APP_OP_COND" },
	{ SD_APP_SET_BUS_WIDTH, "SD_APP_SET_BUS_WIDTH" },
};

static char *get_cmd_name(int cmd)
{
	int i;
	int len = sizeof(cmd_names) / sizeof(struct cmd_to_name);
	for (i = 0 ; i < len ; i++)
		if ( cmd == cmd_names[i].id)
			return cmd_names[i].name;

	return "UNKNOWN";
}

#endif /* CONFIG_MMC_DEBUG */

static void mmc_asic3_stop_clock(void)
{
	H3900_ASIC3_SD_CTRL_CardClockCtrl = 0; // disable clock
}

static void mmc_asic3_start_clock(void)
{
	H3900_ASIC3_SD_CONFIG_ClockMode = CLOCK_MODE_DIV_CLOCK_DISABLE; // disable div rate clock
	H3900_ASIC3_SD_CTRL_CardClockCtrl = SD_CTRL_CARDCLOCKCONTROL_ENABLE_CLOCK;
	if (g_data.flag == 1)
	    	H3900_ASIC3_SD_CTRL_CardClockCtrl |= SD_CTRL_CARDCLOCKCONTROL_FOR_SD_CARD;
	mdelay(1);
}

static int mmc_asic3_set_clock(u32 rate)
{
	H3900_ASIC3_SD_CONFIG_ClockMode = CLOCK_MODE_DIV_CLOCK_DISABLE; // disable div rate clock
	H3900_ASIC3_SD_CTRL_CardClockCtrl = SD_CTRL_CARDCLOCKCONTROL_ENABLE_CLOCK;
	if (g_data.flag == 1)
    		H3900_ASIC3_SD_CTRL_CardClockCtrl |= SD_CTRL_CARDCLOCKCONTROL_FOR_SD_CARD;
	mdelay(1);

	return MMC_NO_ERROR;
}

static int mmc_asic3_reset_asic(void)
{
	if (machine_is_h3900()) {
		H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_SD_2;  // 24.576 Mhz
		H3900_ASIC3_CLOCK_CDEX |= CLOCK_CDEX_EX0;
		mdelay(1);

		/* Enable the host clock */
		H3900_ASIC3_CLOCK_SEL    |= CLOCK_SEL_SD_HCLK_SEL;
		H3900_ASIC3_CLOCK_SEL    &= ~CLOCK_SEL_SD_BCLK_SEL;
		H3900_ASIC3_CLOCK_CDEX   |= CLOCK_CDEX_SD_HOST;
		H3900_ASIC3_CLOCK_CDEX   &= ~CLOCK_CDEX_SD_BUS;
		mdelay(1);
	}

        H3900_ASIC3_CLOCK_CDEX |= CLOCK_CDEX_EX1;
	mdelay(1);
	H3900_ASIC3_EXTCF_Select |= ASIC3_EXTCF_SD_MEM_ENABLE; // enable possibility SD/MMC card in CF slot ?
	H3900_ASIC3_SDHWCTRL_SDConf &= ~(ASIC3_SDHWCTRL_SUSPEND | ASIC3_SDHWCTRL_PCLR); // wake up
	mdelay(10);

	H3900_ASIC3_SD_CONFIG_Addr0 = 0x0800; // offset to host controller
	H3900_ASIC3_SD_CONFIG_Addr1 = 0x0000;

	H3900_ASIC3_SD_CONFIG_ClkStop = SD_CONFIG_CLKSTOP_ENABLE_ALL; // setup stop clock (i think)
	H3900_ASIC3_SD_CONFIG_SDHC_CardDetect = 2; // two cycles (possible 512ms)
	H3900_ASIC3_SD_CONFIG_Command = SD_CONFIG_COMMAND_MAE; // access to control registers enabled

	H3900_ASIC3_SD_CTRL_SoftwareReset = 0;
	mdelay(2);

	H3900_ASIC3_SD_CTRL_SoftwareReset = 1;
	mdelay(2);

	H3900_ASIC3_SDIO_CTRL_SoftwareReset = 0;
	mdelay(2);

	H3900_ASIC3_SDIO_CTRL_SoftwareReset = 1;
	mdelay(2);

	H3900_ASIC3_SD_CTRL_IntMaskCard = 0x0; // mask all interrupts, they are not used
	H3900_ASIC3_SD_CTRL_IntMaskBuffer = 0x0; //       --- // ---

	H3900_ASIC3_SDHWCTRL_SDConf &= ~ASIC3_SDHWCTRL_PCLR; // clear all SDIO registers

	H3900_ASIC3_SD_CTRL_MemCardOptionSetup = MEM_CARD_OPTION_REQUIRED |
						MEM_CARD_OPTION_DATA_RESPONSE_TIMEOUT(14) |
						MEM_CARD_OPTION_C2_MODULE_NOT_PRESENT |
						MEM_CARD_OPTION_DATA_XFR_WIDTH_1;

	H3900_ASIC3_SD_CTRL_ErrorStatus0 = 0; // clear detail error status register
	H3900_ASIC3_SD_CTRL_ErrorStatus1 = 0; //       --- // ---

	H3900_ASIC3_SD_CTRL_CardStatus = 0; // clear card status
	H3900_ASIC3_SD_CTRL_BufferCtrl = 0; // clear buffer controller status
	H3900_ASIC3_SD_CTRL_StopInternal = 0; // clear stop transfer register
	H3900_ASIC3_SDIO_CTRL_ClocknWaitCtrl = 0x0100; // setup clock for SDIO

	H3900_ASIC3_SDHWCTRL_SDConf |= ASIC3_SDHWCTRL_SDPWR; // turn on power at controller(?)
	mdelay(10);
	H3900_ASIC3_SD_CONFIG_SDHC_Power3 = 0; // turn off auto power down after card remove
	H3900_ASIC3_SD_CONFIG_SDHC_Power1 = SD_CONFIG_POWER1_PC_33V; 	// turn on power to card,
									// turn off auto power up after card inserted
	mdelay(10);

	H3900_ASIC3_SD_CONFIG_ClockMode = CLOCK_MODE_DIV_CLOCK_ENABLE; // enable div clock for change rate
	H3900_ASIC3_SD_CTRL_CardClockCtrl = 0; // reset card clock ?
	H3900_ASIC3_SD_CTRL_CardClockCtrl = SD_CTRL_CARDCLOCKCONTROL_ENABLE_CLOCK | SD_CTRL_CARDCLOCKCONTROL_CLK_DIV_64;
	mdelay(1);

	MMC_DEBUG(0, "Card status 0x%04x\n", H3900_ASIC3_SD_CTRL_CardStatus);
	MMC_DEBUG(0, "Buffer status 0x%04x\n", H3900_ASIC3_SD_CTRL_BufferCtrl);

	g_data.flag = 0;

	return MMC_NO_ERROR;
}

static void mmc_asic3_set_command(u16 cmd, u32 arg)
{
	MMC_DEBUG(1, "cmd=%d arg=0x%08x", cmd, arg);

	H3900_ASIC3_SD_CTRL_Arg1 = arg >> 16;
	H3900_ASIC3_SD_CTRL_Arg0 = arg & 0xffff;
	H3900_ASIC3_SD_CTRL_Cmd = cmd;
}

static void mmc_asic3_set_transfer(u16 block_len, u16 nob)
{
	MMC_DEBUG(1, "block_len=%d nob=%d", block_len, nob);

	H3900_ASIC3_SD_CTRL_MemCardXferDataLen = block_len;	// size of block (default 512 bytes)
	H3900_ASIC3_SD_CTRL_TransferSectorCount = nob;		// number blocks to transfer
}

static void mmc_asic3_transmit_data(struct mmc_request *request)
{
	u8 *buf = request->buffer;
	u16 data;
	int i;

	MMC_DEBUG(1, "nob=%d block_len=%d", request->nob, request->block_len);
	if (request->nob <= 0) {
		MMC_DEBUG(1, "*** nob already at 0 ***");
		return;
	}

	for (i = 0; i < request->block_len; i += 2, buf += 2) {
		data = ((u16) *buf) | (*(buf + 1) << 8);
		H3900_ASIC3_SD_CTRL_DataPort = data; // write two bytes to SD/MMC controller buffer
	}

	START_MMC_DEBUG(3) {
		u8 *b = request->buffer;
		for (i = 0 ; i < request->block_len ; i++) {
			printk(" %02x", *b++);
			if (((i + 1) % 16) == 0)
				printk("\n");
		}
	} END_MMC_DEBUG;

	// Updated the request buffer to reflect the current state
	request->buffer = (u8 *)buf;
	request->nob--;
}

static void mmc_asic3_receive_data(struct mmc_request *request)
{
	u8 *buf = request->buffer;
	u16 data;
	int i;

	MMC_DEBUG(1, "nob=%d block_len=%d buf=%p", request->nob, request->block_len, buf);
	if (request->nob <= 0) {
		MMC_DEBUG(1, "*** nob already at 0 ***");
		return;
	}

	for (i = 0; i < request->block_len; i += 2) {
		data = H3900_ASIC3_SD_CTRL_DataPort; // read two bytes from SD/MMC controller buffer
		*buf++ = data & 0xff;
		*buf++ = data >> 8;
	}

	START_MMC_DEBUG(3) {
		u8 *b = request->buffer;
		for (i = 0; i < request->block_len ; i++) {
			printk(" %02x", *b++);
			if ( ((i + 1) % 16) == 0 )
				printk("\n");
		}
	} END_MMC_DEBUG;

	// Updated the request buffer to reflect the current state 
	request->buffer = (u8 *)buf;
	request->nob--;
}

static void mmc_asic3_get_response(struct mmc_request *request)
{
	int i, len = rinfo[request->rtype].length;
	u8 *buf = request->response;
	u16 data;

	request->result = MMC_NO_ERROR;    // Mark this as having a request result of some kind
	if (len <= 0) {
		MMC_DEBUG(1, "no response");
		return;
	}

	if (len > 5) { // R2 handle
		data = H3900_ASIC3_SD_CTRL_Response7;
		buf[0] = 0x3f;
		buf[1] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response6;
		buf[2] = data >> 8;
		buf[3] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response5;
		buf[4] = data >> 8;
		buf[5] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response4;
		buf[6] = data >> 8;
		buf[7] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response3;
		buf[8] = data >> 8;
		buf[9] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response2;
		buf[10] = data >> 8;
		buf[11] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response1;
		buf[12] = data >> 8;
		buf[13] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response0;
		buf[14] = data >> 8;
		buf[15] = data & 0xff;
		buf[16] = 1; // ecc flag
	} else { // R1, R1B, R3 handle
		if (request->rtype == RESPONSE_R3)
			buf[0] = 0x3f;
		else
			buf[0] = request->cmd;
		data = H3900_ASIC3_SD_CTRL_Response1;
		buf[1] = data >> 8;
		buf[2] = data & 0xff;
		data = H3900_ASIC3_SD_CTRL_Response0;
		buf[3] = data >> 8;
		buf[4] = data & 0xff;
	}


	START_MMC_DEBUG(3) {
		printk(__FILE__ ":%s - Response byte stream: ", __FUNCTION__);
		for (i = 0; i < len; i++)
			printk("%02x ", buf[i]);
		printk("\n");
	} END_MMC_DEBUG;
}

static void mmc_asic3_send_command(struct mmc_request *request)
{
	u16 card_status, buffer_status;
	int retval = MMC_NO_ERROR, timeout, res_end = 0, rw_end = 0;
	int read_enable = 0, write_enable = 0, transfer = 0, cdc = 0;

	MMC_DEBUG(2, "request=%p cmd=%d (%s) arg=%08x", request, 
	      request->cmd, get_cmd_name(request->cmd), request->arg);

	g_data.request = request;
	request->result = MMC_NO_RESPONSE;   // Flag to indicate don't have a result yet

	if (request->cmd == MMC_CIM_RESET) {
		retval = mmc_asic3_reset_asic();
		goto end_command;
	}

	if (request->cmd == MMC_STOP_TRANSMISSION) {
		H3900_ASIC3_SD_CTRL_StopInternal = SD_CTRL_STOP_INTERNAL_ISSSUE_CMD12;
                request->response[0] = request->cmd;
                request->response[1] = 0;
                request->response[2] = 0;
                request->response[3] = 0;
                request->response[4] = 0;
		retval = MMC_NO_ERROR;
		goto end_command;
	}

	// reset card status
	card_status = H3900_ASIC3_SD_CTRL_CardStatus; // detect if state of card is changed first
	card_status &= (SD_CTRL_CARDSTATUS_CARD_REMOVED_0 |
			SD_CTRL_CARDSTATUS_CARD_INSERTED_0 |
			SD_CTRL_CARDSTATUS_SIGNAL_STATE_PRESENT_0 |
			SD_CTRL_CARDSTATUS_WRITE_PROTECT);
	H3900_ASIC3_SD_CTRL_CardStatus = card_status;
	// reset buffer status
	H3900_ASIC3_SD_CTRL_BufferCtrl = 0;
	H3900_ASIC3_SD_CTRL_ErrorStatus0 = 0;
	H3900_ASIC3_SD_CTRL_ErrorStatus1 = 0;

	timeout = 0;
	do {	// check and wait if SD/MMC controller is ready for receive commands
		buffer_status = H3900_ASIC3_SD_CTRL_BufferCtrl;
		if ((buffer_status & (SD_CTRL_BUFFERSTATUS_ILLEGAL_ACCESS |
				SD_CTRL_BUFFERSTATUS_CMD_INDEX_ERROR |
				SD_CTRL_BUFFERSTATUS_CRC_ERROR |
				SD_CTRL_BUFFERSTATUS_DATA_TIMEOUT |
				SD_CTRL_BUFFERSTATUS_STOP_BIT_END_ERROR |
				SD_CTRL_BUFFERSTATUS_BUFFER_OVERFLOW |
				SD_CTRL_BUFFERSTATUS_BUFFER_UNDERFLOW |
				SD_CTRL_BUFFERSTATUS_CMD_TIMEOUT)) != 0) {
			MMC_DEBUG(0, "Buffer status ERROR 0x%04x - inside check buffer\n", buffer_status);
			MMC_DEBUG(0, "detail0 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus0);
			MMC_DEBUG(0, "detail1 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus1);
			retval = MMC_ERROR_GENERAL;
			goto end_command;
		}
		if (timeout > 3000000) {
			MMC_DEBUG(0, "TIMEOUT - inside check buffer\n");
			retval = MMC_ERROR_TIMEOUT;
			goto end_command;
		}	
		timeout++;
		udelay(1);
	} while ((buffer_status & SD_CTRL_BUFFERSTATUS_CMD_BUSY) != 0);

	MMC_DEBUG(3, "Buffer status 0x%04x - after check buffer\n", buffer_status);

	switch (request->cmd) {
	case MMC_READ_MULTIPLE_BLOCK:
		H3900_ASIC3_SD_CTRL_StopInternal = SD_CTRL_STOP_INTERNAL_AUTO_ISSUE_CMD12;
		cdc = SD_CTRL_COMMAND_TRANSFER_READ | SD_CTRL_COMMAND_DATA_PRESENT | SD_CTRL_COMMAND_MULTI_BLOCK;
		cdc |= rinfo[request->rtype].cdc_flags;
		mmc_asic3_set_transfer(request->block_len, request->nob);
		mmc_asic3_set_command(request->cmd | cdc, request->arg);
		transfer = 1;
		break;
	case MMC_WRITE_MULTIPLE_BLOCK:
		H3900_ASIC3_SD_CTRL_StopInternal = SD_CTRL_STOP_INTERNAL_AUTO_ISSUE_CMD12;
		cdc = SD_CTRL_COMMAND_TRANSFER_WRITE | SD_CTRL_COMMAND_DATA_PRESENT | SD_CTRL_COMMAND_MULTI_BLOCK;
		cdc |= rinfo[request->rtype].cdc_flags;
		mmc_asic3_set_transfer(request->block_len, request->nob);
		mmc_asic3_set_command(request->cmd | cdc, request->arg);
		transfer = 1;
		break;
	case MMC_READ_SINGLE_BLOCK:
		cdc = SD_CTRL_COMMAND_TRANSFER_READ | SD_CTRL_COMMAND_DATA_PRESENT;
		cdc |= rinfo[request->rtype].cdc_flags;
		mmc_asic3_set_transfer(request->block_len, request->nob);
		mmc_asic3_set_command(request->cmd | cdc, request->arg);
		transfer = 1;
		break;
	case MMC_WRITE_BLOCK:
		cdc = SD_CTRL_COMMAND_TRANSFER_WRITE | SD_CTRL_COMMAND_DATA_PRESENT;
		cdc |= rinfo[request->rtype].cdc_flags;
		mmc_asic3_set_transfer(request->block_len, request->nob);
		mmc_asic3_set_command(request->cmd | cdc, request->arg);
		transfer = 1;
		break;
	default:
		if (request->cmd == MMC_GO_IDLE_STATE)
		    cdc |= SD_CTRL_COMMAND_TYPE_IDLE;
		if (request->cmd == MMC_APP_CMD)
		    cdc |= SD_CTRL_COMMAND_TYPE_ACMD;
		cdc |= rinfo[request->rtype].cdc_flags;
		mmc_asic3_set_command(request->cmd | cdc, request->arg);
		rw_end = 1;
		break;
	}

	timeout = 0;
	do {
		buffer_status = H3900_ASIC3_SD_CTRL_BufferCtrl;
		if ((buffer_status & SD_CTRL_BUFFERSTATUS_CMD_TIMEOUT) != 0) {
			MMC_DEBUG(0, "got status buffer COMMAND TIMEOUT - inside loop after send cmd\n");
			MMC_DEBUG(0, "detail0 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus0);
			MMC_DEBUG(0, "detail1 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus1);
			retval = MMC_ERROR_TIMEOUT;
			goto end_command;
		}
		if ((buffer_status & SD_CTRL_BUFFERSTATUS_DATA_TIMEOUT) != 0) {
			MMC_DEBUG(0, "got status buffer DATA TIMEOUT - inside loop after send cmd\n");
			MMC_DEBUG(0, "detail0 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus0);
			MMC_DEBUG(0, "detail1 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus1);
			retval = MMC_ERROR_TIMEOUT;
			goto end_command;
		}
		if ((buffer_status & (SD_CTRL_BUFFERSTATUS_ILLEGAL_ACCESS |
				SD_CTRL_BUFFERSTATUS_CMD_INDEX_ERROR |
				SD_CTRL_BUFFERSTATUS_CRC_ERROR |
				SD_CTRL_BUFFERSTATUS_STOP_BIT_END_ERROR |
				SD_CTRL_BUFFERSTATUS_BUFFER_OVERFLOW |
				SD_CTRL_BUFFERSTATUS_BUFFER_UNDERFLOW)) != 0) {
			retval = MMC_ERROR_GENERAL;
			MMC_DEBUG(0, "Buffer status ERROR 0x%04x - inside loop after send cmd\n", buffer_status);
			MMC_DEBUG(0, "detail0 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus0);
			MMC_DEBUG(0, "detail1 error status 0x%04x\n", H3900_ASIC3_SD_CTRL_ErrorStatus1);
			goto end_command;
		}
		if ((buffer_status & SD_CTRL_BUFFERSTATUS_BUFFER_READ_ENABLE) != 0) {
			MMC_DEBUG(3, "got status buffer READ ENABLE\n");
			read_enable = 1;
			H3900_ASIC3_SD_CTRL_BufferCtrl &= ~SD_CTRL_BUFFERSTATUS_BUFFER_READ_ENABLE;
		}
		if ((buffer_status & SD_CTRL_BUFFERSTATUS_BUFFER_WRITE_ENABLE) != 0) {
			MMC_DEBUG(3, "got status buffer WRITE ENABLE\n");
			write_enable = 1;
			H3900_ASIC3_SD_CTRL_BufferCtrl &= ~SD_CTRL_BUFFERSTATUS_BUFFER_WRITE_ENABLE;
		}
		if ((read_enable == 1) && (transfer == 1) && (request->nob > 0)) {
			MMC_DEBUG(3, "starting receive data\n");
			mmc_asic3_receive_data(request);
			timeout = 0;
		}
		if ((write_enable == 1) && (transfer == 1) && (request->nob > 0)) {
			MMC_DEBUG(3, "starting transit data\n");
			mmc_asic3_transmit_data(request);
			timeout = 0;
		}
		if ((H3900_ASIC3_SD_CTRL_CardStatus & SD_CTRL_CARDSTATUS_RW_END) != 0) {
			MMC_DEBUG(3, "got END of RW buffer operation - inside loop after send cmd\n");
			rw_end = 1;
			H3900_ASIC3_SD_CTRL_CardStatus &= ~SD_CTRL_CARDSTATUS_RW_END;
			H3900_ASIC3_SD_CTRL_StopInternal = 0;
		}
		if ((H3900_ASIC3_SD_CTRL_CardStatus & SD_CTRL_CARDSTATUS_RESPONSE_END) != 0) {
			MMC_DEBUG(2, "got RESPONSE - inside loop after send cmd\n");
			res_end = 1;
			H3900_ASIC3_SD_CTRL_CardStatus &= ~SD_CTRL_CARDSTATUS_RESPONSE_END;
			mmc_asic3_get_response(request);
		}
		if (timeout > 3000000) {
			MMC_DEBUG(0, "TIMEOUT - inside loop after send cmd\n");
			retval = MMC_ERROR_TIMEOUT;
			goto end_command;
		}
		timeout++;
		udelay(1);
	} while ((res_end == 0) || (rw_end == 0));

	MMC_DEBUG(3, "Card status 0x%04x - after cmd, before response parse\n", H3900_ASIC3_SD_CTRL_CardStatus);
	MMC_DEBUG(3, "Buffer status 0x%04x - after cmd, before response parse\n", H3900_ASIC3_SD_CTRL_BufferCtrl);

	if ((request->cmd == SD_APP_OP_COND) && (retval == MMC_NO_ERROR)) {
		g_data.flag = 1;
	}

	if ((request->cmd == SD_APP_SET_BUS_WIDTH) && (retval == MMC_NO_ERROR)) {
		H3900_ASIC3_SD_CTRL_MemCardOptionSetup = MEM_CARD_OPTION_REQUIRED |
						MEM_CARD_OPTION_DATA_RESPONSE_TIMEOUT(14) |
						MEM_CARD_OPTION_C2_MODULE_NOT_PRESENT |
						MEM_CARD_OPTION_DATA_XFR_WIDTH_4;
	}

end_command:
        request->result = retval;
        mmc_cmd_complete(request);
}

static void mmc_asic3_enable_hardware(void)
{
}

static void mmc_asic3_disable_hardware(void)
{
}

static int mmc_asic3_slot_is_empty(int slot)
{
	return 0;
}

static void mmc_asic3_slot_up(void)
{
}

static void mmc_asic3_slot_down(void)
{
}

static int mmc_asic3_slot_init(void)
{
	MMC_DEBUG(2, "");

//	mmc_asic3_slot_up();

	return 0;
}

static void mmc_asic3_slot_cleanup(void)
{
	MMC_DEBUG(2, "");

//	mmc_asic3_slot_down();
}

int mmc_asic3_suspend(void)
{
//	mmc_asic3_slot_down();
	return 0;
}

void mmc_asic3_resume(void)
{
//	mmc_asic3_slot_up();
}

static struct mmc_slot_driver dops = {
	owner:     THIS_MODULE,
	name:      "ASIC3 MMC/SD",
	ocr:       0x00ffc000,
	flags:     MMC_SDFLAG_MMC_MODE | MMC_SDFLAG_SD_MODE | MMC_SDFLAG_4BIT_BUS,

	init:      mmc_asic3_slot_init,
	cleanup:   mmc_asic3_slot_cleanup,
	is_empty:  mmc_asic3_slot_is_empty,
	send_cmd:  mmc_asic3_send_command,
	set_clock: mmc_asic3_set_clock,
};

int __init mmc_asic3_init(void)
{
	int retval = 0;

	MMC_DEBUG(2, "");

	if (!machine_is_h3900 ()) {	/* seems to put h3900 into some bad state */
		H3900_ASIC3_SDHWCTRL_SDConf = ASIC3_SDHWCTRL_CLKSEL | ASIC3_SDHWCTRL_SUSPEND;
		mdelay(10);
	}

	g_data.flag = 0;

	retval = mmc_register_slot_driver(&dops, 1);
	if (retval < 0)
		PALERT("unable to register slot");

	return retval;
}

void __exit mmc_asic3_cleanup(void)
{
	MMC_DEBUG(2, "");

	H3900_ASIC3_SD_CONFIG_SDHC_Power1 = SD_CONFIG_POWER1_PC_OFF;
	mdelay(1);
	H3900_ASIC3_SDHWCTRL_SDConf &= ~ASIC3_SDHWCTRL_SDPWR;
	mdelay(1);
        H3900_ASIC3_CLOCK_CDEX &= ~CLOCK_CDEX_EX1;
	mdelay(1);
        H3900_ASIC3_SDHWCTRL_SDConf |= ASIC3_SDHWCTRL_SUSPEND;
	mdelay(10);

	mmc_unregister_slot_driver(&dops);
}

module_init(mmc_asic3_init);
module_exit(mmc_asic3_cleanup);

MODULE_AUTHOR("Pawel Kolodziejski");
MODULE_DESCRIPTION("ASIC3 MMC/SD controller driver");
MODULE_LICENSE("GPL");
