/*
 * Glue audio driver for the S3C2410 & Philips UDA1341 codec.
 *
 * Copyright (c) 2000 John Dorsey
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * History:
 *
 * 2000-09-04	John Dorsey	SA-1111 Serial Audio Controller support
 * 				was initially added to the sa1100-uda1341.c
 * 				driver.
 *
 * 2001-06-03	Nicolas Pitre	Made this file a separate module, based on
 * 				the former sa1100-uda1341.c driver.
 *
 * 2001-09-23	Russell King	Remove old L3 bus driver.
 * 2002-02-26   Sangwook Lee    Modifed for S3C2400 <hitchcar@samsung.co.kr>
 *
 * 2002-06-10   Sangwook Lee    Modified for S3C2410 <hitchcar@sec.samsung.com>
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/pm.h>
#include <linux/l3/l3.h>
#include <linux/l3/uda1341.h>

#include <asm/semaphore.h>
#include <asm/mach-types.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>

#include "s3c2410-audio.h"

#undef DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#define swldebug( x... ) printk( ##x)
#else
#define DPRINTK( x... )
#define swldebug( x...) 
#endif


static struct l3_client uda1341;


#ifdef YOU_WANT_CHANGE_MPLL

static unsigned long SAVE_MPLLCON,SAVE_CLKDIVN;
/************************** MPLL *******************************/
static void ChangeMPllValue(int mdiv,int pdiv,int sdiv)
{
    rMPLLCON=(mdiv<<12)|(pdiv<<4)|sdiv;
}

/************************ HCLK, PCLK ***************************/

static void ChangeClockDivider(int hdivn,int pdivn)
// hdivn:pdivn FCLK:HCLK:PCLK
// 0:0         1:1:1 
// 0:1         1:1:2 
// 1:0         1:2:2
// 1:1         1:2:4
{
    rCLKDIVN=(hdivn<<1)|pdivn;    
}
#endif


static int
mixer_ioctl(struct inode *inode, struct file *file, uint cmd, ulong arg)
{
	/*
	 * We only accept mixer (type 'M') ioctls.
	 */
	if (_IOC_TYPE(cmd) != 'M')
		return -EINVAL;
//	printk("%s[%d] %s : before l3_command  \n", __FILE__,__LINE__,__FUNCTION__);
	return l3_command(&uda1341, cmd, (void *)arg);
}

static struct file_operations uda1341_mixer_fops = {
	ioctl:		mixer_ioctl,
	owner:		THIS_MODULE
};



/*
 * Audio interface
 */

static void s3c2410_audio_init(void *dummy)
{
	/* Initialize the UDA1341 internal state */
	l3_open(&uda1341);
}


static void s3c2410_audio_shutdown(void *dummy)
{
#ifdef SWL_CHANGE_CLOCK
        rCLKDIVN = SAVE_CLKDIVN;
        rMPLLCON = SAVE_MPLLCON;
#endif
	l3_close(&uda1341);
}

static int s3c2410_audio_ioctl( struct inode *inode, struct file *file,
				uint cmd, ulong arg)
{
	long val = 0;
	int ret = 0;

	switch (cmd) {
	case SNDCTL_DSP_STEREO:
		ret = get_user(val, (int *) arg);
		if (ret)
			return ret;
		/* the UDA1341 is stereo only */
		ret = (val == 0) ? -EINVAL : 1;
		return put_user(ret, (int *) arg);

	case SNDCTL_DSP_CHANNELS:
	case SOUND_PCM_READ_CHANNELS:
		/* the UDA1341 is stereo only */
		return put_user(2, (long *) arg);

	case SNDCTL_DSP_SPEED:
		ret = get_user(val, (long *) arg);
		if (ret) break;
		if (val < 8000) val = 8000;
		if (val > 44100) val = 44100;

/* For example
 * my wav file 8.0 khz sampling rate.
 * codeclk is 384fs = 8.0*384 = 3.0720Mhz
 * IISPSR :
 *        Prescaler control A  :  PCLK / ( N +1) = 3.072 Mhz  
 *                                N = 50.75/3.072 -1 = 15.67 = 0x10
 *        Prescaler control B  : A=B could be 
 *  
 */
		switch(val) {
		case 8000:
#if (AUDIO_CODEC_CLOCK == 256)        
		        rIISPSR=(24<<5)+24;
#else
		        rIISPSR=(15<<5)+15;
#endif
			break;
		case 11025:
#if (AUDIO_CODEC_CLOCK == 256)        
  		        rIISPSR=(17<<5)+17;
#else
		        rIISPSR=(11<<5)+11;
#endif
			break;
		case 22050:
#if (AUDIO_CODEC_CLOCK == 256)        
		        rIISPSR=(16<<5)+16;
#else 
   #if (LCD_TYPE == TFT640_480 )
			rIISPSR=(10<<5)+10; 

   #else
			rIISPSR=(5<<5)+5;	
			//Prescaler_A/B= for 8.4672MHz, 1:2:4, Fclk = 203MHz
   #endif	
#endif
			break;
		case 44100:
#if (AUDIO_CODEC_CLOCK == 256)        
		        rIISPSR=(7<<5)+7;	
			//Prescaler_A/B= for 11.2896MHz, 1:2:4, Fclk = 203MHz, 640*480
#else
			rIISPSR=(2<<5)+2;
#endif                
			break;
		default:
#if (AUDIO_CODEC_CLOCK == 256)        
		        rIISPSR=(16<<5)+16;
#else
			rIISPSR=(2<<5)+2;	
#endif                
			break;
		} 

	case SOUND_PCM_READ_RATE: /* I am n't sure what it is  */
		return put_user(val , (long *) arg);

	case SNDCTL_DSP_SETFMT:
	case SNDCTL_DSP_GETFMTS:
	        /*printk("%s[%d] %s : DSP_SAMPLESIZE  \n", __FILE__,__LINE__,__FUNCTION__); */
		/* we can do 16-bit only */
		return put_user(AFMT_S16_LE, (long *) arg);

	default:
		/* Maybe this is meant for the mixer (as per OSS Docs) */
		return mixer_ioctl(inode, file, cmd, arg);
	}

	return ret;
}

static audio_stream_t output_stream; 

static audio_state_t audio_state = {
	output_stream:	&output_stream,
	skip_dma_init:	0,  /* done locally */
	hw_init:	s3c2410_audio_init,
	hw_shutdown:	s3c2410_audio_shutdown,
	client_ioctl:	s3c2410_audio_ioctl,
	sem:		__MUTEX_INITIALIZER(audio_state.sem),
};

static int s3c2410_audio_open(struct inode *inode, struct file *file)
{
	/* Acquire and initialize DMA for transfer */
	output_stream.dma_ch = 2;/* used by s3c2410_audio_attach function  */
        audio_state.output_id = "S3C2410 Audio Out";
	audio_state.output_dma = DMA2_SOURCE0;      

	/*******   !!! VERY IMPORTANT !!!!! *************************/
	/* 
	 * THIS FUNCTION ENTRY POINT TO drivers/sound/s3c2410-audio.c
	 */
        return s3c2410_audio_attach(inode, file, &audio_state);
}

/*
 * Missing fields of this structure will be patched with the call
 * to sa1100_audio_attach().
 */
static struct file_operations s3c2410_audio_fops = {
	open:		s3c2410_audio_open,
	owner:		THIS_MODULE
};


static int audio_dev_id, mixer_dev_id;


static int __init s3c2410_uda1341_init(void)
{
	struct uda1341_cfg cfg;
	int ret;
	
	DPRINTK(" s3c2410_uda1341_init \n");
	ret = l3_attach_client(&uda1341, "l3-s3c2410", "uda1341");
	if (ret) goto out;

#if (AUDIO_CODEC_CLOCK == 256)	
	cfg.fs     = 256;
#else
      	cfg.fs     = 384;
#endif	
	cfg.format = FMT_I2S;

	/*---------------!!! VERY IMPORTANT !!!!---------------
	 * 
	 * THIS FUNCTION ENTRY POINT TO drivers/l3/l3-s3c2410.c
	 * l3_command  defined in include/linux/l3/l3.h
	 */
	
	/*
	 *
	 l3_command(&uda1341, L3_UDA1341_CONFIGURE, &cfg);
	 ------>     drivers/l3/l3-s3c2410.c init1341();
	*
	*/
	
	l3_command(&uda1341, L3_UDA1341_CONFIGURE, &cfg);

	/* register devices */
	audio_dev_id = register_sound_dsp(&s3c2410_audio_fops, -1);
	mixer_dev_id = register_sound_mixer(&uda1341_mixer_fops, -1);

	printk(KERN_INFO "S3C2410 UDA1341 / IIS  initialized\n");
	
	return 0;

out:
	return ret;
}

static void __exit s3c2410_uda1341_exit(void)
{
	unregister_sound_dsp(audio_dev_id);
	unregister_sound_mixer(mixer_dev_id);
	s3c2410_free_dma(output_stream.dma_ch);
/*  	s3c2410_free_dma(input_stream.dma_ch); */
	l3_detach_client(&uda1341);
}

module_init(s3c2410_uda1341_init);
module_exit(s3c2410_uda1341_exit);

MODULE_AUTHOR("Sangwook Lee ");
MODULE_DESCRIPTION("Audio driver for the S3C2410 & Philips UDA1341 codec.");

EXPORT_NO_SYMBOLS;
