/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#ifndef Transport_H
#define Transport_H

#include <zlib.h>
#include <errno.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "Misc.h"
#include "Control.h"

#include "Types.h"
#include "Timestamp.h"
#include "Socket.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Define this to know when a socket
// is created or destroyed.
//

#undef  REFERENCES

//
// Size of buffer if not set by user.
//

#define TRANSPORT_BUFFER_DEFAULT_SIZE         16384

//
// This class handles buffered
// IO of the socket stream.
//

class Transport
{
  public:

  //
  // Member functions.
  //

  Transport(int fd);

  virtual ~Transport();

  int fd() const
  {
    return fd_;
  }

  //
  // Virtual members redefined by proxy
  // and 'memory-to-memory' IO layers.
  //

  virtual int read(unsigned char *data, unsigned int size);

  virtual int write(T_write type, const unsigned char *data, const unsigned int size);

  virtual int flush();

  virtual int force(int limit = 0);

  virtual void finish()
  {
    w_buffer_.length_ = 0;
    w_buffer_.start_  = 0;

    finish_ = 1;
  }

  virtual int length() const
  {
    return w_buffer_.length_;
  }

  virtual int pending() const
  {
    return 0;
  }

  virtual int readable() const
  {
    return GetBytesReadable(fd_);
  }

  virtual int writable() const
  {
    return GetBytesWritable(fd_);
  }

  virtual int queued() const
  {
    return GetBytesQueued(fd_);
  }

  virtual int flushable() const
  {
    return 0;
  }

  int forced() const
  {
    return forced_;
  }

  virtual int wait(int timeout) const;

  void setSize(unsigned int initialSize,
                   unsigned int thresholdSize,
                       unsigned int maximumSize);

  virtual int partialReset()
  {
    forced_ = 0;

    return partialReset(w_buffer_);
  }

  virtual int fullReset();

  int blocked() const
  {
    return blocked_;
  }

  protected:

  //
  // Make room in the buffer to accomodate 
  // at least size bytes.
  //

  int resize(T_buffer &buffer, const int &size);

  int partialReset(T_buffer &buffer)
  {
    if (buffer.length_ == 0 &&
            (buffer.data_.size() > initialSize_ ||
                 buffer.data_.capacity() > initialSize_))
    {
      return fullReset(buffer);
    }

    return 0;
  }

  int fullReset(T_buffer &buffer);

  //
  // Data members.
  //

  int fd_;

  int blocked_;
  int forced_;
  int finish_;

  T_buffer w_buffer_;

  unsigned int initialSize_;
  unsigned int thresholdSize_;
  unsigned int maximumSize_;

  private:

  #ifdef REFERENCES

  static int references_;

  #endif
};

//
// This class handles buffered IO and 
// compression of the proxy stream.
//

class ProxyTransport : public Transport
{
  public:

  ProxyTransport(int fd);

  virtual ~ProxyTransport();

  virtual int read(unsigned char *data, unsigned int size);

  virtual int write(T_write type, const unsigned char *data, const unsigned int size);

  virtual int flush();

  virtual void finish()
  {
    w_buffer_.length_ = 0;
    w_buffer_.start_  = 0;

    r_buffer_.length_ = 0;
    r_buffer_.start_  = 0;

    finish_ = 1;
  }

  virtual int length() const
  {
    return w_buffer_.length_ + flush_;
  }

  virtual int pending() const
  {
    return r_buffer_.length_;
  }

  virtual int flushable() const
  {
    return flush_;
  }

  virtual int partialReset()
  {
    forced_ = 0;

    return (Transport::partialReset(r_buffer_) &&
                Transport::partialReset(w_buffer_));
  }

  virtual int fullReset();

  protected:

  int flush_;

  T_buffer r_buffer_;

  z_stream r_stream_;
  z_stream w_stream_;

  private:

  #ifdef REFERENCES

  static int references_;

  #endif
};

#endif /* Transport_H */
