/****************************************************************************
**
** Copyright (C) 1992-2004 Trolltech AS. All rights reserved.
**
** This file is part of the core module of the Qt Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef QDIR_H
#define QDIR_H

#include "qstring.h"
#include "qfileinfo.h"
#include "qstringlist.h"

class QDirPrivate;

class Q_CORE_EXPORT QDir
{
protected:
    QDirPrivate *d_ptr;
private:
    Q_DECLARE_PRIVATE(QDir)
public:
    enum Filter { Dirs        = 0x001,
                  Files       = 0x002,
                  Drives      = 0x004,
                  NoSymLinks  = 0x008,
                  TypeMask    = 0x00f,
#ifdef QT_COMPAT
                  All         = TypeMask,
#endif

                  Readable    = 0x010,
                  Writable    = 0x020,
                  Executable  = 0x040,
                  PermissionMask    = 0x070,
#ifdef QT_COMPAT
                  RWEMask     = 0x070,
#endif

                  Modified    = 0x080,
                  Hidden      = 0x100,
                  System      = 0x200,
                  AccessMask  = 0x3F0,

                  AllDirs       = 0x400,
                  CaseSensitive = 0x800,

                  NoFilter = -1
#ifdef QT_COMPAT
                  ,DefaultFilter = NoFilter
#endif
    };
    Q_DECLARE_FLAGS(Filters, Filter)
#ifdef QT_COMPAT
    typedef Filters FilterSpec;
#endif

    enum SortFlag { Name        = 0x00,
                    Time        = 0x01,
                    Size        = 0x02,
                    Unsorted    = 0x03,
                    SortByMask  = 0x03,

                    DirsFirst   = 0x04,
                    Reversed    = 0x08,
                    IgnoreCase  = 0x10,
                    DirsLast    = 0x20,
                    NoSort = -1
#ifdef QT_COMPAT
                  ,DefaultSort = NoSort
#endif
    };
    Q_DECLARE_FLAGS(SortFlags, SortFlag)

    QDir(const QDir &);
    QDir(const QString &path = QString());
    QDir(const QString &path, const QString &nameFilter,
         SortFlags sort = SortFlags(Name | IgnoreCase), Filters filter = TypeMask);
    ~QDir();

    QDir &operator=(const QDir &);
    QDir &operator=(const QString &path);

    void setPath(const QString &path);
    QString path() const;
    QString absolutePath() const;
    QString canonicalPath() const;

    QString dirName() const;
    QString filePath(const QString &fileName) const;
    QString absoluteFilePath(const QString &fileName) const;

    static QString convertSeparators(const QString &pathName);

    bool cd(const QString &dirName);
    bool cdUp();

    QStringList nameFilters() const;
    void setNameFilters(const QStringList &nameFilters);

    Filters filter() const;
    void setFilter(Filters filter);
    SortFlags sorting() const;
    void setSorting(SortFlags sort);

    uint count() const;
    QString operator[](int) const;

    static QStringList nameFiltersFromString(const QString &nameFilter);

    QStringList entryList(Filters filters = NoFilter, SortFlags sort = NoSort) const;
    QStringList entryList(const QStringList &nameFilters, Filters filters = NoFilter,
                          SortFlags sort = NoSort) const;

    QFileInfoList entryInfoList(Filters filters = NoFilter, SortFlags sort = NoSort) const;
    QFileInfoList entryInfoList(const QStringList &nameFilters, Filters filters = NoFilter,
                                SortFlags sort = NoSort) const;

    enum Recursion { Recursive = 0, NonRecursive = 1 };
    bool mkdir(const QString &dirName, Recursion recurse = NonRecursive) const;
    bool rmdir(const QString &dirName, Recursion recurse = NonRecursive) const;

    bool isReadable() const;
    bool exists() const;
    bool isRoot() const;

    static bool isRelativePath(const QString &path);
    inline static bool isAbsolutePath(const QString &path) { return !isRelativePath(path); }
    bool isRelative() const;
    inline bool isAbsolute() const { return !isRelative(); }
    bool makeAbsolute();

    bool operator==(const QDir &dir) const;
    inline bool operator!=(const QDir &dir) const {  return !operator==(dir); }

    bool remove(const QString &fileName);
    bool rename(const QString &oldName, const QString &newName);
    bool exists(const QString &name) const;

    static QFileInfoList drives();

    static QChar separator();

    static bool setCurrent(const QString &path);
    static inline QDir current() { return QDir(currentPath()); }
    static QString currentPath();

    static inline QDir home() { return QDir(homePath()); }
    static QString homePath();
    static inline QDir root() { return QDir(rootPath()); }
    static QString rootPath();
    static inline QDir temp() { return QDir(tempPath()); }
    static QString tempPath();

#ifndef QT_NO_REGEXP
    bool match(const QStringList &filters, const QString &fileName);
    bool match(const QString &filter, const QString &fileName);
#endif
    static QString cleanPath(const QString &path);
    void refresh() const;

#ifdef QT_COMPAT
    typedef SortFlags SortSpec;
    inline QT_COMPAT QString absPath() const { return absolutePath(); }
    inline QT_COMPAT QString absFilePath(const QString &fileName, bool acceptAbsPath = true) const
       { Q_UNUSED(acceptAbsPath); return absoluteFilePath(fileName); }
    inline QT_COMPAT bool matchAllDirs() const
        { return filter() & AllDirs; }
    inline QT_COMPAT void setMatchAllDirs(bool on)
    {
        if(on)
            setFilter(filter() | AllDirs);
        else
            setFilter(filter() & ~(int)AllDirs);
    }
    inline QT_COMPAT QStringList entryList(const QString &nameFilter, Filters filters = NoFilter,
                                           SortFlags sort = NoSort) const
    { return entryList(nameFiltersFromString(nameFilter), filters, sort); }
    inline QT_COMPAT QFileInfoList entryInfoList(const QString &nameFilter,
                                                 Filters filters = NoFilter,
                                                 SortFlags sort = NoSort) const
    { return entryInfoList(nameFiltersFromString(nameFilter), filters, sort); }

    QT_COMPAT QString nameFilter() const;
    QT_COMPAT void setNameFilter(const QString &nameFilter);

    inline QT_COMPAT bool mkdir(const QString &dirName, bool acceptAbsPath) const
        { Q_UNUSED(acceptAbsPath); return mkdir(dirName, NonRecursive); }
    inline QT_COMPAT bool rmdir(const QString &dirName, bool acceptAbsPath) const
        { Q_UNUSED(acceptAbsPath); return rmdir(dirName, NonRecursive); }

    inline QT_COMPAT void convertToAbs() { makeAbsolute(); }
    inline QT_COMPAT static QString currentDirPath() { return currentPath(); }
    inline QT_COMPAT static QString homeDirPath() { return homePath(); }
    inline QT_COMPAT static QString rootDirPath() { return rootPath(); }
    inline QT_COMPAT static QString cleanDirPath(const QString &name) { return cleanPath(name); }
#endif
};

Q_DECLARE_OPERATORS_FOR_FLAGS(QDir::Filters);

#endif // QDIR_H
