/*--------------------------------*-C-*---------------------------------*
 * File:	misc.C
 *----------------------------------------------------------------------*
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (c) 1996      mj olesen <olesen@me.QueensU.CA> Queen's Univ at Kingston
 * Copyright (c) 1997,1998 Oezguer Kesim <kesim@math.fu-berlin.de>
 * Copyright (c) 1998-2000 Geoff Wing <gcw@pobox.com>
 * Copyright (c) 2003-2004 Marc Lehmann <pcg@goof.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *----------------------------------------------------------------------*/

#include "../config.h"		/* NECESSARY */
#include "rxvt.h"		/* NECESSARY */

char *
rxvt_wcstombs (const wchar_t *str, int len)
{
  if (len < 0) len = wcslen (str);

  mbstate mbs;
  char *r = (char *)rxvt_malloc (len * MB_CUR_MAX + 1);

  char *dst = r;
  while (len--)
    {
      ssize_t l = wcrtomb (dst, *str++, mbs);

      if (l < 0)
        *dst++ = '?';
      else
        dst += l;
    }

  *dst++ = 0;

  return (char *)rxvt_realloc (r, dst - r);
}

wchar_t *
rxvt_mbstowcs (const char *str, int len)
{
  if (len < 0) len = strlen (str);

  wchar_t *r = (wchar_t *)rxvt_malloc ((len + 1) * sizeof (wchar_t));

  if ((ssize_t)mbstowcs (r, str, len + 1) < 0)
    *r = 0;

  return r;
}

char *
rxvt_wcstoutf8 (const wchar_t *str, int len)
{
  if (len < 0) len = wcslen (str);

  char *r = (char *)rxvt_malloc (len * 4 + 1);
  char *p = r;

  while (len--)
    {
      unicode_t w = *str++ & UNICODE_MASK;

      if      (w < 0x000080)
        *p++ = w;
      else if (w < 0x000800)
        *p++ = 0xc0 | ( w >>  6),
        *p++ = 0x80 | ( w        & 0x3f);
      else if (w < 0x010000)
        *p++ = 0xe0 | ( w >> 12       ),
        *p++ = 0x80 | ((w >>  6) & 0x3f),
        *p++ = 0x80 | ( w        & 0x3f);
      else if (w < 0x110000)
        *p++ = 0xf0 | ( w >> 18),
        *p++ = 0x80 | ((w >> 12) & 0x3f),
        *p++ = 0x80 | ((w >>  6) & 0x3f),
        *p++ = 0x80 | ( w        & 0x3f);
      else
        *p++ = '?';
    }

  *p++ = 0;

  return (char *)rxvt_realloc (r, p - r);
}

wchar_t *
rxvt_utf8towcs (const char *str, int len)
{
  if (len < 0) len = strlen (str);

  wchar_t *r = (wchar_t *)rxvt_malloc ((len + 1) * sizeof (wchar_t)),
          *p = r;

  unsigned char *s = (unsigned char *)str,
                *e = s + len;

  for (;;)
    {
      len = e - s;

      if (len == 0)
        break;
      else if (s[0] < 0x80)
        *p++ = *s++;
      else if (len >= 2
               && s[0] >= 0xc2 && s[0] <= 0xdf 
               && (s[1] & 0xc0) == 0x80)
        {
          *p++ = ((s[0] & 0x1f) << 6)
               |  (s[1] & 0x3f);
          s += 2;
        }
      else if (len >= 3
               && (   (s[0] == 0xe0                 && s[1] >= 0xa0 && s[1] <= 0xbf)
                   || (s[0] >= 0xe1 && s[0] <= 0xec && s[1] >= 0x80 && s[1] <= 0xbf)
                   || (s[0] == 0xed                 && s[1] >= 0x80 && s[1] <= 0x9f)
                   || (s[0] >= 0xee && s[0] <= 0xef && s[1] >= 0x80 && s[1] <= 0xbf)
                  )
               && (s[2] & 0xc0) == 0x80)
        {
          *p++ = ((s[0] & 0x0f) << 12)
               | ((s[1] & 0x3f) <<  6)
               |  (s[2] & 0x3f);
          s += 3;
        }
      else if (len >= 4
               && (   (s[0] == 0xf0                 && s[1] >= 0x90 && s[1] <= 0xbf)
                   || (s[0] >= 0xf1 && s[0] <= 0xf3 && s[1] >= 0x80 && s[1] <= 0xbf)
                   || (s[0] == 0xf4                 && s[1] >= 0x80 && s[1] <= 0x8f)
                  )
               && (s[2] & 0xc0) == 0x80
               && (s[3] & 0xc0) == 0x80)
        {
          *p++ = ((s[0] & 0x07) << 18) 
               | ((s[1] & 0x3f) << 12) 
               | ((s[2] & 0x3f) <<  6) 
               |  (s[3] & 0x3f);
          s += 4;
        }
      else
        {
          *p++ = 0xfffd;
          s++;
        }
    }

  *p = 0;

  return r;
}

char *
rxvt_r_basename (const char *str) NOTHROW
{
  char *base = strrchr (str, '/');

  return (char *) (base ? base + 1 : str);
}

/*
 * Print an error message
 */
void
rxvt_vlog (const char *fmt, va_list arg_ptr) NOTHROW
{
  char msg[1024];

  vsnprintf (msg, sizeof msg, fmt, arg_ptr);

  if (GET_R && GET_R->log_hook)
    (*GET_R->log_hook) (msg);
  else
    write (STDOUT_FILENO, msg, strlen (msg));
}

void
rxvt_log (const char *fmt,...) NOTHROW
{
  va_list arg_ptr;

  va_start (arg_ptr, fmt);
  rxvt_vlog (fmt, arg_ptr);
  va_end (arg_ptr);
}

/*
 * Print an error message
 */
void
rxvt_warn (const char *fmt,...) NOTHROW
{
  va_list arg_ptr;

  rxvt_log ("%s: ", RESNAME);

  va_start (arg_ptr, fmt);
  rxvt_vlog (fmt, arg_ptr);
  va_end (arg_ptr);
}

void
rxvt_fatal (const char *fmt,...) THROW ((class rxvt_failure_exception))
{
  va_list arg_ptr;

  rxvt_log ("%s: ", RESNAME);

  va_start (arg_ptr, fmt);
  rxvt_vlog (fmt, arg_ptr);
  va_end (arg_ptr);

  rxvt_exit_failure ();
}

void
rxvt_exit_failure () THROW ((class rxvt_failure_exception))
{
  static class rxvt_failure_exception rxvt_failure_exception;
  throw (rxvt_failure_exception);
}

/*
 * check that the first characters of S1 match S2
 *
 * No Match
 *      return: 0
 * Match
 *      return: strlen (S2)
 */
int
rxvt_Str_match (const char *s1, const char *s2) NOTHROW
{
  int n = strlen (s2);

  return ((strncmp (s1, s2, n) == 0) ? n : 0);
}

const char *
rxvt_Str_skip_space (const char *str) NOTHROW
{
  if (str)
    while (*str && isspace (*str))
      str++;

  return str;
}

/*
 * remove leading/trailing space and strip-off leading/trailing quotes.
 * in place.
 */
char           *
rxvt_Str_trim (char *str) NOTHROW
{
  char *r, *s;

  if (!str || !*str)		/* shortcut */
    return str;

  /* skip leading spaces */
  for (s = str; *s && isspace (*s); s++) ;

  /* goto end of string */
  r = s + strlen (s) - 1;

  /* dump return and other trailing whitespace */
  while (r > s && isspace (*r))
    r--;

#if 0
  /* skip matching leading/trailing quotes */
  if (*s == '"' && *r == '"' && n > 1)
    {
      s++;
      n -= 2;
    }
#endif

  memmove (str, s, r + 1 - s);
  str[r + 1 - s] = 0;

  return str;
}

/*
 * in-place interpretation of string:
 *
 *      backslash-escaped:      "\a\b\E\e\n\r\t", "\octal"
 *      Ctrl chars:     ^@ .. ^_, ^?
 *
 *      Emacs-style:    "M-" prefix
 *
 * Also,
 *      "M-x" prefixed strings, append "\r" if needed
 *      "\E]" prefixed strings (XTerm escape sequence) append ST if needed
 *
 * returns the converted string length
 */
int
rxvt_Str_escaped (char *str) NOTHROW
{
  char            ch, *s, *d;
  int             i, num, append = 0;

  if (!str || !*str)
    return 0;

  d = s = str;

  if (*s == 'M' && s[1] == '-')
    {
      /* Emacs convenience, replace leading `M-..' with `\E..' */
      *d++ = C0_ESC;
      s += 2;
      if (toupper (*s) == 'X')
        /* append carriage-return for `M-xcommand' */
        for (*d++ = 'x', append = '\r', s++; isspace (*s); s++) ;
    }
  for (; (ch = *s++);)
    {
      if (ch == '\\')
        {
          ch = *s++;
          if (ch >= '0' && ch <= '7')
            {	/* octal */
              num = ch - '0';
              for (i = 0; i < 2; i++, s++)
                {
                  ch = *s;
                  if (ch < '0' || ch > '7')
                    break;
                  num = num * 8 + ch - '0';
                }
              ch = (char)num;
            }
          else if (ch == 'a')
            ch = C0_BEL;	/* bell */
          else if (ch == 'b')
            ch = C0_BS;	/* backspace */
          else if (ch == 'E' || ch == 'e')
            ch = C0_ESC;	/* escape */
          else if (ch == 'n')
            ch = '\n';	/* newline */
          else if (ch == 'r')
            ch = '\r';	/* carriage-return */
          else if (ch == 't')
            ch = C0_HT;	/* tab */
        }
      else if (ch == '^')
        {
          ch = *s++;
          ch = toupper (ch);
          ch = (ch == '?' ? 127 : (ch - '@'));
        }
      *d++ = ch;
    }

  /* ESC] is an XTerm escape sequence, must be terminated */
  if (*str == '\0' && str[1] == C0_ESC && str[2] == ']')
    append = CHAR_ST;

  /* add trailing character as required */
  if (append && d[-1] != append)
    *d++ = append;
  *d = '\0';

  return (d - str);
}

/*
 * Split a comma-separated string into an array, stripping leading and
 * trailing spaces from each entry.  Empty strings are properly returned
 * Caller should free each entry and array when done
 */
char          **
rxvt_splitcommastring (const char *cs) NOTHROW
{
  int             l, n, p;
  const char     *s, *t;
  char          **ret;

  if ((s = cs) == NULL)
    s = "";

  for (n = 1, t = s; *t; t++)
    if (*t == ',')
      n++;

  ret = (char **)malloc ((n + 1) * sizeof (char *));
  ret[n] = NULL;

  for (l = 0, t = s; l < n; l++)
    {
      for ( ; *t && *t != ','; t++) ;
      p = t - s;
      ret[l] = (char *)malloc (p + 1);
      strncpy (ret[l], s, p);
      ret[l][p] = '\0';
      rxvt_Str_trim (ret[l]);
      s = ++t;
    }

  return ret;
}

void
rxvt_freecommastring (char **cs) NOTHROW
{
  for (int i = 0; cs[i]; ++i)
    free (cs[i]);

  free (cs);
}

/*----------------------------------------------------------------------*
 * file searching
 */

#ifdef XPM_BACKGROUND
/*
 * search for FILE in the current working directory, and within the
 * colon-delimited PATHLIST, adding the file extension EXT if required.
 *
 * FILE is either semi-colon or zero terminated
 */
char *
rxvt_File_search_path (const char *pathlist, const char *file, const char *ext) NOTHROW
{
  int             maxpath, len;
  const char     *p, *path;
  char            name[256];

  if (!access (file, R_OK))	/* found (plain name) in current directory */
    return strdup (file);

  /* semi-colon delimited */
  if ((p = strchr (file, ';')))
    len = (p - file);
  else
    len = strlen (file);

  /* leave room for an extra '/' and trailing '\0' */
  maxpath = sizeof (name) - (len + (ext ? strlen (ext) : 0) + 2);
  if (maxpath <= 0)
    return NULL;

  /* check if we can find it now */
  strncpy (name, file, len);
  name[len] = '\0';

  if (!access (name, R_OK))
    return strdup (name);
  if (ext)
    {
      strcat (name, ext);
      if (!access (name, R_OK))
        return strdup (name);
    }
  for (path = pathlist; path != NULL && *path != '\0'; path = p)
    {
      int             n;

      /* colon delimited */
      if ((p = strchr (path, ':')) == NULL)
        p = strchr (path, '\0');

      n = (p - path);
      if (*p != '\0')
        p++;

      if (n > 0 && n <= maxpath)
        {
          strncpy (name, path, n);
          if (name[n - 1] != '/')
            name[n++] = '/';
          name[n] = '\0';
          strncat (name, file, len);

          if (!access (name, R_OK))
            return strdup (name);
          if (ext)
            {
              strcat (name, ext);
              if (!access (name, R_OK))
                return strdup (name);
            }
        }
    }
  return NULL;
}

char *
rxvt_File_find (const char *file, const char *ext, const char *path) NOTHROW
{
  char           *f;

  if (file == NULL || *file == '\0')
    return NULL;

  f = rxvt_File_search_path (path, file, ext);

  return f;
}
#endif


