/*****************************************************************************
 * colorspace.h: colorspace functions and definitions
 *****************************************************************************
 * $Id: colorspace.h,v 1.7 2004/09/19 14:15:39 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 1998 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/
#ifndef _XDTV_COLORSPACE_H_
#define _XDTV_COLORSPACE_H_

/* 
   same convention than:
   http://www.thedirks.org/v4l2/v4l2fmt.htm 
   except that the RGB formats are endian dependant
*/
typedef enum  {
  VIDEO_NOFORMAT = 0,

  /* rgb based formats */
  VIDEO_RGB08,  /* B|(G<<3)|(R<<6) */
  VIDEO_RGB15,  /* B|(G<<5)|(R<<10)   endian-dependant */
  VIDEO_RGB16,  /* B|(G<<5)|(R<<11)   endian-dependant */
  VIDEO_RGB24,  /* le: BGR   be: RGB  endian-dependant */
  VIDEO_RGB32,  /* le: BGR0  be: 0RGB endian-dependant */
  VIDEO_RGB32P, /* le: 0BGR  be: RGB0 endian-dependant */

  /* yuv based formats (endian-independant) */
  VIDEO_GRAY1,  /* Y (monochrome)     endian-dependant */
  VIDEO_GRAY8,  /* Y */
  VIDEO_YUYV,   /* Y1_U1_Y2_V2 (packed422) */
  VIDEO_UYVY,
  VIDEO_Y41P,
  VIDEO_YVU420, /* Y........YU..UV..V (planar411) */
  VIDEO_YUV420, /* Y........YU..UV..V (planar411) */
  VIDEO_YVU410,
  VIDEO_YUV410,
  VIDEO_YUV422P,
  VIDEO_YUV411P,
  VIDEO_NV12,
  
  /* reversed-endian formats */
  VIDEO_GRAY1X,
  VIDEO_RGB15X,
  VIDEO_RGB16X,
  VIDEO_RGB24X,
  VIDEO_RGB32X,
  VIDEO_RGB32PX,
  VIDEO_RGB32B, /* starts from the bottom...*/
  VIDEO_RGB24B,

  /* card-specific formats */
  VIDEO_HI240,   /* 16+B+5*R+25*G (remain 31 colors) */

  /* xdtv-specific formats */
  VIDEO_GRAY4,   /* Y          depth=4 bpp=8 (for vga) */ 
  VIDEO_RGB03,   /* B+2*G+4*R  depth=3 bpp=8 (for vga) */
  VIDEO_RGB04b,  /* R|(G<<2)|(B<<3) */
  VIDEO_RGB08b,  /* R|(G<<3)|(B<<6) */

  MAX_VIDEO_FMT  /* MUST ALWAYS BE THE LAST */
} video_fmt;

/* some encoders like jpeg/uncompressed
   need endianness-independant formats */
#ifdef WORDS_BIGENDIAN
#define VIDEO_GRAY1_BE VIDEO_GRAY1
#define VIDEO_RGB15_BE VIDEO_RGB15
#define VIDEO_RGB16_BE VIDEO_RGB16
#define VIDEO_RGB24_BE VIDEO_RGB24
#define VIDEO_RGB32_BE VIDEO_RGB32
#define VIDEO_RGB32P_BE VIDEO_RGB32P
#define VIDEO_GRAY1_LE VIDEO_GRAY1X
#define VIDEO_RGB15_LE VIDEO_RGB15X
#define VIDEO_RGB16_LE VIDEO_RGB16X
#define VIDEO_RGB24_LE VIDEO_RGB24X
#define VIDEO_RGB32_LE VIDEO_RGB32X
#define VIDEO_RGB32P_LE VIDEO_RGB32PX
#else
#define VIDEO_GRAY1_BE VIDEO_GRAY1X
#define VIDEO_RGB15_BE VIDEO_RGB15X
#define VIDEO_RGB16_BE VIDEO_RGB16X
#define VIDEO_RGB24_BE VIDEO_RGB24X
#define VIDEO_RGB32_BE VIDEO_RGB32X
#define VIDEO_RGB32P_BE VIDEO_RGB32PX
#define VIDEO_GRAY1_LE VIDEO_GRAY1
#define VIDEO_RGB15_LE VIDEO_RGB15
#define VIDEO_RGB16_LE VIDEO_RGB16
#define VIDEO_RGB24_LE VIDEO_RGB24
#define VIDEO_RGB32_LE VIDEO_RGB32
#define VIDEO_RGB32P_LE VIDEO_RGB32P
#endif

extern struct STRTAB video_fmt_names[];
int size_img(video_fmt f, int width, int height);
void* convert1(char *src, video_fmt f_src, video_fmt f_dest,
	       int width, int height);
void convert2(char *src, video_fmt f_src, char *dest, video_fmt f_dest,
	      int width, int height);
void swap24(unsigned char *m, int t);
int setpreferred_list(video_fmt formats[], video_fmt preferred[]);
int setpreferred(int formats[], video_fmt preferred[]);

extern video_fmt overlay_format;    // used by grabber.grab_overlay
extern video_fmt x11_native_format; // used by grab_one (mozaic)
extern video_fmt x11_pixmap_format; // used by video_displayframe
extern video_fmt grab_format;       // used by grabber.get_image

#endif // _XDTV_COLORSPACE_H_
