/*****************************************************************************
 * grab-avi.c: avi interface, reading frames from an avi file, only for test
 *  THE AVI FILE MUST BE IN XVID FORMAT.
 *****************************************************************************
 * $Id: grab-avi.c,v 1.43 2004/09/12 11:59:14 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2003 Alainjj
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include "config.h"
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/times.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef NOTHAVE_SEM 
#undef SEM_FAST_REPLACE
  #ifdef SEM_FAST_REPLACE
  /* The following replacement is not correct because
     a mutex should always be unlocked
     by the thread which has locked it. (see pthread_mutex(3)) 
     (in fact it works on linux but NOT on openbsd)
  */
  #undef NOTHAVE_SEM
  #define sem_t pthread_mutex_t
  #define sem_post pthread_mutex_unlock
  #define sem_wait pthread_mutex_lock
  #define sem_trywait pthread_mutex_trylock
  #define sem_init(x,y,z) \
     pthread_mutex_init(x,NULL); pthread_mutex_lock(x);
  #define sem_destroy pthread_mutex_destroy
  #endif
#else
#include <semaphore.h>
#endif
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <math.h>
#ifdef HAVE_XVID
#include <xvid.h>
#ifdef XVID_API
#define HAVE_XVID10 /* XVID 1.0 */
#else
#define HAVE_XVID09 /* XVID 0.9 */
#endif
#endif
#ifdef HAVE_LAME
#include <lame/lame.h>
#endif
#include "strtab.h"
#include "grab.h"
#include "avilib.h"
#include "memcpy.h"
#include "divx.h"
#include "mixer.h"
#include "colorspace.h"
#include "audio.h"


extern void video_set_capture_size(int w, int h);
extern int nbufs;
#ifdef HAVE_XVID
extern int fmts_xaw_to_xvidcolorspace[]; /* from divx.c */
#endif

/* variable specific for grab-avi */
char *avifilename;
int avibench = 0, avinoaudioout = 0;
#ifndef HAVE_SUNAUDIO
int avi_audio_fragments = 2;
#else
int avi_audio_fragments = 16;
#endif
int avi_audio_sizefragment = 2048;
static avi_t *   avifile = NULL;
#ifdef HAVE_XVID
#ifdef HAVE_XVID09
static XVID_DEC_PARAM xvid_dec_p;
#else
static xvid_dec_create_t xvid_dec_p;
#endif
#endif
void **buffers;
static char* bufcompress;
static double delay; // delay between two images
static clock_t t_start;
struct tms start;
static pthread_t audio_thread;
static int use_avi_audio = 0;
static pthread_mutex_t divx_mutex;
#ifndef NOTHAVE_SEM
sem_t avi_sem,avi_sem2;
#else
pthread_mutex_t avi_mut,avi_mut2,*avi_mut_audio,*avi_mut_video;
#endif
extern divx_t divx;
extern double audiostamp;
void ExitCB (Widget, XtPointer, XtPointer);
extern Widget app_shell;
   

/* in http client mode, the server has a priori any norm */
static struct STRTAB norms[] = {
  {0,   "PAL"},
  {1,  "NTSC"},
  {2, "SECAM"},
  {3, "PAL-Nc"},
  {4, "PAL-M"},
  {5, "PAL-N"},
  {6, "NTSC-JP"},
  {7, "PAL-60"},
  {-1, NULL}
};

/* idem */
static struct STRTAB inputs[] = {
  {0,   "Television"},
  {1,   "Composite1"},
  {2,   "Composite2"},
  {3,   "Composite3"},
  {4,   "Composite4"},
  {5,   "S-Video"},
  {6,   "other"},
  {-1, NULL}
};

extern struct GRABBER grab_avi;

static int raw_direct = 0;

void avi_audio_stop(void) {
  if(!use_avi_audio) return;
  pthread_cancel (audio_thread);
#ifndef NOTHAVE_SEM
  sem_post(&avi_sem);
#else
  pthread_mutex_unlock(avi_mut_video);
#endif
  pthread_join (audio_thread, NULL);
  if(!avinoaudioout) audio_close();
  use_avi_audio=0;
  pthread_mutex_destroy(&divx_mutex);
}

#define AUDIOBUF 200000
static char audiobuf[AUDIOBUF];
static int audiobuflen;
static int avireaduncompressedframe(char *buf) {
  static char audiobufcompressed[AUDIOBUF];
  int r;
  long len;
  struct timeval tv;
  char *audiobuf2;
  if(avifile->a_fmt==WAVE_FORMAT_PCM) 
    audiobuf2=audiobuf;
  else
    audiobuf2=audiobufcompressed;
  do {
    r=AVI_read_data(avifile, buf, 778*576*4, audiobuf2, AUDIOBUF, &len);
    if(debug) 
      fprintf(stderr,"AVI_read_data: av=%d len=%ld\n",r,len);
    if(r==2) {
      gettimeofday (&tv, NULL);
      audiostamp=tv.tv_usec/1000000.0+tv.tv_sec;
      if(audiobuf==audiobuf2)
	audiobuflen=len;
      else {
#ifdef HAVE_LAME
	if(avifile->a_fmt==WAVE_FORMAT_MPEGLAYER3) {
	  if(avifile->a_chans==1)
	    audiobuflen=lame_decode(audiobufcompressed,len,
				    (short*)audiobuf,(short*)audiobuf);
	  else {
	    static short pcm_l[AUDIOBUF/2],pcm_r[AUDIOBUF/2];
	    int i;
	    audiobuflen=lame_decode(audiobufcompressed,len,pcm_l,pcm_r);
	    for(i=0;i<audiobuflen;i++) {
	      ((short *)audiobuf)[2*i]=pcm_l[i];
	      ((short *)audiobuf)[2*i+1]=pcm_r[i];
	    }
	    audiobuflen*=2;
	  }
	  if(audiobuflen==0) 
	    if(debug>=2) {fprintf(stderr,"@@ mp3 frame -- need more data\n");}
	  if(audiobuflen<0) {
	    fprintf(stderr,"PB lame_decode\n");
	    exit(1);
	  }
	  audiobuflen*=2;
	} else 
#endif
	  {
	    fprintf(stderr, "Audio format not implemented \n");
	    fprintf(stderr, "convert with \"mencoder -oac pcm -ovc xvid\"\n");
	    exit(1);
	  }
      }
      if(debug>=2) fprintf(stderr,"audiobuflen=%d\n",audiobuflen);
      if(audiobuflen>0) {
#ifndef NOTHAVE_SEM
	sem_post(&avi_sem);
	sem_wait(&avi_sem2);
#else
	pthread_mutex_unlock(avi_mut_video);
	avi_mut_video = avi_mut_video=&avi_mut ? &avi_mut2 : &avi_mut;
#ifdef _POSIX_THREAD_IS_GNU_PTH
	pthread_yield_np();
#endif
	pthread_mutex_lock(avi_mut_video);
#endif
      }
    }
    if(r==0) {
      fprintf(stderr,"END OF AVI FILE REACHED\n");
      AVI_seek_start(avifile);
      if(avibench || !avifile->seekable) {
	int clockpersec = sysconf(_SC_CLK_TCK);
	clock_t t_end;
	struct tms end;
	avi_audio_stop();
	t_end = times(&end);
	printf("total  %.3lf secs\n",(double)(t_end-t_start)/clockpersec);
	printf("user   %.3lf secs\n",
	       (double)(end.tms_utime-start.tms_utime)/clockpersec);
	printf("system %.3lf secs\n",
	       (double)(end.tms_stime-start.tms_stime)/clockpersec);
	ExitCB(app_shell, NULL, NULL);
	exit(0);
      }
    }
  } while(r==2 || r==0);
  if(r!=1) return r; else return len;
}

int avi_extract_audiobuf(void *buf) {
  int r;
#ifdef _POSIX_THREAD_IS_GNU_PTH
	  pthread_yield_np();
#endif
#ifndef NOTHAVE_SEM
  sem_wait(&avi_sem);
#else
  pthread_mutex_lock(avi_mut_audio);
#endif
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  /* The thread must exit at this  point to be sure
     otherwise sem could have the value 2 */
  pthread_testcancel ();
  pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL);
  if(audiobuflen<=0) {
    /* happens with gdb ? */
    int x=-1;
    fprintf(stderr,"PB synchro\n");
#ifndef NOTHAVE_SEM
    while(sem_trywait(&avi_sem));
#else
    while(pthread_mutex_trylock(avi_mut_audio));
#endif
    pthread_exit(&x);
  }
  fast_memcpy(buf,audiobuf,audiobuflen);
  r=audiobuflen;
  audiobuflen=0;
#ifndef NOTHAVE_SEM
  sem_post(&avi_sem2);
#else
  pthread_mutex_unlock(avi_mut_audio);
  avi_mut_audio = avi_mut_audio=&avi_mut ? &avi_mut2 : &avi_mut;
#endif
  return r;
}


static void* audio_main(void *arg) {
  int l;
  static char audiobuf[AUDIOBUF];
  pthread_setcancelstate(PTHREAD_CANCEL_DISABLE,NULL);
  while(1) {
    l=avi_extract_audiobuf(audiobuf);
    if(!avinoaudioout) audio_write(audiobuf, l);
    pthread_mutex_lock(&divx_mutex);
    if (is_divx_in_progress()) {
      write_audio_to_avi((unsigned short*)audiobuf, l/(avifile->a_bits/8));
    }
    pthread_mutex_unlock(&divx_mutex);
  }
  pthread_exit(NULL);
}

void avi_audio_start(void) {
  if(avifile==NULL) return;
  use_avi_audio=1;
  if(!avinoaudioout) {
    if(!audio_open(AUDIO_WO, avifile->a_bits==8 ? AUDIO_U8 : AUDIO_S16LE,
		   avifile->a_chans, avifile->a_rate, avi_audio_fragments,
		   avi_audio_sizefragment)<0) {
      fprintf(stderr, "audio_open failed\n");
      pthread_exit(NULL);
    }
  }
  mixer_set_mode(MIXER_MODE_PCM);
  pthread_mutex_init (&divx_mutex, NULL);
  pthread_create (&audio_thread, NULL, audio_main, NULL);
}

static int
grab_open(struct device_t *device)
{
  int i;
#ifdef HAVE_XVID
#ifdef HAVE_XVID09
  XVID_INIT_PARAM ini;
#else
  xvid_gbl_init_t ini;
#endif
#endif

  avifile = AVI_open_input_file(avifilename, 1);
  if(avifile == NULL) {
    AVI_print_error("AVI_open_input_filename");
    return -1;
  }
  if(debug) {
    fprintf(stderr,"##AVI: width=%ld height=%ld fourcc=%4s\n",
	    avifile->width,avifile->height,avifile->compressor);
    fprintf(stderr,"##AVI: a_chans=%ld a_rate=%ld a_fmt=%ld a_bits=%ld\n",
	    avifile->a_chans, avifile->a_rate, avifile->a_fmt, avifile->a_bits);
  }
  video_set_capture_size(avifile->width, avifile->height);
  if(str_to_int(avifile->compressor, fourccs_raw) != -1) {
    raw_direct = 1;
  }
#ifdef HAVE_XVID
  else if(!strncasecmp(avifile->compressor,"XVID",4)) {
    memset(&ini,0,sizeof(ini));
#ifdef HAVE_XVID09
    if(xvid_init(NULL, 0, &ini, NULL)) {
      fprintf(stderr,"initialization xvid failed\n");
      return -1;
    }
    if(ini.api_version != API_VERSION) {
      fprintf(stderr,"xvid has changed since compilation, recompile !\n");
      return -1;
    }
#else
    ini.version = XVID_VERSION;
    if(xvid_global(NULL, 0, &ini, NULL)<0) {
      fprintf(stderr,"initialization xvid failed\n");
      return -1;
    }
    xvid_dec_p.version = XVID_VERSION;
#endif
    xvid_dec_p.width = avifile->width;
    xvid_dec_p.height = avifile->height;
    if(xvid_decore(NULL, XVID_DEC_CREATE, &xvid_dec_p, NULL)) {
      fprintf(stderr, "xvid init failed\n");
      return -1;
    }
    bufcompress=(char *) malloc(6*avifile->width*avifile->height);
  }
#endif
  else {
    fprintf(stderr,"video format (fourcc=%4s) not implemented\n",avifile->compressor);
    fprintf(stderr, "convert with \"mencoder -oac pcm -ovc xvid\"\n");
    exit(1);
  }
  nbufs = nbufs_default;
  if(nbufs > 3) nbufs = 3;
  buffers = (void **)(malloc(nbufs*sizeof(void*)));
  for(i=0;i<nbufs;i++)
    buffers[i] = malloc((avifile->width+15)*avifile->height*4);
  delay = (double)avifile->fps_den / avifile->fps_num;
#ifdef HAVE_LAME
  if(avifile->a_fmt==WAVE_FORMAT_MPEGLAYER3) {
    lame_decode_init();
    avifile->a_bits=16;
  }
#endif
  t_start = times(&start);
#ifndef NOTHAVE_SEM
  sem_init(&avi_sem ,0, 0);
  sem_init(&avi_sem2,0,0);
#else
  pthread_mutex_init (&avi_mut, NULL);
  pthread_mutex_init (&avi_mut2, NULL);
  avi_mut_audio = avi_mut_video = &avi_mut;
  pthread_mutex_lock(&avi_mut);
#endif
  avi_audio_start();
  return 0;
}

void avi_wait_write_audio(void) {
  pthread_mutex_lock(&divx_mutex);
  pthread_mutex_unlock(&divx_mutex);
}

static int
grab_close ()
{
  int i;
  avi_audio_stop();
#ifndef NOTHAVE_SEM
  sem_destroy(&avi_sem);
  sem_destroy(&avi_sem2);
#else
  pthread_mutex_destroy(&avi_mut);
  pthread_mutex_destroy(&avi_mut2);
#endif
#ifdef HAVE_XVID
  if(!strncasecmp(avifile->compressor,"XVID",4)) 
    xvid_decore(xvid_dec_p.handle,XVID_DEC_DESTROY, NULL, NULL);
#endif
#ifdef HAVE_LAME_DECODE_EXIT
  lame_decode_exit();
#endif
  AVI_close(avifile);
  avifile=NULL;
  for(i=0;i<nbufs;i++) free(buffers[i]);
  free(buffers);
  return 0;
}

/*
static int
grab_overlay (int x, int y, int width, int height, int format,
              struct OVERLAY_CLIP *oc, int count)
{
  return 0;
}
*/

void * get_img(video_fmt format, int width, int height) {
  struct timeval tv;
#ifdef HAVE_XVID
  int r;
#ifdef HAVE_XVID09
  XVID_DEC_FRAME dec;
#else
  xvid_dec_frame_t dec;
#endif
#endif
  int imgprec=img;

  img++;
  if(img==nbufs) img=0;
#ifdef HAVE_XVID
  if(!strncasecmp(avifile->compressor,"XVID",4)) {
    if(width<avifile->width || width>= avifile->width+16) {
      fprintf(stderr,
	      "avi_width=%ld requested_width in [%d %d]\n",avifile->width,
	      width-15,width);
      return NULL;
    }
  } else 
#endif
    if(width!=avifile->width) {
      fprintf(stderr,
	      "avi_width=%ld requested_width=%d !\n",avifile->width,width);
      return NULL;
    }
  if(height!=avifile->height) {
    fprintf(stderr,
	    "avi_height=%ld requested_height=%d !\n",avifile->height,height);
    return NULL;
  }
  if(debug) 
    fprintf(stderr,"extracting frame %ld\n",avifile->video_pos);
  
  if(raw_direct) {
    int l=avireaduncompressedframe(buffers[img]);
    if(l<0) {
      fprintf(stderr, "PB avireaduncompressedframe\n");
      return NULL;
    }
    if(l==0) {
      fast_memcpy(buffers[img], buffers[imgprec], size_img(format,width,height));
      if(debug) fprintf(stderr, "duplicated frame ?\n");
    }
  }
#ifdef HAVE_XVID
  else if(!strncasecmp(avifile->compressor,"XVID",4)) {
    int l=avireaduncompressedframe(bufcompress);
    if(l<0) {
      fprintf(stderr, "PB avireaduncompressedframe\n");
      return NULL;
    }
    if(l==0) {
      fast_memcpy(buffers[img], buffers[imgprec], size_img(format,width,height));
      if(debug) fprintf(stderr, "duplicated frame ?\n");
    } else {
      memset(&dec,0,sizeof(dec));
#ifdef HAVE_XVID09
      dec.colorspace = fmts_xaw_to_xvidcolorspace[format];
      dec.image = buffers[img];
      dec.stride = width;
#else
      dec.version = XVID_VERSION;
      dec.output.csp = fmts_xaw_to_xvidcolorspace[format];
      dec.output.plane[0] = buffers[img];
      dec.output.stride[0] = size_img(format,width,1);
#endif
      dec.bitstream = bufcompress;
      dec.length = l;
      r=xvid_decore(xvid_dec_p.handle, XVID_DEC_DECODE,&dec,NULL);
#ifdef HAVE_XVID09
      if(r>0) r=-1000-r;
#endif
      if(r<0) {
	fprintf(stderr, "xvid decoding error, the file is probably not XVID r=%d\n",r);
	fprintf(stderr, "  convert with 'mencoder -ovc xvid -oac copy'\n");
	return NULL;
      }
    }
  }
#endif
  gettimeofday (&tv, NULL);
  videostampmax=tv.tv_usec/1000000.0+tv.tv_sec;
  if(!avibench && videostampmax<videostampmin+delay) {
    usleep(1000000*(delay-videostampmax+videostampmin));
  }
  videostampmin=videostampmax;
  return buffers[img];
}  

static int fmt_available(video_fmt f) {
  if(raw_direct)
    return f == str_to_int(avifile->compressor,fourccs_raw);
#ifdef HAVE_XVID
  else
    return fmts_xaw_to_xvidcolorspace[f]!=0;
#else
  return 0;
#endif
}


/* ---------------------------------------------------------------------- */

static int
grab_tune (unsigned long freq)
{
  if(strncmp(avifilename,"http://",7)==0) {
    char buf[100];
    sprintf(buf,"freq %lu",freq);
    write(avifile->fdes,buf,strlen(buf)+1);
  }
  return 0;
}

static int
grab_tuned ()
{
  return 1;
}

static int
grab_input (int input, int norm)
{
  if(strncmp(avifilename,"http://",7)==0) {
    char buf[100];
    if(input!=-1) {
      sprintf(buf,"input %s",int_to_str(input,inputs));
      write(avifile->fdes,buf,strlen(buf)+1);
    }
    if(norm!=-1) {
      sprintf(buf,"norm %s",int_to_str(norm,norms));
      write(avifile->fdes,buf,strlen(buf)+1);
    }      
  }
  return 0;
}

static int
grab_picture (int color, int bright, int hue, int contrast)
{
  /* TODO */
  return 0;
}

static int
grab_audio (int mute, int volume, int *mode)
{
  return 0;
}

static int is525(int norm) {
  return norm == 1 || norm == 6 || norm==7;
}

static int issecam(int norm) {
  return norm == 2;
}

static void* get_buf(int i){
  return buffers[i];
}

struct GRABBER grab_avi = {
  "avi",
  norms, inputs,
  grab_open,
  grab_close,
  NULL,
  fmt_available,
  get_img,
  grab_tune,
  grab_tuned,
  grab_input,
  grab_picture,
  grab_audio,
  is525,
  issecam,
  get_buf
};
