/* Ack pipe
 * Ben Lynn
 */
/*
Copyright (C) 2002 Benjamin Lynn (blynn@cs.stanford.edu)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <xmms/util.h>

#include "common.h"
#include "config.h"
#include "pipe.h"
#include "ack.h"

static pipe_t ackpipe;

static GString *ackfile;

static jumptable_t ack_jt;

static void add_ack_command(gchar *s, void (* function)())
{
    jumptable_add(ack_jt, s, function);
}

static void ack_open()
{
    pipe_assign(ackpipe, ackfile->str);

    pipe_open(ackpipe);
    if (!ackpipe->is_open) {
	xmms_show_message("XMMSPipe Error", "Error opening ack pipe", "Ok",
		FALSE, NULL, NULL);
    }
}

static void ack_close()
{
    pipe_close(ackpipe);
}

static void ack_setfile()
{
    if (arg) g_string_assign(ackfile, arg);
}

static void ack_flush()
{
    pipe_readflush(ackpipe);
}

static void ack_setauto()
{
    setauto(&config->ackauto);
}

static void ack_on()
{
    if (!ackpipe->is_open) ack_open();
}

static void ack_off()
{
    if (ackpipe->is_open) ack_close();
}

static void ack_setecho()
{
    config->ackecho = 1;
}

static void ack_setmessage()
{
    if (arg) {
	config->ackecho = 0;
	g_string_assign(config->ackmessage, arg);
    }
}

static void ack_setprefix()
{
    if (arg) {
	g_string_assign(config->ackprefix, arg);
    }
}

static void ack_command()
{
    subcommand(ack_jt);
}

void ack_init()
{
    jumptable_init(ack_jt);

    ackfile = g_string_new(config->ackprefix->str);
    g_string_append(ackfile, filesuffix);

    if (config->ackauto) ack_open();

    add_command("ack", ack_command);
    add_ack_command("flush", ack_flush);
    add_ack_command("on", ack_on);
    add_ack_command("start", ack_on);
    add_ack_command("off", ack_off);
    add_ack_command("autostart", ack_setauto);
    add_ack_command("auto", ack_setauto);
    add_ack_command("echo", ack_setecho);
    add_ack_command("return", ack_setmessage);
    add_ack_command("message", ack_setmessage);
    add_ack_command("prefix", ack_setprefix);
    add_ack_command("file", ack_setfile);
}

void ack_free()
{
    pipe_free(ackpipe);
    jumptable_free(ack_jt);
}

void ack_send(char *s)
{
    gchar *message;

    if (!ackpipe->is_open) return;

    if (config->ackecho) {
	message = g_strconcat(s, "\n", NULL);
    } else {
	message = g_strconcat(config->ackmessage->str, "\n", NULL);
    }

    pipe_write(ackpipe, message);

    g_free(message);
}
