#!/usr/bin/perl -w

# doc.pl is a document viewer. The only supported format is MultiMarkDown.
#
# Accepts the following parameters:
#   doc     - URI of a MultiMarkdown document to be displayed (mandatory).
#   pgnum   - number of a page to be displayed in case of category view mode
#           (optional).
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

BEGIN {
    push @INC, "../lib";
}

use 5.8.8;
use strict;
use CGI;
use Date::Format;
use Date::Parse;
use CMSsh::Presentation::TemplateHandler;
use CMSsh::Presentation::Utils qw(cgi_die);
use CMSsh::DocumentMode;

my $mode = CMSsh::DocumentMode->new;
my $msg = $mode->get_messages;
my $cfg = $mode->get_config;

my $cgi = CGI->new;
my $uri = '/' . ($cgi->param('path') || '');
my $page_num = $cgi->param('pgnum');

my $vars; # Template variables

my $publication = $mode->find_publication($uri)
    or cgi_die $msg->get('CANT_FIND_PUB');
if ($publication->get_type eq 'DocumentMode::Document') {
# Show a single document
    my $doc = $publication;

    cgi_die $msg->get('BAD_REQUEST') if (defined $page_num);

    my $metadata = $doc->get_metadata
        or cgi_die $msg->get('CANT_GET_PUB_MTDATA', $doc->get_uri);
    if ($metadata->{Date}) {
        my @time = localtime str2time($metadata->{Date});
        $metadata->{Date} = strftime($cfg->get('DATE_TMPL'), @time);
    }
    
    my $markup = $doc->get_markup
        or cgi_die $msg->get('CANT_GET_PUB_MARKUP', $doc->get_uri);

    $vars = {
        title => $metadata->{Title},
        content => $markup,
        source => $doc->get_source_path,
        date => $metadata->{Date},
    };
} elsif ($publication->get_type eq 'DocumentMode::Category') {
# List a category
    my $cat = $publication;

    my $metadata = $cat->get_metadata
        or cgi_die $msg->get('CANT_GET_PUB_MTDATA', $cat->get_uri);

    my $prev_ref = $cat->get_inner_preview
        or cgi_die $msg->send('CANT_PREV_PUB', $cat->get_uri);

    my @previews = sort {
        return $b->{is_category} <=> $a->{is_category}
            if $a->{is_category} != $b->{is_category};
        if ($metadata->{'Sort by'} &&
            $metadata->{'Sort by'} eq 'Date') {
# Sort by date if it's explicitly requested
            return $b->{date} <=> $a->{date};
        } else {
# Sort by title by default
            return $a->{title} cmp $b->{title};
        }
    } @$prev_ref;

    my @curr_previews = @previews;
    my $shift = 0;
    my $last_id = $#previews;

    $page_num = 0 unless defined $page_num;
    my $prev_per_page = $cfg->get('PREVIEWS_PER_PAGE');
    if (@previews && $prev_per_page) {
# Fetch only previews located on the current page
        $shift = $page_num * $prev_per_page;
        $last_id = ($shift + $prev_per_page - 1 <= $#previews) ?
                      $shift + $prev_per_page - 1 : $#previews;
# We are out of boudaries, pageNum is too large
        cgi_die $msg->get('BAD_REQUEST') if ($last_id < $shift);

        @curr_previews = @previews[$shift..$last_id];
    }

    $vars = {
        title => $metadata->{Title},
        content => $cat->get_markup,
        curr_uri => $uri,
        previews => \@curr_previews,
        page_num => $page_num,
        is_first_page => $shift == 0,
        is_last_page => $last_id == $#previews,
    };
} else {
    cgi_die $msg->get('UNKNOWN_PUB_TYPE',
                      $publication->get_uri, $publication->get_type);
}

my $th = CMSsh::Presentation::TemplateHandler->new($mode->get_view);
$th->process('doc', $vars, $publication);

