# The View class represents the DocumentMode to the presentation layer.
# Basically it provides links to the root subcategories (for the menu) and
# links to all publications in the hierarchy from the current publication up to
# the root category, except the top category, which is displayed in the menu
# (for the path).
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::DocumentMode::View;

our @ISA = qw(CMSsh::Mode::View);

use strict;
use CMSsh::Mode::View;
use CMSsh::DocumentMode::Category;
use CMSsh::DocumentMode::Messages;
use CMSsh::Presentation::Utils qw(cgi_die);

my $Msg = CMSsh::DocumentMode::Messages->new;

sub get_menu_items ($$) {
    my ($self, $publication) = @_;

    my @result;
    my $root_cat = CMSsh::DocumentMode::Category->new('/');
    unless ($root_cat) {
        warn $Msg->get('CANT_INST_PUB', '/');
        return [];
    }
    my $listing = $root_cat->get_listing;
    unless ($listing) {
        warn $Msg->get('CANT_LIST_COLL', '/');
        return [];
    }

    foreach my $cat ($root_cat,
                     @{$listing->{CMSsh::DocumentMode::Category->get_type}}) {
        my $metadata = $cat->get_metadata;
        unless ($metadata) {
            warn $Msg->get('CANT_GET_PUB_MTDATA', '/');
            next;
        }
        my $uri = $cat->get_uri;
        my $is_active = defined $publication && (
                $cat->is_root &&
                    ($publication->get_type eq 'DocumentMode::Category' &&
                         $publication->is_root ||
                     $publication->get_type eq 'DocumentMode::Document' &&
                         ($publication->get_parent_uri eq '/')) ||
                !$cat->is_root && $publication->get_uri =~ /^$uri(\/.*)?$/);
        push @result, { href => $uri,
                        title => $metadata->{Name} || $metadata->{Title},
                        is_active => $is_active,
                        priority => $metadata->{Priority} || 0, };
    }

    return \@result;
}

sub get_path ($$) {
    my ($self, $publication) = @_;

    my @result;
    my $curr_path = '/';
    my @items = split('/', $publication->get_uri);
    foreach my $i (1..$#items) {
        my $curr_pub = $self->{_mode}->find_publication($curr_path . $items[$i])
            or cgi_die $Msg->get('PUB_DOESNT_EXIST', $curr_path . $items[$i]);
# Only the last item can be a document
        if ($i == $#items and $curr_pub->get_type eq 'DocumentMode::Document') {
            my $metadata = $curr_pub->get_metadata
                or cgi_die $Msg->get('CANT_GET_PUB_MTDATA',
                                     $curr_pub->get_uri);
            push @result, { href => $curr_path . $items[$i],
                            title => $metadata->{Title} };
        } elsif ($curr_pub->get_type eq 'DocumentMode::Category') {
# Skip the top category, because it is displayed in the menu
            if ($curr_pub->get_parent_uri ne '/') {
                my $metadata = $curr_pub->get_metadata
                    or cgi_die $Msg->get('CANT_GET_PUB_MTDATA',
                                         $curr_pub->get_uri);
                push @result, { href => $curr_path . $items[$i],
                                title => $metadata->{Name} ||
                                         $metadata->{Title} };
            }
        } else {
            cgi_die $Msg->get('UNKNOWN_PUB_TYPE',
                              $curr_pub->get_uri, $curr_pub->get_type);
        }
        $curr_path .= $items[$i] . '/';
    }

    return \@result;
}

1
