# File is a wrapper for low-level Perl's I/O operations with files.
#
# This file is part of cms.sh.
#
# cms.sh is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# cms.sh is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# cms.sh. If not, see <http://www.gnu.org/licenses/>.
#
# (c) 2010-2011 Vitaly Minko <vitaly.minko@gmail.com>

package CMSsh::FileSystem::File;

our @ISA = qw(CMSsh::FileSystem::Object);

use strict;
use File::Copy qw();
use CMSsh::FileSystem::Object;
use CMSsh::Messages;

my $Msg = CMSsh::Messages->new;

sub new ($$) {
    my ($class_name, $abs_path) = @_;

    my $self = CMSsh::FileSystem::Object->new($abs_path);

    unless (-f $self->{_path}) {
        warn $Msg->get('FSO_DOESNT_EXIST', $abs_path);
        return undef;
    }

    bless $self, $class_name;
    return $self;
}

sub get_content ($) {
    my $self = shift;

    my ($text, $line);
    open(TXT, '<', $self->{_path})
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return undef;
    };
    $text .= $line while ($line = <TXT>);
    close TXT;

    return $text;
}

sub get_basename ($) {
    my $self = shift;
    return undef unless ($self->{_name} =~ /^(.*?)(:?\.[^\.]+)?$/);
    return $1;
}

sub get_ext ($) {
    my $self = shift;
    return undef unless ($self->{_name} =~ /\.([^\.]+)$/);
    return $1;
}

sub get_basepath ($) {
    my $self = shift;
    return $self->get_parent_path . '/' . $self->get_basename;
}

sub move ($$) {
    my ($self, $dest_dir) = @_;

    my $abs_dest = $dest_dir->get_path;
    my $new_path = $abs_dest . '/' . $self->{_name};

# Do not overwrite exisitng files
    if (-f $new_path) {
        warn $Msg->get('FSO_ALRD_EXISTS', $new_path);
        return 0;
    }

    File::Copy::move($self->{_path}, $abs_dest)
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return 0;
    };
    
    $self->{_path} = $new_path;

    return 1;
}

sub rename ($$) {
    my ($self, $new_name) = @_;

    return 1 if ($new_name eq $self->{_name});

    if ($new_name =~ /\//) {
        warn $Msg->get('WRONG_NAME', $new_name);
        return 0;
    }

    my $new_path = $self->get_parent_path . '/' . $new_name;
    if (-f $new_path) {
        warn $Msg->get('FSO_ALRD_EXISTS', $new_path);
        return 0;
    }

    File::Copy::move($self->{_path}, $new_path)
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return 0;
    };

    $self->{_path} = $new_path;
    $self->_update_name;

    return 1;
}

sub remove ($) {
    my $self = shift;

    unlink($self->{_path})
    or do {
        warn $Msg->get('LOW_LEVEL_IO_ERR', $!);
        return 0;
    };

    return 1;
}

1
