/* PDB2ADR converts the HP 200LX PDB database (phone book) in conjunction
 * with GDBIO so that the output can directly be used as address book in
 * D&A Software's email client Post/LX.
 *
 * (c) 2001 by Daniel Hertrich http://www.daniel-hertrich.de/pdb2x
 *             Thanks to Grzegorz Wroblewski who helped to debug the program
 * 
 * needed: GDBIO (http://www.palmtop.net/super.html)
 *         Post/LX from D&A Software (http://www.dasoft.com)
 *
 * usage:  prepare phone book file with
 *
 *           GDBIO C:\_DAT\PHONE.PDB > PHONE.CDF
 *
 *         use PHONE.CDF as input file for PDB2ADR and give
 *         five more arguments (described below). Redirect the 
 *         output into a file of your choice (preferrable post.adr
 *         in the Post/LX directory):
 * 
 *           PDB2ADR PHONE.CDF N E P C CATEGORY > c:\WWWLX\POST.ADR
 *
 *         This will create a text file in the format required 
 *         by Post/LX as email address book. Email addresses are
 *         written into the output file as
 *
 *           name <email>
 *   
 *         and mobile phone numbers are written into the output 
 *         file as
 *
 *           number | name
 * 
 *         The latter is only for use with Stefan Peichl's PDU
 *         if you use Post/LX with PDU to send SMS messages over
 *         your mobile phone.
 *
 *         THE PARAMETERS N, E, P, C and CATEGORY:
 *  
 *         N is the number of the column of the CDF file which 
 *           holds the NAME of each entry (probably it is 1)
 *
 *         E is the number of the column of the CDF file which 
 *           holds the EMAIL of each entry (set to 0 if you don't
 *           have an email field any only want to export mobile
 *           phone numbers)
 *
 *         P is the number of the column of the CDF file which 
 *           holds the MOBILE PHONE NUMBER of each entry 
 *           (set to 0 if you don't want to export phone numbers)
 * 
 *         C is the number of the column of the CDF file which
 *           holds the CATEGORY of each entry
 *           (set to 0 if you don't need category check)
 *           If you set a category, only entries matching the given
 *           category will be written into the output file.
 *
 *         CATEGORY: This is the name of the category which you want
 *           PDB2ADR to check for. The category has to be spelled
 *           *exactly* as it is spelled in the phone book, otherwise
 *           PDB2ADR will not recognize it (care about upper- and 
 *           lower case letters etc.). PDB2ADR checks only if the
 *           string given in CATEGORY is equal or A PART OF the category
 *           field of the input file. So if you have for example two
 *           different cagegories "my family" and "my wife's familiy",
 *           and you give "family" as the parameter CATEGORY, PDB2ADR 
 *           will process all entries from these TWO categories.
 *           Or if you have entries matching the categories "business"
 *           and "private", PDB2ADR will process them if you have either
 *           "business" or "private" given as CATEGORY.
 *           Set CATEGORY to "0" if you set C to "0" and don't want 
 *           to use the category check.
 *
 *           So you can even let PDB2ADR create your distribution lists, 
 *           controlled by the category entries in your phone book 
 *           application.
 *
 *           If you prefer two seperate ADR files for SMS and for email
 *           (i.e. one containing only email addresses and on containing 
 *           only phone numbers), execute PDB2ADR twice. Once with
 *           P set to 0 (creates email address list) and once with 
 *           E set to 0 (creates the SMS phone number list).
 *           In POST.CFG, use a section like this to choose between 
 *           the different address books when composing a message:
 *
 *           [ADDRESSES]
 *           &Email=email.adr
 *           &SMS=sms.adr
 *           
 *
 */
 
#define VERSION "2.2"
#define YEAR "2003"
#define TOKLENGTH 80    // maximal length for a token, such as email, phone no, name of phone.cdf
#define LINELENGTH 1024 // maximal length of a line in cdf file
#define SMSNUMLEN 24    // maximal length of SMS phone numbers

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


FILE *fpi;
int NC, EC, PC, CC;
int linecount=0;
char testchar;
char line[LINELENGTH];
char name[TOKLENGTH];
char email[TOKLENGTH];
char phone[TOKLENGTH];
char category[TOKLENGTH];
char *dup, *dup2;

void hello(char *filename, int NC, int EC, int PC, int CC, char *cat);
void usage (void);
char *extr_cdf_col(char *cdfline, int c);
void makeemail(int namecol, int emailcol);
void makephone(int namecol, int phonecol);
void makedistrib(int namecol, int catcol, int emailcol, char *catcomp);

/* MAIN ============================================== */

int main(int argc, char *argv[])
{
  if (argc != 7) usage();

  NC = atoi(argv[2]);
  EC = atoi(argv[3]);
  PC = atoi(argv[4]);
  CC = atoi(argv[5]);

  hello(argv[1], NC, EC, PC, CC, argv[6]);
  
  if ((fpi=fopen(argv[1],"r"))!=NULL)
  {
    if ((CC == 0) && ((PC != 0) || (EC != 0)))
    {
      fprintf(stderr,"Category check DISABLED. Creating post.adr compatible output.\n");
      if (PC != 0)
      {
        printf("========= CTRL-ENTER TO CHOOSE NUMBER FOR SMS! =========\n");
        makephone(NC, PC);
        if ((PC != 0) && (EC != 0)) 
        { 
          printf("\n========= END PHONE NUMBERS, START EMAIL ADDRESSES ===========\n\n");
          rewind(fpi);
        }
      }
      if (EC != 0) makeemail(NC, EC);
    }

    if (CC != 0)
    {
      fprintf(stderr,"Category check ENABLED, creating distribution list compatible output!\n");
      makedistrib(NC, CC, EC, argv[6]);
    }


    
    if ((PC == 0) && (EC == 0))
    {
      fprintf(stderr,"PDB2ADR:\a\nNothing to do, no column for email or phone number given!\n\n");
    }

    fclose(fpi);
  }
  else 
  {
    fprintf(stderr,"\n\aError opening input file \"%s\"!\nAbnormal program termination.\n\n",argv[1]);
    return(1);
  }
  return(0);
}



/* HELLO ============================================= */

void hello(char *filename, int NC, int EC, int PC, int CC, char *cat)
{
  if (filename == NULL)
  {
    filename="NO INPUT FILE NAME GIVEN!";
  }
  fprintf(stderr,"\n*** This is PDB2ADR, Version %s, (c) %s Daniel Hertrich\n",VERSION,YEAR);
  fprintf(stderr,"*** Input CDF file: %s\n",filename);
  fprintf(stderr,"*** Columns used for: NAME: %i, EMAIL: %i, PHONE NO: %i\n", NC, EC, PC);
  fprintf(stderr,"*** Category check options: column %i, category %s\n", CC, cat);
  fprintf(stderr,"*** Please wait while processing input file.....\n\n");
}

/* USAGE =========================================== */

void usage (void)
{
  printf("\n*** This is PDB2ADR, Version %s, (c) %s Daniel Hertrich\n",VERSION,YEAR);
  printf("\a\n\nError!\nCall of PDB2ADR: \nPDB2ADR <input CDF file> <N> <E> <P> <C> <CATEGORY> > <output ADR file>\n\n");
  printf("N: Name col, E: Email col, P: Phone col, C: Category col.\n");
  printf("See pdb2adr.c source file for details.\n");
  printf("Are you sure you called PDB2ADR with a CDF file as input? PDB2ADR doesn't\n");
  printf("work with original PDB files, you have to use GDBIO first! See .C file!\n\n");
  exit(1);
}


/* EXTR_CDF_COL ===================================== */
/* give parameters:
      1. copy of a line of a cdf file (made by gdbio)
      2. number of column you want to extract
   get back:
      pointer to string containing string of given column
*/

char *extr_cdf_col(char *cdfline, int c)
{
  static char token[TOKLENGTH];
  char *loc;
  char *temptok = malloc(LINELENGTH*sizeof(char));
  int i = 0;

  temptok[0]=0;
    
  for (i=0; i<c; i++)
  {
    if (i>0) cdfline=loc+3;
    loc=strstr(cdfline,"\",\"");
    if (loc != NULL)
    { 
      strncpy(temptok,cdfline,loc-cdfline);
      temptok[loc-cdfline]='\0';
    }
    else
    {
      fprintf(stderr,"ERROR! Contact Author! loc is NULL!, c is %i\nCDFLINE is: %s\nLINECOUNT: %i\n\a\a",c,cdfline,linecount);
      exit(1);
    }
  }
  strcpy(token,temptok);
  free(temptok);
  return(token);
}


/* MAKEEMAIL ========================================= */

void makeemail(int namecol, int emailcol)
{
  while((testchar=fgetc(fpi))!= EOF)
  {
    linecount++;
    fgets(line, LINELENGTH, fpi);
    dup=strdup(line);
    strcpy(name,extr_cdf_col(dup,namecol));
    strcpy(email,extr_cdf_col(line,emailcol));
    free(dup);
    if (strlen(email) > 0) 
    {
      printf("%s <%s>\n",name,email);
    }
  }
}



/* MAKEPHONE ========================================= */

void makephone(int namecol, int phonecol)
{
  int i;
  
  while((testchar=fgetc(fpi))!= EOF)
  {
    fgets(line, LINELENGTH, fpi);
    dup=strdup(line);
    strcpy(name,extr_cdf_col(dup,namecol));
    strcpy(phone,extr_cdf_col(line,phonecol));
    free(dup);
    if (strlen(phone) > 0) 
    {
      printf("%s",phone);
      for (i=strlen(phone);i<SMSNUMLEN;i++) printf(" ");
      printf("| %s \n",name);
    }
  }

  
}

/* MAKEDISTRIIB ======================================= */

void makedistrib(int namecol, int catcol, int emailcol, char *catcomp)
{
  while((testchar=fgetc(fpi))!= EOF)
  {
    linecount++;
    fgets(line, LINELENGTH, fpi);
    dup=strdup(line);
    dup2=strdup(line);
    strcpy(category,extr_cdf_col(dup,catcol));
    strcpy(email,extr_cdf_col(line,emailcol));
    strcpy(name,extr_cdf_col(line,namecol));
    free(dup);
    free(dup2);
    if ((strlen(email) > 0) && (strstr(category, catcomp) != NULL))
    {
      printf("%s <%s>\n",name,email);
    }
  }
  
}

